#include <stddef.h>

struct check_item {
	const char *str;
	int expected;
};

static const char * const dev_patterns[] = {
	"loop/[0-9]+",
	"hd[a-d][0-5]+",
	NULL
};

static const char * const nonprint_patterns[] = {
	"foo\x80" "bar",
	"foo\xc2" "b",
	"\x80",
	NULL
};

static const struct check_item nonprint[] = {
	{ "foo\x2e" "bar", 0 },
	{ "foo\x80" "bar", 3 },
	{ "foo\xc2" "b", 2 },
	{ "\x80", 3 },
	{ NULL, 0 }
};

static const char * const random_patterns[] = {
	"(((a?)(([Ub]*)|z))((([qr]|X)+)([Qn]*)))+",
	"[HZejtuw]*",
	"((B|s)*)|(((([Fv]l)(N+))(([el]|C)(tJ)))?)",
	"((([Ma]?)|(t*))*)|((([cm]E)|(M?))|(([BE][EV])|([Qj][Mh])))",
	"(((([bw]*)|([IO]*))((zK)*))|(((pU)|(i|q))|((z?)|([HL]?))))*",
	"((([Pt]?)|[Tr])?)((Hq)*)",
	"[HOXcfgikosvwxz]",
	"[BCEFGHNPTUWfjlprsy]",
	"((((aD)*)|([Xo]+))+)(([HKn](([Eq]|[JQ])(I*)))*)",
	"([LNWYeghv]|e)*",
	"(((y(L*))*)|((([EP]+)(W+))*))*",
	"U*",
	"((((R+)(W|[Qr]))|([py]+))+)([LM]*)",
	"(([DOjx](D(b?)))|([Ke]*))*",
	"((([ls](c|[FT]))*)([JS]*))*",
	"((l?)|(([Gz]+)|(D*)))*",
	"[ABgjn]",
	"(((q|[dg])?)|([Uk]*))((([Fl]?)|([Ry]+))|(([IR]|c)|(T?)))",
	"((([an]|P)|[Jw])((a*)|(m*)))*",
	"((((R[ht])(h+))?)|(([pz](n?))+))+",
	"(((([Dc]b)([Sp][Ii]))|((k|F)*))|[Uiovz])*",
	"[Res]*",
	"[Zl]|a",
	"^[ANZdf]$",
	"[En]|(((Q+)(U+))([pt]*))",
	"[ADEIMQUWXZhklrsvz]",
	"(((S(y*))*)|(j*))*",
	"n*",
	"[NUau]*",
	"((((Z*)(D|[Nd]))|(([np]|B)+))|(([Xy][Fi])*))+",
	"((([EZ]?)|(d[HR]))*)((([Hg]|q)(P+))*)",
	"q",
	"((m*)|(p|B))|((((x?)|(t+))(([Sb][PX])(O|[HM])))+)",
	"((((A*)(z[RS]))*)|(((z+)(Q*))+))*",
	"(((M*)([Uu]*))+)|[Uk]",
	"[imv]",
	"[GLSchtw](([Yw]((F[Dd])|([Tw]+)))?)",
	"([MOZj]*)(S|[Wknr])",
	"((G|q)*)[BHKN]",
	"((((NW)|([Ao]?))|((l|[UV])+))+)|((i|(z*))*)",
	"((((Z+)|([IR]?))|(L*))|([JKQ]+))+",
	"([Bdin](S*))+",
	"[HLNSTp]*",
	"(((J*)([Bq]|[Yu]))*)|([Kv]*)",
	"(((([BJ]|[Zy])(wI))*)(y*))+",
	"(((hF)+)|(H*))*",
	"((([QU][Pj])([GQ]?))+)|[PWo]",
	"(((([cq][BX])?)|((f[DI])*))*)(([GM]*)[SVYr])",
	"(([Zt]*)|((qx)|(([BV]+)(f?))))*",
	"[ILWYhsx]*",
	"(([Uy]*)|[sv])|([NSc]*)",
	"((c*)|([JUfhy]?))+",
	"(((q*)([So]*))(((g[jq])(j?))+))*",
	"((b+)|(((T+)([fw]T))?))*",
	"((([DS]?)|([Th]|u))(Q*))*",
	"[FKLX]|((([fw](L?))(([gq]*)|(O?)))?)",
	"((([HZ]+)u)*)|[APWijn]",
	"(e*)|(((v?)|((J+)(Hb)))?)",
	"(e|((w+)f))*",
	"[BEHKPQVdelnqy]",
	"((((B|N)(s*))|[Rr])(((g?)|([rv]+))+))+",
	"(((s*)|(K*))([AP]G))*",
	"[CELTp]",
	"(([Fq]?)|([Al]+))*",
	"((((r?)|(y[jx]))|([mp]*))+)|((B(S*))*)",
	"((([Eq]+)|(Y[ds]))|(x|(i|[Ku])))[IJNrvy]",
	"((([NO]*)[Ix])+)([Jenq]+)",
	"(((([HP]*)(j|y))*)[Ylqvy])*",
	"[PTv]+",
	"[AINSZhpx]|([EOYZ]*)",
	"([ABCFQv]*)((([Zx]|h)+)|([ej]*))",
	"((([pr]*)|(([Dq]|p)|(H?)))?)([NRUXmoq]*)",
	"(([er]*)|([mx]*))(((nV)([am]?))+)",
	"[BHPRlpu]",
	"(((([Ah]|[tx])|(e|[uy]))?)((([fl]+)([Vz]|v))*))*",
	"[AGdm]",
	"(((K*)^(O*)$)|(B?))*",
	"((([Ks]|[Ka])*)|([FSTab]?))?",
	"(([kw]+)[ei])(([Hy]*)(([Mc]*)|(G|f)))",
	"((((e*)|(Zf))|(R|[nq]))((([Jz]v)([Rj]+))+))*",
	"(((a?)|(e?))(([Uc]*)(S+)))*",
	"((((E+)([MZ]?))+)|(((s|[Az])|z)*))?",
	"((((i[MO])*)|((LH)*))|(((BA)|([AI]+))|[Ug]))*",
	"[EGHILcho]*",
	"(((Z[vw])?)((z|g)+))(((H|U)([iv]Q))|([qw]?))",
	"(([ehmr]|((L[Uw])*))+)((a+)I)",
	"[EKNSWYagj](((v|[TX])|([Uk]+))*)",
	"(((R[Mo])|(O*))|([Fm]|([qw]*)))((m*)|((S|[Ki])?))",
	"((((kP)|c)?)((([do]+)|([Gi]?))*))*",
	"((^(B|W)$|([Ww]+))([no]*))|((([iv]?)|(M*))|((x|L)?))",
	"[AEGPRSbcfhsy]",
	"[Wbcf]|((([MO]?)|([NT]|m))(([Oo]?)([Wg]*)))",
	"(((YZ)*)[PQVei])*",
	"[GJKYt][AEGWdegmnt]",
	"^[CDEGJKNUVYZagkv]$",
	"([DPWbx]*)|(((q|B)|(P|u))((M[Bq])*))",
	"[FHIJRTVYZdiorsuvz]*",
	"([MWoqvz]*)|^(l*)",
	"(((I|[Rx])*)((X[Mf])([Xa]L)))([Ha]|([HY]*))",
	"(((l|[Sd])*)((([Ix]+)|([XY]?))(Z*)))+",
	NULL
};

static const struct check_item devices[] = {
	{ "/dev", 0 },
	{ "/dev/.devfsd", 0 },
	{ "/dev/cpu", 0 },
	{ "/dev/cpu/mtrr", 0 },
	{ "/dev/netlink", 0 },
	{ "/dev/netlink/route", 0 },
	{ "/dev/netlink/skip", 0 },
	{ "/dev/netlink/USERSOCK", 0 },
	{ "/dev/netlink/fwmonitor", 0 },
	{ "/dev/netlink/ARPD", 0 },
	{ "/dev/netlink/ROUTE6", 0 },
	{ "/dev/netlink/IP6_FW", 0 },
	{ "/dev/netlink/tap0", 0 },
	{ "/dev/netlink/tap1", 0 },
	{ "/dev/netlink/tap2", 0 },
	{ "/dev/netlink/tap3", 0 },
	{ "/dev/netlink/tap4", 0 },
	{ "/dev/netlink/tap5", 0 },
	{ "/dev/netlink/tap6", 0 },
	{ "/dev/netlink/tap7", 0 },
	{ "/dev/netlink/tap8", 0 },
	{ "/dev/netlink/tap9", 0 },
	{ "/dev/netlink/tap10", 0 },
	{ "/dev/netlink/tap11", 0 },
	{ "/dev/netlink/tap12", 0 },
	{ "/dev/netlink/tap13", 0 },
	{ "/dev/netlink/tap14", 0 },
	{ "/dev/netlink/tap15", 0 },
	{ "/dev/shm", 0 },
	{ "/dev/mem", 0 },
	{ "/dev/kmem", 0 },
	{ "/dev/null", 0 },
	{ "/dev/port", 0 },
	{ "/dev/zero", 0 },
	{ "/dev/full", 0 },
	{ "/dev/random", 0 },
	{ "/dev/urandom", 0 },
	{ "/dev/tty", 0 },
	{ "/dev/console", 0 },
	{ "/dev/vc", 0 },
	{ "/dev/vc/1", 0 },
	{ "/dev/vc/2", 0 },
	{ "/dev/vc/3", 0 },
	{ "/dev/vc/4", 0 },
	{ "/dev/vc/5", 0 },
	{ "/dev/vc/6", 0 },
	{ "/dev/vc/7", 0 },
	{ "/dev/vc/8", 0 },
	{ "/dev/vc/9", 0 },
	{ "/dev/vc/10", 0 },
	{ "/dev/vc/11", 0 },
	{ "/dev/vc/12", 0 },
	{ "/dev/vc/13", 0 },
	{ "/dev/vc/14", 0 },
	{ "/dev/vc/15", 0 },
	{ "/dev/vc/16", 0 },
	{ "/dev/vc/17", 0 },
	{ "/dev/vc/18", 0 },
	{ "/dev/vc/19", 0 },
	{ "/dev/vc/20", 0 },
	{ "/dev/vc/21", 0 },
	{ "/dev/vc/22", 0 },
	{ "/dev/vc/23", 0 },
	{ "/dev/vc/24", 0 },
	{ "/dev/vc/25", 0 },
	{ "/dev/vc/26", 0 },
	{ "/dev/vc/27", 0 },
	{ "/dev/vc/28", 0 },
	{ "/dev/vc/29", 0 },
	{ "/dev/vc/30", 0 },
	{ "/dev/vc/31", 0 },
	{ "/dev/vc/32", 0 },
	{ "/dev/vc/33", 0 },
	{ "/dev/vc/34", 0 },
	{ "/dev/vc/35", 0 },
	{ "/dev/vc/36", 0 },
	{ "/dev/vc/37", 0 },
	{ "/dev/vc/38", 0 },
	{ "/dev/vc/39", 0 },
	{ "/dev/vc/40", 0 },
	{ "/dev/vc/41", 0 },
	{ "/dev/vc/42", 0 },
	{ "/dev/vc/43", 0 },
	{ "/dev/vc/44", 0 },
	{ "/dev/vc/45", 0 },
	{ "/dev/vc/46", 0 },
	{ "/dev/vc/47", 0 },
	{ "/dev/vc/48", 0 },
	{ "/dev/vc/49", 0 },
	{ "/dev/vc/50", 0 },
	{ "/dev/vc/51", 0 },
	{ "/dev/vc/52", 0 },
	{ "/dev/vc/53", 0 },
	{ "/dev/vc/54", 0 },
	{ "/dev/vc/55", 0 },
	{ "/dev/vc/56", 0 },
	{ "/dev/vc/57", 0 },
	{ "/dev/vc/58", 0 },
	{ "/dev/vc/59", 0 },
	{ "/dev/vc/60", 0 },
	{ "/dev/vc/61", 0 },
	{ "/dev/vc/62", 0 },
	{ "/dev/vc/63", 0 },
	{ "/dev/vc/0", 0 },
	{ "/dev/ptmx", 0 },
	{ "/dev/misc", 0 },
	{ "/dev/misc/psaux", 0 },
	{ "/dev/pty", 0 },
	{ "/dev/pty/m0", 0 },
	{ "/dev/pty/m1", 0 },
	{ "/dev/pty/m2", 0 },
	{ "/dev/pty/m3", 0 },
	{ "/dev/pty/m4", 0 },
	{ "/dev/pty/m5", 0 },
	{ "/dev/pty/m6", 0 },
	{ "/dev/pty/m7", 0 },
	{ "/dev/pty/m8", 0 },
	{ "/dev/pty/m9", 0 },
	{ "/dev/pty/m10", 0 },
	{ "/dev/pty/m11", 0 },
	{ "/dev/pty/m12", 0 },
	{ "/dev/pty/m13", 0 },
	{ "/dev/pty/m14", 0 },
	{ "/dev/pty/m15", 0 },
	{ "/dev/pty/m16", 0 },
	{ "/dev/pty/m17", 0 },
	{ "/dev/pty/m18", 0 },
	{ "/dev/pty/m19", 0 },
	{ "/dev/pty/m20", 0 },
	{ "/dev/pty/m21", 0 },
	{ "/dev/pty/m22", 0 },
	{ "/dev/pty/m23", 0 },
	{ "/dev/pty/m24", 0 },
	{ "/dev/pty/m25", 0 },
	{ "/dev/pty/m26", 0 },
	{ "/dev/pty/m27", 0 },
	{ "/dev/pty/m28", 0 },
	{ "/dev/pty/m29", 0 },
	{ "/dev/pty/m30", 0 },
	{ "/dev/pty/m31", 0 },
	{ "/dev/pty/m32", 0 },
	{ "/dev/pty/m33", 0 },
	{ "/dev/pty/m34", 0 },
	{ "/dev/pty/m35", 0 },
	{ "/dev/pty/m36", 0 },
	{ "/dev/pty/m37", 0 },
	{ "/dev/pty/m38", 0 },
	{ "/dev/pty/m39", 0 },
	{ "/dev/pty/m40", 0 },
	{ "/dev/pty/m41", 0 },
	{ "/dev/pty/m42", 0 },
	{ "/dev/pty/m43", 0 },
	{ "/dev/pty/m44", 0 },
	{ "/dev/pty/m45", 0 },
	{ "/dev/pty/m46", 0 },
	{ "/dev/pty/m47", 0 },
	{ "/dev/pty/m48", 0 },
	{ "/dev/pty/m49", 0 },
	{ "/dev/pty/m50", 0 },
	{ "/dev/pty/m51", 0 },
	{ "/dev/pty/m52", 0 },
	{ "/dev/pty/m53", 0 },
	{ "/dev/pty/m54", 0 },
	{ "/dev/pty/m55", 0 },
	{ "/dev/pty/m56", 0 },
	{ "/dev/pty/m57", 0 },
	{ "/dev/pty/m58", 0 },
	{ "/dev/pty/m59", 0 },
	{ "/dev/pty/m60", 0 },
	{ "/dev/pty/m61", 0 },
	{ "/dev/pty/m62", 0 },
	{ "/dev/pty/m63", 0 },
	{ "/dev/pty/m64", 0 },
	{ "/dev/pty/m65", 0 },
	{ "/dev/pty/m66", 0 },
	{ "/dev/pty/m67", 0 },
	{ "/dev/pty/m68", 0 },
	{ "/dev/pty/m69", 0 },
	{ "/dev/pty/m70", 0 },
	{ "/dev/pty/m71", 0 },
	{ "/dev/pty/m72", 0 },
	{ "/dev/pty/m73", 0 },
	{ "/dev/pty/m74", 0 },
	{ "/dev/pty/m75", 0 },
	{ "/dev/pty/m76", 0 },
	{ "/dev/pty/m77", 0 },
	{ "/dev/pty/m78", 0 },
	{ "/dev/pty/m79", 0 },
	{ "/dev/pty/m80", 0 },
	{ "/dev/pty/m81", 0 },
	{ "/dev/pty/m82", 0 },
	{ "/dev/pty/m83", 0 },
	{ "/dev/pty/m84", 0 },
	{ "/dev/pty/m85", 0 },
	{ "/dev/pty/m86", 0 },
	{ "/dev/pty/m87", 0 },
	{ "/dev/pty/m88", 0 },
	{ "/dev/pty/m89", 0 },
	{ "/dev/pty/m90", 0 },
	{ "/dev/pty/m91", 0 },
	{ "/dev/pty/m92", 0 },
	{ "/dev/pty/m93", 0 },
	{ "/dev/pty/m94", 0 },
	{ "/dev/pty/m95", 0 },
	{ "/dev/pty/m96", 0 },
	{ "/dev/pty/m97", 0 },
	{ "/dev/pty/m98", 0 },
	{ "/dev/pty/m99", 0 },
	{ "/dev/pty/m100", 0 },
	{ "/dev/pty/m101", 0 },
	{ "/dev/pty/m102", 0 },
	{ "/dev/pty/m103", 0 },
	{ "/dev/pty/m104", 0 },
	{ "/dev/pty/m105", 0 },
	{ "/dev/pty/m106", 0 },
	{ "/dev/pty/m107", 0 },
	{ "/dev/pty/m108", 0 },
	{ "/dev/pty/m109", 0 },
	{ "/dev/pty/m110", 0 },
	{ "/dev/pty/m111", 0 },
	{ "/dev/pty/m112", 0 },
	{ "/dev/pty/m113", 0 },
	{ "/dev/pty/m114", 0 },
	{ "/dev/pty/m115", 0 },
	{ "/dev/pty/m116", 0 },
	{ "/dev/pty/m117", 0 },
	{ "/dev/pty/m118", 0 },
	{ "/dev/pty/m119", 0 },
	{ "/dev/pty/m120", 0 },
	{ "/dev/pty/m121", 0 },
	{ "/dev/pty/m122", 0 },
	{ "/dev/pty/m123", 0 },
	{ "/dev/pty/m124", 0 },
	{ "/dev/pty/m125", 0 },
	{ "/dev/pty/m126", 0 },
	{ "/dev/pty/m127", 0 },
	{ "/dev/pty/m128", 0 },
	{ "/dev/pty/m129", 0 },
	{ "/dev/pty/m130", 0 },
	{ "/dev/pty/m131", 0 },
	{ "/dev/pty/m132", 0 },
	{ "/dev/pty/m133", 0 },
	{ "/dev/pty/m134", 0 },
	{ "/dev/pty/m135", 0 },
	{ "/dev/pty/m136", 0 },
	{ "/dev/pty/m137", 0 },
	{ "/dev/pty/m138", 0 },
	{ "/dev/pty/m139", 0 },
	{ "/dev/pty/m140", 0 },
	{ "/dev/pty/m141", 0 },
	{ "/dev/pty/m142", 0 },
	{ "/dev/pty/m143", 0 },
	{ "/dev/pty/m144", 0 },
	{ "/dev/pty/m145", 0 },
	{ "/dev/pty/m146", 0 },
	{ "/dev/pty/m147", 0 },
	{ "/dev/pty/m148", 0 },
	{ "/dev/pty/m149", 0 },
	{ "/dev/pty/m150", 0 },
	{ "/dev/pty/m151", 0 },
	{ "/dev/pty/m152", 0 },
	{ "/dev/pty/m153", 0 },
	{ "/dev/pty/m154", 0 },
	{ "/dev/pty/m155", 0 },
	{ "/dev/pty/m156", 0 },
	{ "/dev/pty/m157", 0 },
	{ "/dev/pty/m158", 0 },
	{ "/dev/pty/m159", 0 },
	{ "/dev/pty/m160", 0 },
	{ "/dev/pty/m161", 0 },
	{ "/dev/pty/m162", 0 },
	{ "/dev/pty/m163", 0 },
	{ "/dev/pty/m164", 0 },
	{ "/dev/pty/m165", 0 },
	{ "/dev/pty/m166", 0 },
	{ "/dev/pty/m167", 0 },
	{ "/dev/pty/m168", 0 },
	{ "/dev/pty/m169", 0 },
	{ "/dev/pty/m170", 0 },
	{ "/dev/pty/m171", 0 },
	{ "/dev/pty/m172", 0 },
	{ "/dev/pty/m173", 0 },
	{ "/dev/pty/m174", 0 },
	{ "/dev/pty/m175", 0 },
	{ "/dev/pty/m176", 0 },
	{ "/dev/pty/m177", 0 },
	{ "/dev/pty/m178", 0 },
	{ "/dev/pty/m179", 0 },
	{ "/dev/pty/m180", 0 },
	{ "/dev/pty/m181", 0 },
	{ "/dev/pty/m182", 0 },
	{ "/dev/pty/m183", 0 },
	{ "/dev/pty/m184", 0 },
	{ "/dev/pty/m185", 0 },
	{ "/dev/pty/m186", 0 },
	{ "/dev/pty/m187", 0 },
	{ "/dev/pty/m188", 0 },
	{ "/dev/pty/m189", 0 },
	{ "/dev/pty/m190", 0 },
	{ "/dev/pty/m191", 0 },
	{ "/dev/pty/m192", 0 },
	{ "/dev/pty/m193", 0 },
	{ "/dev/pty/m194", 0 },
	{ "/dev/pty/m195", 0 },
	{ "/dev/pty/m196", 0 },
	{ "/dev/pty/m197", 0 },
	{ "/dev/pty/m198", 0 },
	{ "/dev/pty/m199", 0 },
	{ "/dev/pty/m200", 0 },
	{ "/dev/pty/m201", 0 },
	{ "/dev/pty/m202", 0 },
	{ "/dev/pty/m203", 0 },
	{ "/dev/pty/m204", 0 },
	{ "/dev/pty/m205", 0 },
	{ "/dev/pty/m206", 0 },
	{ "/dev/pty/m207", 0 },
	{ "/dev/pty/m208", 0 },
	{ "/dev/pty/m209", 0 },
	{ "/dev/pty/m210", 0 },
	{ "/dev/pty/m211", 0 },
	{ "/dev/pty/m212", 0 },
	{ "/dev/pty/m213", 0 },
	{ "/dev/pty/m214", 0 },
	{ "/dev/pty/m215", 0 },
	{ "/dev/pty/m216", 0 },
	{ "/dev/pty/m217", 0 },
	{ "/dev/pty/m218", 0 },
	{ "/dev/pty/m219", 0 },
	{ "/dev/pty/m220", 0 },
	{ "/dev/pty/m221", 0 },
	{ "/dev/pty/m222", 0 },
	{ "/dev/pty/m223", 0 },
	{ "/dev/pty/m224", 0 },
	{ "/dev/pty/m225", 0 },
	{ "/dev/pty/m226", 0 },
	{ "/dev/pty/m227", 0 },
	{ "/dev/pty/m228", 0 },
	{ "/dev/pty/m229", 0 },
	{ "/dev/pty/m230", 0 },
	{ "/dev/pty/m231", 0 },
	{ "/dev/pty/m232", 0 },
	{ "/dev/pty/m233", 0 },
	{ "/dev/pty/m234", 0 },
	{ "/dev/pty/m235", 0 },
	{ "/dev/pty/m236", 0 },
	{ "/dev/pty/m237", 0 },
	{ "/dev/pty/m238", 0 },
	{ "/dev/pty/m239", 0 },
	{ "/dev/pty/m240", 0 },
	{ "/dev/pty/m241", 0 },
	{ "/dev/pty/m242", 0 },
	{ "/dev/pty/m243", 0 },
	{ "/dev/pty/m244", 0 },
	{ "/dev/pty/m245", 0 },
	{ "/dev/pty/m246", 0 },
	{ "/dev/pty/m247", 0 },
	{ "/dev/pty/m248", 0 },
	{ "/dev/pty/m249", 0 },
	{ "/dev/pty/m250", 0 },
	{ "/dev/pty/m251", 0 },
	{ "/dev/pty/m252", 0 },
	{ "/dev/pty/m253", 0 },
	{ "/dev/pty/m254", 0 },
	{ "/dev/pty/m255", 0 },
	{ "/dev/pts", 0 },
	{ "/dev/pts/0", 0 },
	{ "/dev/pts/1", 0 },
	{ "/dev/pts/2", 0 },
	{ "/dev/pts/3", 0 },
	{ "/dev/pts/4", 0 },
	{ "/dev/pts/5", 0 },
	{ "/dev/pts/6", 0 },
	{ "/dev/pts/7", 0 },
	{ "/dev/vcc", 0 },
	{ "/dev/vcc/0", 0 },
	{ "/dev/vcc/a", 0 },
	{ "/dev/vcc/1", 0 },
	{ "/dev/vcc/a1", 0 },
	{ "/dev/vcc/2", 0 },
	{ "/dev/vcc/a2", 0 },
	{ "/dev/vcc/3", 0 },
	{ "/dev/vcc/a3", 0 },
	{ "/dev/vcc/5", 0 },
	{ "/dev/vcc/a5", 0 },
	{ "/dev/vcc/4", 0 },
	{ "/dev/vcc/a4", 0 },
	{ "/dev/vcc/6", 0 },
	{ "/dev/vcc/a6", 0 },
	{ "/dev/vcc/7", 0 },
	{ "/dev/vcc/a7", 0 },
	{ "/dev/tts", 0 },
	{ "/dev/tts/0", 0 },
	{ "/dev/cua", 0 },
	{ "/dev/cua/0", 0 },
	{ "/dev/ide", 0 },
	{ "/dev/ide/host0", 0 },
	{ "/dev/ide/host0/bus0", 0 },
	{ "/dev/ide/host0/bus0/target0", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/disc", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/part1", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/part2", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/part3", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/part4", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/part5", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/part6", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/part7", 0 },
	{ "/dev/ide/host0/bus0/target0/lun0/part8", 0 },
	{ "/dev/ide/host0/bus0/target1", 0 },
	{ "/dev/ide/host0/bus0/target1/lun0", 0 },
	{ "/dev/ide/host0/bus0/target1/lun0/disc", 0 },
	{ "/dev/ide/host0/bus0/target1/lun0/part1", 0 },
	{ "/dev/ide/host0/bus1", 0 },
	{ "/dev/ide/host0/bus1/target0", 0 },
	{ "/dev/ide/host0/bus1/target0/lun0", 0 },
	{ "/dev/ide/host0/bus1/target0/lun0/disc", 0 },
	{ "/dev/ide/host0/bus1/target0/lun0/part1", 0 },
	{ "/dev/ide/host0/bus1/target1", 0 },
	{ "/dev/ide/host0/bus1/target1/lun0", 0 },
	{ "/dev/discs", 0 },
	{ "/dev/discs/disc0", 0 },
	{ "/dev/discs/disc1", 0 },
	{ "/dev/discs/disc2", 0 },
	{ "/dev/floppy", 0 },
	{ "/dev/floppy/0u1440", 0 },
	{ "/dev/floppy/0u1680", 0 },
	{ "/dev/floppy/0u1722", 0 },
	{ "/dev/floppy/0u1743", 0 },
	{ "/dev/floppy/0u1760", 0 },
	{ "/dev/floppy/0u1920", 0 },
	{ "/dev/floppy/0u1840", 0 },
	{ "/dev/floppy/0u1600", 0 },
	{ "/dev/floppy/0u360", 0 },
	{ "/dev/floppy/0u720", 0 },
	{ "/dev/floppy/0u820", 0 },
	{ "/dev/floppy/0u830", 0 },
	{ "/dev/floppy/0u1040", 0 },
	{ "/dev/floppy/0u1120", 0 },
	{ "/dev/floppy/0u800", 0 },
	{ "/dev/floppy/0", 0 },
	{ "/dev/loop", 0 },
	{ "/dev/loop/0", 1 },
	{ "/dev/loop/1", 1 },
	{ "/dev/loop/2", 1 },
	{ "/dev/loop/3", 1 },
	{ "/dev/loop/4", 1 },
	{ "/dev/loop/5", 1 },
	{ "/dev/loop/6", 1 },
	{ "/dev/loop/7", 1 },
	{ "/dev/cdroms", 0 },
	{ "/dev/sound", 0 },
	{ "/dev/sound/dsp", 0 },
	{ "/dev/sound/dsp1", 0 },
	{ "/dev/sound/mixer", 0 },
	{ "/dev/sound/midi", 0 },
	{ "/dev/usb", 0 },
	{ "/dev/root", 0 },
	{ "/dev/initctl", 0 },
	{ "/dev/xconsole", 0 },
	{ "/dev/fd", 0 },
	{ "/dev/stdin", 0 },
	{ "/dev/stdout", 0 },
	{ "/dev/stderr", 0 },
	{ "/dev/route", 0 },
	{ "/dev/skip", 0 },
	{ "/dev/USERSOCK", 0 },
	{ "/dev/fwmonitor", 0 },
	{ "/dev/ARPD", 0 },
	{ "/dev/ROUTE6", 0 },
	{ "/dev/IP6_FW", 0 },
	{ "/dev/tap0", 0 },
	{ "/dev/tap1", 0 },
	{ "/dev/tap2", 0 },
	{ "/dev/tap3", 0 },
	{ "/dev/tap4", 0 },
	{ "/dev/tap5", 0 },
	{ "/dev/tap6", 0 },
	{ "/dev/tap7", 0 },
	{ "/dev/tap8", 0 },
	{ "/dev/tap9", 0 },
	{ "/dev/tap10", 0 },
	{ "/dev/tap11", 0 },
	{ "/dev/tap12", 0 },
	{ "/dev/tap13", 0 },
	{ "/dev/tap14", 0 },
	{ "/dev/tap15", 0 },
	{ "/dev/tty1", 0 },
	{ "/dev/tty2", 0 },
	{ "/dev/tty3", 0 },
	{ "/dev/tty4", 0 },
	{ "/dev/tty5", 0 },
	{ "/dev/tty6", 0 },
	{ "/dev/tty7", 0 },
	{ "/dev/tty8", 0 },
	{ "/dev/tty9", 0 },
	{ "/dev/tty10", 0 },
	{ "/dev/tty11", 0 },
	{ "/dev/tty12", 0 },
	{ "/dev/tty13", 0 },
	{ "/dev/tty14", 0 },
	{ "/dev/tty15", 0 },
	{ "/dev/tty16", 0 },
	{ "/dev/tty17", 0 },
	{ "/dev/tty18", 0 },
	{ "/dev/tty19", 0 },
	{ "/dev/tty20", 0 },
	{ "/dev/tty21", 0 },
	{ "/dev/tty22", 0 },
	{ "/dev/tty23", 0 },
	{ "/dev/tty24", 0 },
	{ "/dev/tty25", 0 },
	{ "/dev/tty26", 0 },
	{ "/dev/tty27", 0 },
	{ "/dev/tty28", 0 },
	{ "/dev/tty29", 0 },
	{ "/dev/tty30", 0 },
	{ "/dev/tty31", 0 },
	{ "/dev/tty32", 0 },
	{ "/dev/tty33", 0 },
	{ "/dev/tty34", 0 },
	{ "/dev/tty35", 0 },
	{ "/dev/tty36", 0 },
	{ "/dev/tty37", 0 },
	{ "/dev/tty38", 0 },
	{ "/dev/tty39", 0 },
	{ "/dev/tty40", 0 },
	{ "/dev/tty41", 0 },
	{ "/dev/tty42", 0 },
	{ "/dev/tty43", 0 },
	{ "/dev/tty44", 0 },
	{ "/dev/tty45", 0 },
	{ "/dev/tty46", 0 },
	{ "/dev/tty47", 0 },
	{ "/dev/tty48", 0 },
	{ "/dev/tty49", 0 },
	{ "/dev/tty50", 0 },
	{ "/dev/tty51", 0 },
	{ "/dev/tty52", 0 },
	{ "/dev/tty53", 0 },
	{ "/dev/tty54", 0 },
	{ "/dev/tty55", 0 },
	{ "/dev/tty56", 0 },
	{ "/dev/tty57", 0 },
	{ "/dev/tty58", 0 },
	{ "/dev/tty59", 0 },
	{ "/dev/tty60", 0 },
	{ "/dev/tty61", 0 },
	{ "/dev/tty62", 0 },
	{ "/dev/tty63", 0 },
	{ "/dev/tty0", 0 },
	{ "/dev/psaux", 0 },
	{ "/dev/ptyp0", 0 },
	{ "/dev/ptyp1", 0 },
	{ "/dev/ptyp2", 0 },
	{ "/dev/ptyp3", 0 },
	{ "/dev/ptyp4", 0 },
	{ "/dev/ptyp5", 0 },
	{ "/dev/ptyp6", 0 },
	{ "/dev/ptyp7", 0 },
	{ "/dev/ptyp8", 0 },
	{ "/dev/ptyp9", 0 },
	{ "/dev/ptypa", 0 },
	{ "/dev/ptypb", 0 },
	{ "/dev/ptypc", 0 },
	{ "/dev/ptypd", 0 },
	{ "/dev/ptype", 0 },
	{ "/dev/ptypf", 0 },
	{ "/dev/ptyq0", 0 },
	{ "/dev/ptyq1", 0 },
	{ "/dev/ptyq2", 0 },
	{ "/dev/ptyq3", 0 },
	{ "/dev/ptyq4", 0 },
	{ "/dev/ptyq5", 0 },
	{ "/dev/ptyq6", 0 },
	{ "/dev/ptyq7", 0 },
	{ "/dev/ptyq8", 0 },
	{ "/dev/ptyq9", 0 },
	{ "/dev/ptyqa", 0 },
	{ "/dev/ptyqb", 0 },
	{ "/dev/ptyqc", 0 },
	{ "/dev/ptyqd", 0 },
	{ "/dev/ptyqe", 0 },
	{ "/dev/ptyqf", 0 },
	{ "/dev/ptyr0", 0 },
	{ "/dev/ptyr1", 0 },
	{ "/dev/ptyr2", 0 },
	{ "/dev/ptyr3", 0 },
	{ "/dev/ptyr4", 0 },
	{ "/dev/ptyr5", 0 },
	{ "/dev/ptyr6", 0 },
	{ "/dev/ptyr7", 0 },
	{ "/dev/ptyr8", 0 },
	{ "/dev/ptyr9", 0 },
	{ "/dev/ptyra", 0 },
	{ "/dev/ptyrb", 0 },
	{ "/dev/ptyrc", 0 },
	{ "/dev/ptyrd", 0 },
	{ "/dev/ptyre", 0 },
	{ "/dev/ptyrf", 0 },
	{ "/dev/ptys0", 0 },
	{ "/dev/ptys1", 0 },
	{ "/dev/ptys2", 0 },
	{ "/dev/ptys3", 0 },
	{ "/dev/ptys4", 0 },
	{ "/dev/ptys5", 0 },
	{ "/dev/ptys6", 0 },
	{ "/dev/ptys7", 0 },
	{ "/dev/ptys8", 0 },
	{ "/dev/ptys9", 0 },
	{ "/dev/ptysa", 0 },
	{ "/dev/ptysb", 0 },
	{ "/dev/ptysc", 0 },
	{ "/dev/ptysd", 0 },
	{ "/dev/ptyse", 0 },
	{ "/dev/ptysf", 0 },
	{ "/dev/ptyt0", 0 },
	{ "/dev/ptyt1", 0 },
	{ "/dev/ptyt2", 0 },
	{ "/dev/ptyt3", 0 },
	{ "/dev/ptyt4", 0 },
	{ "/dev/ptyt5", 0 },
	{ "/dev/ptyt6", 0 },
	{ "/dev/ptyt7", 0 },
	{ "/dev/ptyt8", 0 },
	{ "/dev/ptyt9", 0 },
	{ "/dev/ptyta", 0 },
	{ "/dev/ptytb", 0 },
	{ "/dev/ptytc", 0 },
	{ "/dev/ptytd", 0 },
	{ "/dev/ptyte", 0 },
	{ "/dev/ptytf", 0 },
	{ "/dev/ptyu0", 0 },
	{ "/dev/ptyu1", 0 },
	{ "/dev/ptyu2", 0 },
	{ "/dev/ptyu3", 0 },
	{ "/dev/ptyu4", 0 },
	{ "/dev/ptyu5", 0 },
	{ "/dev/ptyu6", 0 },
	{ "/dev/ptyu7", 0 },
	{ "/dev/ptyu8", 0 },
	{ "/dev/ptyu9", 0 },
	{ "/dev/ptyua", 0 },
	{ "/dev/ptyub", 0 },
	{ "/dev/ptyuc", 0 },
	{ "/dev/ptyud", 0 },
	{ "/dev/ptyue", 0 },
	{ "/dev/ptyuf", 0 },
	{ "/dev/ptyv0", 0 },
	{ "/dev/ptyv1", 0 },
	{ "/dev/ptyv2", 0 },
	{ "/dev/ptyv3", 0 },
	{ "/dev/ptyv4", 0 },
	{ "/dev/ptyv5", 0 },
	{ "/dev/ptyv6", 0 },
	{ "/dev/ptyv7", 0 },
	{ "/dev/ptyv8", 0 },
	{ "/dev/ptyv9", 0 },
	{ "/dev/ptyva", 0 },
	{ "/dev/ptyvb", 0 },
	{ "/dev/ptyvc", 0 },
	{ "/dev/ptyvd", 0 },
	{ "/dev/ptyve", 0 },
	{ "/dev/ptyvf", 0 },
	{ "/dev/ptyw0", 0 },
	{ "/dev/ptyw1", 0 },
	{ "/dev/ptyw2", 0 },
	{ "/dev/ptyw3", 0 },
	{ "/dev/ptyw4", 0 },
	{ "/dev/ptyw5", 0 },
	{ "/dev/ptyw6", 0 },
	{ "/dev/ptyw7", 0 },
	{ "/dev/ptyw8", 0 },
	{ "/dev/ptyw9", 0 },
	{ "/dev/ptywa", 0 },
	{ "/dev/ptywb", 0 },
	{ "/dev/ptywc", 0 },
	{ "/dev/ptywd", 0 },
	{ "/dev/ptywe", 0 },
	{ "/dev/ptywf", 0 },
	{ "/dev/ptyx0", 0 },
	{ "/dev/ptyx1", 0 },
	{ "/dev/ptyx2", 0 },
	{ "/dev/ptyx3", 0 },
	{ "/dev/ptyx4", 0 },
	{ "/dev/ptyx5", 0 },
	{ "/dev/ptyx6", 0 },
	{ "/dev/ptyx7", 0 },
	{ "/dev/ptyx8", 0 },
	{ "/dev/ptyx9", 0 },
	{ "/dev/ptyxa", 0 },
	{ "/dev/ptyxb", 0 },
	{ "/dev/ptyxc", 0 },
	{ "/dev/ptyxd", 0 },
	{ "/dev/ptyxe", 0 },
	{ "/dev/ptyxf", 0 },
	{ "/dev/ptyy0", 0 },
	{ "/dev/ptyy1", 0 },
	{ "/dev/ptyy2", 0 },
	{ "/dev/ptyy3", 0 },
	{ "/dev/ptyy4", 0 },
	{ "/dev/ptyy5", 0 },
	{ "/dev/ptyy6", 0 },
	{ "/dev/ptyy7", 0 },
	{ "/dev/ptyy8", 0 },
	{ "/dev/ptyy9", 0 },
	{ "/dev/ptyya", 0 },
	{ "/dev/ptyyb", 0 },
	{ "/dev/ptyyc", 0 },
	{ "/dev/ptyyd", 0 },
	{ "/dev/ptyye", 0 },
	{ "/dev/ptyyf", 0 },
	{ "/dev/ptyz0", 0 },
	{ "/dev/ptyz1", 0 },
	{ "/dev/ptyz2", 0 },
	{ "/dev/ptyz3", 0 },
	{ "/dev/ptyz4", 0 },
	{ "/dev/ptyz5", 0 },
	{ "/dev/ptyz6", 0 },
	{ "/dev/ptyz7", 0 },
	{ "/dev/ptyz8", 0 },
	{ "/dev/ptyz9", 0 },
	{ "/dev/ptyza", 0 },
	{ "/dev/ptyzb", 0 },
	{ "/dev/ptyzc", 0 },
	{ "/dev/ptyzd", 0 },
	{ "/dev/ptyze", 0 },
	{ "/dev/ptyzf", 0 },
	{ "/dev/ptya0", 0 },
	{ "/dev/ptya1", 0 },
	{ "/dev/ptya2", 0 },
	{ "/dev/ptya3", 0 },
	{ "/dev/ptya4", 0 },
	{ "/dev/ptya5", 0 },
	{ "/dev/ptya6", 0 },
	{ "/dev/ptya7", 0 },
	{ "/dev/ptya8", 0 },
	{ "/dev/ptya9", 0 },
	{ "/dev/ptyaa", 0 },
	{ "/dev/ptyab", 0 },
	{ "/dev/ptyac", 0 },
	{ "/dev/ptyad", 0 },
	{ "/dev/ptyae", 0 },
	{ "/dev/ptyaf", 0 },
	{ "/dev/ptyb0", 0 },
	{ "/dev/ptyb1", 0 },
	{ "/dev/ptyb2", 0 },
	{ "/dev/ptyb3", 0 },
	{ "/dev/ptyb4", 0 },
	{ "/dev/ptyb5", 0 },
	{ "/dev/ptyb6", 0 },
	{ "/dev/ptyb7", 0 },
	{ "/dev/ptyb8", 0 },
	{ "/dev/ptyb9", 0 },
	{ "/dev/ptyba", 0 },
	{ "/dev/ptybb", 0 },
	{ "/dev/ptybc", 0 },
	{ "/dev/ptybd", 0 },
	{ "/dev/ptybe", 0 },
	{ "/dev/ptybf", 0 },
	{ "/dev/ptyc0", 0 },
	{ "/dev/ptyc1", 0 },
	{ "/dev/ptyc2", 0 },
	{ "/dev/ptyc3", 0 },
	{ "/dev/ptyc4", 0 },
	{ "/dev/ptyc5", 0 },
	{ "/dev/ptyc6", 0 },
	{ "/dev/ptyc7", 0 },
	{ "/dev/ptyc8", 0 },
	{ "/dev/ptyc9", 0 },
	{ "/dev/ptyca", 0 },
	{ "/dev/ptycb", 0 },
	{ "/dev/ptycc", 0 },
	{ "/dev/ptycd", 0 },
	{ "/dev/ptyce", 0 },
	{ "/dev/ptycf", 0 },
	{ "/dev/ptyd0", 0 },
	{ "/dev/ptyd1", 0 },
	{ "/dev/ptyd2", 0 },
	{ "/dev/ptyd3", 0 },
	{ "/dev/ptyd4", 0 },
	{ "/dev/ptyd5", 0 },
	{ "/dev/ptyd6", 0 },
	{ "/dev/ptyd7", 0 },
	{ "/dev/ptyd8", 0 },
	{ "/dev/ptyd9", 0 },
	{ "/dev/ptyda", 0 },
	{ "/dev/ptydb", 0 },
	{ "/dev/ptydc", 0 },
	{ "/dev/ptydd", 0 },
	{ "/dev/ptyde", 0 },
	{ "/dev/ptydf", 0 },
	{ "/dev/ptye0", 0 },
	{ "/dev/ptye1", 0 },
	{ "/dev/ptye2", 0 },
	{ "/dev/ptye3", 0 },
	{ "/dev/ptye4", 0 },
	{ "/dev/ptye5", 0 },
	{ "/dev/ptye6", 0 },
	{ "/dev/ptye7", 0 },
	{ "/dev/ptye8", 0 },
	{ "/dev/ptye9", 0 },
	{ "/dev/ptyea", 0 },
	{ "/dev/ptyeb", 0 },
	{ "/dev/ptyec", 0 },
	{ "/dev/ptyed", 0 },
	{ "/dev/ptyee", 0 },
	{ "/dev/ptyef", 0 },
	{ "/dev/vcs", 0 },
	{ "/dev/vcsa", 0 },
	{ "/dev/vcs1", 0 },
	{ "/dev/vcsa1", 0 },
	{ "/dev/ttyS0", 0 },
	{ "/dev/cua0", 0 },
	{ "/dev/hda", 0 },
	{ "/dev/hda1", 2 },
	{ "/dev/hda2", 2 },
	{ "/dev/hda3", 2 },
	{ "/dev/hda4", 2 },
	{ "/dev/hda5", 2 },
	{ "/dev/hda6", 0 },
	{ "/dev/hda7", 0 },
	{ "/dev/hda8", 0 },
	{ "/dev/hdb", 0 },
	{ "/dev/hdb1", 2 },
	{ "/dev/hdc", 0 },
	{ "/dev/hdc1", 2 },
	{ "/dev/fd0u1440", 0 },
	{ "/dev/fd0u1680", 0 },
	{ "/dev/fd0u1722", 0 },
	{ "/dev/fd0u1743", 0 },
	{ "/dev/fd0u1760", 0 },
	{ "/dev/fd0u1920", 0 },
	{ "/dev/fd0u1840", 0 },
	{ "/dev/fd0u1600", 0 },
	{ "/dev/fd0u360", 0 },
	{ "/dev/fd0u720", 0 },
	{ "/dev/fd0u820", 0 },
	{ "/dev/fd0u830", 0 },
	{ "/dev/fd0u1040", 0 },
	{ "/dev/fd0u1120", 0 },
	{ "/dev/fd0u800", 0 },
	{ "/dev/fd0", 0 },
	{ "/dev/loop0", 0 },
	{ "/dev/loop1", 0 },
	{ "/dev/loop2", 0 },
	{ "/dev/loop3", 0 },
	{ "/dev/loop4", 0 },
	{ "/dev/loop5", 0 },
	{ "/dev/loop6", 0 },
	{ "/dev/loop7", 0 },
	{ "/dev/dsp", 0 },
	{ "/dev/dsp1", 0 },
	{ "/dev/mixer", 0 },
	{ "/dev/midi", 0 },
	{ "/dev/lvm", 0 },
	{ "/dev/vg0", 0 },
	{ "/dev/vg0/group", 0 },
	{ "/dev/vg0/packages", 0 },
	{ "/dev/vg0/photos", 0 },
	{ "/dev/vg0/music", 0 },
	{ "/dev/log", 0 },
	{ "/dev/MAKEDEV", 0 },
	{ "/dev/printer", 0 },
	{ "/dev/vcs2", 0 },
	{ "/dev/vcsa2", 0 },
	{ "/dev/vcs3", 0 },
	{ "/dev/vcsa3", 0 },
	{ "/dev/vcs5", 0 },
	{ "/dev/vcsa5", 0 },
	{ "/dev/vcs4", 0 },
	{ "/dev/vcsa4", 0 },
	{ "/dev/vcs6", 0 },
	{ "/dev/vcsa6", 0 },
	{ "/dev/nvidia0", 0 },
	{ "/dev/nvidia1", 0 },
	{ "/dev/nvidia2", 0 },
	{ "/dev/nvidia3", 0 },
	{ "/dev/nvidiactl", 0 },
	{ "/dev/vcs7", 0 },
	{ "/dev/vcsa7", 0 },
	{ NULL, 0 }
};
