/* -*- c-file-style: "gnu"; indent-tabs-mode: nil; -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/**
 * @file
 *
 * Gnome 2.x monitor for distcc.
 *
 * This program is very small.  Please useful features to it.
 *
 * Because repainting the table can be a bit expensive, we don't do
 * that unless the list of processes has actually changed.
 **/


#include "config.h"

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <unistd.h>

#include <gtk/gtk.h>

#include "types.h"
#include "distcc.h"
#include "rpc.h"
#include "io.h"
#include "trace.h"
#include "exitcode.h"
#include "mon.h"
#include "netutil.h"
#include "state.h"

const char *rs_program_name = "distccmon-gnome";

enum {
  COLUMN_PID = 0,
  COLUMN_STATUS,
  COLUMN_FILE,
  COLUMN_HOST,
  NUM_COLUMNS
};


static gint update_cb_tag;


/**
 * Create a GtkTreeModel that will contain information about all
 * running processes.
 **/
static GtkTreeModel *dcc_gnome_make_proc_model (void)
{
  GtkListStore *proc_store;
  
  /* Create a table for process status */
  proc_store = gtk_list_store_new (NUM_COLUMNS,
                                   G_TYPE_INT,
                                   G_TYPE_STRING,
                                   G_TYPE_STRING,
                                   G_TYPE_STRING);
  return GTK_TREE_MODEL (proc_store);
}


static void dcc_gnome_make_proc_view (GtkTreeModel *proc_model,
                                      GtkWidget **align_return,
                                      GtkWidget **tree_return)
{
  GtkWidget *proc_tree, *proc_scroll;
  GtkCellRenderer   *renderer;
  GtkTreeViewColumn *column;
  GtkWidget *align;
  GtkTreeSelection *selection;

  /* Make cell renderer */
  renderer = gtk_cell_renderer_text_new ();

  proc_tree = gtk_tree_view_new_with_model (proc_model);
  gtk_object_set (GTK_OBJECT (proc_tree),
                  "headers-visible", TRUE,
                  "rules-hint", TRUE,
                  NULL);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (proc_tree));

  gtk_tree_selection_set_mode (selection, GTK_SELECTION_NONE);
  
  /* Add PID */
  column = gtk_tree_view_column_new_with_attributes ("Pid",
                                                     renderer,
                                                     "text", COLUMN_PID,
                                                     NULL);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (proc_tree), column);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_PID);
  
  /* Add Status */
  column = gtk_tree_view_column_new_with_attributes ("Status",
                                                     renderer,
                                                     "text", COLUMN_STATUS,
                                                     NULL);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (proc_tree), column);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_STATUS);

  /* File */
  column = gtk_tree_view_column_new_with_attributes ("File",
                                                     renderer,
                                                     "text", COLUMN_FILE,
                                                     NULL);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (proc_tree), column);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_FILE);
  
  /* Host */
  column = gtk_tree_view_column_new_with_attributes ("Host",
                                                     renderer,
                                                     "text", COLUMN_HOST,
                                                     NULL);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (proc_tree), column);
  gtk_tree_view_column_set_sort_column_id (column, COLUMN_HOST);

  proc_scroll = gtk_scrolled_window_new (NULL, NULL);

  /* no horizontal scrolling; let the table stretch */
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (proc_scroll),
                                  GTK_POLICY_NEVER,
                                  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (proc_scroll), proc_tree);


  /* Expands to fill all space */
  align = gtk_alignment_new (0, 0, 1, 1);
  gtk_container_add (GTK_CONTAINER (align), proc_scroll);

  *align_return = align;
  *tree_return = proc_tree;
}


/**
 * Return TRUE if the window containing @p w is not minimized or in
 * other ways hidden.
 *
 * Used to suppress scans for processes when the window is not
 * visible.
 *
 * Explicitly depending on the state this way feels a bit ugly; it
 * might be nice to directly discover if we're mapped or visible.
 * Note that GTK_WIDGET_DRAWABLE and GTK_WIDGET_MAPPED seem to return
 * TRUE even when we're minimized.  I suspect this is because it's the
 * parent decoration window that is getting unmapped, and GTK+ doesn't
 * check that.
 **/
static gint dcc_gnome_is_onscreen (GtkWidget *w)
{
  GdkWindow *gdkw;

  gdkw = gtk_widget_get_parent_window (w);
  if (!gdkw)
    return FALSE;               /* eh? */

  return !(gdk_window_get_state (gdkw)
           & (GDK_WINDOW_STATE_WITHDRAWN | GDK_WINDOW_STATE_ICONIFIED));
}


/**
 * Callback when the timer triggers, causing a refresh.
 *
 * Empties the tree model and refills from the contents of the status
 * monitor.
 **/
static gint dcc_gnome_update_cb (gpointer view_void)
{
  GtkWidget *view;
  GtkListStore *store;
  struct dcc_mon_list *ml, *il;

  view = GTK_WIDGET (view_void);

  if (!dcc_gnome_is_onscreen (view))
    return TRUE;

  ml = NULL;
  if (dcc_mon_poll (&ml)) 
    return TRUE;

  store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (view)));

  /* TODO: As a special optimization, if the list model was previously
     empty and there are currently no entries, we avoid a pointless
     update. */
  
  gtk_list_store_clear (store);

  for (il = ml; il; il = il->next)
    {
      GtkTreeIter iter;

      gtk_list_store_append (store, &iter);
      gtk_list_store_set (store, &iter,
                          COLUMN_PID, il->cpid,
                          COLUMN_STATUS, il->state,
                          COLUMN_FILE, il->file,
                          COLUMN_HOST, il->host,
                          -1);
    }

  dcc_mon_list_free (ml);
  
  return TRUE;                  /* please call again */
}


static gchar *dcc_gnome_get_title (void)
{
  char host[256];
  const char *user;
  struct passwd *pw;
  
  if (gethostname(host, sizeof host) == -1)
    strcpy (host, "localhost");

  /* We need to look up from our pid rather than using $LOGIN because
     that's consistent with the monitor routines.  Otherwise you might
     get strange results from "sudo distccmon-gnome". */
  user = NULL;
  pw = getpwuid (getuid ());
  if (pw)
    user = pw->pw_name;
  if (!user)
    user = "";
  
  return g_strdup_printf ("distcc Monitor - %s@%s",
                          user, host);
}


static gint dcc_gnome_load_update_db (gpointer label_void)
{
  gchar message[200];
  double loadavg[3];

  if (getloadavg (loadavg, 3) == -1)
    {
      rs_log_error ("getloadavg failed: %s", strerror (errno));
      return FALSE;             /* give up */
    }

  snprintf (message, sizeof message,
            "Load average: %.2f, %.2f, %.2f",
            loadavg[0], loadavg[1], loadavg[2]);

  gtk_label_set_text (GTK_LABEL (label_void),
                      message);

  return TRUE;                  /* please call again */
}


static GtkWidget * dcc_gnome_make_load_bar (void)
{
  GtkWidget *bar, *box;

  bar = gtk_label_new ("Load: ");
  box = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (box), bar, FALSE, FALSE, 6);

  g_timeout_add (2000,          /* ms */
                 dcc_gnome_load_update_db,
                 bar);

  /* Call once to get started */
  dcc_gnome_load_update_db (bar);
  
  return box;
}


static GtkWidget * dcc_gnome_make_mainwin (void)
{
  GtkWidget *mainwin;

  mainwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  gtk_window_set_title (GTK_WINDOW (mainwin),
                        dcc_gnome_get_title());
  
  /* Set a reasonable default size that allows all columns and a few
     rows to be seen with a typical theme */
  gtk_window_set_default_size (GTK_WINDOW (mainwin), 500, 300);

  /* Quit when it's closed */
  g_signal_connect (GTK_OBJECT(mainwin), "delete-event", 
                    G_CALLBACK (gtk_main_quit), NULL);
  g_signal_connect (GTK_OBJECT(mainwin), "destroy", 
                    G_CALLBACK (gtk_main_quit), NULL);

  return mainwin;
}


static int dcc_gnome_make_app (void)
{
  GtkWidget *topbox, *proc_align, *proc_tree, *load_bar;
  GtkTreeModel *proc_model;
  GtkWidget *mainwin;

  /* Create the main window */
  mainwin = dcc_gnome_make_mainwin ();

  /* Create a vbox for the contents */
  topbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (mainwin),
                     topbox);

  proc_model = dcc_gnome_make_proc_model ();
  dcc_gnome_make_proc_view (proc_model, &proc_align, &proc_tree);

  load_bar = dcc_gnome_make_load_bar ();
    
  gtk_container_add (GTK_CONTAINER (topbox),
                     proc_align);

  /* Standard gnome padding is 12px */

  gtk_box_pack_end (GTK_BOX (topbox),
                    load_bar,
                    FALSE, /* expand */
                    FALSE,
                    6);
    
  /* Show the application window */
  gtk_widget_show_all (mainwin);

  update_cb_tag = g_timeout_add (300 /* ms */,
                                 dcc_gnome_update_cb,
                                 proc_tree);

  return 0;
}



int main(int argc, char **argv)
{
  /* We don't want to take too much time away from the real work of compilation */
  nice(5);
  
  /* Initialize i18n support */
  gtk_set_locale ();

  /* Initialize the widget set */
  gtk_init (&argc, &argv);

  dcc_gnome_make_app ();

  /* Keep running until quit */
  gtk_main ();

  return 0;
}
