/*
 * getrpmtype.c - Determines the RPM version and RPM type of an RPM
 *                package.
 *
 * Author:  David Cantrell <david@burdell.org>, <david@slackware.com>
 *
 * Copyright (C) 2002 David L. Cantrell Jr., Atlanta, GA, USA
 * All rights reserved.
 *
 * Redistribution and use of this script, with or without modification, is
 * permitted provided that the following conditions are met:
 *
 * 1. Redistributions of this script must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO
 *  EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <rpm/rpmlib.h>

extern char *optarg;
extern int optind, opterr, optopt;

/*
 * bfn:  Returns the base filename
 */
char *bfn( char *fn ) {
   char *t;

   if( (t=strrchr(fn, '/')) ) {
      return( t+1 );
   } else {
      return( fn );
   }
}

/*
 * usage:  Program help screen
 */
void usage( char *pn ) {
   printf( "Usage: %s [-n] <RPM package> ...\n", bfn(pn) );
   return;
}

/*
 * MAIN
 */
int main( int argc, char **argv ) {
   int c;
   int noname = 0;
   int i, e, src, major, minor;
   Header h;
   FD_t fd;

   if( argc <= 1 ) {
      usage( argv[0] );
      return( EXIT_SUCCESS );
   }

   while( (c=getopt(argc, argv, "nh?")) != -1 ) {
      switch( c ) {
         case 'n':
            noname = 1;
            break;
         case 'h':
         case '?':
         default:
            usage( argv[0] );
            return( EXIT_SUCCESS );
            break;
      }
   }

   /* iterate over the packages specified */
   for( i=optind; i < argc; i++ ) {
      /* open the package */
      if( (fd = fdOpen(argv[i], O_RDONLY, 0644)) == NULL ) {
         fprintf( stderr, "ERROR: Cannot open %s\n", argv[i] );
         continue;
      }

      /* read the package header */
      if( (e = rpmReadPackageHeader(fd, &h, &src, &major, &minor)) ) {
         fprintf( stderr, "rpmReadPackageHeader error: %d\n%s\n",
                  e, strerror(errno) );
      }

      /* print the type */
      if( noname == 0 ) {
         printf( "%s: ", bfn(argv[i]) );
      }

      if( src ) {
         printf( "RPM %d.%d source package.\n", major, minor );
      } else {
         printf( "RPM %d.%d binary package.\n", major, minor );
      }
   }

   return( EXIT_SUCCESS );
}
