/*
 * @(#)PushbackInputStream.java	1.15 97/01/22
 * 
 * Copyright (c) 1995, 1996 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 * 
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 * 
 * CopyrightVersion 1.1_beta
 * 
 */

package java.io;

/**
 * This class is an input stream filter that provides a 1-byte 
 * pushback buffer. This feature allows an application to 
 * "unread" the last character that it read. The next 
 * time that a read is performed on the input stream filter, the 
 * "unread" character is re-read. 
 * <p>
 * This functionality is useful when a fragment of code should read 
 * an indefinite number of data bytes that are delimited by 
 * particular byte values. After reading the terminating byte, the 
 * code fragment can "unread" it, so that the next read 
 * operation on the input stream will re-read the byte that was pushed back.
 *
 * @author  David Connelly
 * @author  Jonathan Payne
 * @version 1.15, 22 Jan 1997
 * @since   JDK1.0
 */
public
class PushbackInputStream extends FilterInputStream {
    /**
     * The push back buffer.
     * @since   JDK1.1
     */
    protected byte[] buf;

    /**
     * The current position within the buffer.
     * @since   JDK1.1
     */
    protected int pos;

    /**
     * Creates a new input stream with a push back buffer of specified size.
     *
     * @param  size  the size of the push back buffer
     * @since  JDK1.1
     */
    public PushbackInputStream(InputStream in, int size) {
	super(in);
	this.buf = new byte[size];
	this.pos = size;
    }

    /**
     * Constructs a new pushback input stream that reads its input from 
     * the specified input stream. 
     *
     * @param   in   the underlying input stream.
     * @since   JDK1.0
     */
    public PushbackInputStream(InputStream in) {
	this(in, 1);
    }

    /**
     * Reads the next byte of data from this input stream. The value 
     * byte is returned as an <code>int</code> in the range 
     * <code>0</code> to <code>255</code>. If no byte is available 
     * because the end of the stream has been reached, the value 
     * <code>-1</code> is returned. This method blocks until input data 
     * is available, the end of the stream is detected, or an exception 
     * is thrown. 
     * <p>
     * The <code>read</code> method of <code>PushbackInputStream</code> 
     * returns the just pushed-back character, if there is one, and 
     * otherwise calls the <code>read</code> method of its underlying 
     * input stream and returns whatever value that method returns. 
     *
     * @return     the next byte of data, or <code>-1</code> if the end of the
     *             stream is reached.
     * @exception  IOException  if an I/O error occurs.
     * @see        java.io.InputStream#read()
     * @since      JDK1.0
     */
    public int read() throws IOException {
	if (pos < buf.length) {
	    return buf[pos++] & 0xff;
	}
	return super.read();
    }

    /**
     * Reads up to <code>len</code> bytes of data from this input stream 
     * into an array of bytes. This method blocks until at least 1 byte 
     * of input is available. 
     *
     * @param      b     the buffer into which the data is read.
     * @param      off   the start offset of the data.
     * @param      len   the maximum number of bytes read.
     * @return     the total number of bytes read into the buffer, or
     *             <code>-1</code> if there is no more data because the end of
     *             the stream has been reached.
     * @exception  IOException  if an I/O error occurs.
     * @since      JDK1.0
     */
    public int read(byte[] b, int off, int len) throws IOException {
	if (len <= 0) {
	    return 0;
	}
	int avail = buf.length - pos;
	if (avail > 0) {
	    if (len < avail) {
		avail = len;
	    }
	    System.arraycopy(buf, pos, b, off, avail);
	    pos += avail;
	    off += avail;
	    len -= avail;
	}
	if (len > 0) {
	    len = super.read(b, off, len);
	    if (len == -1) {
		return avail == 0 ? -1 : avail;
	    }
	    return avail + len;
	}
	return avail;
    }

    /**
     * Pushes back a character so that it is read again by the next call 
     * to the <code>read</code> method on this input stream. 
     *
     * @param      b   the character to push back.
     * @exception  IOException  if the application attempts to push back a
     *               character before the previously pushed-back character has
     *               been read.
     * @since      JDK1.0
     */
    public void unread(int b) throws IOException {
	if (pos == 0) {
	    throw new IOException("Push back buffer is full");
	}
	buf[--pos] = (byte)b;
    }

    /**
     * Pushes back an array of bytes.
     * @param b the bytes to push back
     * @param off the start offset of the data
     * @param len the number of bytes to push back.
     * @exception IOException If there is not enough room in the push back
     *			      buffer for the specified number of bytes.
     * @since     JDK1.1
     */
    public void unread(byte[] b, int off, int len) throws IOException {
	if (len > pos) {
	    throw new IOException("Push back buffer is full");
	}
	pos -= len;
	System.arraycopy(b, off, buf, pos, len);
    }

    /**
     * Pushes back an array of bytes.
     * @param b the bytes to push back
     * @exception IOException If there is not enough room in the push back
     *			      buffer for the specified number of bytes.
     * @since     JDK1.1
     */
    public void unread(byte[] b) throws IOException {
	unread(b, 0, b.length);
    }

    /**
     * Returns the number of bytes that can be read from this input 
     * stream without blocking. 
     * <p>
     * The <code>available</code> method of 
     * <code>PushbackInputStream</code> calls the <code>available</code> 
     * method of its underlying input stream; it returns that value if 
     * there is no character that has been pushed back, or that value 
     * plus <code>1</code> if there is a character that has been pushed back.
     *
     * @return     the number of bytes that can be read from the input stream
     *             without blocking.
     * @exception  IOException  if an I/O error occurs.
     * @see        java.io.FilterInputStream#in
     * @since      JDK1.0
     */
    public int available() throws IOException {
	return pos + super.available();
    }

    /**
     * Tests if the input stream supports the <code>mark</code> and 
     * <code>reset</code> methods. The <code>markSupported</code> method 
     * of <code>PushbackInputStream</code> always returns <code>false</code>.
     *
     * @return   <code>true</code> if this stream type supports the
     *           <code>mark</code> and <code>reset</code> methods;
     *           <code>false</code> otherwise.
     * @see     java.io.InputStream#mark(int)
     * @see     java.io.InputStream#reset()
     * @since   JDK1.0
     */
    public boolean markSupported() {
	return false;
    }
}
