/*
 * @(#)HexUtil.java
 *
 * Copyright (C) 2004 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.util;

import java.util.Comparator;
import java.util.zip.CRC32;
import java.util.zip.Checksum;


/**
 * A simple utility to provide the Hexidecimal conversions for the log files.
 * This contains the string-length checking and other verification steps
 * that might be left out of some implementations.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/07/08 05:29:20 $
 * @since     July 7, 2004
 */
public class HexUtil
{
    protected static HexUtil s_instance = new HexUtil();
    
    
    public static class TwoShorts {
        public short a;
        public short b;
        
        public TwoShorts() {}
        public TwoShorts( short a, short b )
        {
            this.a = a;
            this.b = b;
        }
    }
    
    
    /**
     * Creates a new finder using the default (system) classpath.
     */
    protected HexUtil()
    {
        // do nothing
    }
    
    
    public static final HexUtil getInstance()
    {
        return s_instance;
    }
    
    
    /**
     * Parses two hex strings separated by a character.  The string should
     * not start with the separator char after the startPos.
     *
     * @return <code>true</code> if <code>ts</code> was populated correctly,
     *      or <code>false</code> if there was an error parsing the string.
     */
    public boolean parseTwoHex( String text, final TwoShorts ts,
            final char sep, final int startPos )
    {
        if (text == null || ts == null || startPos < 0)
        {
            return false;
        }
        
        int sepPos = text.indexOf( sep, startPos );
        if (sepPos <= 0 || sepPos >= text.length())
        {
            return false;
        }
        
        try
        {
            ts.a = (short)Integer.parseInt(
                text.substring( startPos, sepPos ), 16 );
            ts.b = (short)Integer.parseInt(
                text.substring( sepPos + 1 ).trim(), 16 );
        }
        catch (NumberFormatException e)
        {
            return false;
        }
        
        return true;
    }
    
    
    /* This logic for creating the HEX string will be contained only in the
    DirectoryChannelLogger class.
    
     * Make static final so that the invocation time is minimized.
     * <p>
     * This now returns a character array, for performance reasons.  The
     * array's format is in hexidecimal.
    public static final char[] createCoverString(
            short methodIndex, short markIndex )
    {
        char c[] = new char[10];
        c[9] = '\n';
        
        // unroll the loop
        c[8] = HEX[ methodIndex & 0xf ]; methodIndex >>= 4;
        c[7] = HEX[ methodIndex & 0xf ]; methodIndex >>= 4;
        c[6] = HEX[ methodIndex & 0xf ]; methodIndex >>= 4;
        c[5] = HEX[ methodIndex & 0xf ];
        
        c[4] = ' ';
        
        c[3] = HEX[ markIndex & 0xf ]; markIndex >>= 4;
        c[2] = HEX[ markIndex & 0xf ]; markIndex >>= 4;
        c[1] = HEX[ markIndex & 0xf ]; markIndex >>= 4;
        c[0] = HEX[ markIndex & 0xf ];
        
        return c;
    }
    */
}
