/*
 * unity-webapps-dirs.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 * 
 unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <sys/stat.h>
#include <errno.h>

#include <glib/gstdio.h>

#include "unity-webapps-dirs.h"

/*
 * Get the unity webapps user storage directory (and ensure it exists).
 * Currently this is $HOME/.local/share/unity-webapps
 */

static gchar *
unity_webapps_dirs_try_make (gchar *dir)
{
  int status;

  if (g_file_test (dir, G_FILE_TEST_EXISTS) == TRUE)
    {
      return dir;
    }

  status = g_mkdir (dir, 0700);
  
  if (status < -1)
    {
      g_critical("Error making Unity Webapps directory (%s): %s", dir, g_strerror(errno));
      
      return NULL;
    }
  return dir;
}

const gchar *
unity_webapps_dirs_get_user_dir ()
{
  const gchar *envdir;
  static gchar *userdir = NULL;
  
  if (userdir != NULL)
    return userdir;
  
  envdir = g_getenv ("UNITY_WEBAPPS_CONTEXT_USER_DIR");
  if (envdir != NULL)
    {
      userdir = g_strdup (envdir);
      return userdir;
    }
  
  userdir = g_build_filename(g_get_user_data_dir(), "unity-webapps", NULL);
  userdir = unity_webapps_dirs_try_make (userdir);
  
  return userdir;  
}

/*
 * Get the unity webapps icon storage directory and ensure it exists, 
 * currently this is a folder "icons" inside the user storage directory. */
const gchar *
unity_webapps_dirs_get_icon_dir () 
{
  static gchar *icondir = NULL;
  const gchar *envdir;
  
  if (icondir != NULL)
    return icondir;
  
  envdir = g_getenv("UNITY_WEBAPPS_CONTEXT_ICON_DIR");
  if (envdir != NULL)
    {
      icondir = g_strdup (envdir);
      
      return icondir;
    }
  
  icondir = g_build_filename (g_get_home_dir(), ".icons", NULL);
  
  icondir = unity_webapps_dirs_try_make (icondir);
  
  return icondir;
}

const gchar *
unity_webapps_dirs_get_icon_theme_dir () 
{
  static gchar *iconthemedir = NULL;
  gchar *icondir;
  const gchar *envdir;
  
  if (iconthemedir != NULL)
    return iconthemedir;
  
  envdir = g_getenv("UNITY_WEBAPPS_CONTEXT_ICON_THEME_DIR");
  if (envdir != NULL)
    {
      return g_strdup (envdir);
    }
  
  icondir = g_build_filename (g_get_home_dir(), ".local", "share", "icons", NULL);
  icondir = unity_webapps_dirs_try_make (icondir);
  
  g_free (icondir);

  iconthemedir = g_build_filename (g_get_home_dir(), ".local", "share", "icons", "unity-webapps", NULL);
  
  iconthemedir = unity_webapps_dirs_try_make (iconthemedir);
  
  return iconthemedir;
}

/* 
 * Get the directory to store desktop files in
 */
const gchar *
unity_webapps_dirs_get_application_dir () 
{
  static gchar *applicationdir = NULL;
  const gchar *envdir;
  
  if (applicationdir != NULL)
    return applicationdir;

  envdir = g_getenv("UNITY_WEBAPPS_CONTEXT_APPLICATION_DIR");
  if (envdir != NULL)
    {
      applicationdir = g_strdup(envdir);
      
      return applicationdir;
    }
 
  /* 
   * This is mostly useful for testing, i.e. we want to write the .desktop file to a temporary directory
   * and verify its contents.
   * TODO: We should have a test that does this!
   */
  applicationdir = g_build_filename (g_get_user_data_dir(), "applications", NULL);
  applicationdir = unity_webapps_dirs_try_make (applicationdir);
  
  return applicationdir;
}



const gchar *
unity_webapps_dirs_get_resource_dir () 
{
  static gchar *resourcedir = NULL;
  const gchar *userdir;
  const gchar *envdir;

  envdir = g_getenv("UNITY_WEBAPPS_CONTEXT_RESOURCE_DIR");
  if (envdir != NULL)
    {
      resourcedir = g_strdup(envdir);
      return resourcedir;
    }
 
  userdir = unity_webapps_dirs_get_user_dir ();
  resourcedir = g_build_filename (userdir, "resources", NULL);
  
  resourcedir = unity_webapps_dirs_try_make (resourcedir);
  
  
  return resourcedir;
}


