/*
 * This file is part of LibEuFin.
 * Copyright (C) 2025 Taler Systems S.A.

 * LibEuFin is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation; either version 3, or
 * (at your option) any later version.

 * LibEuFin is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General
 * Public License for more details.

 * You should have received a copy of the GNU Affero General Public
 * License along with LibEuFin; see the file COPYING.  If not, see
 * <http://www.gnu.org/licenses/>
 */

import io.ktor.http.*
import io.ktor.server.testing.*
import io.ktor.client.request.*
import io.ktor.client.request.forms.*
import org.junit.Test
import tech.libeufin.common.*
import tech.libeufin.common.test.*
import tech.libeufin.ebics.test.*
import tech.libeufin.ebics.*
import tech.libeufin.ebisync.api.*
import kotlin.test.*

class SynApiTest {
    // GET /config
    @Test
    fun config() = serverSetup { db, bank ->
        client.get("/config").assertOkJson<TalerEbiSyncConfig>()
    }

    // GET /submit
    @Test
    fun orders() = serverSetup { db, bank ->
        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
        })
        val date = client.get("/submit").assertOkJson<ListSubmitOrders>()
        assertContentEquals(date.orders, listOf(
            SubmitOrder("BTU-SCT-pain.001", "Direct Debit"),
            SubmitOrder("BTU-SCI-DE-pain.001", "Instant Direct Debit"),
        ))
    }

    // POST /submit
    @Test
    fun submit() = serverSetup { db, bank ->
        client.submitFormWithBinaryData(
            url = "/submit",
            formData = formData {
            }
        ).assertBadRequest(TalerErrorCode.GENERIC_PARAMETER_MISSING)
        client.submitFormWithBinaryData(
            url = "/submit",
            formData = formData {
                append("order", "UNKNOWN")
            }
        ).assertBadRequest(TalerErrorCode.GENERIC_PARAMETER_MISSING)

        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
        })
        client.submitFormWithBinaryData(
            url = "/submit",
            formData = formData {
                append("order", "UNKNOWN")
                append("file", "test", Headers.build {
                    append(HttpHeaders.ContentType, "application/xml")
                    append(HttpHeaders.ContentDisposition, "filename=\"content.xml\"")
                })
            }
        ).assertNotFound(TalerErrorCode.END)

        setMock(sequence {
            yield(bank::hkd)
            yield(bank::receiptOk)
            yield(bank::btuInit)
            yield(bank::btuPayload)
        })
        client.submitFormWithBinaryData(
            url = "/submit",
            formData = formData {
                append("order", "BTU-SCI-DE-pain.001")
                append("file", "test", Headers.build {
                    append(HttpHeaders.ContentType, "application/xml")
                    append(HttpHeaders.ContentDisposition, "filename=\"content.xml\"")
                })
            }
        ).assertOkJson<SyncSubmit>()
    }
}