﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codeconnections/CodeConnectionsRequest.h>
#include <aws/codeconnections/CodeConnections_EXPORTS.h>
#include <aws/codeconnections/model/ProviderType.h>
#include <aws/codeconnections/model/Tag.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CodeConnections {
namespace Model {

/**
 */
class CreateConnectionRequest : public CodeConnectionsRequest {
 public:
  AWS_CODECONNECTIONS_API CreateConnectionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateConnection"; }

  AWS_CODECONNECTIONS_API Aws::String SerializePayload() const override;

  AWS_CODECONNECTIONS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the external provider where your third-party code repository is
   * configured.</p>
   */
  inline ProviderType GetProviderType() const { return m_providerType; }
  inline bool ProviderTypeHasBeenSet() const { return m_providerTypeHasBeenSet; }
  inline void SetProviderType(ProviderType value) {
    m_providerTypeHasBeenSet = true;
    m_providerType = value;
  }
  inline CreateConnectionRequest& WithProviderType(ProviderType value) {
    SetProviderType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the connection to be created.</p>
   */
  inline const Aws::String& GetConnectionName() const { return m_connectionName; }
  inline bool ConnectionNameHasBeenSet() const { return m_connectionNameHasBeenSet; }
  template <typename ConnectionNameT = Aws::String>
  void SetConnectionName(ConnectionNameT&& value) {
    m_connectionNameHasBeenSet = true;
    m_connectionName = std::forward<ConnectionNameT>(value);
  }
  template <typename ConnectionNameT = Aws::String>
  CreateConnectionRequest& WithConnectionName(ConnectionNameT&& value) {
    SetConnectionName(std::forward<ConnectionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key-value pair to use when tagging the resource.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateConnectionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateConnectionRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the host associated with the connection to
   * be created.</p>
   */
  inline const Aws::String& GetHostArn() const { return m_hostArn; }
  inline bool HostArnHasBeenSet() const { return m_hostArnHasBeenSet; }
  template <typename HostArnT = Aws::String>
  void SetHostArn(HostArnT&& value) {
    m_hostArnHasBeenSet = true;
    m_hostArn = std::forward<HostArnT>(value);
  }
  template <typename HostArnT = Aws::String>
  CreateConnectionRequest& WithHostArn(HostArnT&& value) {
    SetHostArn(std::forward<HostArnT>(value));
    return *this;
  }
  ///@}
 private:
  ProviderType m_providerType{ProviderType::NOT_SET};

  Aws::String m_connectionName;

  Aws::Vector<Tag> m_tags;

  Aws::String m_hostArn;
  bool m_providerTypeHasBeenSet = false;
  bool m_connectionNameHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_hostArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeConnections
}  // namespace Aws
