import { useCallback, useEffect } from 'react';
import { autoScroller } from '@atlaskit/pragmatic-drag-and-drop-react-beautiful-dnd-autoscroll';
import { monitorForElements } from '@atlaskit/pragmatic-drag-and-drop/element/adapter';
import { preventUnhandled } from '@atlaskit/pragmatic-drag-and-drop/prevent-unhandled';
import { isDraggableData } from '../../draggable/data';
import { isDroppableData } from '../../droppable/data';
import { getDraggableLocation } from '../draggable-location';
import { rbdInvariant } from '../rbd-invariant';
/**
 * Sets up listeners for pointer dragging.
 */
export function usePointerControls(_ref) {
  var dragController = _ref.dragController,
    contextId = _ref.contextId;
  var updatePointerDrag = useCallback(function (location) {
    dragController.updateDrag({
      targetLocation: getDraggableLocation(location.current)
    });
  }, [dragController]);
  useEffect(function () {
    return monitorForElements({
      canMonitor: function canMonitor(_ref2) {
        var initial = _ref2.initial,
          source = _ref2.source;
        if (!isDraggableData(source.data)) {
          // not dragging something from the migration layer
          // we should not monitor it
          return false;
        }
        var isValidDraggable = source.data.contextId === contextId;
        if (!isValidDraggable) {
          return false;
        }
        var droppable = initial.dropTargets.find(function (target) {
          return isDroppableData(target.data);
        });
        if (!droppable) {
          /**
           * There may be no droppable in the `dropTargets` if it is disabled.
           *
           * This is still valid.
           */
          return true;
        }
        var isValidDroppable = droppable.data.contextId === contextId;
        return isValidDroppable;
      },
      onDragStart: function onDragStart(_ref3) {
        var location = _ref3.location,
          source = _ref3.source;
        autoScroller.start({
          input: location.current.input
        });

        /**
         * We use `preventUnhandled` because we are rendering a custom drag
         * preview.
         */
        preventUnhandled.start();
        var data = source.data;
        rbdInvariant(isDraggableData(data));
        var draggableId = data.draggableId,
          droppableId = data.droppableId,
          getIndex = data.getIndex,
          type = data.type;
        dragController.startDrag({
          draggableId: draggableId,
          type: type,
          getSourceLocation: function getSourceLocation() {
            return {
              droppableId: droppableId,
              index: getIndex()
            };
          },
          sourceElement: source.element,
          mode: 'FLUID'
        });
      },
      onDrag: function onDrag(_ref4) {
        var location = _ref4.location;
        autoScroller.updateInput({
          input: location.current.input
        });
        updatePointerDrag(location);
      },
      onDropTargetChange: function onDropTargetChange(_ref5) {
        var location = _ref5.location;
        updatePointerDrag(location);
      },
      onDrop: function onDrop() {
        autoScroller.stop();
        preventUnhandled.stop();
        dragController.stopDrag({
          reason: 'DROP'
        });
      }
    });
  }, [dragController, contextId, updatePointerDrag]);
}