#! /usr/bin/env python
#
# example4_mpl.py -- Load a fits file into a Ginga widget with a
#          matplotlib backend.
#
# This is open-source software licensed under a BSD license.
# Please see the file LICENSE.txt for details.
#
#
"""
   $ ./example4_mpl.py [fits file]

A Ginga object rendering to a generic matplotlib Figure.  In short,
this allows you to have all the interactive UI goodness of a Ginga widget
window in a matplotlib figure.  You can interactively flip, rotate, pan, zoom,
set cut levels and color map warp a FITS image.  Furthermore, you can plot
using matplotlib plotting on top of the image and the plots will follow all
the transformations.

See the Ginga quick reference
(http://ginga.readthedocs.io/en/latest/quickref.html)
for a list of the interactive features in the standard ginga widget.

example4 produces a simple matplotlib fits view with a couple of overplots.
This shows how you can use the functionality with straight python/matplotlib
sessions.  Run this by supplying a single FITS file on the command line.
"""

import sys
import os
import platform
# just in case you want to use qt
os.environ['QT_API'] = 'pyqt'

import matplotlib
MPL_V1 = matplotlib.__version__.startswith('1')
options = ['Qt4Agg', 'GTK', 'GTKAgg', 'MacOSX', 'GTKCairo', 'WXAgg',
           'TkAgg', 'QtAgg', 'FltkAgg', 'WX']
# Force a specific toolkit on mac
macos_ver = platform.mac_ver()[0]
if len(macos_ver) > 0:
    # change this to "pass" if you want to force a different backend
    # On Mac OS X I found the default choice for matplotlib is not stable
    # with ginga
    matplotlib.use('Qt4Agg')

import matplotlib.pyplot as plt
import matplotlib.patches as patches

from ginga.mplw.ImageViewMpl import CanvasView
from ginga.misc import log
from ginga import cmap
from ginga.util.loader import load_data

# add matplotlib colormaps to ginga's own set
cmap.add_matplotlib_cmaps()

# Set to True to get diagnostic logging output
use_logger = True
logger = log.get_logger(null=not use_logger, log_stderr=True)

# create a regular matplotlib figure
fig = plt.figure()

# create a ginga object, initialize some defaults and
# tell it about the figure
fi = CanvasView(logger=logger)
fi.enable_autocuts('on')
fi.set_autocut_params('zscale')
fi.show_mode_indicator(True, corner='ur')
#fi.set_cmap(cmap.get_cmap('rainbow3'))
fi.set_figure(fig)
fi.show_mode_indicator(True, corner='ur')

# enable all interactive ginga features
fi.get_bindings().enable_all(True)

# load an image
if len(sys.argv) < 2:
    print("Please provide a FITS file on the command line")
    sys.exit(1)

image = load_data(sys.argv[1], logger=logger)
fi.set_image(image)
#fi.rotate(45)

# plot some example graphics via matplotlib

# Note adding axis from ginga (mpl backend) object
ax = fi.add_axes()
if MPL_V1:
    # old matplotlib API
    ax.hold(True)

wd, ht = image.get_size()

# plot a line
l = ax.plot((wd * 0.33, wd * 0.75), (ht * 0.5, ht * 0.75), 'go-', c="g",
            label='line1')
# a rect
r = patches.Rectangle((wd * 0.10, ht * 0.10), wd * 0.6, ht * 0.5, ec='b',
                      fill=False)
ax.add_patch(r)

# if you rotate, flip, zoom or pan the the ginga image the graphics
# stay properly plotted.  See quickref of interactive ginga commands here:
#    http://ginga.readthedocs.io/en/latest/quickref.html
plt.show()
