package Pluto::Engine;

use 5.014;

use strict;
use warnings;

use registry;
use routines;

use Data::Object::Class;
use Data::Object::ClassHas;
use Data::Object::Space;
use Rewire;

our $VERSION = '0.01'; # VERSION

# ATTRIBUTES

has 'package' => (
  is => 'ro',
  isa => 'Str',
  req => 1,
);

has 'config' => (
  is => 'ro',
  isa => 'HashRef',
  opt => 1,
);

has 'library' => (
  is => 'ro',
  isa => 'Object',
  new => 1,
);

fun new_library($self) {
  $self->space->package->meta;
}

has 'services' => (
  is => 'ro',
  isa => 'Object',
  new => 1,
);

fun new_services($self) {
  Rewire->new(services => $self->config);
}

has 'space' => (
  is => 'ro',
  isa => 'Object',
  new => 1,
);

fun new_space($self) {
  Data::Object::Space->new($self->package);
}

# BUILD

my %seen;

fun BUILD($self, $args) {
  my $space = $self->space;

  if ($seen{$space->package}++) {
    return $self;
  }

  # defaults
  $space->use('5.014');
  $space->eval('use strict');
  $space->eval('use warnings');

  # types and signatures
  $space->eval('use Type::Library -base');
  $space->use('registry', $space->package);
  $space->use('routines');

  return $self;
}

# METHODS

method resolve($pack, @args) {
  $self->services->process($pack->[1], @args);
}

method execute($next, $name, @args) {
  if (Scalar::Util::blessed($next)) {
    return $next->$name(@args);
  }
  else {
    my $space = Data::Object::Space->new($next);
    my $package = $space->load;

    if (my $routine = $package->can($name)) {
      return $routine->(@args);
    }

    if ($package->can('AUTOLOAD')) {
      no strict 'refs';
      return &{"${package}::${name}"}(@args);
    }
  }
}

1;

=encoding utf8

=head1 NAME

Pluto::Engine

=cut

=head1 ABSTRACT

Functional Scripting for Perl 5

=cut

=head1 SYNOPSIS

  package Example;

  use Pluto::Engine;

  my $engine = Pluto::Engine->new(
    package => 'example',
    config => { tempdir => { package => 'File/Temp' } },
  );

=cut

=head1 DESCRIPTION

This package provides an engine for resolving dependencies and dispatching
function calls, enabling a functional programming environment for Perl 5. See
L<Rewire> for more information on dependency injection.

=cut

=head1 LIBRARIES

This package uses type constraints from:

L<Types::Standard>

=cut

=head1 ATTRIBUTES

This package has the following attributes:

=cut

=head2 config

  config(HashRef)

This attribute is read-only, accepts C<(HashRef)> values, and is optional.

=cut

=head2 library

  library(Object)

This attribute is read-only, accepts C<(Object)> values, and is optional.

=cut

=head2 package

  package(Str)

This attribute is read-only, accepts C<(Str)> values, and is required.

=cut

=head2 services

  services(Object)

This attribute is read-only, accepts C<(Object)> values, and is optional.

=cut

=head2 space

  space(Object)

This attribute is read-only, accepts C<(Object)> values, and is optional.

=cut

=head1 METHODS

This package implements the following methods:

=cut

=head2 execute

  execute(ClassName | Object $next, Str $name, Any @args) : Any

The execute method dispatches a routine call.

=over 4

=item execute example #1

  # given: synopsis

  $engine->execute('File::Temp', 'new', 'File::Temp');

=back

=cut

=head2 resolve

  resolve(Tuple[Str, Str] $pack, Any @args) : Any

The resolve method performs service lookup and resolution, and dependency
injection.

=over 4

=item resolve example #1

  # given: synopsis

  $engine->resolve([main => 'tempdir'], cleanup => 1);

=back

=cut

=head1 AUTHOR

Al Newkirk, C<awncorp@cpan.org>

=head1 LICENSE

Copyright (C) 2011-2019, Al Newkirk, et al.

This is free software; you can redistribute it and/or modify it under the terms
of the The Apache License, Version 2.0, as elucidated in the L<"license
file"|https://github.com/iamalnewkirk/pluto/blob/master/LICENSE>.

=head1 PROJECT

L<Wiki|https://github.com/iamalnewkirk/pluto/wiki>

L<Project|https://github.com/iamalnewkirk/pluto>

L<Initiatives|https://github.com/iamalnewkirk/pluto/projects>

L<Milestones|https://github.com/iamalnewkirk/pluto/milestones>

L<Contributing|https://github.com/iamalnewkirk/pluto/blob/master/CONTRIBUTE.md>

L<Issues|https://github.com/iamalnewkirk/pluto/issues>

=cut