# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module FeatureReadiness
      module Concerns
        module WorkItemConcern
          OPERATIONAL_READINESS_NOTE_ID = '<!-- OPERATIONAL READINESS PRECHECK COMMENT -->'
          OPERATIONAL_READINESS_TRACKING_LABEL = 'tracking operational readiness'

          def add_operational_readiness_precheck_comment(work_item, work_items_client, label_client)
            comment = <<~COMMENT
              #{OPERATIONAL_READINESS_NOTE_ID}
              ## Operational Readiness Pre-Check

              @#{work_item[:author][:username]} This is an automated comment to help determine if an
              [operational readiness check](https://gitlab.com/gitlab-org/gitlab/-/blob/master/.gitlab/issue_templates/Operational%20Readiness.md?ref_type=heads)
              is needed for this feature.

              Please respond with the ✅ emoji on this comment if your feature meets any of the below criteria. If not, please respond with the ❌ emoji.

              1. Requires new infrastructure components, or significant changes to existing components that have dependencies on the GitLab application.
              2. Requires changes to our application architecture that change how the infrastructure scales, GitLab is deployed or how data is processed or stored.
              3. Adds new services or changes existing services that will factor into the availability of the GitLab application.
            COMMENT

            add_labels(ids_for_labels([OPERATIONAL_READINESS_TRACKING_LABEL], label_client), work_item[:id], work_items_client)

            discussion = existing_note_containing_text(OPERATIONAL_READINESS_NOTE_ID, work_item[:iid], work_items_client)

            return discussion if discussion

            work_items_client.create_discussion(id: work_item[:id], note: comment)

            puts "\nAdded operational readiness comment to epic work item: #{work_item[:webUrl]}\n"

            existing_note_containing_text(OPERATIONAL_READINESS_NOTE_ID, work_item[:iid], work_items_client)
          end

          def existing_note_containing_text(text, work_item_iid, client)
            work_item = fetch_work_item(work_item_iid, client, [:notes])

            work_item[:widgets]
              .find { |widget| widget.key?(:discussions) }
              .dig(:discussions, :nodes)
              .find { |node| node[:notes][:nodes].any? { |node| node[:body].include?(text) } }
              &.dig(:notes, :nodes, 0)
          end

          def link_operation_readiness_issue(issue, work_item, link_type, client)
            client.create_linked_items(work_item_id: work_item[:id], item_ids: ["gid://gitlab/Issue/#{issue.id}"], link_type: link_type)
          end

          def note_has_emoji?(note, emoji_name)
            note&.dig(:awardEmoji, :nodes)&.any? { |node| node[:name] == emoji_name }
          end

          def post_comment_about_operation_readiness_issue_created(work_item, issue, precheck_comment, client)
            comment_text = <<~COMMENT
              @#{work_item[:author][:username]} Thanks for confirming that your feature requires an operational readiness check.
              Based on your response, an operational readiness check issue has been created and linked to this issue: #{issue.web_url}.
            COMMENT

            client.create_discussion_note(work_item_id: work_item[:id], discussion_id: precheck_comment.dig(:discussion, :id), text: comment_text)
          end

          def get_labels(work_item)
            labels_node = work_item[:widgets]&.find { |widget| widget.key?(:labels) }
            labels_node && labels_node[:labels][:nodes].map { |label| label[:title] }
          end

          def get_issue_iids(work_item, project)
            childern_node = work_item[:widgets]&.find { |widget| widget.key?(:children) }
            childern_node && childern_node[:children][:nodes].filter_map { |issue| issue[:iid] if issue[:workItemType][:name] == "Issue" && issue[:project][:fullPath] == project }
          end

          def has_label?(work_item, label)
            get_labels(work_item).include?(label)
          end

          def add_labels(label_ids, work_item_id, client)
            client.add_labels(work_item_id: work_item_id, label_ids: label_gids(label_ids))
          end

          def fetch_work_item(iid, client, widgets = [])
            client.work_item(workitem_iid: iid, widgets: widgets)
          end

          def has_a_child_epic?(epic)
            epic[:widgets]
              .find { |widget| widget.has_key?(:children) }[:children][:nodes]
              .any? { |child| child[:workItemType][:name] == "Epic" }
          end

          def work_item_author_id(work_item)
            extract_id_from_gid(work_item[:author][:id])
          end

          def linked_issue_iids(work_item)
            work_item[:widgets].find { |widget| widget.key?(:linkedItems) }
                               .dig(:linkedItems, :nodes)
                               .map { |node| node.dig(:workItem, :iid) }
          end

          def label_gids(label_ids = [])
            label_ids.map { |label_id| "gid://gitlab/Label/#{label_id}" }
          end

          def ids_for_labels(labels, label_client)
            labels.map { |label| get_id_for_label(label, label_client) }
          end

          def get_id_for_label(label_name, label_client)
            labels = label_client.labels(options: { search: label_name })

            raise "No labels found with name: '#{label_name}'" if labels.empty?

            labels.first.id
          end

          def ids_for_group_labels(labels, group_labels_client)
            labels.filter_map { |label| get_id_for_group_label(label, group_labels_client) }
          end

          def get_id_for_group_label(label_name, group_labels_client)
            labels = group_labels_client.group_labels(options: { search: label_name })
            return nil if labels.empty?

            labels.first.id
          end

          def extract_id_from_gid(gid)
            gid.to_s.split('/').last.to_i
          end
        end
      end
    end
  end
end
