# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class LcovFile
        # @param [String] lcov_file_content The content of the lcov file
        def initialize(lcov_file_content)
          @lcov_file_content = lcov_file_content
        end

        # @return [Hash<String, Hash>] The parsed content of the lcov file
        # @example Return value
        #   {
        #     "path/to/file1.rb" => {
        #       line_coverage: { 1 => 1, 2 => 0 },
        #       branch_coverage: {},
        #       total_lines: 2,
        #       covered_lines: 1,
        #       percentage: 50.0,
        #       total_branches: 4,
        #       covered_branches: 3,
        #       branch_percentage: 75.0,
        #       total_functions: 2,
        #       covered_functions: 1,
        #       function_percentage: 50.0
        #     },
        #     ...
        #  }
        def parsed_content
          return @parsed_content if @parsed_content

          @parsed_content = {}
          @current_file = nil

          @lcov_file_content.each_line { |line| parse_line(line) }

          include_coverage
          @parsed_content
        end

        private

        def parse_line(line)
          case line
          when /^SF:(.+)$/
            register_source_file(::Regexp.last_match(1))
          when /^DA:(\d+),(\d+)$/
            register_line_data(::Regexp.last_match(1), ::Regexp.last_match(2))
          when /^BRDA:(\d+),(\d+),(\d+),(-|\d+)$/
            register_branch_data(::Regexp.last_match(1), ::Regexp.last_match(4))
          when /^FNF:(\d+)$/
            register_functions_found(::Regexp.last_match(1))
          when /^FNH:(\d+)$/
            register_functions_hit(::Regexp.last_match(1))
          when /^BRF:(\d+)$/
            register_branches_found(::Regexp.last_match(1))
          when /^BRH:(\d+)$/
            register_branches_hit(::Regexp.last_match(1))
          when /^end_of_record$/
            @current_file = nil
          end
        end

        def include_coverage
          @parsed_content.each_key do |file|
            @parsed_content[file].merge!(line_coverage_for(file))
            @parsed_content[file].merge!(branch_coverage_for(file))
            @parsed_content[file].merge!(function_coverage_for(file))
          end
        end

        def line_coverage_for(file)
          data = @parsed_content[file]
          return unless data

          total_lines = data[:line_coverage].size
          covered_lines = data[:line_coverage].values.count(&:positive?)

          {
            total_lines: total_lines,
            covered_lines: covered_lines,
            percentage: total_lines.zero? ? 0.0 : (covered_lines.to_f / total_lines * 100).round(2)
          }
        end

        def register_source_file(filename)
          @current_file = normalize_path(filename)
          @parsed_content[@current_file] = {
            line_coverage: {},
            branch_coverage: {},
            branches: { found: 0, hit: 0 },
            functions: { found: 0, hit: 0 }
          }
        end

        def normalize_path(filename)
          # Remove leading ./ if present
          path = filename.gsub(%r{^\./}, '')

          # Handle GDK/CI paths like "../../../home/gdk/gitlab-development-kit/gitlab/app/..."
          # Extract path starting from known root directories
          match = path.match(%r{((?:ee/)?(?:app|lib|config|db|spec|scripts|tooling|workhorse|vendor)/.+)$})
          match ? match[1] : path
        end

        def register_line_data(line_no, count)
          return unless @current_file

          @parsed_content[@current_file][:line_coverage][line_no.to_i] = count.to_i
        end

        def register_branch_data(line_no, taken)
          return unless @current_file

          taken_count = taken == '-' ? 0 : taken.to_i
          @parsed_content[@current_file][:branch_coverage][line_no.to_i] ||= []
          @parsed_content[@current_file][:branch_coverage][line_no.to_i] << taken_count
        end

        def register_functions_found(count)
          return unless @current_file

          @parsed_content[@current_file][:functions][:found] = count.to_i
        end

        def register_functions_hit(count)
          return unless @current_file

          @parsed_content[@current_file][:functions][:hit] = count.to_i
        end

        def register_branches_found(count)
          return unless @current_file

          @parsed_content[@current_file][:branches][:found] = count.to_i
        end

        def register_branches_hit(count)
          return unless @current_file

          @parsed_content[@current_file][:branches][:hit] = count.to_i
        end

        def branch_coverage_for(file)
          data = @parsed_content[file]
          return {} unless data && data[:branches]

          total = data[:branches][:found]
          covered = data[:branches][:hit]

          return {} if total.nil? || total.zero?

          {
            total_branches: total,
            covered_branches: covered,
            branch_percentage: (covered.to_f / total * 100).round(2)
          }
        end

        def function_coverage_for(file)
          data = @parsed_content[file]
          return {} unless data && data[:functions]

          total = data[:functions][:found]
          covered = data[:functions][:hit]

          return {} if total.nil? || total.zero?

          {
            total_functions: total,
            covered_functions: covered,
            function_percentage: (covered.to_f / total * 100).round(2)
          }
        end
      end
    end
  end
end
