#############################################
# Class to handle standard deviations for
# PSP processes
# See POD for more details
#############################################
package Utility::StandardDeviation;

sub new{
	my ($class, @list) = @_;
	return bless \@list, $class;
}

sub compute{
	my $self = shift;
	sqrt($self->_variance());
}

sub _mean{
	my $self = shift;
	my $sum = 0;
	my $count = @{$self};
	foreach my $num (@{$self}){
		$sum += $num;	
	}
	return $sum / $count;
}

sub _variance{
	my $self = shift;
	my ($mean, $sum) = ($self->_mean(), 0);
	my $count =  @{$self};
	foreach my $num (@{$self}){
		$sum += (($num - $mean) ** 2);
	}
	return $sum / ($count-1);
}

1;

__END__

=pod

=head1 NAME

Utility::StandardDeviation

=head1 SYNOPSIS

use Utility::StandardDeviation;

my @test1 = (160,591,114,229,230,270,128,1657,624,1503);
my $sd1 = new Utility::StandardDeviation(@test1);

print $sd1->compute(), "\n";
# prints 572.027 (~)

=head1 SUMMARY

This package provides a very simple method to compute standard deviations.
It is intended to be used with the Personal Software Process, but it may have other utilities for you.

=head2 Utility::StandardDeviation

This is a Perl class blessed from an array reference passed in on creation.

=head3 Utility::StandardDeviation::new(@)

Creation method.  Takes array to compute as argument.

=head3 Utility::StandardDeviation::_mean()

Calculates the mean of the list.
Mean is a fancy word for array.

=head3 Utility::StandardDeviation::_variance()

Calculates the variance of the list.

=head3 Utility::StandardDeviation::compute()

Performs a sqrt on the variance of the classes list value.

=head1 AUTHOR

John Fraser <raptnor@yahoo.com<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (c) 2005, John Fraser C<< <raptnor@yahoo.com> >>. All rights reserved.

This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself. See L<perlartistic>.


=head1 DISCLAIMER OF WARRANTY

BECAUSE THIS SOFTWARE IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE SOFTWARE, TO THE EXTENT PERMITTED BY APPLICABLE LAW. EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE SOFTWARE "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOFTWARE IS WITH
YOU. SHOULD THE SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL
NECESSARY SERVICING, REPAIR, OR CORRECTION.

IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE SOFTWARE AS PERMITTED BY THE ABOVE LICENCE, BE
LIABLE TO YOU FOR DAMAGES, INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL,
OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OR INABILITY TO USE
THE SOFTWARE (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING
RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A
FAILURE OF THE SOFTWARE TO OPERATE WITH ANY OTHER SOFTWARE), EVEN IF
SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

=cut