/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 */

#define SIMDE_TEST_X86_AVX512_INSN adds

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/adds.h>

static int
test_simde_mm_mask_adds_epi8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int8_t src[16];
    const simde__mmask16 k;
    const int8_t a[16];
    const int8_t b[16];
    const int8_t r[16];
  } test_vec[] = {
    { { -INT8_C(  31), -INT8_C(   5),  INT8_C(  46),  INT8_C(   1),  INT8_C(  19), -INT8_C(  53),  INT8_C(  28),  INT8_C(  66),
        -INT8_C(  30),  INT8_C( 100),  INT8_C(  35), -INT8_C(  33),  INT8_C(  39), -INT8_C(  73),  INT8_C(  19),  INT8_C(  92) },
      UINT16_C( 2426),
      { -INT8_C(  95),  INT8_C(  26), -INT8_C(  72),  INT8_C(  19),  INT8_C(  14),  INT8_C( 102),  INT8_C(  82), -INT8_C(   2),
        -INT8_C(  59), -INT8_C(  10),  INT8_C( 113),  INT8_C(  15),  INT8_C(  27),  INT8_C(  69), -INT8_C( 110),  INT8_C( 115) },
      {  INT8_C(  15),  INT8_C(  75),  INT8_C( 123), -INT8_C( 127), -INT8_C( 110),  INT8_C(  47),  INT8_C(  62), -INT8_C( 126),
         INT8_C(  41),  INT8_C(  26),  INT8_C(  55),  INT8_C(   3), -INT8_C( 110), -INT8_C(  34), -INT8_C(  49), -INT8_C(  55) },
      { -INT8_C(  31),  INT8_C( 101),  INT8_C(  46), -INT8_C( 108), -INT8_C(  96),      INT8_MAX,      INT8_MAX,  INT8_C(  66),
        -INT8_C(  18),  INT8_C( 100),  INT8_C(  35),  INT8_C(  18),  INT8_C(  39), -INT8_C(  73),  INT8_C(  19),  INT8_C(  92) } },
    { {  INT8_C( 118),  INT8_C(  85),  INT8_C(  76), -INT8_C(  47), -INT8_C(  19),  INT8_C(  81),  INT8_C(  94),  INT8_C(  77),
         INT8_C(  47), -INT8_C( 123), -INT8_C( 116), -INT8_C( 110), -INT8_C(  57),  INT8_C(  55), -INT8_C( 111), -INT8_C(   4) },
      UINT16_C(32116),
      { -INT8_C(  69),  INT8_C( 126),  INT8_C(  82), -INT8_C(  14), -INT8_C(  58),  INT8_C(  10),  INT8_C(  64),      INT8_MAX,
         INT8_C(  23), -INT8_C( 109),  INT8_C(  72), -INT8_C(  20),  INT8_C(  64),  INT8_C(  73),  INT8_C(  75), -INT8_C(  38) },
      { -INT8_C(  24),  INT8_C(  58), -INT8_C(  98),  INT8_C( 121),  INT8_C(  74),  INT8_C(  61), -INT8_C(  26), -INT8_C( 116),
         INT8_C( 100),  INT8_C(  59), -INT8_C( 113), -INT8_C( 115), -INT8_C(  23),  INT8_C(  33),  INT8_C(  58), -INT8_C(   8) },
      {  INT8_C( 118),  INT8_C(  85), -INT8_C(  16), -INT8_C(  47),  INT8_C(  16),  INT8_C(  71),  INT8_C(  38),  INT8_C(  77),
         INT8_C( 123), -INT8_C( 123), -INT8_C(  41),      INT8_MIN,  INT8_C(  41),  INT8_C( 106),      INT8_MAX, -INT8_C(   4) } },
    { { -INT8_C(  47), -INT8_C(  23),  INT8_C(  39),  INT8_C(  51), -INT8_C(  61), -INT8_C(  44), -INT8_C( 113),  INT8_C(  35),
        -INT8_C(  52), -INT8_C(  41),  INT8_C(  98),  INT8_C(  85), -INT8_C( 104),  INT8_C( 102), -INT8_C(  36), -INT8_C(  31) },
      UINT16_C(20717),
      { -INT8_C( 104), -INT8_C(  53), -INT8_C( 112), -INT8_C(  37),  INT8_C(  87),  INT8_C(  78), -INT8_C(  51), -INT8_C(  79),
         INT8_C(   5),  INT8_C( 119),  INT8_C(  69), -INT8_C(  74), -INT8_C(  10),  INT8_C(  26),  INT8_C(  95), -INT8_C( 100) },
      { -INT8_C(  17), -INT8_C( 107), -INT8_C(   7), -INT8_C(  58),  INT8_C(  10),  INT8_C(  88),  INT8_C(  60), -INT8_C(  46),
         INT8_C(   2),  INT8_C(  90),  INT8_C(  57),  INT8_C(  85), -INT8_C(  77),  INT8_C(  49), -INT8_C(  36), -INT8_C(  31) },
      { -INT8_C( 121), -INT8_C(  23), -INT8_C( 119), -INT8_C(  95), -INT8_C(  61),      INT8_MAX,  INT8_C(   9), -INT8_C( 125),
        -INT8_C(  52), -INT8_C(  41),  INT8_C(  98),  INT8_C(  85), -INT8_C(  87),  INT8_C( 102),  INT8_C(  59), -INT8_C(  31) } },
    { {  INT8_C(  85),  INT8_C( 118), -INT8_C( 103),      INT8_MIN,  INT8_C(  39), -INT8_C(  43), -INT8_C(  67), -INT8_C(  57),
        -INT8_C(  62),  INT8_C(  83), -INT8_C( 118), -INT8_C( 116), -INT8_C(  79),  INT8_C(  25),  INT8_C( 118),  INT8_C(  95) },
      UINT16_C(62809),
      {  INT8_C(  82), -INT8_C( 124),  INT8_C(  65),  INT8_C(  56), -INT8_C(  94), -INT8_C(   9),  INT8_C(  63), -INT8_C( 116),
        -INT8_C(  68),  INT8_C(  40),  INT8_C(   6), -INT8_C( 104),  INT8_C(   1), -INT8_C( 120), -INT8_C(  34), -INT8_C(  27) },
      {  INT8_C(  86), -INT8_C( 105),  INT8_C(  63), -INT8_C(  53), -INT8_C(  76), -INT8_C( 100),  INT8_C(   3),  INT8_C( 113),
        -INT8_C(  76),  INT8_C(  30), -INT8_C( 104), -INT8_C(  53),  INT8_C(   4), -INT8_C(  45), -INT8_C(  26), -INT8_C(   6) },
      {      INT8_MAX,  INT8_C( 118), -INT8_C( 103),  INT8_C(   3),      INT8_MIN, -INT8_C(  43),  INT8_C(  66), -INT8_C(  57),
             INT8_MIN,  INT8_C(  83), -INT8_C(  98), -INT8_C( 116),  INT8_C(   5),      INT8_MIN, -INT8_C(  60), -INT8_C(  33) } },
    { { -INT8_C(  90), -INT8_C( 117), -INT8_C(  26), -INT8_C(   9), -INT8_C(  54),  INT8_C( 123),  INT8_C(  17), -INT8_C(  40),
        -INT8_C(  24),  INT8_C(  12), -INT8_C(   5),  INT8_C( 120),  INT8_C(  71),  INT8_C(   4),  INT8_C(  10),  INT8_C(   1) },
      UINT16_C(44733),
      { -INT8_C(  54),  INT8_C(  98), -INT8_C(  59),  INT8_C( 107), -INT8_C(  95),  INT8_C(  79), -INT8_C(  23),  INT8_C(  35),
        -INT8_C(  87),  INT8_C(   3),  INT8_C(  87), -INT8_C( 117), -INT8_C( 101),  INT8_C(  66), -INT8_C( 100), -INT8_C(  98) },
      {  INT8_C(   3), -INT8_C(  14), -INT8_C(  39),  INT8_C(  50),  INT8_C(  93), -INT8_C(  35), -INT8_C(  58),  INT8_C(  81),
         INT8_C( 108),  INT8_C(  97),  INT8_C( 119), -INT8_C(   4), -INT8_C(  32), -INT8_C(  56),  INT8_C(  12),  INT8_C(  20) },
      { -INT8_C(  51), -INT8_C( 117), -INT8_C(  98),      INT8_MAX, -INT8_C(   2),  INT8_C(  44),  INT8_C(  17),  INT8_C( 116),
        -INT8_C(  24),  INT8_C( 100),      INT8_MAX, -INT8_C( 121),  INT8_C(  71),  INT8_C(  10),  INT8_C(  10), -INT8_C(  78) } },
    { {  INT8_C(  19), -INT8_C(  64), -INT8_C(  43), -INT8_C(   1), -INT8_C( 111),  INT8_C(  62), -INT8_C( 109),  INT8_C( 118),
        -INT8_C(  60), -INT8_C(  20),  INT8_C( 126), -INT8_C( 114), -INT8_C( 121), -INT8_C(  42), -INT8_C(  28), -INT8_C(  20) },
      UINT16_C(55639),
      { -INT8_C(  71),  INT8_C(  65),  INT8_C(   2), -INT8_C(  81), -INT8_C(  74),  INT8_C(  31),  INT8_C( 108), -INT8_C(  43),
         INT8_C( 116),  INT8_C(  15),  INT8_C(  97), -INT8_C(  97), -INT8_C(  27),  INT8_C( 102), -INT8_C( 108),  INT8_C(  86) },
      { -INT8_C(  89), -INT8_C(  92),  INT8_C( 119), -INT8_C(  97), -INT8_C(  64), -INT8_C(  44),  INT8_C(  57),  INT8_C(  45),
        -INT8_C(  53),  INT8_C(  21),  INT8_C(  88), -INT8_C( 119), -INT8_C(  42),  INT8_C( 100), -INT8_C(  41),  INT8_C(  82) },
      {      INT8_MIN, -INT8_C(  27),  INT8_C( 121), -INT8_C(   1),      INT8_MIN,  INT8_C(  62),      INT8_MAX,  INT8_C( 118),
         INT8_C(  63), -INT8_C(  20),  INT8_C( 126),      INT8_MIN, -INT8_C(  69), -INT8_C(  42),      INT8_MIN,      INT8_MAX } },
    { {  INT8_C(   0), -INT8_C(  56), -INT8_C(  74), -INT8_C(  70),  INT8_C(   8),  INT8_C(  43),  INT8_C( 101), -INT8_C( 121),
         INT8_C(   0),  INT8_C(  92), -INT8_C(  58),  INT8_C(  42),  INT8_C(  93),  INT8_C( 109),  INT8_C(  64),  INT8_C( 125) },
      UINT16_C(21225),
      {  INT8_C( 113),  INT8_C(  49), -INT8_C(  88),  INT8_C( 112), -INT8_C(  39), -INT8_C(  82),  INT8_C(  13), -INT8_C(  74),
         INT8_C(  91), -INT8_C( 122), -INT8_C(  99),  INT8_C( 112), -INT8_C(  37),  INT8_C(  19), -INT8_C(  46), -INT8_C(  30) },
      { -INT8_C(  88),  INT8_C(  60),  INT8_C(  21), -INT8_C(  52),  INT8_C(  33), -INT8_C(  37), -INT8_C(  92),  INT8_C(  25),
         INT8_C(  63),  INT8_C(  50), -INT8_C(  57),  INT8_C(  91),  INT8_C(  73),  INT8_C( 120),  INT8_C(  62),  INT8_C(  79) },
      {  INT8_C(  25), -INT8_C(  56), -INT8_C(  74),  INT8_C(  60),  INT8_C(   8), -INT8_C( 119), -INT8_C(  79), -INT8_C(  49),
         INT8_C(   0), -INT8_C(  72), -INT8_C(  58),  INT8_C(  42),  INT8_C(  36),  INT8_C( 109),  INT8_C(  16),  INT8_C( 125) } },
    { { -INT8_C(  10), -INT8_C(  50), -INT8_C(  47),  INT8_C(  74),  INT8_C(  57), -INT8_C(  26), -INT8_C(  66),  INT8_C(  94),
         INT8_C(  84), -INT8_C( 107), -INT8_C(  57), -INT8_C(  48), -INT8_C(  69), -INT8_C(  65),  INT8_C(  38),  INT8_C(  80) },
      UINT16_C(52423),
      {  INT8_C(  55), -INT8_C(   9),  INT8_C(  51),  INT8_C(  94),  INT8_C(  84), -INT8_C( 116),  INT8_C(  78), -INT8_C(  33),
        -INT8_C(  36), -INT8_C(  21), -INT8_C(  69), -INT8_C(  98),  INT8_C(  52),  INT8_C(  73),  INT8_C(  52), -INT8_C(  72) },
      { -INT8_C(  69), -INT8_C(  10),  INT8_C(   2),  INT8_C(  68), -INT8_C( 124), -INT8_C(  68), -INT8_C(  24), -INT8_C(   4),
         INT8_C(  40), -INT8_C(  98),  INT8_C(  58),  INT8_C(  89), -INT8_C(  59), -INT8_C(  66),  INT8_C(  90), -INT8_C(  84) },
      { -INT8_C(  14), -INT8_C(  19),  INT8_C(  53),  INT8_C(  74),  INT8_C(  57), -INT8_C(  26),  INT8_C(  54), -INT8_C(  37),
         INT8_C(  84), -INT8_C( 107), -INT8_C(  11), -INT8_C(   9), -INT8_C(  69), -INT8_C(  65),      INT8_MAX,      INT8_MIN } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi8(test_vec[i].src);
    simde__m128i a = simde_x_mm_loadu_epi8(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi8(test_vec[i].b);
    simde__m128i r = simde_mm_mask_adds_epi8(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x16(r, simde_x_mm_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_maskz_adds_epi8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const int8_t a[16];
    const int8_t b[16];
    const int8_t r[16];
  } test_vec[] = {
    { UINT16_C(12102),
      {  INT8_C(   1),  INT8_C(  83), -INT8_C(  45),  INT8_C(   8), -INT8_C( 110), -INT8_C(  61),      INT8_MIN, -INT8_C( 118),
        -INT8_C(  32), -INT8_C(   8), -INT8_C(  28),  INT8_C(  82), -INT8_C(  13), -INT8_C(  72), -INT8_C(  60),  INT8_C(  13) },
      { -INT8_C( 121), -INT8_C(  88), -INT8_C(  66), -INT8_C(  50),  INT8_C(  67),  INT8_C( 126),      INT8_MIN,  INT8_C(  29),
        -INT8_C( 107),  INT8_C(  42), -INT8_C(  47), -INT8_C(  51),  INT8_C( 119),  INT8_C(  13),  INT8_C(  63), -INT8_C(  24) },
      {  INT8_C(   0), -INT8_C(   5), -INT8_C( 111),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),      INT8_MIN,  INT8_C(   0),
             INT8_MIN,  INT8_C(  34), -INT8_C(  75),  INT8_C(  31),  INT8_C(   0), -INT8_C(  59),  INT8_C(   0),  INT8_C(   0) } },
    { UINT16_C(65315),
      {  INT8_C(  98), -INT8_C( 110),  INT8_C(  37),  INT8_C( 119), -INT8_C(  45), -INT8_C( 116),  INT8_C(  33),  INT8_C( 110),
         INT8_C( 102),  INT8_C(  43), -INT8_C(  95), -INT8_C(  73), -INT8_C(  10),  INT8_C(  70),  INT8_C(  39),  INT8_C(  96) },
      {  INT8_C(  77), -INT8_C(  80),  INT8_C(  43), -INT8_C(  64),  INT8_C(  51), -INT8_C(  18),  INT8_C(  91),  INT8_C(  16),
        -INT8_C(  40),  INT8_C(  55), -INT8_C(  39), -INT8_C(  38), -INT8_C(  69), -INT8_C(  67),  INT8_C(  53), -INT8_C( 103) },
      {      INT8_MAX,      INT8_MIN,  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),      INT8_MIN,  INT8_C(   0),  INT8_C(   0),
         INT8_C(  62),  INT8_C(  98),      INT8_MIN, -INT8_C( 111), -INT8_C(  79),  INT8_C(   3),  INT8_C(  92), -INT8_C(   7) } },
    { UINT16_C(29490),
      { -INT8_C(  35), -INT8_C(  98),  INT8_C(   6), -INT8_C(  75),  INT8_C( 113), -INT8_C(  22),  INT8_C(  62),  INT8_C(   7),
        -INT8_C(  56), -INT8_C(  16),  INT8_C( 122), -INT8_C( 115), -INT8_C(  47),  INT8_C(  54), -INT8_C( 125), -INT8_C(  94) },
      { -INT8_C(  82),  INT8_C(  75), -INT8_C(  88),  INT8_C(  26),  INT8_C(  97), -INT8_C(  72),  INT8_C(  34), -INT8_C(  46),
        -INT8_C(  23),  INT8_C(  62),  INT8_C(  30),  INT8_C( 110), -INT8_C(  34), -INT8_C( 121), -INT8_C(  34),  INT8_C(  14) },
      {  INT8_C(   0), -INT8_C(  23),  INT8_C(   0),  INT8_C(   0),      INT8_MAX, -INT8_C(  94),  INT8_C(   0),  INT8_C(   0),
        -INT8_C(  79),  INT8_C(  46),  INT8_C(   0),  INT8_C(   0), -INT8_C(  81), -INT8_C(  67),      INT8_MIN,  INT8_C(   0) } },
    { UINT16_C(14341),
      { -INT8_C(  19), -INT8_C(  46), -INT8_C(  73),  INT8_C(  51), -INT8_C( 101),  INT8_C(  37), -INT8_C(  80),  INT8_C(  41),
         INT8_C(  38),  INT8_C(  20), -INT8_C(  33), -INT8_C(  54),  INT8_C(  67),  INT8_C(  20),  INT8_C(  21),  INT8_C( 117) },
      {  INT8_C(  33), -INT8_C(  99),  INT8_C(  79), -INT8_C( 113), -INT8_C(  63),  INT8_C(   7), -INT8_C( 124), -INT8_C( 105),
         INT8_C(  64), -INT8_C(  85), -INT8_C(  62), -INT8_C(  17), -INT8_C(  96), -INT8_C(  25), -INT8_C(  93),  INT8_C( 124) },
      {  INT8_C(  14),  INT8_C(   0),  INT8_C(   6),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0), -INT8_C(  71), -INT8_C(  29), -INT8_C(   5),  INT8_C(   0),  INT8_C(   0) } },
    { UINT16_C( 3703),
      {      INT8_MIN, -INT8_C( 125),  INT8_C(  45), -INT8_C( 105),  INT8_C( 125), -INT8_C(  41),  INT8_C(  59), -INT8_C(  89),
         INT8_C(  14), -INT8_C(  92), -INT8_C(  90), -INT8_C( 114),  INT8_C(  96), -INT8_C(  78), -INT8_C( 109),  INT8_C(  39) },
      { -INT8_C(  91),  INT8_C(  23), -INT8_C(  57),  INT8_C(  59),  INT8_C(  66),  INT8_C(  66),  INT8_C(  12), -INT8_C(  94),
        -INT8_C( 108), -INT8_C(  63),  INT8_C(  99),  INT8_C(  23),  INT8_C(  86),  INT8_C(  80),  INT8_C(  24),  INT8_C(  42) },
      {      INT8_MIN, -INT8_C( 102), -INT8_C(  12),  INT8_C(   0),      INT8_MAX,  INT8_C(  25),  INT8_C(  71),  INT8_C(   0),
         INT8_C(   0),      INT8_MIN,  INT8_C(   9), -INT8_C(  91),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT16_C(23183),
      { -INT8_C( 119), -INT8_C(  26), -INT8_C(  57),  INT8_C(  13), -INT8_C(  45),  INT8_C(  56),  INT8_C(  92),  INT8_C(  18),
         INT8_C(  77), -INT8_C(  82),  INT8_C( 104),  INT8_C(  71), -INT8_C(  99), -INT8_C( 127), -INT8_C(  39), -INT8_C(  14) },
      {  INT8_C(  71),  INT8_C(  93),  INT8_C(  46), -INT8_C(  20),  INT8_C(   5), -INT8_C(  14), -INT8_C(  92), -INT8_C( 125),
        -INT8_C(  73), -INT8_C(  53), -INT8_C(  18),  INT8_C(   0),  INT8_C(  48), -INT8_C(  84), -INT8_C(  86), -INT8_C(  53) },
      { -INT8_C(  48),  INT8_C(  67), -INT8_C(  11), -INT8_C(   7),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0), -INT8_C( 107),
         INT8_C(   0),      INT8_MIN,  INT8_C(   0),  INT8_C(  71), -INT8_C(  51),  INT8_C(   0), -INT8_C( 125),  INT8_C(   0) } },
    { UINT16_C( 4280),
      {  INT8_C(  10),  INT8_C(  78),      INT8_MAX, -INT8_C(   8),  INT8_C( 108), -INT8_C(  11), -INT8_C(  97), -INT8_C( 111),
        -INT8_C(  46),  INT8_C(  13), -INT8_C(  47), -INT8_C(  45), -INT8_C( 105), -INT8_C(  25),  INT8_C(   0), -INT8_C(   7) },
      { -INT8_C(  16), -INT8_C(  22),  INT8_C(  88), -INT8_C(  71),  INT8_C(   8), -INT8_C(  70), -INT8_C(  80),  INT8_C( 103),
         INT8_C( 123),  INT8_C(  93), -INT8_C(  35),  INT8_C(  85),  INT8_C(  10),  INT8_C(  77),  INT8_C(   0),  INT8_C(  47) },
      {  INT8_C(   0),  INT8_C(   0),  INT8_C(   0), -INT8_C(  79),  INT8_C( 116), -INT8_C(  81),  INT8_C(   0), -INT8_C(   8),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0), -INT8_C(  95),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT16_C( 4325),
      {  INT8_C(  65),  INT8_C(  62),  INT8_C(  42), -INT8_C(  68),  INT8_C(  90),  INT8_C(  73),  INT8_C(  19), -INT8_C(  56),
        -INT8_C(   4),  INT8_C(  31),  INT8_C(   4),  INT8_C(   3),  INT8_C(  44),  INT8_C(  90),  INT8_C(  99), -INT8_C(  25) },
      {  INT8_C(  89),  INT8_C(  52),  INT8_C( 104), -INT8_C(  86),  INT8_C( 121), -INT8_C(  27),  INT8_C(  75), -INT8_C(   9),
        -INT8_C(  48),  INT8_C( 121),  INT8_C(  40), -INT8_C(   3), -INT8_C(  53), -INT8_C( 114), -INT8_C( 100),  INT8_C(  81) },
      {      INT8_MAX,  INT8_C(   0),      INT8_MAX,  INT8_C(   0),  INT8_C(   0),  INT8_C(  46),  INT8_C(  94), -INT8_C(  65),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0), -INT8_C(   9),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi8(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi8(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_adds_epi8(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x16(r, simde_x_mm_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_mask_adds_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t src[8];
    const simde__mmask8 k;
    const int16_t a[8];
    const int16_t b[8];
    const int16_t r[8];
  } test_vec[] = {
    { {  INT16_C( 19830), -INT16_C( 20263),  INT16_C(  6365),  INT16_C(  9514),  INT16_C(  8029), -INT16_C( 31012), -INT16_C( 27351),  INT16_C( 19606) },
      UINT8_C(166),
      { -INT16_C( 21201), -INT16_C( 16983), -INT16_C( 28913),  INT16_C(   479), -INT16_C( 27699), -INT16_C(   698), -INT16_C( 21664),  INT16_C( 12978) },
      { -INT16_C( 31630), -INT16_C( 27895), -INT16_C( 31919), -INT16_C( 25946),  INT16_C(  5384),  INT16_C( 31469),  INT16_C( 29499), -INT16_C(  8608) },
      {  INT16_C( 19830),       INT16_MIN,       INT16_MIN,  INT16_C(  9514),  INT16_C(  8029),  INT16_C( 30771), -INT16_C( 27351),  INT16_C(  4370) } },
    { {  INT16_C( 14956), -INT16_C( 24206),  INT16_C( 10279),  INT16_C(   662), -INT16_C( 16911), -INT16_C( 17106), -INT16_C(  4862), -INT16_C(  4374) },
      UINT8_C(253),
      {  INT16_C( 25492), -INT16_C(  2767), -INT16_C(  5158), -INT16_C(  5013),  INT16_C( 23080),  INT16_C( 24538),  INT16_C( 24678), -INT16_C( 27578) },
      { -INT16_C(  5284),  INT16_C(  1255),  INT16_C( 12121),  INT16_C( 12017),  INT16_C( 13281),  INT16_C(  1685),  INT16_C( 18360), -INT16_C( 24670) },
      {  INT16_C( 20208), -INT16_C( 24206),  INT16_C(  6963),  INT16_C(  7004),       INT16_MAX,  INT16_C( 26223),       INT16_MAX,       INT16_MIN } },
    { { -INT16_C( 14832), -INT16_C( 19736),  INT16_C(  5222), -INT16_C(  1427),  INT16_C( 24020),  INT16_C( 13226), -INT16_C( 22015),  INT16_C(  8143) },
      UINT8_C(219),
      { -INT16_C( 20552), -INT16_C(  5853),  INT16_C( 22872), -INT16_C(  6013), -INT16_C( 16137), -INT16_C( 27569),  INT16_C( 19276), -INT16_C( 26420) },
      {  INT16_C(  1430),  INT16_C( 15804),  INT16_C( 13643),  INT16_C( 13486),  INT16_C( 28580),  INT16_C( 26652),  INT16_C( 15895), -INT16_C( 25828) },
      { -INT16_C( 19122),  INT16_C(  9951),  INT16_C(  5222),  INT16_C(  7473),  INT16_C( 12443),  INT16_C( 13226),       INT16_MAX,       INT16_MIN } },
    { {  INT16_C( 18056), -INT16_C(  6311), -INT16_C( 10823),  INT16_C(  8297),  INT16_C( 24132),  INT16_C( 31019), -INT16_C( 20821),  INT16_C(  1073) },
      UINT8_C(155),
      {  INT16_C( 25312), -INT16_C( 18725), -INT16_C( 31168), -INT16_C( 13304), -INT16_C( 19949),  INT16_C( 28347),  INT16_C(  3674),  INT16_C( 11284) },
      { -INT16_C( 32624),  INT16_C( 14806),  INT16_C( 29854), -INT16_C( 13394),  INT16_C( 19939),  INT16_C( 14150), -INT16_C( 32469), -INT16_C( 16490) },
      { -INT16_C(  7312), -INT16_C(  3919), -INT16_C( 10823), -INT16_C( 26698), -INT16_C(    10),  INT16_C( 31019), -INT16_C( 20821), -INT16_C(  5206) } },
    { {  INT16_C( 16411),  INT16_C( 26770), -INT16_C( 15770),  INT16_C( 11567), -INT16_C( 25386), -INT16_C( 21370), -INT16_C( 18951),  INT16_C(  5013) },
      UINT8_C(175),
      {  INT16_C( 22074), -INT16_C(  6096),  INT16_C( 12562), -INT16_C(  2681), -INT16_C(  8474),  INT16_C(  7981),  INT16_C( 29859), -INT16_C(  5915) },
      { -INT16_C( 26649), -INT16_C( 14667), -INT16_C( 21700),  INT16_C( 23924),  INT16_C( 23041), -INT16_C( 12329), -INT16_C( 26844),  INT16_C( 12057) },
      { -INT16_C(  4575), -INT16_C( 20763), -INT16_C(  9138),  INT16_C( 21243), -INT16_C( 25386), -INT16_C(  4348), -INT16_C( 18951),  INT16_C(  6142) } },
    { {  INT16_C( 31823), -INT16_C( 16416),  INT16_C( 10547), -INT16_C( 20936), -INT16_C( 21717),  INT16_C(  2429), -INT16_C( 14089),  INT16_C( 20415) },
      UINT8_C(116),
      { -INT16_C( 20670), -INT16_C( 18929), -INT16_C(  9110), -INT16_C( 27658),  INT16_C(  5659), -INT16_C( 22755),  INT16_C( 30401),  INT16_C( 12241) },
      { -INT16_C(  2428), -INT16_C(  3085),  INT16_C( 20629), -INT16_C( 29739), -INT16_C( 19289),  INT16_C( 29198),  INT16_C( 13379), -INT16_C( 26784) },
      {  INT16_C( 31823), -INT16_C( 16416),  INT16_C( 11519), -INT16_C( 20936), -INT16_C( 13630),  INT16_C(  6443),       INT16_MAX,  INT16_C( 20415) } },
    { {  INT16_C( 10682),  INT16_C( 30723), -INT16_C( 25631),  INT16_C( 12870), -INT16_C(  7733), -INT16_C(  3023),  INT16_C(  8882),  INT16_C( 24910) },
      UINT8_C( 45),
      { -INT16_C( 15846), -INT16_C( 19943), -INT16_C( 32694), -INT16_C(  6090),  INT16_C( 14036), -INT16_C( 30104), -INT16_C( 22309),  INT16_C( 32750) },
      { -INT16_C( 16035),  INT16_C(  6633),  INT16_C( 31017), -INT16_C( 29512),  INT16_C( 20156),  INT16_C( 28149),  INT16_C(  9311), -INT16_C( 22877) },
      { -INT16_C( 31881),  INT16_C( 30723), -INT16_C(  1677),       INT16_MIN, -INT16_C(  7733), -INT16_C(  1955),  INT16_C(  8882),  INT16_C( 24910) } },
    { {  INT16_C( 11801),  INT16_C( 30522), -INT16_C( 28771),  INT16_C( 18595), -INT16_C( 26062),  INT16_C(  7549),  INT16_C( 19180), -INT16_C( 19581) },
      UINT8_C( 42),
      {  INT16_C( 21562), -INT16_C( 15180),  INT16_C( 17197),  INT16_C( 26341),  INT16_C( 15144), -INT16_C( 11790),  INT16_C( 26114), -INT16_C( 26150) },
      { -INT16_C( 16280), -INT16_C( 27280),  INT16_C( 27387), -INT16_C( 12348),  INT16_C( 28781),  INT16_C( 23348),  INT16_C( 22336),  INT16_C(  2436) },
      {  INT16_C( 11801),       INT16_MIN, -INT16_C( 28771),  INT16_C( 13993), -INT16_C( 26062),  INT16_C( 11558),  INT16_C( 19180), -INT16_C( 19581) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_x_mm_loadu_epi16(test_vec[i].src);
    simde__m128i a = simde_x_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_mask_adds_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x8(r, simde_x_mm_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_maskz_adds_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const int16_t a[8];
    const int16_t b[8];
    const int16_t r[8];
  } test_vec[] = {
    { UINT8_C(126),
      {  INT16_C( 18530), -INT16_C( 15866), -INT16_C( 15400), -INT16_C( 26342), -INT16_C( 14737), -INT16_C( 15030),  INT16_C( 21680),  INT16_C( 29882) },
      {  INT16_C( 27962),  INT16_C( 13123), -INT16_C( 10479),  INT16_C(  3537), -INT16_C(  7715),  INT16_C( 27275), -INT16_C(  4308),  INT16_C(  2341) },
      {  INT16_C(     0), -INT16_C(  2743), -INT16_C( 25879), -INT16_C( 22805), -INT16_C( 22452),  INT16_C( 12245),  INT16_C( 17372),  INT16_C(     0) } },
    { UINT8_C(175),
      { -INT16_C( 18089), -INT16_C( 11763), -INT16_C(  7340), -INT16_C( 26300),  INT16_C( 14485),  INT16_C( 27175),  INT16_C(  8158), -INT16_C( 20335) },
      { -INT16_C(  8103),  INT16_C( 16065), -INT16_C(  4453),  INT16_C(  5089),  INT16_C( 29515), -INT16_C(  2790), -INT16_C( 21763),  INT16_C( 22019) },
      { -INT16_C( 26192),  INT16_C(  4302), -INT16_C( 11793), -INT16_C( 21211),  INT16_C(     0),  INT16_C( 24385),  INT16_C(     0),  INT16_C(  1684) } },
    { UINT8_C( 14),
      { -INT16_C( 20827), -INT16_C( 18732), -INT16_C( 10152), -INT16_C(  2933),  INT16_C( 31842), -INT16_C(  8748), -INT16_C( 11815),  INT16_C( 21410) },
      { -INT16_C(  1528), -INT16_C( 13596), -INT16_C( 25943), -INT16_C( 28335),  INT16_C( 25848),  INT16_C(  9269), -INT16_C( 15402),  INT16_C( 26007) },
      {  INT16_C(     0), -INT16_C( 32328),       INT16_MIN, -INT16_C( 31268),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { UINT8_C( 15),
      { -INT16_C( 27507), -INT16_C(  9398),  INT16_C( 28915), -INT16_C(  2956), -INT16_C( 14608), -INT16_C( 10466),  INT16_C( 28978), -INT16_C(  7517) },
      {  INT16_C( 20451), -INT16_C( 22940),  INT16_C( 26080),  INT16_C( 12379), -INT16_C(  5954), -INT16_C( 27171), -INT16_C( 18512), -INT16_C( 15809) },
      { -INT16_C(  7056), -INT16_C( 32338),       INT16_MAX,  INT16_C(  9423),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { UINT8_C( 30),
      {  INT16_C( 17121), -INT16_C( 27683),  INT16_C( 32718),  INT16_C( 24958), -INT16_C( 20150), -INT16_C(  1717), -INT16_C(  8726), -INT16_C( 15593) },
      {  INT16_C( 21952), -INT16_C(  7015), -INT16_C( 16049), -INT16_C( 27828),  INT16_C( 16846),  INT16_C( 20356),  INT16_C( 31961), -INT16_C( 30752) },
      {  INT16_C(     0),       INT16_MIN,  INT16_C( 16669), -INT16_C(  2870), -INT16_C(  3304),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { UINT8_C(176),
      { -INT16_C(  2095),  INT16_C( 22806),  INT16_C( 10643),  INT16_C( 17957), -INT16_C(  4712), -INT16_C( 17466), -INT16_C( 31523),  INT16_C( 23464) },
      { -INT16_C(  9670),  INT16_C(  4356), -INT16_C( 25926),  INT16_C( 21354), -INT16_C(  3587),  INT16_C( 16434), -INT16_C(  8316), -INT16_C( 28686) },
      {  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C(  8299), -INT16_C(  1032),  INT16_C(     0), -INT16_C(  5222) } },
    { UINT8_C(196),
      {  INT16_C( 23417), -INT16_C(  1452), -INT16_C( 25974), -INT16_C(  1716),  INT16_C( 22247), -INT16_C( 10292),  INT16_C(  7502), -INT16_C(   839) },
      { -INT16_C(  1858), -INT16_C(  2942), -INT16_C( 11236),  INT16_C( 15194), -INT16_C(  6079),  INT16_C(  6575), -INT16_C( 15784), -INT16_C( 25633) },
      {  INT16_C(     0),  INT16_C(     0),       INT16_MIN,  INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C(  8282), -INT16_C( 26472) } },
    { UINT8_C(  6),
      { -INT16_C(  1086),  INT16_C( 31230), -INT16_C( 15014), -INT16_C(  7078), -INT16_C( 11452),  INT16_C( 13045),  INT16_C( 23661),  INT16_C( 14428) },
      { -INT16_C( 29896), -INT16_C( 10249), -INT16_C(  5921), -INT16_C( 11132), -INT16_C(  3726), -INT16_C( 12393),  INT16_C(  5350),  INT16_C( 20495) },
      {  INT16_C(     0),  INT16_C( 20981), -INT16_C( 20935),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_adds_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x8(r, simde_x_mm_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm_mask_adds_epu8(SIMDE_MUNIT_TEST_ARGS) {
#if 1
  const struct {
    uint8_t src[16];
    simde__mmask16 k;
    uint8_t a[16];
    uint8_t b[16];
    uint8_t r[16];
  } test_vec[8] = {
    { { UINT8_C(254), UINT8_C(232), UINT8_C(119), UINT8_C(124), UINT8_C( 24), UINT8_C( 70), UINT8_C(190), UINT8_C(106),
        UINT8_C(154), UINT8_C(202), UINT8_C(214), UINT8_C(114), UINT8_C(137), UINT8_C(135), UINT8_C(222), UINT8_C( 16) },
      UINT16_C(30381),
      { UINT8_C( 39), UINT8_C(110), UINT8_C( 25), UINT8_C(196), UINT8_C( 65), UINT8_C(  4), UINT8_C(227), UINT8_C(220),
        UINT8_C(154), UINT8_C( 26), UINT8_C( 87), UINT8_C(181), UINT8_C( 13), UINT8_C( 86), UINT8_C(157), UINT8_C(133) },
      { UINT8_C(210), UINT8_C(181), UINT8_C(203), UINT8_C(144), UINT8_C( 31), UINT8_C(101), UINT8_C( 91), UINT8_C(245),
        UINT8_C(215), UINT8_C(228), UINT8_C(124), UINT8_C(182), UINT8_C(244), UINT8_C( 41), UINT8_C( 44), UINT8_C( 27) },
      { UINT8_C(249), UINT8_C(232), UINT8_C(228),    UINT8_MAX, UINT8_C( 24), UINT8_C(105), UINT8_C(190),    UINT8_MAX,
        UINT8_C(154), UINT8_C(254), UINT8_C(211), UINT8_C(114),    UINT8_MAX, UINT8_C(127), UINT8_C(201), UINT8_C( 16) } },
    { { UINT8_C(152), UINT8_C( 69), UINT8_C(223), UINT8_C(217), UINT8_C( 73), UINT8_C(195), UINT8_C(182), UINT8_C(227),
        UINT8_C(221), UINT8_C( 13), UINT8_C(152), UINT8_C(235), UINT8_C( 99), UINT8_C( 53), UINT8_C(112), UINT8_C( 53) },
      UINT16_C(15338),
      { UINT8_C(198), UINT8_C(  9), UINT8_C(160), UINT8_C( 33), UINT8_C(254), UINT8_C(119), UINT8_C(  5), UINT8_C(122),
        UINT8_C( 45), UINT8_C(249), UINT8_C(163), UINT8_C( 89), UINT8_C( 21), UINT8_C( 59), UINT8_C(158), UINT8_C(244) },
      { UINT8_C( 21), UINT8_C(231), UINT8_C(183), UINT8_C(203), UINT8_C(202), UINT8_C(149), UINT8_C(216), UINT8_C( 98),
        UINT8_C(128), UINT8_C( 60), UINT8_C(151), UINT8_C(240), UINT8_C(113), UINT8_C(129), UINT8_C( 43), UINT8_C( 55) },
      { UINT8_C(152), UINT8_C(240), UINT8_C(223), UINT8_C(236), UINT8_C( 73),    UINT8_MAX, UINT8_C(221), UINT8_C(220),
        UINT8_C(173),    UINT8_MAX, UINT8_C(152),    UINT8_MAX, UINT8_C(134), UINT8_C(188), UINT8_C(112), UINT8_C( 53) } },
    { { UINT8_C(138), UINT8_C(203), UINT8_C( 88), UINT8_C(136), UINT8_C( 66), UINT8_C( 93), UINT8_C(  2), UINT8_C(112),
        UINT8_C( 87), UINT8_C(166), UINT8_C(201), UINT8_C(108), UINT8_C(225), UINT8_C(104), UINT8_C( 96), UINT8_C(246) },
      UINT16_C( 6223),
      { UINT8_C(193), UINT8_C( 26), UINT8_C(173), UINT8_C(154), UINT8_C(124), UINT8_C( 45), UINT8_C(214), UINT8_C( 20),
        UINT8_C( 29), UINT8_C( 71), UINT8_C(149), UINT8_C( 72), UINT8_C(127), UINT8_C( 32), UINT8_C( 19), UINT8_C(215) },
      { UINT8_C(168), UINT8_C( 85), UINT8_C( 53), UINT8_C(171), UINT8_C(197), UINT8_C(140), UINT8_C( 81), UINT8_C(143),
        UINT8_C(248), UINT8_C( 50), UINT8_C(247), UINT8_C( 88), UINT8_C( 41), UINT8_C( 70), UINT8_C(112), UINT8_C(234) },
      {    UINT8_MAX, UINT8_C(111), UINT8_C(226),    UINT8_MAX, UINT8_C( 66), UINT8_C( 93),    UINT8_MAX, UINT8_C(112),
        UINT8_C( 87), UINT8_C(166), UINT8_C(201), UINT8_C(160), UINT8_C(168), UINT8_C(104), UINT8_C( 96), UINT8_C(246) } },
    { { UINT8_C( 96), UINT8_C( 29), UINT8_C(132), UINT8_C(221), UINT8_C( 74), UINT8_C( 90), UINT8_C(241), UINT8_C(103),
        UINT8_C(162), UINT8_C(134), UINT8_C(175), UINT8_C( 33), UINT8_C(166), UINT8_C(194), UINT8_C(248), UINT8_C( 79) },
      UINT16_C(11544),
      { UINT8_C(250), UINT8_C(221), UINT8_C(185), UINT8_C( 75), UINT8_C(108), UINT8_C(177), UINT8_C(125), UINT8_C( 99),
        UINT8_C( 10), UINT8_C(166), UINT8_C(170), UINT8_C(122), UINT8_C(145), UINT8_C( 10), UINT8_C(152), UINT8_C( 21) },
      { UINT8_C(231), UINT8_C(226), UINT8_C(112), UINT8_C(216), UINT8_C( 74), UINT8_C( 18), UINT8_C( 95), UINT8_C(249),
        UINT8_C( 51), UINT8_C(  5), UINT8_C(188), UINT8_C( 43), UINT8_C( 84), UINT8_C(212), UINT8_C( 89), UINT8_C( 78) },
      { UINT8_C( 96), UINT8_C( 29), UINT8_C(132),    UINT8_MAX, UINT8_C(182), UINT8_C( 90), UINT8_C(241), UINT8_C(103),
        UINT8_C( 61), UINT8_C(134),    UINT8_MAX, UINT8_C(165), UINT8_C(166), UINT8_C(222), UINT8_C(248), UINT8_C( 79) } },
    { { UINT8_C(177), UINT8_C( 18), UINT8_C(153), UINT8_C( 30), UINT8_C(196), UINT8_C( 23), UINT8_C(129), UINT8_C(206),
        UINT8_C(189), UINT8_C( 43), UINT8_C( 72), UINT8_C( 78), UINT8_C( 54), UINT8_C(224), UINT8_C(100), UINT8_C( 29) },
      UINT16_C(54467),
      { UINT8_C(246), UINT8_C( 13), UINT8_C(230), UINT8_C( 85), UINT8_C(  6), UINT8_C( 25), UINT8_C( 90), UINT8_C(194),
        UINT8_C( 68), UINT8_C(175), UINT8_C(150), UINT8_C(157), UINT8_C(253), UINT8_C( 72), UINT8_C(176), UINT8_C(151) },
      { UINT8_C(102), UINT8_C(116), UINT8_C(174), UINT8_C(231), UINT8_C( 66), UINT8_C(107), UINT8_C( 19), UINT8_C(138),
        UINT8_C(186), UINT8_C( 73), UINT8_C(107), UINT8_C( 30), UINT8_C(102), UINT8_C( 46), UINT8_C(242), UINT8_C( 92) },
      {    UINT8_MAX, UINT8_C(129), UINT8_C(153), UINT8_C( 30), UINT8_C(196), UINT8_C( 23), UINT8_C(109),    UINT8_MAX,
        UINT8_C(189), UINT8_C( 43),    UINT8_MAX, UINT8_C( 78),    UINT8_MAX, UINT8_C(224),    UINT8_MAX, UINT8_C(243) } },
    { { UINT8_C( 59), UINT8_C(216), UINT8_C(177), UINT8_C( 65), UINT8_C(241), UINT8_C( 12), UINT8_C(  4), UINT8_C( 53),
        UINT8_C(187), UINT8_C(154), UINT8_C(211), UINT8_C(184), UINT8_C(226), UINT8_C(131), UINT8_C( 79), UINT8_C( 72) },
      UINT16_C(65015),
      { UINT8_C( 48), UINT8_C( 57), UINT8_C(105), UINT8_C( 67), UINT8_C(195), UINT8_C( 35), UINT8_C(140), UINT8_C( 46),
        UINT8_C( 65), UINT8_C(242), UINT8_C( 92), UINT8_C( 51), UINT8_C( 79), UINT8_C(151), UINT8_C( 11), UINT8_C(  0) },
      { UINT8_C(217), UINT8_C(252), UINT8_C( 12), UINT8_C(221), UINT8_C( 49), UINT8_C(199), UINT8_C(119), UINT8_C(  4),
        UINT8_C(128), UINT8_C( 90), UINT8_C(135), UINT8_C(207), UINT8_C(162), UINT8_C(126), UINT8_C(205), UINT8_C(210) },
      {    UINT8_MAX,    UINT8_MAX, UINT8_C(117), UINT8_C( 65), UINT8_C(244), UINT8_C(234),    UINT8_MAX, UINT8_C( 50),
        UINT8_C(193), UINT8_C(154), UINT8_C(227),    UINT8_MAX, UINT8_C(241),    UINT8_MAX, UINT8_C(216), UINT8_C(210) } },
    { { UINT8_C(183), UINT8_C( 54), UINT8_C( 21), UINT8_C(123), UINT8_C( 89), UINT8_C(161), UINT8_C(169), UINT8_C(154),
        UINT8_C(148), UINT8_C(  6), UINT8_C(205), UINT8_C(227), UINT8_C(157), UINT8_C(216), UINT8_C(227), UINT8_C(118) },
      UINT16_C(61652),
      { UINT8_C( 83), UINT8_C(  5), UINT8_C(183), UINT8_C(203), UINT8_C( 10), UINT8_C( 55), UINT8_C( 37), UINT8_C(145),
        UINT8_C(  7), UINT8_C(199), UINT8_C( 16), UINT8_C(212), UINT8_C(154), UINT8_C(199), UINT8_C( 10), UINT8_C(175) },
      { UINT8_C( 66), UINT8_C( 99), UINT8_C( 81), UINT8_C(236), UINT8_C(253), UINT8_C(229), UINT8_C(242), UINT8_C(202),
        UINT8_C(200), UINT8_C(143), UINT8_C(162), UINT8_C(171), UINT8_C(  6), UINT8_C(118), UINT8_C(155), UINT8_C( 89) },
      { UINT8_C(183), UINT8_C( 54),    UINT8_MAX, UINT8_C(123),    UINT8_MAX, UINT8_C(161),    UINT8_MAX,    UINT8_MAX,
        UINT8_C(148), UINT8_C(  6), UINT8_C(205), UINT8_C(227), UINT8_C(160),    UINT8_MAX, UINT8_C(165),    UINT8_MAX } },
    { { UINT8_C(123), UINT8_C( 83), UINT8_C( 36), UINT8_C(133), UINT8_C(138), UINT8_C( 73), UINT8_C( 23), UINT8_C(145),
        UINT8_C( 17), UINT8_C( 39), UINT8_C(101), UINT8_C(171), UINT8_C(238), UINT8_C(111), UINT8_C( 90), UINT8_C( 49) },
      UINT16_C(43986),
      { UINT8_C( 29), UINT8_C(207), UINT8_C(144), UINT8_C( 15), UINT8_C(153), UINT8_C( 88), UINT8_C(158), UINT8_C( 59),
        UINT8_C(  4), UINT8_C(164), UINT8_C(177), UINT8_C(159), UINT8_C(254), UINT8_C( 45), UINT8_C(242), UINT8_C( 34) },
      { UINT8_C(178), UINT8_C(125), UINT8_C(108), UINT8_C(201), UINT8_C( 14), UINT8_C(125), UINT8_C(240), UINT8_C(116),
        UINT8_C( 40), UINT8_C(223), UINT8_C(227), UINT8_C(130), UINT8_C( 16), UINT8_C(182), UINT8_C( 46), UINT8_C( 45) },
      { UINT8_C(123),    UINT8_MAX, UINT8_C( 36), UINT8_C(133), UINT8_C(167), UINT8_C( 73),    UINT8_MAX, UINT8_C(175),
        UINT8_C( 44),    UINT8_MAX, UINT8_C(101),    UINT8_MAX, UINT8_C(238), UINT8_C(227), UINT8_C( 90), UINT8_C( 79) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i src = simde_x_mm_loadu_epu8(test_vec[i].src);
    simde__m128i a = simde_x_mm_loadu_epu8(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epu8(test_vec[i].b);
    simde__m128i r = simde_mm_mask_adds_epu8(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x16(r, simde_x_mm_loadu_epu8(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i src = simde_test_x86_random_u8x16();
    simde__mmask32 k = simde_test_x86_random_mmask16();
    simde__m128i a = simde_test_x86_random_u8x16();
    simde__m128i b = simde_test_x86_random_u8x16();
    simde__m128i r = simde_mm_mask_adds_epu8(src, k, a, b);

    simde_test_x86_write_u8x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_maskz_adds_epu8(SIMDE_MUNIT_TEST_ARGS) {
#if 1
  const struct {
    simde__mmask16 k;
    uint8_t a[16];
    uint8_t b[16];
    uint8_t r[16];
  } test_vec[8] = {
    { UINT16_C(40582),
      { UINT8_C(122), UINT8_C(117), UINT8_C(  8), UINT8_C(156), UINT8_C(200), UINT8_C(132), UINT8_C(184), UINT8_C(195),
        UINT8_C(223), UINT8_C(249), UINT8_C(209), UINT8_C(182), UINT8_C(201), UINT8_C( 51), UINT8_C( 44), UINT8_C(216) },
      { UINT8_C(100), UINT8_C(193), UINT8_C(172), UINT8_C( 40), UINT8_C(248), UINT8_C( 35), UINT8_C(197), UINT8_C(187),
        UINT8_C(  1), UINT8_C(124), UINT8_C( 39), UINT8_C(200), UINT8_C( 43), UINT8_C(173), UINT8_C(102), UINT8_C(165) },
      { UINT8_C(  0),    UINT8_MAX, UINT8_C(180), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,
        UINT8_C(  0),    UINT8_MAX, UINT8_C(248),    UINT8_MAX, UINT8_C(244), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX } },
    { UINT16_C(28195),
      { UINT8_C( 65), UINT8_C(235), UINT8_C(242), UINT8_C(249), UINT8_C(175), UINT8_C(209), UINT8_C(242), UINT8_C(128),
        UINT8_C(135), UINT8_C(188), UINT8_C(179), UINT8_C(179), UINT8_C(148), UINT8_C( 23), UINT8_C(116), UINT8_C( 65) },
      { UINT8_C( 63), UINT8_C(108), UINT8_C(100), UINT8_C(  5), UINT8_C( 40), UINT8_C(101), UINT8_C(129), UINT8_C( 79),
        UINT8_C( 45), UINT8_C(173), UINT8_C(252), UINT8_C(147), UINT8_C( 82), UINT8_C( 31), UINT8_C(  1), UINT8_C(148) },
      { UINT8_C(128),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),
        UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C( 54), UINT8_C(117), UINT8_C(  0) } },
    { UINT16_C(62219),
      { UINT8_C(141), UINT8_C(186), UINT8_C(196), UINT8_C(128), UINT8_C( 58), UINT8_C( 75), UINT8_C( 60), UINT8_C(238),
        UINT8_C(254), UINT8_C(208), UINT8_C(  5), UINT8_C(115), UINT8_C( 17), UINT8_C( 69), UINT8_C(223), UINT8_C(118) },
      { UINT8_C( 74), UINT8_C(  7), UINT8_C(219), UINT8_C(203), UINT8_C( 86), UINT8_C(  9), UINT8_C(120), UINT8_C( 83),
        UINT8_C(156), UINT8_C(203), UINT8_C(114), UINT8_C(158), UINT8_C( 95), UINT8_C(125), UINT8_C(145), UINT8_C(236) },
      { UINT8_C(215), UINT8_C(193), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
           UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(112), UINT8_C(194),    UINT8_MAX,    UINT8_MAX } },
    { UINT16_C(22071),
      { UINT8_C(108), UINT8_C(114), UINT8_C(161), UINT8_C(168), UINT8_C( 96), UINT8_C(160), UINT8_C(121), UINT8_C(101),
        UINT8_C( 19), UINT8_C(138), UINT8_C(170), UINT8_C(242), UINT8_C(  0), UINT8_C(244), UINT8_C(250), UINT8_C(220) },
      { UINT8_C(192), UINT8_C( 80), UINT8_C(229), UINT8_C( 56), UINT8_C(163), UINT8_C(129), UINT8_C(  3), UINT8_C( 22),
        UINT8_C( 31), UINT8_C( 98), UINT8_C(147), UINT8_C(177), UINT8_C( 79), UINT8_C(203), UINT8_C(  7), UINT8_C(187) },
      {    UINT8_MAX, UINT8_C(194),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(236),    UINT8_MAX, UINT8_C(  0), UINT8_C( 79), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0) } },
    { UINT16_C(43069),
      { UINT8_C(100), UINT8_C(157), UINT8_C( 72), UINT8_C(221), UINT8_C(  2), UINT8_C( 91), UINT8_C(103), UINT8_C(173),
        UINT8_C( 78), UINT8_C(104), UINT8_C(161), UINT8_C( 72), UINT8_C( 68), UINT8_C( 97), UINT8_C(152), UINT8_C( 41) },
      { UINT8_C(154), UINT8_C( 60), UINT8_C(170), UINT8_C(157), UINT8_C( 82), UINT8_C(202), UINT8_C(  0), UINT8_C(229),
        UINT8_C(123), UINT8_C( 79), UINT8_C(176), UINT8_C(130), UINT8_C( 10), UINT8_C(237), UINT8_C( 42), UINT8_C(110) },
      { UINT8_C(254), UINT8_C(  0), UINT8_C(242),    UINT8_MAX, UINT8_C( 84),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(202), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(151) } },
    { UINT16_C(29578),
      { UINT8_C( 75), UINT8_C(141), UINT8_C(206), UINT8_C(179), UINT8_C( 58), UINT8_C( 28), UINT8_C( 27), UINT8_C(219),
        UINT8_C(100), UINT8_C( 95), UINT8_C( 61), UINT8_C(253), UINT8_C(136), UINT8_C(215), UINT8_C( 57), UINT8_C( 50) },
      { UINT8_C(116), UINT8_C(139), UINT8_C(252), UINT8_C(116), UINT8_C(112), UINT8_C(119), UINT8_C(195), UINT8_C( 33),
        UINT8_C(249), UINT8_C(206), UINT8_C( 14), UINT8_C( 36), UINT8_C( 60), UINT8_C(153), UINT8_C(151), UINT8_C(136) },
      { UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(252),
           UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(196),    UINT8_MAX, UINT8_C(208), UINT8_C(  0) } },
    { UINT16_C(25894),
      { UINT8_C( 59), UINT8_C( 96), UINT8_C(130), UINT8_C( 86), UINT8_C( 59), UINT8_C(230), UINT8_C(181), UINT8_C(120),
        UINT8_C(227), UINT8_C( 61), UINT8_C( 79), UINT8_C( 28), UINT8_C(111), UINT8_C(196), UINT8_C(167), UINT8_C(108) },
      { UINT8_C( 56), UINT8_C( 24), UINT8_C(227), UINT8_C(252), UINT8_C( 57), UINT8_C(221), UINT8_C(202), UINT8_C( 71),
        UINT8_C(  1), UINT8_C(  6), UINT8_C(224), UINT8_C(152), UINT8_C(142), UINT8_C(  6), UINT8_C(253), UINT8_C(201) },
      { UINT8_C(  0), UINT8_C(120),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),
        UINT8_C(228), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(202),    UINT8_MAX, UINT8_C(  0) } },
    { UINT16_C(32614),
      { UINT8_C( 31), UINT8_C(162), UINT8_C(102), UINT8_C(212), UINT8_C( 26), UINT8_C( 73), UINT8_C( 17), UINT8_C(106),
        UINT8_C(102), UINT8_C(129), UINT8_C( 46), UINT8_C( 13), UINT8_C(237), UINT8_C(102), UINT8_C( 37), UINT8_C(208) },
      { UINT8_C( 98), UINT8_C( 94), UINT8_C(173), UINT8_C( 44), UINT8_C(166), UINT8_C(174), UINT8_C( 51), UINT8_C(134),
        UINT8_C( 70), UINT8_C(193), UINT8_C(141), UINT8_C( 68), UINT8_C(139), UINT8_C(243), UINT8_C(195), UINT8_C(170) },
      { UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(247), UINT8_C( 68), UINT8_C(  0),
        UINT8_C(172),    UINT8_MAX, UINT8_C(187), UINT8_C( 81),    UINT8_MAX,    UINT8_MAX, UINT8_C(232), UINT8_C(  0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i a = simde_x_mm_loadu_epu8(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epu8(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_adds_epu8(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x16(r, simde_x_mm_loadu_epu8(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask32 k = simde_test_x86_random_mmask16();
    simde__m128i a = simde_test_x86_random_u8x16();
    simde__m128i b = simde_test_x86_random_u8x16();
    simde__m128i r = simde_mm_maskz_adds_epu8(k, a, b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_u8x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_mask_adds_epu16(SIMDE_MUNIT_TEST_ARGS) {
#if 1
  const struct {
    uint16_t src[8];
    simde__mmask8 k;
    uint16_t a[8];
    uint16_t b[8];
    uint16_t r[8];
  } test_vec[8] = {
    { { UINT16_C(13255), UINT16_C(51591), UINT16_C(53876), UINT16_C(24597), UINT16_C(31283), UINT16_C(58755), UINT16_C(18634), UINT16_C(50331) },
      UINT8_C(194),
      { UINT16_C(61984), UINT16_C(36594), UINT16_C(63543), UINT16_C(16919), UINT16_C(35553), UINT16_C( 7484), UINT16_C(10843), UINT16_C(36581) },
      { UINT16_C(44721), UINT16_C(33794), UINT16_C(25283), UINT16_C(15799), UINT16_C(40421), UINT16_C(11528), UINT16_C(52280), UINT16_C(23023) },
      { UINT16_C(13255),      UINT16_MAX, UINT16_C(53876), UINT16_C(24597), UINT16_C(31283), UINT16_C(58755), UINT16_C(63123), UINT16_C(59604) } },
    { { UINT16_C(57790), UINT16_C(63207), UINT16_C(65241), UINT16_C(47928), UINT16_C(29833), UINT16_C(58584), UINT16_C(48542), UINT16_C(20338) },
      UINT8_C(108),
      { UINT16_C(54132), UINT16_C(55087), UINT16_C(28043), UINT16_C(10428), UINT16_C(60021), UINT16_C(16736), UINT16_C(47577), UINT16_C(48127) },
      { UINT16_C(62880), UINT16_C(40852), UINT16_C(20269), UINT16_C(41256), UINT16_C( 3112), UINT16_C(58687), UINT16_C(36734), UINT16_C(62033) },
      { UINT16_C(57790), UINT16_C(63207), UINT16_C(48312), UINT16_C(51684), UINT16_C(29833),      UINT16_MAX,      UINT16_MAX, UINT16_C(20338) } },
    { { UINT16_C(33122), UINT16_C(60873), UINT16_C(34542), UINT16_C(25365), UINT16_C(30320), UINT16_C(18852), UINT16_C(41775), UINT16_C(53252) },
      UINT8_C(153),
      { UINT16_C(28569), UINT16_C(59590), UINT16_C(26775), UINT16_C(41744), UINT16_C(63143), UINT16_C(13857), UINT16_C( 4935), UINT16_C(51353) },
      { UINT16_C(34525), UINT16_C(25526), UINT16_C( 6556), UINT16_C( 4819), UINT16_C( 7357), UINT16_C(24897), UINT16_C( 4385), UINT16_C(47866) },
      { UINT16_C(63094), UINT16_C(60873), UINT16_C(34542), UINT16_C(46563),      UINT16_MAX, UINT16_C(18852), UINT16_C(41775),      UINT16_MAX } },
    { { UINT16_C(49280), UINT16_C( 6050), UINT16_C(45864), UINT16_C(53434), UINT16_C(56233), UINT16_C(61446), UINT16_C(40943), UINT16_C(52409) },
      UINT8_C( 38),
      { UINT16_C(12143), UINT16_C(35266), UINT16_C(54274), UINT16_C( 7750), UINT16_C(42773), UINT16_C(10047), UINT16_C(63905), UINT16_C(25255) },
      { UINT16_C(49052), UINT16_C(20362), UINT16_C(23161), UINT16_C(22008), UINT16_C(59489), UINT16_C(   68), UINT16_C( 4257), UINT16_C( 4390) },
      { UINT16_C(49280), UINT16_C(55628),      UINT16_MAX, UINT16_C(53434), UINT16_C(56233), UINT16_C(10115), UINT16_C(40943), UINT16_C(52409) } },
    { { UINT16_C(59455), UINT16_C(16794), UINT16_C(57532), UINT16_C(53855), UINT16_C(40840), UINT16_C(10745), UINT16_C(41112), UINT16_C(13451) },
      UINT8_C( 95),
      { UINT16_C(33558), UINT16_C(28889), UINT16_C(11899), UINT16_C(25809), UINT16_C(53874), UINT16_C(33285), UINT16_C( 5880), UINT16_C(57793) },
      { UINT16_C(  688), UINT16_C(37277), UINT16_C(28513), UINT16_C(   25), UINT16_C(17000), UINT16_C( 2457), UINT16_C(52686), UINT16_C(58472) },
      { UINT16_C(34246),      UINT16_MAX, UINT16_C(40412), UINT16_C(25834),      UINT16_MAX, UINT16_C(10745), UINT16_C(58566), UINT16_C(13451) } },
    { { UINT16_C(16721), UINT16_C(52308), UINT16_C( 9839), UINT16_C(57648), UINT16_C(14072), UINT16_C(61539), UINT16_C( 9292), UINT16_C(64977) },
      UINT8_C( 38),
      { UINT16_C(36463), UINT16_C(56968), UINT16_C(34983), UINT16_C(59719), UINT16_C(20513), UINT16_C(61367), UINT16_C(39864), UINT16_C(64064) },
      { UINT16_C( 3312), UINT16_C( 5737), UINT16_C(19261), UINT16_C(29454), UINT16_C(65198), UINT16_C(54207), UINT16_C(48336), UINT16_C(16377) },
      { UINT16_C(16721), UINT16_C(62705), UINT16_C(54244), UINT16_C(57648), UINT16_C(14072),      UINT16_MAX, UINT16_C( 9292), UINT16_C(64977) } },
    { { UINT16_C(33098), UINT16_C(61725), UINT16_C(25610), UINT16_C(11227), UINT16_C(37556), UINT16_C(27930), UINT16_C(23086), UINT16_C( 7783) },
      UINT8_C(103),
      { UINT16_C(13520), UINT16_C( 7076), UINT16_C( 5954), UINT16_C(16586), UINT16_C(40406), UINT16_C(37648), UINT16_C(20374), UINT16_C( 6365) },
      { UINT16_C(53101), UINT16_C(53538), UINT16_C(19882), UINT16_C(15494), UINT16_C(62312), UINT16_C(49770), UINT16_C(34906), UINT16_C(10793) },
      {      UINT16_MAX, UINT16_C(60614), UINT16_C(25836), UINT16_C(11227), UINT16_C(37556),      UINT16_MAX, UINT16_C(55280), UINT16_C( 7783) } },
    { { UINT16_C(52668), UINT16_C(65094), UINT16_C( 4324), UINT16_C(47935), UINT16_C(20397), UINT16_C(17230), UINT16_C(11167), UINT16_C( 3163) },
      UINT8_C(250),
      { UINT16_C(56701), UINT16_C(52132), UINT16_C(57699), UINT16_C(22067), UINT16_C(62795), UINT16_C(54448), UINT16_C(56095), UINT16_C(60560) },
      { UINT16_C(36641), UINT16_C(12753), UINT16_C(36046), UINT16_C( 7646), UINT16_C( 8666), UINT16_C( 1468), UINT16_C(51325), UINT16_C(64000) },
      { UINT16_C(52668), UINT16_C(64885), UINT16_C( 4324), UINT16_C(29713),      UINT16_MAX, UINT16_C(55916),      UINT16_MAX,      UINT16_MAX } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i src = simde_x_mm_loadu_epu16(test_vec[i].src);
    simde__m128i a = simde_x_mm_loadu_epu16(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epu16(test_vec[i].b);
    simde__m128i r = simde_mm_mask_adds_epu16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x8(r, simde_x_mm_loadu_epu16(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i src = simde_test_x86_random_u16x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_u16x8();
    simde__m128i b = simde_test_x86_random_u16x8();
    simde__m128i r = simde_mm_mask_adds_epu16(src, k, a, b);

    simde_test_x86_write_u16x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_maskz_adds_epu16(SIMDE_MUNIT_TEST_ARGS) {
#if 1
  const struct {
    simde__mmask8 k;
    uint16_t a[8];
    uint16_t b[8];
    uint16_t r[8];
  } test_vec[8] = {
    { UINT8_C(254),
      { UINT16_C(23919), UINT16_C(49544), UINT16_C(19896), UINT16_C(21846), UINT16_C(32275), UINT16_C(42402), UINT16_C( 2787), UINT16_C(46236) },
      { UINT16_C(14462), UINT16_C( 5599), UINT16_C(59796), UINT16_C( 3243), UINT16_C(29483), UINT16_C(43046), UINT16_C(29390), UINT16_C(16038) },
      { UINT16_C(    0), UINT16_C(55143),      UINT16_MAX, UINT16_C(25089), UINT16_C(61758),      UINT16_MAX, UINT16_C(32177), UINT16_C(62274) } },
    { UINT8_C(207),
      { UINT16_C(65326), UINT16_C(31624), UINT16_C(56661), UINT16_C(54415), UINT16_C(13440), UINT16_C(35511), UINT16_C(27600), UINT16_C( 2057) },
      { UINT16_C( 7755), UINT16_C(13469), UINT16_C(43466), UINT16_C(15711), UINT16_C( 2255), UINT16_C(16651), UINT16_C(18862), UINT16_C(56593) },
      {      UINT16_MAX, UINT16_C(45093),      UINT16_MAX,      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(46462), UINT16_C(58650) } },
    { UINT8_C( 73),
      { UINT16_C(22681), UINT16_C(30366), UINT16_C(29415), UINT16_C( 7158), UINT16_C(33065), UINT16_C(38379), UINT16_C(62602), UINT16_C(43232) },
      { UINT16_C( 5265), UINT16_C(14962), UINT16_C(44916), UINT16_C(31754), UINT16_C(19387), UINT16_C( 1066), UINT16_C( 1884), UINT16_C(62797) },
      { UINT16_C(27946), UINT16_C(    0), UINT16_C(    0), UINT16_C(38912), UINT16_C(    0), UINT16_C(    0), UINT16_C(64486), UINT16_C(    0) } },
    { UINT8_C( 96),
      { UINT16_C(27884), UINT16_C(24135), UINT16_C(25442), UINT16_C(58248), UINT16_C( 7502), UINT16_C(17005), UINT16_C( 5885), UINT16_C( 4563) },
      { UINT16_C( 3720), UINT16_C(14469), UINT16_C(  280), UINT16_C(25587), UINT16_C(63276), UINT16_C(13248), UINT16_C(46405), UINT16_C(12691) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(30253), UINT16_C(52290), UINT16_C(    0) } },
    { UINT8_C( 33),
      { UINT16_C(36827), UINT16_C(16004), UINT16_C(26391), UINT16_C(13452), UINT16_C(53205), UINT16_C(60209), UINT16_C(17314), UINT16_C(45171) },
      { UINT16_C(43976), UINT16_C(51912), UINT16_C(11422), UINT16_C(38646), UINT16_C(10732), UINT16_C(41435), UINT16_C( 3261), UINT16_C(39107) },
      {      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0) } },
    { UINT8_C(155),
      { UINT16_C(54855), UINT16_C(44723), UINT16_C(59234), UINT16_C(12675), UINT16_C(28185), UINT16_C(23764), UINT16_C(34018), UINT16_C(36132) },
      { UINT16_C(61005), UINT16_C(31020), UINT16_C(49892), UINT16_C( 3685), UINT16_C( 1693), UINT16_C(43467), UINT16_C(25545), UINT16_C( 4164) },
      {      UINT16_MAX,      UINT16_MAX, UINT16_C(    0), UINT16_C(16360), UINT16_C(29878), UINT16_C(    0), UINT16_C(    0), UINT16_C(40296) } },
    { UINT8_C( 57),
      { UINT16_C(49143), UINT16_C(57243), UINT16_C(52546), UINT16_C(45560), UINT16_C(21665), UINT16_C( 9619), UINT16_C( 8312), UINT16_C(26482) },
      { UINT16_C(60236), UINT16_C( 3659), UINT16_C(22864), UINT16_C(22443), UINT16_C(21540), UINT16_C(34592), UINT16_C(12697), UINT16_C(37056) },
      {      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(43205), UINT16_C(44211), UINT16_C(    0), UINT16_C(    0) } },
    { UINT8_C(240),
      { UINT16_C(28508), UINT16_C(10546), UINT16_C(58215), UINT16_C(48074), UINT16_C(61302), UINT16_C(38708), UINT16_C(39778), UINT16_C(19939) },
      { UINT16_C(62182), UINT16_C(16542), UINT16_C(62877), UINT16_C(62052), UINT16_C(60437), UINT16_C(18059), UINT16_C( 7084), UINT16_C( 2102) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(56767), UINT16_C(46862), UINT16_C(22041) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i a = simde_x_mm_loadu_epu16(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epu16(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_adds_epu16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x8(r, simde_x_mm_loadu_epu16(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_u16x8();
    simde__m128i b = simde_test_x86_random_u16x8();
    simde__m128i r = simde_mm_maskz_adds_epu16(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_u16x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_mask_adds_epi8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int8_t src[32];
    const simde__mmask32 k;
    const int8_t a[32];
    const int8_t b[32];
    const int8_t r[32];
  } test_vec[] = {
      { {  INT8_C(  63),  INT8_C(  79),  INT8_C(  62), -INT8_C(  24),  INT8_C(  99),  INT8_C(  34),  INT8_C(  33),  INT8_C(  55),
         INT8_C(  34),  INT8_C( 107),  INT8_C( 116),  INT8_C(  40), -INT8_C(  20),  INT8_C(  26),  INT8_C(  82), -INT8_C( 107),
        -INT8_C(  52),      INT8_MIN, -INT8_C(  36),  INT8_C(  51), -INT8_C(  21), -INT8_C( 124),  INT8_C( 112),  INT8_C( 122),
        -INT8_C(  57), -INT8_C(  30), -INT8_C(  11), -INT8_C(  90), -INT8_C(  59),  INT8_C(  91),  INT8_C( 106), -INT8_C( 116) },
      UINT32_C(     17939),
      { -INT8_C( 123), -INT8_C(  71),  INT8_C(   3), -INT8_C(  27),  INT8_C(  90), -INT8_C(  35),  INT8_C(  53), -INT8_C(  70),
         INT8_C(  37), -INT8_C(  93),  INT8_C(  39), -INT8_C(  99), -INT8_C(  73),  INT8_C(  99), -INT8_C(  90), -INT8_C(  99),
         INT8_C(  34),  INT8_C(  74),  INT8_C(  66), -INT8_C(  87), -INT8_C(  35),  INT8_C( 109), -INT8_C(  31),  INT8_C(  57),
         INT8_C(  29), -INT8_C(   1), -INT8_C( 109), -INT8_C(  29), -INT8_C( 102), -INT8_C( 122), -INT8_C(  40), -INT8_C(  64) },
      {  INT8_C(   1), -INT8_C(  57), -INT8_C(  42), -INT8_C(  31),  INT8_C(  49), -INT8_C(  49),  INT8_C(  95), -INT8_C( 124),
        -INT8_C( 108),  INT8_C(  82), -INT8_C( 127), -INT8_C(  41),  INT8_C( 123),  INT8_C(  60),  INT8_C(  18),  INT8_C(   4),
        -INT8_C(  92), -INT8_C( 115),  INT8_C(  22), -INT8_C(  82),  INT8_C(  16), -INT8_C(  95),  INT8_C(  51), -INT8_C(   3),
         INT8_C(  91),  INT8_C(  71), -INT8_C(  33), -INT8_C( 101), -INT8_C( 115), -INT8_C(   4),  INT8_C(  91), -INT8_C( 104) },
      { -INT8_C( 122),      INT8_MIN,  INT8_C(  62), -INT8_C(  24),      INT8_MAX,  INT8_C(  34),  INT8_C(  33),  INT8_C(  55),
         INT8_C(  34), -INT8_C(  11), -INT8_C(  88),  INT8_C(  40), -INT8_C(  20),  INT8_C(  26), -INT8_C(  72), -INT8_C( 107),
        -INT8_C(  52),      INT8_MIN, -INT8_C(  36),  INT8_C(  51), -INT8_C(  21), -INT8_C( 124),  INT8_C( 112),  INT8_C( 122),
        -INT8_C(  57), -INT8_C(  30), -INT8_C(  11), -INT8_C(  90), -INT8_C(  59),  INT8_C(  91),  INT8_C( 106), -INT8_C( 116) } },
    { { -INT8_C( 123),  INT8_C( 123),  INT8_C(  97),  INT8_C(  38), -INT8_C(  67), -INT8_C(  49),  INT8_C(  30), -INT8_C(  76),
        -INT8_C(  84),  INT8_C(  84),  INT8_C(  23),  INT8_C(  86),  INT8_C(   8), -INT8_C(  26), -INT8_C(  64), -INT8_C(  51),
         INT8_C( 112),  INT8_C(  87),  INT8_C( 119),  INT8_C(   1),  INT8_C(  43), -INT8_C(  21),  INT8_C(  90),  INT8_C(  91),
         INT8_C( 125), -INT8_C(  98),  INT8_C(  20),  INT8_C(  24),  INT8_C( 126), -INT8_C(  27), -INT8_C( 117), -INT8_C(  87) },
      UINT32_C(     56101),
      { -INT8_C(  29),  INT8_C(  83),  INT8_C( 102), -INT8_C(  68), -INT8_C(  94),  INT8_C(  98), -INT8_C( 113), -INT8_C(  98),
        -INT8_C(  91), -INT8_C(  30),  INT8_C(  20),  INT8_C( 105),  INT8_C(  46), -INT8_C(  21), -INT8_C(  97), -INT8_C(  66),
        -INT8_C( 109), -INT8_C(  73),  INT8_C(  45),  INT8_C(  30), -INT8_C(  72), -INT8_C( 108), -INT8_C(  85), -INT8_C(  45),
         INT8_C(  78),  INT8_C(  69),  INT8_C(  14),  INT8_C(  75), -INT8_C(  71), -INT8_C(  49), -INT8_C(  55),  INT8_C(  42) },
      {  INT8_C(  52), -INT8_C(  69), -INT8_C( 103), -INT8_C(  82),  INT8_C(  90),  INT8_C(  51), -INT8_C(  42),      INT8_MIN,
         INT8_C( 102),  INT8_C( 100),  INT8_C(   6),  INT8_C(  50), -INT8_C( 114),  INT8_C( 106),  INT8_C( 119), -INT8_C(  17),
        -INT8_C(  46), -INT8_C(  52),  INT8_C(  25), -INT8_C(  71),  INT8_C( 100),  INT8_C(  75),  INT8_C(  70),  INT8_C(   2),
        -INT8_C(  20),  INT8_C(  69), -INT8_C(   9), -INT8_C(  68),  INT8_C( 119), -INT8_C(  98), -INT8_C(  25), -INT8_C(  53) },
      {  INT8_C(  23),  INT8_C( 123), -INT8_C(   1),  INT8_C(  38), -INT8_C(  67),      INT8_MAX,  INT8_C(  30), -INT8_C(  76),
         INT8_C(  11),  INT8_C(  70),  INT8_C(  23),      INT8_MAX, -INT8_C(  68), -INT8_C(  26),  INT8_C(  22), -INT8_C(  83),
         INT8_C( 112),  INT8_C(  87),  INT8_C( 119),  INT8_C(   1),  INT8_C(  43), -INT8_C(  21),  INT8_C(  90),  INT8_C(  91),
         INT8_C( 125), -INT8_C(  98),  INT8_C(  20),  INT8_C(  24),  INT8_C( 126), -INT8_C(  27), -INT8_C( 117), -INT8_C(  87) } },
    { { -INT8_C(  26),  INT8_C(  33), -INT8_C(  40),  INT8_C(  31), -INT8_C(  46),  INT8_C(  71),  INT8_C(  81), -INT8_C(  30),
        -INT8_C(  45), -INT8_C(   8), -INT8_C(   2),      INT8_MAX, -INT8_C(  99),  INT8_C(  87),  INT8_C(  64), -INT8_C(  20),
        -INT8_C(  83), -INT8_C(  40), -INT8_C(  19), -INT8_C( 114),  INT8_C(  34),  INT8_C( 105),  INT8_C(  90),  INT8_C(  35),
         INT8_C(  75),  INT8_C( 112), -INT8_C(  89), -INT8_C( 127),  INT8_C(  95),  INT8_C(  79),  INT8_C(  40),  INT8_C(   8) },
      UINT32_C(     23780),
      { -INT8_C(  46), -INT8_C(  84), -INT8_C(  66), -INT8_C(  62),  INT8_C(  68), -INT8_C( 119), -INT8_C( 126), -INT8_C(  81),
        -INT8_C( 113),  INT8_C(  88), -INT8_C(  93), -INT8_C( 126),  INT8_C(  25),  INT8_C( 108), -INT8_C( 109),  INT8_C(  86),
        -INT8_C(  76),  INT8_C(  45),  INT8_C(  25),  INT8_C(  69),  INT8_C(  27), -INT8_C(  54),  INT8_C(  66), -INT8_C( 126),
         INT8_C( 100), -INT8_C(  10), -INT8_C(  44), -INT8_C(  28),  INT8_C( 113), -INT8_C( 119), -INT8_C(  10), -INT8_C(  93) },
      {  INT8_C( 109), -INT8_C(  64), -INT8_C(  31), -INT8_C(  47),  INT8_C( 104), -INT8_C(  13),  INT8_C(  71),  INT8_C(  10),
        -INT8_C(  90),  INT8_C(  62),  INT8_C( 102), -INT8_C(  45),  INT8_C(  60), -INT8_C(  39), -INT8_C( 121), -INT8_C(  60),
         INT8_C( 117), -INT8_C(  59),  INT8_C(  37), -INT8_C(  71), -INT8_C(  20),  INT8_C(  65), -INT8_C(  54), -INT8_C(  58),
        -INT8_C(  96), -INT8_C(  70), -INT8_C(  87),  INT8_C(  79), -INT8_C(  17),  INT8_C(  50), -INT8_C(  51),  INT8_C(  52) },
      { -INT8_C(  26),  INT8_C(  33), -INT8_C(  97),  INT8_C(  31), -INT8_C(  46),      INT8_MIN, -INT8_C(  55), -INT8_C(  71),
        -INT8_C(  45), -INT8_C(   8),  INT8_C(   9),      INT8_MIN,  INT8_C(  85),  INT8_C(  87),      INT8_MIN, -INT8_C(  20),
        -INT8_C(  83), -INT8_C(  40), -INT8_C(  19), -INT8_C( 114),  INT8_C(  34),  INT8_C( 105),  INT8_C(  90),  INT8_C(  35),
         INT8_C(  75),  INT8_C( 112), -INT8_C(  89), -INT8_C( 127),  INT8_C(  95),  INT8_C(  79),  INT8_C(  40),  INT8_C(   8) } },
    { { -INT8_C( 127), -INT8_C( 111), -INT8_C(  31),  INT8_C( 126),  INT8_C( 108),  INT8_C(  93), -INT8_C(  18),  INT8_C( 111),
        -INT8_C( 105),  INT8_C(  57),  INT8_C(  30), -INT8_C(  40),  INT8_C(  11), -INT8_C( 124), -INT8_C(  72),  INT8_C(  21),
         INT8_C( 105),  INT8_C(  64), -INT8_C(   9),  INT8_C( 116), -INT8_C(  97), -INT8_C(  95),  INT8_C(   6), -INT8_C(  94),
         INT8_C(  64), -INT8_C(  46),  INT8_C(  55),  INT8_C(  98),  INT8_C(  21), -INT8_C(  72), -INT8_C( 125),  INT8_C(  46) },
      UINT32_C(     49881),
      { -INT8_C( 114), -INT8_C(  58),  INT8_C(   3),  INT8_C(  38), -INT8_C(  42), -INT8_C( 116),  INT8_C(  75),  INT8_C(  22),
        -INT8_C(  60),  INT8_C(  47),  INT8_C(  34),  INT8_C(  47), -INT8_C( 105),  INT8_C(  78), -INT8_C(  59), -INT8_C(  90),
        -INT8_C(  37),  INT8_C( 116), -INT8_C(  87), -INT8_C(  26),  INT8_C( 120),  INT8_C(  99), -INT8_C(  89),  INT8_C(  40),
        -INT8_C(   5), -INT8_C(  91), -INT8_C(  78),  INT8_C(  22),  INT8_C(  10), -INT8_C(  39), -INT8_C(  35),  INT8_C(  56) },
      {  INT8_C(   4),  INT8_C(  92),  INT8_C(  58), -INT8_C(  42),  INT8_C( 109),  INT8_C(  85), -INT8_C(  76),  INT8_C( 113),
         INT8_C(  58), -INT8_C(  80),  INT8_C(  16),  INT8_C(  52),  INT8_C(   8),  INT8_C(  35), -INT8_C(  21),  INT8_C( 108),
         INT8_C( 126), -INT8_C(  36), -INT8_C(  43), -INT8_C(  64),  INT8_C(  95), -INT8_C( 115), -INT8_C(  91),  INT8_C(  38),
         INT8_C(  96),  INT8_C(  13),  INT8_C(  93),  INT8_C(  14),  INT8_C( 107), -INT8_C(  90),  INT8_C(  99),  INT8_C(  71) },
      { -INT8_C( 110), -INT8_C( 111), -INT8_C(  31), -INT8_C(   4),  INT8_C(  67),  INT8_C(  93), -INT8_C(   1),      INT8_MAX,
        -INT8_C( 105), -INT8_C(  33),  INT8_C(  30), -INT8_C(  40),  INT8_C(  11), -INT8_C( 124), -INT8_C(  80),  INT8_C(  18),
         INT8_C( 105),  INT8_C(  64), -INT8_C(   9),  INT8_C( 116), -INT8_C(  97), -INT8_C(  95),  INT8_C(   6), -INT8_C(  94),
         INT8_C(  64), -INT8_C(  46),  INT8_C(  55),  INT8_C(  98),  INT8_C(  21), -INT8_C(  72), -INT8_C( 125),  INT8_C(  46) } },
    { { -INT8_C(  92), -INT8_C(  56), -INT8_C(  56), -INT8_C(   3),  INT8_C( 106), -INT8_C( 125), -INT8_C(  24), -INT8_C( 120),
        -INT8_C(  42), -INT8_C(   1),  INT8_C(  74),  INT8_C(  47),  INT8_C(  71), -INT8_C(  15), -INT8_C(   8),  INT8_C(  57),
         INT8_C( 109),  INT8_C(  68), -INT8_C(  69),  INT8_C(  98), -INT8_C(  52),  INT8_C(  55),  INT8_C(  55),  INT8_C( 101),
        -INT8_C(  58),  INT8_C( 121), -INT8_C(  76), -INT8_C( 112),  INT8_C(  50), -INT8_C(  36), -INT8_C(  36),  INT8_C(  96) },
      UINT32_C(     33993),
      { -INT8_C(   4), -INT8_C(   8),  INT8_C(  72),  INT8_C( 121),  INT8_C(  22), -INT8_C(  61), -INT8_C(  95),  INT8_C(  91),
         INT8_C(  13), -INT8_C( 124),  INT8_C(  34), -INT8_C(  59),  INT8_C(  96),  INT8_C(  14),  INT8_C(  20),  INT8_C(  69),
         INT8_C(  18),  INT8_C(  37),  INT8_C(  87), -INT8_C(  61), -INT8_C(  47), -INT8_C(  17),  INT8_C(  37), -INT8_C(  73),
        -INT8_C(  66),  INT8_C(  43), -INT8_C(  63),  INT8_C( 101), -INT8_C( 107), -INT8_C(  56), -INT8_C(  72), -INT8_C(  87) },
      {  INT8_C(  82),  INT8_C( 121), -INT8_C(  76),  INT8_C(  57), -INT8_C(  81),  INT8_C(  56), -INT8_C(  72),  INT8_C(  67),
         INT8_C(  58),  INT8_C(  91),  INT8_C(  66), -INT8_C(  59), -INT8_C(  79), -INT8_C(  29), -INT8_C(  96),      INT8_MIN,
         INT8_C(  89), -INT8_C(  52), -INT8_C(  48), -INT8_C( 102), -INT8_C(  77), -INT8_C(  68),  INT8_C(   9),  INT8_C( 120),
         INT8_C(  28), -INT8_C(  42),  INT8_C(  38), -INT8_C(  89), -INT8_C(  78),  INT8_C( 109),  INT8_C(  34),  INT8_C(  43) },
      {  INT8_C(  78), -INT8_C(  56), -INT8_C(  56),      INT8_MAX,  INT8_C( 106), -INT8_C( 125),      INT8_MIN,      INT8_MAX,
        -INT8_C(  42), -INT8_C(   1),  INT8_C( 100),  INT8_C(  47),  INT8_C(  71), -INT8_C(  15), -INT8_C(   8), -INT8_C(  59),
         INT8_C( 109),  INT8_C(  68), -INT8_C(  69),  INT8_C(  98), -INT8_C(  52),  INT8_C(  55),  INT8_C(  55),  INT8_C( 101),
        -INT8_C(  58),  INT8_C( 121), -INT8_C(  76), -INT8_C( 112),  INT8_C(  50), -INT8_C(  36), -INT8_C(  36),  INT8_C(  96) } },
    { { -INT8_C( 119),  INT8_C(   0),  INT8_C(  65), -INT8_C(  53), -INT8_C(  18),  INT8_C(  41),  INT8_C(  12),  INT8_C(  72),
        -INT8_C(  86),  INT8_C(  39),  INT8_C(  80), -INT8_C(  86), -INT8_C(  98),  INT8_C(  71), -INT8_C(  10),  INT8_C(  26),
         INT8_C(   2),  INT8_C(   2),  INT8_C(  81), -INT8_C(  15), -INT8_C( 127),  INT8_C(  89),  INT8_C( 120), -INT8_C( 108),
         INT8_C(  37),  INT8_C(  34),  INT8_C(  62),  INT8_C(   1),  INT8_C(  36),  INT8_C(  86),  INT8_C( 109),  INT8_C( 112) },
      UINT32_C(      9379),
      {  INT8_C( 104),  INT8_C(  79),  INT8_C(  45),  INT8_C(  74), -INT8_C(   6), -INT8_C(  53),  INT8_C(  39), -INT8_C(  66),
         INT8_C(  44), -INT8_C(  68),  INT8_C(  67),  INT8_C(  50), -INT8_C(  91), -INT8_C(  88),  INT8_C(  82), -INT8_C(  87),
         INT8_C(  16),  INT8_C(  37),  INT8_C(  87), -INT8_C(  66), -INT8_C(  77),  INT8_C(  46),  INT8_C(  25),  INT8_C(  15),
         INT8_C(  45), -INT8_C(  13), -INT8_C(  39), -INT8_C( 113),  INT8_C(  16),  INT8_C(  13), -INT8_C(  73), -INT8_C( 102) },
      {  INT8_C(  96),  INT8_C(  32),  INT8_C(  87), -INT8_C(  68),  INT8_C( 115), -INT8_C(  15), -INT8_C(  38), -INT8_C(  33),
        -INT8_C(  61), -INT8_C( 113), -INT8_C( 118),  INT8_C(  12),  INT8_C(  78), -INT8_C(  29), -INT8_C(  43),  INT8_C(  48),
        -INT8_C(  55),  INT8_C(  71), -INT8_C(  43),  INT8_C(  21), -INT8_C(  98), -INT8_C(  50),  INT8_C( 121),  INT8_C( 116),
         INT8_C(  65),  INT8_C(  79),      INT8_MAX,  INT8_C(  62),  INT8_C(   2),  INT8_C( 115),  INT8_C(  10),  INT8_C(  17) },
      {      INT8_MAX,  INT8_C( 111),  INT8_C(  65), -INT8_C(  53), -INT8_C(  18), -INT8_C(  68),  INT8_C(  12), -INT8_C(  99),
        -INT8_C(  86),  INT8_C(  39), -INT8_C(  51), -INT8_C(  86), -INT8_C(  98), -INT8_C( 117), -INT8_C(  10),  INT8_C(  26),
         INT8_C(   2),  INT8_C(   2),  INT8_C(  81), -INT8_C(  15), -INT8_C( 127),  INT8_C(  89),  INT8_C( 120), -INT8_C( 108),
         INT8_C(  37),  INT8_C(  34),  INT8_C(  62),  INT8_C(   1),  INT8_C(  36),  INT8_C(  86),  INT8_C( 109),  INT8_C( 112) } },
    { {  INT8_C( 106), -INT8_C(  38),  INT8_C( 113), -INT8_C( 114), -INT8_C( 121),  INT8_C(  92), -INT8_C(  30),  INT8_C( 110),
         INT8_C(  56), -INT8_C(  93), -INT8_C(  61),  INT8_C(  32),  INT8_C(   8),  INT8_C(  89),  INT8_C(  67), -INT8_C( 108),
        -INT8_C(  52),  INT8_C( 104), -INT8_C(  67), -INT8_C(   2),  INT8_C( 105),  INT8_C(  96), -INT8_C(  60),  INT8_C(  83),
        -INT8_C( 123), -INT8_C(  16),  INT8_C(  90),  INT8_C(  53),  INT8_C(   8), -INT8_C(  83), -INT8_C(   2), -INT8_C(  72) },
      UINT32_C(     60777),
      {  INT8_C(  87),  INT8_C( 109), -INT8_C(  14), -INT8_C(  56), -INT8_C(  36),  INT8_C(  50),  INT8_C( 112), -INT8_C( 106),
         INT8_C( 115),  INT8_C(  42),  INT8_C( 109),  INT8_C(   2), -INT8_C(  46), -INT8_C(  67),  INT8_C(  26),  INT8_C( 102),
        -INT8_C( 104), -INT8_C( 104),  INT8_C(   5), -INT8_C(  69), -INT8_C(  12), -INT8_C(  38), -INT8_C( 100), -INT8_C(  59),
         INT8_C(  92), -INT8_C(  63),  INT8_C(  99),  INT8_C(  33),  INT8_C(  56), -INT8_C(  66), -INT8_C(  59), -INT8_C(  33) },
      { -INT8_C(  65),  INT8_C(  98),  INT8_C(  34),  INT8_C( 105),  INT8_C(  11),  INT8_C(  40), -INT8_C(  94),  INT8_C(  55),
         INT8_C(  93),  INT8_C(  32),  INT8_C(  16), -INT8_C( 123), -INT8_C(  53),  INT8_C( 118),  INT8_C(  76), -INT8_C(   7),
        -INT8_C( 126), -INT8_C(  96),  INT8_C( 111),      INT8_MIN, -INT8_C(  88), -INT8_C(  35),  INT8_C( 107),  INT8_C(  19),
         INT8_C(  54), -INT8_C(  31),  INT8_C(  97),  INT8_C(  15), -INT8_C(  66),  INT8_C( 116),  INT8_C(  96), -INT8_C(   1) },
      {  INT8_C(  22), -INT8_C(  38),  INT8_C( 113),  INT8_C(  49), -INT8_C( 121),  INT8_C(  90),  INT8_C(  18),  INT8_C( 110),
             INT8_MAX, -INT8_C(  93),  INT8_C( 125), -INT8_C( 121),  INT8_C(   8),  INT8_C(  51),  INT8_C( 102),  INT8_C(  95),
        -INT8_C(  52),  INT8_C( 104), -INT8_C(  67), -INT8_C(   2),  INT8_C( 105),  INT8_C(  96), -INT8_C(  60),  INT8_C(  83),
        -INT8_C( 123), -INT8_C(  16),  INT8_C(  90),  INT8_C(  53),  INT8_C(   8), -INT8_C(  83), -INT8_C(   2), -INT8_C(  72) } },
    { {  INT8_C( 100), -INT8_C(  55),      INT8_MIN,  INT8_C(  72),  INT8_C(  87),  INT8_C(  65),  INT8_C(  86), -INT8_C(  76),
         INT8_C(  80),  INT8_C( 106), -INT8_C( 121),      INT8_MAX, -INT8_C( 112), -INT8_C(  35), -INT8_C(   2), -INT8_C(  44),
         INT8_C(  49), -INT8_C(  82), -INT8_C(  22), -INT8_C(  13), -INT8_C(  17), -INT8_C(   8), -INT8_C(  14),  INT8_C( 108),
         INT8_C( 126),  INT8_C(  95), -INT8_C(  50), -INT8_C(  76),  INT8_C( 116), -INT8_C(   6),  INT8_C(  17), -INT8_C(  97) },
      UINT32_C(     17935),
      { -INT8_C(  36),  INT8_C( 107),  INT8_C(  99), -INT8_C(  88), -INT8_C( 108), -INT8_C(  54), -INT8_C( 107), -INT8_C(  60),
         INT8_C(  33),  INT8_C( 111),  INT8_C(  96),  INT8_C( 117),  INT8_C(  43), -INT8_C(  46), -INT8_C(  19), -INT8_C( 103),
        -INT8_C( 107), -INT8_C(  51),  INT8_C( 122),  INT8_C(  90),  INT8_C(  34), -INT8_C(  40), -INT8_C( 107),  INT8_C( 117),
         INT8_C(  95),  INT8_C(  30), -INT8_C(  30),  INT8_C( 112),  INT8_C(  86), -INT8_C(  46), -INT8_C(   2),  INT8_C( 109) },
      {  INT8_C(  69),  INT8_C(  50),  INT8_C( 112),  INT8_C(  36), -INT8_C(  68),  INT8_C(  41),  INT8_C(  61),  INT8_C(  95),
        -INT8_C(  71),  INT8_C(  65), -INT8_C(  90), -INT8_C(  46),  INT8_C(  10),  INT8_C(  89), -INT8_C(  35), -INT8_C(  68),
        -INT8_C(  54),  INT8_C(  48),      INT8_MIN, -INT8_C( 108), -INT8_C(  48), -INT8_C(  34), -INT8_C(  69), -INT8_C(  69),
         INT8_C(   0),  INT8_C(  32),  INT8_C(   8),  INT8_C(  21),  INT8_C(  10),  INT8_C(  48), -INT8_C(  60),  INT8_C(  55) },
      {  INT8_C(  33),      INT8_MAX,      INT8_MAX, -INT8_C(  52),  INT8_C(  87),  INT8_C(  65),  INT8_C(  86), -INT8_C(  76),
         INT8_C(  80),      INT8_MAX,  INT8_C(   6),      INT8_MAX, -INT8_C( 112), -INT8_C(  35), -INT8_C(  54), -INT8_C(  44),
         INT8_C(  49), -INT8_C(  82), -INT8_C(  22), -INT8_C(  13), -INT8_C(  17), -INT8_C(   8), -INT8_C(  14),  INT8_C( 108),
         INT8_C( 126),  INT8_C(  95), -INT8_C(  50), -INT8_C(  76),  INT8_C( 116), -INT8_C(   6),  INT8_C(  17), -INT8_C(  97) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi8(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epi8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi8(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_adds_epi8(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_maskz_adds_epi8 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask32 k;
    const int8_t a[32];
    const int8_t b[32];
    const int8_t r[32];
  } test_vec[] = {
    { UINT32_C(     14766),
      {  INT8_C(  55),  INT8_C(  62),  INT8_C(  61), -INT8_C(  18),  INT8_C(  70), -INT8_C(  29),  INT8_C(  26), -INT8_C(  17),
         INT8_C(  21),  INT8_C(  30), -INT8_C(  94), -INT8_C(   9),  INT8_C(   2),  INT8_C( 115), -INT8_C(  89), -INT8_C(  32),
         INT8_C(  70), -INT8_C( 105),  INT8_C(  70),  INT8_C(  70),  INT8_C(  61),  INT8_C(  10),  INT8_C(   9),  INT8_C(  43),
         INT8_C(  68), -INT8_C(  98), -INT8_C( 100), -INT8_C( 127), -INT8_C(  29),  INT8_C(  26), -INT8_C(  67), -INT8_C(  57) },
      {  INT8_C(  66),  INT8_C(  16),  INT8_C( 100), -INT8_C(  70),  INT8_C(  89), -INT8_C(  69), -INT8_C(  68), -INT8_C( 110),
         INT8_C(  64), -INT8_C( 113), -INT8_C(  40),  INT8_C(  48),  INT8_C(  79), -INT8_C(  27),  INT8_C(  25),  INT8_C(  42),
        -INT8_C(  19),  INT8_C(  67),  INT8_C(  63), -INT8_C(  22), -INT8_C( 125),  INT8_C(  59), -INT8_C(  74),  INT8_C(  64),
        -INT8_C(  96),  INT8_C(  66),  INT8_C(  71), -INT8_C(  80),  INT8_C(  23), -INT8_C( 102),  INT8_C(  65),  INT8_C( 126) },
      {  INT8_C(   0),  INT8_C(  78),      INT8_MAX, -INT8_C(  88),  INT8_C(   0), -INT8_C(  98),  INT8_C(   0), -INT8_C( 127),
         INT8_C(  85),  INT8_C(   0),  INT8_C(   0),  INT8_C(  39),  INT8_C(  81),  INT8_C(  88),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT32_C(     48711),
      { -INT8_C(  23), -INT8_C( 106),  INT8_C(  46), -INT8_C(  92), -INT8_C(  53), -INT8_C( 100), -INT8_C(  19), -INT8_C(  86),
        -INT8_C(   3),  INT8_C( 102),  INT8_C(  27),  INT8_C(  95),  INT8_C(  80),  INT8_C(  93),  INT8_C( 104), -INT8_C(  65),
        -INT8_C( 117), -INT8_C(  64),  INT8_C( 111), -INT8_C(  87), -INT8_C(  92),  INT8_C(  32),  INT8_C(  69), -INT8_C(  39),
         INT8_C(  50), -INT8_C(  78),  INT8_C(  69),  INT8_C(  71), -INT8_C(  10), -INT8_C(  95),  INT8_C(  19),  INT8_C(  78) },
      { -INT8_C(  13),  INT8_C(  64),  INT8_C(  95), -INT8_C( 107),  INT8_C(  32),  INT8_C(  21),  INT8_C( 116), -INT8_C(  18),
        -INT8_C(  61),  INT8_C(  88),  INT8_C( 111), -INT8_C(  60), -INT8_C(   5),  INT8_C(   0), -INT8_C(  87), -INT8_C(  89),
        -INT8_C(  75), -INT8_C( 119),  INT8_C( 100),  INT8_C(  56),  INT8_C( 112), -INT8_C( 120), -INT8_C( 115), -INT8_C(  17),
        -INT8_C(  19),  INT8_C(  30), -INT8_C(  93),  INT8_C(  18),  INT8_C(  40), -INT8_C(  94), -INT8_C(  36), -INT8_C(  32) },
      { -INT8_C(  36), -INT8_C(  42),      INT8_MAX,  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(  97),  INT8_C(   0),
         INT8_C(   0),      INT8_MAX,      INT8_MAX,  INT8_C(  35),  INT8_C(  75),  INT8_C(  93),  INT8_C(   0),      INT8_MIN,
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT32_C(     22629),
      {  INT8_C(   0),  INT8_C(  69), -INT8_C(  39), -INT8_C( 106), -INT8_C(  84),  INT8_C(  19),  INT8_C(  31), -INT8_C(  16),
        -INT8_C( 113), -INT8_C(  23),  INT8_C(  40),  INT8_C(  69), -INT8_C(   7), -INT8_C(  85), -INT8_C(  89), -INT8_C( 117),
         INT8_C(  70),  INT8_C(  44),  INT8_C( 115),  INT8_C(  20),  INT8_C(  16),  INT8_C(  45), -INT8_C(  30),  INT8_C(  36),
         INT8_C(  38), -INT8_C(  49),  INT8_C(  88),  INT8_C(  80),  INT8_C(  78),  INT8_C(  52),  INT8_C(  70),  INT8_C(  47) },
      { -INT8_C(  37), -INT8_C(  51), -INT8_C(  78), -INT8_C(  33),  INT8_C(  76),  INT8_C(  24),  INT8_C(   2),  INT8_C(  44),
         INT8_C(  68), -INT8_C(  42),  INT8_C(   1), -INT8_C(  82), -INT8_C(   7), -INT8_C( 101), -INT8_C( 121), -INT8_C(   5),
         INT8_C(  42),  INT8_C(  31), -INT8_C(  59), -INT8_C( 111),      INT8_MIN, -INT8_C(  74), -INT8_C(  80), -INT8_C(  43),
        -INT8_C(  92), -INT8_C(   5),  INT8_C(  56), -INT8_C(  57), -INT8_C(  14),  INT8_C(  35), -INT8_C(  94), -INT8_C(  10) },
      { -INT8_C(  37),  INT8_C(   0), -INT8_C( 117),  INT8_C(   0),  INT8_C(   0),  INT8_C(  43),  INT8_C(  33),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0), -INT8_C(  13), -INT8_C(  14),  INT8_C(   0),      INT8_MIN,  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT32_C(     34647),
      {  INT8_C(  22), -INT8_C( 101), -INT8_C(   1),  INT8_C(  44),  INT8_C(  87),  INT8_C(  24), -INT8_C(  84), -INT8_C(  58),
        -INT8_C(  20),  INT8_C(  81), -INT8_C(  60), -INT8_C(  24),  INT8_C(   6),  INT8_C(  70), -INT8_C(  48),  INT8_C( 111),
        -INT8_C(  65),  INT8_C(  60), -INT8_C(  59),  INT8_C(  20),  INT8_C(   6), -INT8_C(  65), -INT8_C(  88),  INT8_C( 103),
         INT8_C(  43),  INT8_C(  22),  INT8_C(  95),  INT8_C(  54),  INT8_C(  91),  INT8_C( 116),  INT8_C( 118),  INT8_C(   1) },
      { -INT8_C(  64), -INT8_C(  72), -INT8_C(  36), -INT8_C(  45), -INT8_C(  62), -INT8_C( 121),  INT8_C(  20),  INT8_C(  98),
         INT8_C(   9), -INT8_C(   3), -INT8_C(  33), -INT8_C(  57), -INT8_C(  77),  INT8_C(  54),  INT8_C(  14), -INT8_C( 101),
         INT8_C(  57), -INT8_C( 123), -INT8_C(  71),  INT8_C(  16),  INT8_C( 102),  INT8_C(   5), -INT8_C( 127),  INT8_C(  46),
         INT8_C( 112), -INT8_C(  64),  INT8_C(   3),  INT8_C(  32), -INT8_C(  70), -INT8_C(  90), -INT8_C(  38), -INT8_C(  60) },
      { -INT8_C(  42),      INT8_MIN, -INT8_C(  37),  INT8_C(   0),  INT8_C(  25),  INT8_C(   0), -INT8_C(  64),  INT8_C(   0),
        -INT8_C(  11),  INT8_C(  78), -INT8_C(  93),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(  10),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT32_C(     33178),
      { -INT8_C( 112), -INT8_C( 114), -INT8_C(  27),  INT8_C(  55),  INT8_C(  41),  INT8_C(  10),  INT8_C( 103), -INT8_C( 121),
         INT8_C(  66), -INT8_C( 119),  INT8_C(  82), -INT8_C( 125), -INT8_C( 120),  INT8_C(   5),  INT8_C(  41), -INT8_C(  55),
        -INT8_C( 114),  INT8_C( 124), -INT8_C( 115),  INT8_C(  57),  INT8_C(  20), -INT8_C(  84), -INT8_C(   7),  INT8_C( 102),
        -INT8_C(  77), -INT8_C(   5), -INT8_C(  59), -INT8_C(  62),  INT8_C( 124),  INT8_C(  14),  INT8_C(  57),  INT8_C(  94) },
      { -INT8_C( 126),  INT8_C(  83),  INT8_C(   7),  INT8_C(  67),  INT8_C(  33),  INT8_C(  82), -INT8_C(  24),  INT8_C(  96),
        -INT8_C( 112), -INT8_C(  43),  INT8_C(  36),  INT8_C(  36), -INT8_C(  99), -INT8_C(  23),  INT8_C(  74),  INT8_C(  27),
        -INT8_C(  72),  INT8_C(  23), -INT8_C(  17), -INT8_C(   4), -INT8_C(  25), -INT8_C(  84), -INT8_C(  55), -INT8_C( 115),
        -INT8_C(  36),  INT8_C(  16),  INT8_C(  33),  INT8_C(  13),  INT8_C(   3),  INT8_C(  29), -INT8_C(  53),      INT8_MAX },
      {  INT8_C(   0), -INT8_C(  31),  INT8_C(   0),  INT8_C( 122),  INT8_C(  74),  INT8_C(   0),  INT8_C(   0), -INT8_C(  25),
        -INT8_C(  46),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0), -INT8_C(  28),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT32_C(     23935),
      {  INT8_C(  37),  INT8_C(  27),  INT8_C(  77), -INT8_C( 113),  INT8_C(   4), -INT8_C(  36), -INT8_C(   1),  INT8_C( 117),
        -INT8_C(  77),  INT8_C(  42),  INT8_C( 104),  INT8_C( 109), -INT8_C(   9),  INT8_C(  94),  INT8_C(  23),  INT8_C(  25),
        -INT8_C(  69), -INT8_C(  80), -INT8_C(  76),  INT8_C(  22), -INT8_C(  44), -INT8_C(  48),  INT8_C( 100), -INT8_C(  87),
         INT8_C(  73),  INT8_C(  64),  INT8_C(  74),  INT8_C(  77), -INT8_C( 104),  INT8_C( 109),  INT8_C(  73), -INT8_C( 100) },
      {  INT8_C(  34),  INT8_C(  89),  INT8_C(  99),  INT8_C(  37),  INT8_C(   2),  INT8_C(  58),  INT8_C(  49),  INT8_C(  10),
         INT8_C(  72), -INT8_C(  35),  INT8_C( 120),  INT8_C(  43),  INT8_C(  84), -INT8_C(  59), -INT8_C(  72),  INT8_C( 106),
         INT8_C(  33),  INT8_C(  86),  INT8_C(  69), -INT8_C(  42), -INT8_C(  14),  INT8_C(  32),  INT8_C( 113), -INT8_C( 101),
        -INT8_C( 108), -INT8_C(  77),  INT8_C( 126), -INT8_C(  15),  INT8_C(  63),  INT8_C( 113),  INT8_C( 108), -INT8_C( 117) },
      {  INT8_C(  71),  INT8_C( 116),      INT8_MAX, -INT8_C(  76),  INT8_C(   6),  INT8_C(  22),  INT8_C(  48),  INT8_C(   0),
        -INT8_C(   5),  INT8_C(   0),      INT8_MAX,      INT8_MAX,  INT8_C(  75),  INT8_C(   0), -INT8_C(  49),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT32_C(     30325),
      { -INT8_C(  38),  INT8_C(  61), -INT8_C(  91),  INT8_C(  83), -INT8_C(  95), -INT8_C(  17),  INT8_C(  39), -INT8_C(  69),
        -INT8_C(  65), -INT8_C(  37), -INT8_C( 112), -INT8_C( 103),  INT8_C(  54), -INT8_C(  77),  INT8_C(  39), -INT8_C(  98),
        -INT8_C(  45), -INT8_C( 124), -INT8_C(  82), -INT8_C( 119), -INT8_C(  44), -INT8_C(  17), -INT8_C(  83), -INT8_C(  33),
         INT8_C(  89), -INT8_C(  29),  INT8_C( 105),  INT8_C(  39), -INT8_C(  32), -INT8_C(  38),  INT8_C(  98), -INT8_C(  47) },
      { -INT8_C( 109),  INT8_C( 105), -INT8_C(  21),  INT8_C(  27),  INT8_C(   2), -INT8_C( 112), -INT8_C(  93),  INT8_C(  15),
        -INT8_C(  75), -INT8_C(   2),  INT8_C(  65), -INT8_C( 112), -INT8_C(   6),  INT8_C(  59), -INT8_C(  94),  INT8_C( 123),
         INT8_C(  35), -INT8_C(  22), -INT8_C(  85),  INT8_C( 118),  INT8_C(  60),  INT8_C(  28),  INT8_C(  72),  INT8_C( 106),
        -INT8_C( 104), -INT8_C(  83), -INT8_C(  41), -INT8_C(  31),  INT8_C(  80),  INT8_C(  85), -INT8_C(  69),  INT8_C(  37) },
      {      INT8_MIN,  INT8_C(   0), -INT8_C( 112),  INT8_C(   0), -INT8_C(  93),      INT8_MIN, -INT8_C(  54),  INT8_C(   0),
         INT8_C(   0), -INT8_C(  39), -INT8_C(  47),  INT8_C(   0),  INT8_C(  48), -INT8_C(  18), -INT8_C(  55),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } },
    { UINT32_C(     38151),
      {  INT8_C(  24),  INT8_C(  22),  INT8_C( 106),  INT8_C(  19), -INT8_C(  11),  INT8_C(  73), -INT8_C(  76), -INT8_C( 100),
        -INT8_C(  66), -INT8_C(  18),  INT8_C(   0),  INT8_C( 116), -INT8_C(  99),  INT8_C(   9), -INT8_C( 107), -INT8_C(  41),
         INT8_C(   2), -INT8_C(  51), -INT8_C( 119), -INT8_C(  50),  INT8_C(  52), -INT8_C(  79), -INT8_C( 112), -INT8_C( 105),
         INT8_C( 121), -INT8_C(   1), -INT8_C(  33), -INT8_C(  59), -INT8_C(  40),  INT8_C( 113), -INT8_C( 108), -INT8_C(  42) },
      { -INT8_C( 111), -INT8_C(   9), -INT8_C(  21),      INT8_MIN,  INT8_C(  68), -INT8_C(  45), -INT8_C(  26),  INT8_C(  55),
         INT8_C(  54), -INT8_C( 115), -INT8_C(  52), -INT8_C( 115),  INT8_C(  11), -INT8_C(  27),  INT8_C(  14),  INT8_C(  69),
         INT8_C( 105), -INT8_C(  79), -INT8_C(  62), -INT8_C(   2),  INT8_C(  99),  INT8_C(  81), -INT8_C( 127),  INT8_C(   0),
        -INT8_C(  25),  INT8_C(  66),  INT8_C(  49), -INT8_C(  21),  INT8_C(   3),  INT8_C( 110), -INT8_C(  88), -INT8_C(  74) },
      { -INT8_C(  87),  INT8_C(  13),  INT8_C(  85),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
        -INT8_C(  12),  INT8_C(   0), -INT8_C(  52),  INT8_C(   0), -INT8_C(  88),  INT8_C(   0),  INT8_C(   0),  INT8_C(  28),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),
         INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0),  INT8_C(   0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi8(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_adds_epi8(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epi8(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_adds_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t src[16];
    const simde__mmask16 k;
    const int16_t a[16];
    const int16_t b[16];
    const int16_t r[16];
  } test_vec[] = {
    { { -INT16_C( 13954), -INT16_C( 13566),  INT16_C( 13346), -INT16_C( 19324),  INT16_C( 20767), -INT16_C( 25591), -INT16_C( 15670),  INT16_C( 22319),
        -INT16_C( 15735),  INT16_C( 15143),  INT16_C( 21746), -INT16_C( 32557),  INT16_C( 16862),  INT16_C( 27014), -INT16_C(  1471), -INT16_C( 16462) },
      UINT16_C(46276),
      { -INT16_C(  6518),  INT16_C(  3817),  INT16_C(  2202), -INT16_C( 23713),  INT16_C( 10660), -INT16_C( 11419), -INT16_C(  4480), -INT16_C( 22379),
        -INT16_C( 30679), -INT16_C(   516), -INT16_C(  9720), -INT16_C( 28866),  INT16_C( 32579), -INT16_C(  2679),  INT16_C( 19774), -INT16_C( 13910) },
      { -INT16_C( 27853), -INT16_C( 12841),  INT16_C( 14235),  INT16_C( 16240), -INT16_C( 10912), -INT16_C(  7917), -INT16_C( 22332), -INT16_C(  4727),
        -INT16_C( 31440),  INT16_C( 14826),  INT16_C( 10591), -INT16_C( 23864),  INT16_C( 20904), -INT16_C(  6249),  INT16_C( 16799), -INT16_C( 11600) },
      { -INT16_C( 13954), -INT16_C( 13566),  INT16_C( 16437), -INT16_C( 19324),  INT16_C( 20767), -INT16_C( 25591), -INT16_C( 26812), -INT16_C( 27106),
        -INT16_C( 15735),  INT16_C( 15143),  INT16_C(   871), -INT16_C( 32557),       INT16_MAX, -INT16_C(  8928), -INT16_C(  1471), -INT16_C( 25510) } },
    { { -INT16_C( 30764),  INT16_C( 28576),  INT16_C(  4286),  INT16_C(  8111), -INT16_C( 15642), -INT16_C( 22016), -INT16_C( 30358), -INT16_C( 25705),
        -INT16_C( 32242),  INT16_C( 28116), -INT16_C( 25429),  INT16_C( 21263), -INT16_C( 22803), -INT16_C( 29638), -INT16_C(  5400), -INT16_C( 17313) },
      UINT16_C(65394),
      {  INT16_C( 12332), -INT16_C(  9457), -INT16_C(  2737),  INT16_C( 20381),  INT16_C(  1951),  INT16_C( 14296), -INT16_C(  6494),  INT16_C( 30393),
         INT16_C( 25683),  INT16_C( 25106),  INT16_C(   183), -INT16_C(  3575), -INT16_C(  3700), -INT16_C(  5156),  INT16_C( 20141), -INT16_C(  9750) },
      { -INT16_C(  1409), -INT16_C( 12620),  INT16_C( 20975), -INT16_C( 28898), -INT16_C(  2471), -INT16_C(  1082),  INT16_C( 32733),  INT16_C( 12402),
        -INT16_C( 31517), -INT16_C( 25965), -INT16_C( 25468),  INT16_C(  4492),  INT16_C( 27021),  INT16_C( 15100), -INT16_C(  6217),  INT16_C( 13844) },
      { -INT16_C( 30764), -INT16_C( 22077),  INT16_C(  4286),  INT16_C(  8111), -INT16_C(   520),  INT16_C( 13214),  INT16_C( 26239), -INT16_C( 25705),
        -INT16_C(  5834), -INT16_C(   859), -INT16_C( 25285),  INT16_C(   917),  INT16_C( 23321),  INT16_C(  9944),  INT16_C( 13924),  INT16_C(  4094) } },
    { { -INT16_C( 14111), -INT16_C( 12283),  INT16_C(  8986),  INT16_C( 29535),  INT16_C(  9497), -INT16_C(  2450), -INT16_C(  8028), -INT16_C( 30937),
        -INT16_C( 17819), -INT16_C(  5854), -INT16_C( 20906), -INT16_C(  7174), -INT16_C(  2281), -INT16_C( 12515),  INT16_C( 12766), -INT16_C( 16635) },
      UINT16_C( 2810),
      {  INT16_C(  5263), -INT16_C(  4307),  INT16_C( 18311), -INT16_C(  2796), -INT16_C( 18115),  INT16_C( 25814),  INT16_C( 15168),  INT16_C( 25118),
         INT16_C( 29732),  INT16_C(  7953),  INT16_C( 10327),  INT16_C( 29974), -INT16_C(  2825), -INT16_C(   602), -INT16_C( 24397),  INT16_C( 16903) },
      {  INT16_C( 13748),  INT16_C( 15153),  INT16_C( 18044), -INT16_C( 18127),  INT16_C(  2047),  INT16_C( 16158),  INT16_C( 15426),  INT16_C( 26274),
        -INT16_C( 19535),  INT16_C(  2181), -INT16_C( 25637), -INT16_C( 11395),  INT16_C(  9359),  INT16_C( 17104), -INT16_C( 10300),  INT16_C( 31109) },
      { -INT16_C( 14111),  INT16_C( 10846),  INT16_C(  8986), -INT16_C( 20923), -INT16_C( 16068),       INT16_MAX,  INT16_C( 30594),       INT16_MAX,
        -INT16_C( 17819),  INT16_C( 10134), -INT16_C( 20906),  INT16_C( 18579), -INT16_C(  2281), -INT16_C( 12515),  INT16_C( 12766), -INT16_C( 16635) } },
    { { -INT16_C( 18932), -INT16_C( 30540), -INT16_C(  6660), -INT16_C(  1214),  INT16_C( 24812),  INT16_C( 11835), -INT16_C(  8804),  INT16_C( 19861),
         INT16_C(  6800),  INT16_C( 27478), -INT16_C( 11338),  INT16_C( 17726),  INT16_C(  3831), -INT16_C( 17272),  INT16_C(  3558), -INT16_C(  3531) },
      UINT16_C(59843),
      { -INT16_C( 16261), -INT16_C( 16945), -INT16_C( 17477), -INT16_C(  2531), -INT16_C( 17942),  INT16_C( 32723),  INT16_C( 25351),  INT16_C( 23961),
         INT16_C( 20431),  INT16_C(  3376),  INT16_C( 10389),  INT16_C(  7452),  INT16_C(   740),  INT16_C(  6442), -INT16_C(  4620),  INT16_C( 28418) },
      { -INT16_C( 11859),  INT16_C( 26924),  INT16_C( 18829),  INT16_C( 30559),  INT16_C( 13059),  INT16_C(  2806), -INT16_C( 28778),  INT16_C( 25959),
        -INT16_C( 26657),  INT16_C( 29811), -INT16_C( 28737), -INT16_C( 23663), -INT16_C( 17519), -INT16_C( 31300), -INT16_C( 16472),  INT16_C( 22261) },
      { -INT16_C( 28120),  INT16_C(  9979), -INT16_C(  6660), -INT16_C(  1214),  INT16_C( 24812),  INT16_C( 11835), -INT16_C(  3427),       INT16_MAX,
        -INT16_C(  6226),  INT16_C( 27478), -INT16_C( 11338), -INT16_C( 16211),  INT16_C(  3831), -INT16_C( 24858), -INT16_C( 21092),       INT16_MAX } },
    { {  INT16_C(  8592),  INT16_C(  7615),  INT16_C(  7787),  INT16_C( 28308), -INT16_C( 30127), -INT16_C(  6024), -INT16_C(  8422), -INT16_C(  1715),
        -INT16_C( 16266),  INT16_C( 13933), -INT16_C(   433), -INT16_C(  7975), -INT16_C( 26951),  INT16_C( 24934),  INT16_C( 23381), -INT16_C(  6729) },
      UINT16_C(30332),
      { -INT16_C(  6397), -INT16_C( 26731), -INT16_C(  6571), -INT16_C( 13022),  INT16_C( 15566),  INT16_C(  7340),  INT16_C(  9013), -INT16_C( 23844),
         INT16_C( 11353),  INT16_C( 12960),  INT16_C( 22796),  INT16_C( 29384),  INT16_C(  7610),  INT16_C( 29389),  INT16_C( 18947),  INT16_C(  1768) },
      {  INT16_C( 32049), -INT16_C( 30819), -INT16_C( 16540),  INT16_C( 12884),  INT16_C(   507),  INT16_C( 12366),  INT16_C( 11044),  INT16_C( 32210),
         INT16_C( 29271),  INT16_C( 25519),  INT16_C( 30923), -INT16_C( 31018), -INT16_C( 23659), -INT16_C( 26376), -INT16_C(  7955),  INT16_C(  8094) },
      {  INT16_C(  8592),  INT16_C(  7615), -INT16_C( 23111), -INT16_C(   138),  INT16_C( 16073),  INT16_C( 19706),  INT16_C( 20057), -INT16_C(  1715),
        -INT16_C( 16266),       INT16_MAX,       INT16_MAX, -INT16_C(  7975), -INT16_C( 16049),  INT16_C(  3013),  INT16_C( 10992), -INT16_C(  6729) } },
    { {  INT16_C( 15454), -INT16_C( 15706), -INT16_C(  1285), -INT16_C(  2060),  INT16_C( 17403),  INT16_C(  7975), -INT16_C(  1426), -INT16_C( 14948),
         INT16_C( 19564),  INT16_C( 14376), -INT16_C(   316),  INT16_C( 22974), -INT16_C( 18782), -INT16_C( 28686), -INT16_C( 28522), -INT16_C(  2898) },
      UINT16_C(21708),
      { -INT16_C( 14154), -INT16_C( 21681),  INT16_C( 19135), -INT16_C(  6418),  INT16_C( 23658),  INT16_C(  1760),  INT16_C( 19745),  INT16_C( 18770),
         INT16_C(  5765),  INT16_C( 17224), -INT16_C(  5520),  INT16_C( 25337), -INT16_C( 28807),  INT16_C( 10482), -INT16_C( 16508),  INT16_C( 14972) },
      { -INT16_C( 13433),  INT16_C( 18149), -INT16_C( 11498), -INT16_C( 32724),  INT16_C(  3375),  INT16_C( 20614), -INT16_C(  9894), -INT16_C(  8294),
        -INT16_C(  7441),  INT16_C( 24354),  INT16_C(  7116),  INT16_C( 17857), -INT16_C( 19286),  INT16_C( 11885), -INT16_C(  5517), -INT16_C(  1431) },
      {  INT16_C( 15454), -INT16_C( 15706),  INT16_C(  7637),       INT16_MIN,  INT16_C( 17403),  INT16_C(  7975),  INT16_C(  9851),  INT16_C( 10476),
         INT16_C( 19564),  INT16_C( 14376),  INT16_C(  1596),  INT16_C( 22974),       INT16_MIN, -INT16_C( 28686), -INT16_C( 22025), -INT16_C(  2898) } },
    { {  INT16_C( 20149), -INT16_C( 13504),  INT16_C( 27682),  INT16_C( 20811), -INT16_C( 11655), -INT16_C( 11358),  INT16_C( 15531), -INT16_C( 25934),
        -INT16_C( 11234), -INT16_C(  5382), -INT16_C( 17425), -INT16_C( 26065), -INT16_C( 25233), -INT16_C(  7480),  INT16_C( 12679),  INT16_C( 15580) },
      UINT16_C( 7296),
      { -INT16_C( 24056),  INT16_C( 21385),  INT16_C(   755), -INT16_C( 27355), -INT16_C( 12074), -INT16_C( 30511), -INT16_C(  4245),  INT16_C( 25949),
         INT16_C( 19673),  INT16_C(  2336), -INT16_C( 28442), -INT16_C( 20570),  INT16_C( 11634),  INT16_C( 20448),  INT16_C( 24681),  INT16_C( 29035) },
      { -INT16_C(  3070), -INT16_C(  2363), -INT16_C(  5385), -INT16_C( 12917),  INT16_C( 23995),  INT16_C(  9813), -INT16_C( 19892),  INT16_C(  9867),
        -INT16_C( 21505), -INT16_C(  6865), -INT16_C( 10949), -INT16_C( 20844),  INT16_C( 29954),  INT16_C( 27645),  INT16_C( 26837), -INT16_C( 10019) },
      {  INT16_C( 20149), -INT16_C( 13504),  INT16_C( 27682),  INT16_C( 20811), -INT16_C( 11655), -INT16_C( 11358),  INT16_C( 15531),       INT16_MAX,
        -INT16_C( 11234), -INT16_C(  5382),       INT16_MIN,       INT16_MIN,       INT16_MAX, -INT16_C(  7480),  INT16_C( 12679),  INT16_C( 15580) } },
    { { -INT16_C( 23971),  INT16_C( 21710),  INT16_C( 22924),  INT16_C( 18209),  INT16_C( 30390),  INT16_C(   877), -INT16_C(  2007),  INT16_C( 10281),
         INT16_C( 22692), -INT16_C(  8435), -INT16_C( 24019),  INT16_C( 12173), -INT16_C( 30185), -INT16_C(  4966),  INT16_C( 30707),  INT16_C( 20676) },
      UINT16_C(37401),
      { -INT16_C( 22876), -INT16_C( 14868), -INT16_C( 23827),  INT16_C( 23355),  INT16_C( 25765), -INT16_C( 12717), -INT16_C(  2164), -INT16_C( 26074),
         INT16_C( 21463),  INT16_C( 25660),  INT16_C( 21378),  INT16_C(  7663), -INT16_C(  7617),  INT16_C(  1172), -INT16_C( 20942), -INT16_C( 10602) },
      { -INT16_C( 32172),  INT16_C( 16795), -INT16_C( 10715), -INT16_C( 13668), -INT16_C(  4037), -INT16_C( 14439), -INT16_C( 16409), -INT16_C( 16799),
        -INT16_C( 25325), -INT16_C( 27357),  INT16_C(  4848),  INT16_C( 12466),  INT16_C( 18420),  INT16_C(  9780), -INT16_C( 13579),  INT16_C( 18940) },
      {       INT16_MIN,  INT16_C( 21710),  INT16_C( 22924),  INT16_C(  9687),  INT16_C( 21728),  INT16_C(   877), -INT16_C(  2007),  INT16_C( 10281),
         INT16_C( 22692), -INT16_C(  1697), -INT16_C( 24019),  INT16_C( 12173),  INT16_C( 10803), -INT16_C(  4966),  INT16_C( 30707),  INT16_C(  8338) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i src = simde_x_mm256_loadu_epi16(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_adds_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_maskz_adds_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const int16_t a[16];
    const int16_t b[16];
    const int16_t r[16];
  } test_vec[] = {
    { UINT16_C(31891),
      { -INT16_C( 23134), -INT16_C( 31842),  INT16_C( 15981),  INT16_C( 19589), -INT16_C( 21450), -INT16_C( 21374), -INT16_C(  7737), -INT16_C(   739),
        -INT16_C( 31345),  INT16_C( 27262),  INT16_C( 24141), -INT16_C(   681),  INT16_C( 10726), -INT16_C( 29885), -INT16_C( 10531),  INT16_C( 32519) },
      { -INT16_C( 23173), -INT16_C(  5886), -INT16_C( 30749),  INT16_C(  6453), -INT16_C( 18636), -INT16_C(  1082), -INT16_C(  7272),  INT16_C( 10232),
         INT16_C( 30312), -INT16_C( 19055), -INT16_C(  5932), -INT16_C( 17486), -INT16_C(  2799), -INT16_C(  4538),  INT16_C( 19915),  INT16_C( 18286) },
      {       INT16_MIN,       INT16_MIN,  INT16_C(     0),  INT16_C(     0),       INT16_MIN,  INT16_C(     0),  INT16_C(     0),  INT16_C(  9493),
         INT16_C(     0),  INT16_C(     0),  INT16_C( 18209), -INT16_C( 18167),  INT16_C(  7927),       INT16_MIN,  INT16_C(  9384),  INT16_C(     0) } },
    { UINT16_C(28914),
      { -INT16_C( 10960),  INT16_C( 26104),  INT16_C( 11503), -INT16_C( 19172), -INT16_C( 19417),  INT16_C(  8344),  INT16_C(   475),  INT16_C( 27798),
         INT16_C( 27574),  INT16_C( 26964),  INT16_C( 25894),  INT16_C( 27742),  INT16_C( 10836), -INT16_C( 15687), -INT16_C( 21647), -INT16_C( 24270) },
      {  INT16_C( 10880),  INT16_C( 28422),  INT16_C(  8790),  INT16_C( 32292), -INT16_C( 16938), -INT16_C( 20066),  INT16_C( 13502),  INT16_C( 29725),
         INT16_C( 29087), -INT16_C( 14883),  INT16_C( 15574),  INT16_C( 10801), -INT16_C(  5530), -INT16_C( 10260),  INT16_C(  8085),  INT16_C(  5752) },
      {  INT16_C(     0),       INT16_MAX,  INT16_C(     0),  INT16_C(     0),       INT16_MIN, -INT16_C( 11722),  INT16_C( 13977),       INT16_MAX,
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(  5306), -INT16_C( 25947), -INT16_C( 13562),  INT16_C(     0) } },
    { UINT16_C(32329),
      { -INT16_C( 24443), -INT16_C( 21856),  INT16_C( 30238), -INT16_C( 17305),  INT16_C(  9511),  INT16_C( 17648), -INT16_C( 28519),  INT16_C( 30645),
        -INT16_C( 29867), -INT16_C( 30797),  INT16_C(  6582), -INT16_C( 23951),  INT16_C(  2032),  INT16_C( 26817),  INT16_C(  2845), -INT16_C( 23834) },
      { -INT16_C( 31061), -INT16_C( 14004), -INT16_C( 19460),  INT16_C(  9093),  INT16_C( 30168),  INT16_C( 29287),  INT16_C(  7173),  INT16_C( 23529),
        -INT16_C( 25433),  INT16_C( 24034),  INT16_C( 21429), -INT16_C( 23296), -INT16_C( 16038),  INT16_C( 30477), -INT16_C(  3124),  INT16_C( 30490) },
      {       INT16_MIN,  INT16_C(     0),  INT16_C(     0), -INT16_C(  8212),  INT16_C(     0),  INT16_C(     0), -INT16_C( 21346),  INT16_C(     0),
         INT16_C(     0), -INT16_C(  6763),  INT16_C( 28011),       INT16_MIN, -INT16_C( 14006),       INT16_MAX, -INT16_C(   279),  INT16_C(     0) } },
    { UINT16_C(26233),
      {  INT16_C( 30016), -INT16_C( 15078), -INT16_C(  3432), -INT16_C(   197),  INT16_C( 16484),  INT16_C( 19739), -INT16_C( 15717),  INT16_C( 32233),
        -INT16_C( 25056),  INT16_C(  8401),  INT16_C( 11075),  INT16_C( 20705), -INT16_C( 20829), -INT16_C( 17085), -INT16_C( 17371),  INT16_C( 26147) },
      {  INT16_C( 15665), -INT16_C( 14037),  INT16_C( 26160), -INT16_C( 27448), -INT16_C(  7257),  INT16_C( 17122), -INT16_C( 13402), -INT16_C( 14656),
        -INT16_C( 28310), -INT16_C( 21018), -INT16_C( 14404),  INT16_C( 24574),  INT16_C( 16757), -INT16_C( 25828),  INT16_C( 16638),  INT16_C( 12033) },
      {       INT16_MAX,  INT16_C(     0),  INT16_C(     0), -INT16_C( 27645),  INT16_C(  9227),       INT16_MAX, -INT16_C( 29119),  INT16_C(     0),
         INT16_C(     0), -INT16_C( 12617), -INT16_C(  3329),  INT16_C(     0),  INT16_C(     0),       INT16_MIN, -INT16_C(   733),  INT16_C(     0) } },
    { UINT16_C(11389),
      { -INT16_C( 20999), -INT16_C( 15981),  INT16_C( 14914),  INT16_C(  9381),  INT16_C( 19324),  INT16_C( 15599),  INT16_C( 22801), -INT16_C(  2099),
        -INT16_C( 30201),  INT16_C(  1470),  INT16_C( 13545),  INT16_C(  1606),  INT16_C( 17615), -INT16_C( 12218), -INT16_C( 15500),  INT16_C( 28156) },
      { -INT16_C( 28815), -INT16_C( 19666), -INT16_C( 11319),  INT16_C( 18135), -INT16_C( 14818),  INT16_C( 12162),  INT16_C( 20512),  INT16_C( 10022),
        -INT16_C(  6694), -INT16_C( 15572),  INT16_C( 29209), -INT16_C(  5943),  INT16_C(  4023),  INT16_C( 11192), -INT16_C( 19245),  INT16_C( 17560) },
      {       INT16_MIN,  INT16_C(     0),  INT16_C(  3595),  INT16_C( 27516),  INT16_C(  4506),  INT16_C( 27761),       INT16_MAX,  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),       INT16_MAX, -INT16_C(  4337),  INT16_C(     0), -INT16_C(  1026),  INT16_C(     0),  INT16_C(     0) } },
    { UINT16_C(50756),
      {  INT16_C(  3575), -INT16_C( 12646), -INT16_C( 18349), -INT16_C( 10604), -INT16_C( 19224),  INT16_C(  3622),  INT16_C(   219),  INT16_C(  2035),
         INT16_C(  3267), -INT16_C( 29318),  INT16_C( 12788), -INT16_C( 21348),  INT16_C( 28508), -INT16_C(  2975), -INT16_C( 23117), -INT16_C( 21830) },
      {  INT16_C( 21682),  INT16_C(  1656),  INT16_C(  3341), -INT16_C(  2596),  INT16_C(   705), -INT16_C( 25341), -INT16_C(  2302), -INT16_C( 14940),
         INT16_C(  7683), -INT16_C(  1966), -INT16_C(  4273), -INT16_C( 21596),  INT16_C(  1374),  INT16_C(  4767),  INT16_C( 23210),  INT16_C( 23996) },
      {  INT16_C(     0),  INT16_C(     0), -INT16_C( 15008),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C(  2083),  INT16_C(     0),
         INT16_C(     0), -INT16_C( 31284),  INT16_C(  8515),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(    93),  INT16_C(  2166) } },
    { UINT16_C(13742),
      { -INT16_C( 17565),  INT16_C( 16194),  INT16_C(   944), -INT16_C( 19391),  INT16_C( 17312),  INT16_C( 17835), -INT16_C( 20984),  INT16_C( 23395),
        -INT16_C( 19546),  INT16_C( 19274), -INT16_C( 22434), -INT16_C(   432), -INT16_C(  1094),  INT16_C( 30552),  INT16_C(  1624), -INT16_C( 17492) },
      { -INT16_C(  4414),  INT16_C( 29434),  INT16_C( 15345), -INT16_C( 28122), -INT16_C( 11906), -INT16_C( 31017),  INT16_C( 14976),  INT16_C(  9953),
         INT16_C( 11245),  INT16_C( 19569), -INT16_C( 15660), -INT16_C( 29110), -INT16_C( 23875),  INT16_C(  5381), -INT16_C( 20056),  INT16_C( 27344) },
      {  INT16_C(     0),       INT16_MAX,  INT16_C( 16289),       INT16_MIN,  INT16_C(     0), -INT16_C( 13182),  INT16_C(     0),       INT16_MAX,
        -INT16_C(  8301),  INT16_C(     0),       INT16_MIN,  INT16_C(     0), -INT16_C( 24969),       INT16_MAX,  INT16_C(     0),  INT16_C(     0) } },
    { UINT16_C(51871),
      { -INT16_C( 28195),  INT16_C(   773), -INT16_C( 31965), -INT16_C(  1323),  INT16_C( 21769), -INT16_C(  5324),  INT16_C(  8827), -INT16_C(  4842),
        -INT16_C(  5522), -INT16_C( 18257),  INT16_C( 27769),  INT16_C( 32346),  INT16_C(   641),  INT16_C( 20784), -INT16_C( 12435),  INT16_C( 18971) },
      {  INT16_C(  8288), -INT16_C( 31923),  INT16_C(  8867), -INT16_C( 21379), -INT16_C( 19849), -INT16_C(  3177), -INT16_C( 20780),  INT16_C( 17120),
        -INT16_C( 28776),  INT16_C(  4602),  INT16_C( 21755),  INT16_C( 31888), -INT16_C( 16298), -INT16_C( 15411), -INT16_C(  6001), -INT16_C(  4083) },
      { -INT16_C( 19907), -INT16_C( 31150), -INT16_C( 23098), -INT16_C( 22702),  INT16_C(  1920),  INT16_C(     0),  INT16_C(     0),  INT16_C( 12278),
         INT16_C(     0), -INT16_C( 13655),  INT16_C(     0),       INT16_MAX,  INT16_C(     0),  INT16_C(     0), -INT16_C( 18436),  INT16_C( 14888) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi16(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_adds_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm256_mask_adds_epu8(SIMDE_MUNIT_TEST_ARGS) {
#if 1
  const struct {
    uint8_t src[32];
    simde__mmask32 k;
    uint8_t a[32];
    uint8_t b[32];
    uint8_t r[32];
  } test_vec[8] = {
    { { UINT8_C(123), UINT8_C(  8), UINT8_C( 12), UINT8_C(  3), UINT8_C(138), UINT8_C(109), UINT8_C(121), UINT8_C( 76),
        UINT8_C(169), UINT8_C(140), UINT8_C( 66), UINT8_C( 51), UINT8_C(112), UINT8_C( 14), UINT8_C(235), UINT8_C(113),
        UINT8_C( 73), UINT8_C(152), UINT8_C(235), UINT8_C(222), UINT8_C(224), UINT8_C( 57), UINT8_C(110), UINT8_C(196),
        UINT8_C(219), UINT8_C(126), UINT8_C(217), UINT8_C(218), UINT8_C(221), UINT8_C( 50), UINT8_C( 77), UINT8_C( 88) },
      UINT32_C(3311098171),
      { UINT8_C(198), UINT8_C(212), UINT8_C( 17), UINT8_C(111), UINT8_C( 97), UINT8_C( 83), UINT8_C(162), UINT8_C(209),
        UINT8_C( 97), UINT8_C(141), UINT8_C( 66), UINT8_C(171), UINT8_C( 37), UINT8_C( 45), UINT8_C(137), UINT8_C(  6),
        UINT8_C(103), UINT8_C(247), UINT8_C(202), UINT8_C( 66), UINT8_C(117), UINT8_C(163), UINT8_C( 28), UINT8_C( 82),
        UINT8_C(213), UINT8_C(105), UINT8_C(170), UINT8_C( 16), UINT8_C(194), UINT8_C(  6), UINT8_C(213), UINT8_C(136) },
      { UINT8_C(218), UINT8_C(230), UINT8_C(247), UINT8_C( 59), UINT8_C( 57), UINT8_C(154), UINT8_C( 12), UINT8_C(155),
        UINT8_C( 39), UINT8_C( 78), UINT8_C( 70), UINT8_C( 77), UINT8_C(124), UINT8_C(207), UINT8_C( 83), UINT8_C(227),
        UINT8_C(198), UINT8_C( 29), UINT8_C( 37), UINT8_C( 60), UINT8_C(192), UINT8_C( 66), UINT8_C(142), UINT8_C(149),
        UINT8_C(171), UINT8_C( 57), UINT8_C(166), UINT8_C(110), UINT8_C( 63), UINT8_C(123), UINT8_C(246), UINT8_C( 25) },
      {    UINT8_MAX,    UINT8_MAX, UINT8_C( 12), UINT8_C(170), UINT8_C(154), UINT8_C(237), UINT8_C(121), UINT8_C( 76),
        UINT8_C(136), UINT8_C(140), UINT8_C( 66), UINT8_C(248), UINT8_C(161), UINT8_C( 14), UINT8_C(220), UINT8_C(113),
           UINT8_MAX,    UINT8_MAX, UINT8_C(235), UINT8_C(126),    UINT8_MAX, UINT8_C( 57), UINT8_C(170), UINT8_C(196),
           UINT8_MAX, UINT8_C(126),    UINT8_MAX, UINT8_C(218), UINT8_C(221), UINT8_C( 50),    UINT8_MAX, UINT8_C(161) } },
    { { UINT8_C(207), UINT8_C(222), UINT8_C( 89), UINT8_C( 69), UINT8_C( 31), UINT8_C( 64), UINT8_C( 63), UINT8_C( 67),
        UINT8_C(135), UINT8_C(170), UINT8_C(229), UINT8_C(214), UINT8_C(227), UINT8_C(134), UINT8_C(134), UINT8_C(154),
        UINT8_C( 48), UINT8_C( 39), UINT8_C( 14), UINT8_C(  6), UINT8_C(172), UINT8_C( 61), UINT8_C(110), UINT8_C(156),
        UINT8_C(  9), UINT8_C(208), UINT8_C(218), UINT8_C(180), UINT8_C( 28), UINT8_C(178), UINT8_C(100), UINT8_C(235) },
      UINT32_C(2955984272),
      { UINT8_C(254), UINT8_C(112), UINT8_C(243), UINT8_C(133), UINT8_C( 26), UINT8_C(217), UINT8_C( 91), UINT8_C(254),
        UINT8_C( 95), UINT8_C(225), UINT8_C(152), UINT8_C(144), UINT8_C(  8), UINT8_C(166), UINT8_C(150), UINT8_C(180),
        UINT8_C(227), UINT8_C(  5), UINT8_C( 80), UINT8_C(236), UINT8_C(213), UINT8_C( 43), UINT8_C(161), UINT8_C(242),
        UINT8_C(221), UINT8_C(  5), UINT8_C(221), UINT8_C(109), UINT8_C(194), UINT8_C( 14), UINT8_C( 29), UINT8_C(192) },
      { UINT8_C(126), UINT8_C( 17), UINT8_C( 69), UINT8_C(152), UINT8_C(234), UINT8_C(160), UINT8_C(150), UINT8_C( 73),
        UINT8_C(129), UINT8_C( 46), UINT8_C(217), UINT8_C(138), UINT8_C(212), UINT8_C(112), UINT8_C( 62), UINT8_C(184),
        UINT8_C(117), UINT8_C(143), UINT8_C(164), UINT8_C( 74), UINT8_C(186), UINT8_C( 69), UINT8_C( 60), UINT8_C(151),
        UINT8_C( 74), UINT8_C( 26), UINT8_C(  4), UINT8_C( 13), UINT8_C( 40), UINT8_C( 34), UINT8_C(205), UINT8_C(166) },
      { UINT8_C(207), UINT8_C(222), UINT8_C( 89), UINT8_C( 69),    UINT8_MAX, UINT8_C( 64), UINT8_C( 63),    UINT8_MAX,
        UINT8_C(224), UINT8_C(170),    UINT8_MAX,    UINT8_MAX, UINT8_C(220),    UINT8_MAX, UINT8_C(134),    UINT8_MAX,
        UINT8_C( 48), UINT8_C( 39), UINT8_C( 14), UINT8_C(  6),    UINT8_MAX, UINT8_C(112), UINT8_C(110), UINT8_C(156),
        UINT8_C(  9), UINT8_C(208), UINT8_C(218), UINT8_C(180), UINT8_C(234), UINT8_C( 48), UINT8_C(100),    UINT8_MAX } },
    { { UINT8_C( 51), UINT8_C( 19), UINT8_C( 62), UINT8_C( 29), UINT8_C(179), UINT8_C(213), UINT8_C(102), UINT8_C( 53),
        UINT8_C(  3), UINT8_C( 64), UINT8_C(191), UINT8_C(216), UINT8_C(176), UINT8_C(253), UINT8_C(144), UINT8_C( 37),
        UINT8_C(140), UINT8_C( 52), UINT8_C(111), UINT8_C( 70), UINT8_C(122), UINT8_C(172), UINT8_C(221), UINT8_C(196),
        UINT8_C(198), UINT8_C(226), UINT8_C(209), UINT8_C(238), UINT8_C(  4), UINT8_C(159), UINT8_C(148), UINT8_C( 55) },
      UINT32_C(1700057778),
      { UINT8_C(167), UINT8_C(186), UINT8_C(154), UINT8_C(171), UINT8_C(250), UINT8_C( 89), UINT8_C(131), UINT8_C(170),
        UINT8_C( 87), UINT8_C( 19), UINT8_C(207), UINT8_C(227), UINT8_C( 71), UINT8_C( 63), UINT8_C( 42), UINT8_C(193),
        UINT8_C(235), UINT8_C(  7), UINT8_C(134), UINT8_C(177), UINT8_C(233), UINT8_C( 87), UINT8_C(159), UINT8_C(237),
        UINT8_C(246), UINT8_C( 51), UINT8_C( 36), UINT8_C(168), UINT8_C(  5), UINT8_C(120), UINT8_C( 14), UINT8_C(173) },
      { UINT8_C( 51), UINT8_C(168), UINT8_C( 88), UINT8_C( 45), UINT8_C(  2), UINT8_C(219), UINT8_C(216), UINT8_C( 89),
        UINT8_C(238), UINT8_C(167), UINT8_C( 60), UINT8_C( 53), UINT8_C(230), UINT8_C(102), UINT8_C(247), UINT8_C(209),
        UINT8_C(110), UINT8_C(125), UINT8_C(130), UINT8_C( 87), UINT8_C(212), UINT8_C( 33), UINT8_C( 69), UINT8_C(203),
        UINT8_C( 84), UINT8_C(105), UINT8_C(115), UINT8_C( 90), UINT8_C(226), UINT8_C(129), UINT8_C(  7), UINT8_C( 21) },
      { UINT8_C( 51),    UINT8_MAX, UINT8_C( 62), UINT8_C( 29), UINT8_C(252),    UINT8_MAX, UINT8_C(102),    UINT8_MAX,
        UINT8_C(  3), UINT8_C(186), UINT8_C(191), UINT8_C(216),    UINT8_MAX, UINT8_C(253),    UINT8_MAX,    UINT8_MAX,
        UINT8_C(140), UINT8_C( 52),    UINT8_MAX, UINT8_C( 70),    UINT8_MAX, UINT8_C(172), UINT8_C(228), UINT8_C(196),
           UINT8_MAX, UINT8_C(226), UINT8_C(151), UINT8_C(238), UINT8_C(  4), UINT8_C(249), UINT8_C( 21), UINT8_C( 55) } },
    { { UINT8_C( 42), UINT8_C( 95), UINT8_C( 66), UINT8_C( 44), UINT8_C( 58), UINT8_C( 26), UINT8_C(133), UINT8_C( 40),
        UINT8_C(194), UINT8_C(193), UINT8_C( 93), UINT8_C(168), UINT8_C( 40), UINT8_C( 84), UINT8_C(122), UINT8_C(150),
        UINT8_C(209), UINT8_C(252), UINT8_C(237), UINT8_C(166), UINT8_C( 30), UINT8_C( 50), UINT8_C(113), UINT8_C(114),
        UINT8_C(156), UINT8_C(228), UINT8_C(204), UINT8_C(126), UINT8_C(102), UINT8_C(211), UINT8_C(147), UINT8_C(144) },
      UINT32_C(1824314674),
      { UINT8_C(240), UINT8_C( 65), UINT8_C(148), UINT8_C(178), UINT8_C(  2), UINT8_C(242), UINT8_C( 90), UINT8_C( 42),
        UINT8_C( 70), UINT8_C(212), UINT8_C(192), UINT8_C( 24), UINT8_C(209), UINT8_C(174), UINT8_C(190), UINT8_C(239),
        UINT8_C(224), UINT8_C( 47), UINT8_C( 97), UINT8_C(124), UINT8_C( 19), UINT8_C( 46), UINT8_C(250), UINT8_C(121),
        UINT8_C(  1), UINT8_C(141), UINT8_C(  9), UINT8_C( 52), UINT8_C( 99), UINT8_C(197), UINT8_C(160), UINT8_C( 83) },
      { UINT8_C(  6), UINT8_C( 53), UINT8_C(  5), UINT8_C(  9), UINT8_C( 39), UINT8_C( 95), UINT8_C( 51), UINT8_C(109),
        UINT8_C( 52), UINT8_C(244), UINT8_C(133), UINT8_C(  5), UINT8_C(162), UINT8_C( 67), UINT8_C(244), UINT8_C(130),
        UINT8_C(114), UINT8_C( 85),    UINT8_MAX, UINT8_C(134), UINT8_C(131), UINT8_C(249),    UINT8_MAX, UINT8_C(133),
        UINT8_C(135), UINT8_C(  9), UINT8_C(185), UINT8_C(234), UINT8_C(206), UINT8_C( 89), UINT8_C( 61), UINT8_C(213) },
      { UINT8_C( 42), UINT8_C(118), UINT8_C( 66), UINT8_C( 44), UINT8_C( 41),    UINT8_MAX, UINT8_C(133), UINT8_C( 40),
        UINT8_C(122), UINT8_C(193),    UINT8_MAX, UINT8_C(168),    UINT8_MAX, UINT8_C( 84),    UINT8_MAX,    UINT8_MAX,
        UINT8_C(209), UINT8_C(252),    UINT8_MAX,    UINT8_MAX, UINT8_C(150),    UINT8_MAX, UINT8_C(113), UINT8_C(254),
        UINT8_C(156), UINT8_C(228), UINT8_C(194),    UINT8_MAX, UINT8_C(102),    UINT8_MAX, UINT8_C(221), UINT8_C(144) } },
    { { UINT8_C(142), UINT8_C( 66), UINT8_C(222), UINT8_C(181), UINT8_C(161), UINT8_C( 17), UINT8_C( 35), UINT8_C(213),
        UINT8_C(  5), UINT8_C(168), UINT8_C(218), UINT8_C(167), UINT8_C(236), UINT8_C(206), UINT8_C( 42), UINT8_C( 94),
        UINT8_C( 36), UINT8_C( 41), UINT8_C(228), UINT8_C(167), UINT8_C( 34), UINT8_C(228), UINT8_C( 44), UINT8_C(169),
        UINT8_C(237), UINT8_C(229), UINT8_C(147), UINT8_C(187), UINT8_C( 63), UINT8_C(208), UINT8_C(144), UINT8_C(205) },
      UINT32_C(3028512274),
      { UINT8_C(128), UINT8_C(166), UINT8_C(137), UINT8_C(133), UINT8_C( 78), UINT8_C(100), UINT8_C( 45), UINT8_C( 58),
        UINT8_C( 50), UINT8_C( 87), UINT8_C(153), UINT8_C( 86), UINT8_C(128), UINT8_C(125), UINT8_C(254), UINT8_C(162),
        UINT8_C( 97), UINT8_C( 42), UINT8_C( 76), UINT8_C( 78), UINT8_C( 16), UINT8_C(223), UINT8_C( 10), UINT8_C( 79),
        UINT8_C(176), UINT8_C(154), UINT8_C( 28), UINT8_C(194), UINT8_C(  9), UINT8_C(159), UINT8_C(118), UINT8_C(137) },
      { UINT8_C( 69), UINT8_C(  0), UINT8_C( 14), UINT8_C(148), UINT8_C(100), UINT8_C( 59), UINT8_C(206), UINT8_C(150),
        UINT8_C(146), UINT8_C(103), UINT8_C(237), UINT8_C( 18), UINT8_C(229), UINT8_C(235), UINT8_C(181), UINT8_C( 70),
        UINT8_C( 21), UINT8_C(  1), UINT8_C(149), UINT8_C( 37), UINT8_C(224), UINT8_C(159), UINT8_C(116), UINT8_C(144),
        UINT8_C( 57), UINT8_C(145), UINT8_C( 83), UINT8_C( 66), UINT8_C( 48), UINT8_C(201), UINT8_C(203), UINT8_C(118) },
      { UINT8_C(142), UINT8_C(166), UINT8_C(222), UINT8_C(181), UINT8_C(178), UINT8_C( 17), UINT8_C( 35), UINT8_C(213),
        UINT8_C(  5), UINT8_C(190),    UINT8_MAX, UINT8_C(104), UINT8_C(236),    UINT8_MAX,    UINT8_MAX, UINT8_C( 94),
        UINT8_C(118), UINT8_C( 43), UINT8_C(228), UINT8_C(167), UINT8_C( 34), UINT8_C(228), UINT8_C( 44), UINT8_C(223),
        UINT8_C(237), UINT8_C(229), UINT8_C(111), UINT8_C(187), UINT8_C( 57),    UINT8_MAX, UINT8_C(144),    UINT8_MAX } },
    { { UINT8_C(201), UINT8_C(218), UINT8_C( 10), UINT8_C( 45), UINT8_C( 21), UINT8_C(216), UINT8_C(196), UINT8_C(168),
        UINT8_C( 64), UINT8_C(177), UINT8_C(186), UINT8_C( 37), UINT8_C(156), UINT8_C(111), UINT8_C(107), UINT8_C(177),
        UINT8_C(112), UINT8_C(  0), UINT8_C(215), UINT8_C( 81), UINT8_C(159), UINT8_C( 75), UINT8_C(225), UINT8_C(217),
        UINT8_C(220), UINT8_C( 52), UINT8_C( 27), UINT8_C( 13), UINT8_C(254), UINT8_C(231), UINT8_C(131), UINT8_C(199) },
      UINT32_C(3606416833),
      { UINT8_C(101), UINT8_C(185), UINT8_C(126), UINT8_C(165), UINT8_C(106), UINT8_C( 57), UINT8_C(202), UINT8_C(  6),
        UINT8_C(168), UINT8_C( 54), UINT8_C(183), UINT8_C( 25), UINT8_C( 54), UINT8_C(142), UINT8_C(106), UINT8_C(214),
        UINT8_C(218), UINT8_C( 75), UINT8_C(175), UINT8_C(182), UINT8_C(128), UINT8_C(202), UINT8_C(195), UINT8_C(126),
        UINT8_C(177), UINT8_C( 70), UINT8_C( 69), UINT8_C(114), UINT8_C(211), UINT8_C( 58), UINT8_C( 73), UINT8_C( 57) },
      { UINT8_C(243), UINT8_C(199), UINT8_C(222), UINT8_C( 93), UINT8_C(  0), UINT8_C(169), UINT8_C( 99), UINT8_C(169),
        UINT8_C(223), UINT8_C( 27), UINT8_C(194), UINT8_C( 21), UINT8_C(169), UINT8_C( 44), UINT8_C(235), UINT8_C(131),
        UINT8_C(119), UINT8_C(154), UINT8_C( 58), UINT8_C(247), UINT8_C(101), UINT8_C(253), UINT8_C(117), UINT8_C( 22),
        UINT8_C( 68), UINT8_C(187), UINT8_C(137), UINT8_C( 23), UINT8_C(245), UINT8_C(210), UINT8_C( 80), UINT8_C(233) },
      {    UINT8_MAX, UINT8_C(218), UINT8_C( 10), UINT8_C( 45), UINT8_C( 21), UINT8_C(216),    UINT8_MAX, UINT8_C(175),
           UINT8_MAX, UINT8_C(177),    UINT8_MAX, UINT8_C( 46), UINT8_C(156), UINT8_C(111), UINT8_C(107),    UINT8_MAX,
           UINT8_MAX, UINT8_C(  0), UINT8_C(233), UINT8_C( 81), UINT8_C(229),    UINT8_MAX,    UINT8_MAX, UINT8_C(148),
        UINT8_C(220),    UINT8_MAX, UINT8_C(206), UINT8_C( 13),    UINT8_MAX, UINT8_C(231), UINT8_C(153),    UINT8_MAX } },
    { { UINT8_C( 21), UINT8_C(210), UINT8_C(178), UINT8_C( 60), UINT8_C(128), UINT8_C(207), UINT8_C(233), UINT8_C( 60),
        UINT8_C(137), UINT8_C(197), UINT8_C( 97), UINT8_C(217), UINT8_C( 96), UINT8_C(  6), UINT8_C( 62), UINT8_C(231),
        UINT8_C(248), UINT8_C(  6), UINT8_C( 34), UINT8_C( 97), UINT8_C(186), UINT8_C(222), UINT8_C(206), UINT8_C( 19),
        UINT8_C(201), UINT8_C(156), UINT8_C(167), UINT8_C( 75), UINT8_C( 47), UINT8_C(  2), UINT8_C(221), UINT8_C( 68) },
      UINT32_C(1434488788),
      { UINT8_C( 94), UINT8_C(105), UINT8_C(145), UINT8_C(232), UINT8_C( 46), UINT8_C(242), UINT8_C(193), UINT8_C(142),
        UINT8_C(249),    UINT8_MAX, UINT8_C(117), UINT8_C(241), UINT8_C(  5), UINT8_C(151), UINT8_C( 83), UINT8_C(191),
        UINT8_C(117), UINT8_C( 33), UINT8_C(210), UINT8_C( 63), UINT8_C(189), UINT8_C(122), UINT8_C(138), UINT8_C(236),
        UINT8_C(124), UINT8_C(103), UINT8_C( 49), UINT8_C( 80), UINT8_C(246), UINT8_C(177), UINT8_C(165), UINT8_C( 85) },
      { UINT8_C( 27), UINT8_C( 55), UINT8_C( 61), UINT8_C( 73), UINT8_C( 41), UINT8_C(254), UINT8_C(216), UINT8_C( 34),
        UINT8_C(253), UINT8_C( 77), UINT8_C( 20), UINT8_C(  2), UINT8_C(229), UINT8_C(103), UINT8_C(193), UINT8_C( 90),
        UINT8_C(136), UINT8_C(148), UINT8_C(153), UINT8_C( 69), UINT8_C( 14), UINT8_C( 36), UINT8_C( 49), UINT8_C(138),
        UINT8_C(139), UINT8_C( 98), UINT8_C(218), UINT8_C(130), UINT8_C( 20), UINT8_C(128), UINT8_C(215), UINT8_C( 47) },
      { UINT8_C( 21), UINT8_C(210), UINT8_C(206), UINT8_C( 60), UINT8_C( 87), UINT8_C(207),    UINT8_MAX, UINT8_C(176),
           UINT8_MAX,    UINT8_MAX, UINT8_C(137), UINT8_C(243), UINT8_C( 96), UINT8_C(  6), UINT8_C( 62),    UINT8_MAX,
        UINT8_C(248), UINT8_C(  6), UINT8_C( 34), UINT8_C( 97), UINT8_C(186), UINT8_C(222), UINT8_C(206),    UINT8_MAX,
           UINT8_MAX, UINT8_C(156),    UINT8_MAX, UINT8_C( 75),    UINT8_MAX, UINT8_C(  2),    UINT8_MAX, UINT8_C( 68) } },
    { { UINT8_C(183), UINT8_C( 20), UINT8_C(120), UINT8_C(224), UINT8_C( 18), UINT8_C( 80), UINT8_C(  3), UINT8_C( 15),
        UINT8_C(158), UINT8_C( 23), UINT8_C( 17), UINT8_C(131), UINT8_C(126), UINT8_C(210), UINT8_C(221), UINT8_C(  6),
        UINT8_C(102), UINT8_C(119), UINT8_C( 75), UINT8_C(116), UINT8_C(155), UINT8_C(124), UINT8_C(254), UINT8_C( 38),
        UINT8_C(223), UINT8_C(217), UINT8_C(168), UINT8_C(243), UINT8_C( 89), UINT8_C(127), UINT8_C( 34), UINT8_C( 16) },
      UINT32_C(2784008851),
      { UINT8_C(235), UINT8_C(243), UINT8_C(180), UINT8_C(137), UINT8_C( 10), UINT8_C(197), UINT8_C( 12), UINT8_C(136),
        UINT8_C(152), UINT8_C(233), UINT8_C(142), UINT8_C(254), UINT8_C( 96), UINT8_C(217), UINT8_C(115), UINT8_C(251),
        UINT8_C( 86), UINT8_C(113), UINT8_C( 34), UINT8_C( 53), UINT8_C( 74), UINT8_C(202), UINT8_C( 40), UINT8_C(163),
        UINT8_C( 74), UINT8_C( 74), UINT8_C(179), UINT8_C(221), UINT8_C(228), UINT8_C(164), UINT8_C(131), UINT8_C(207) },
      { UINT8_C(151), UINT8_C( 55), UINT8_C( 88), UINT8_C(162), UINT8_C(253), UINT8_C(100), UINT8_C( 42), UINT8_C(149),
        UINT8_C( 78), UINT8_C(185), UINT8_C(147), UINT8_C(174), UINT8_C(146), UINT8_C(  6), UINT8_C(170), UINT8_C(232),
        UINT8_C(120), UINT8_C(204), UINT8_C( 29), UINT8_C(194), UINT8_C(150), UINT8_C( 69), UINT8_C(102), UINT8_C(224),
        UINT8_C(143), UINT8_C( 25), UINT8_C(190), UINT8_C(116), UINT8_C(189), UINT8_C( 65), UINT8_C( 67), UINT8_C( 85) },
      {    UINT8_MAX,    UINT8_MAX, UINT8_C(120), UINT8_C(224),    UINT8_MAX, UINT8_C( 80), UINT8_C(  3),    UINT8_MAX,
        UINT8_C(158),    UINT8_MAX, UINT8_C( 17),    UINT8_MAX, UINT8_C(242), UINT8_C(210), UINT8_C(221),    UINT8_MAX,
        UINT8_C(102), UINT8_C(119), UINT8_C( 75), UINT8_C(116), UINT8_C(224),    UINT8_MAX, UINT8_C(142),    UINT8_MAX,
        UINT8_C(217), UINT8_C(217),    UINT8_MAX, UINT8_C(243), UINT8_C( 89), UINT8_C(229), UINT8_C( 34),    UINT8_MAX } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i src = simde_x_mm256_loadu_epu8(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epu8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epu8(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_adds_epu8(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epu8(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i src = simde_test_x86_random_u8x32();
    simde__mmask32 k = simde_test_x86_random_mmask32();
    simde__m256i a = simde_test_x86_random_u8x32();
    simde__m256i b = simde_test_x86_random_u8x32();
    simde__m256i r = simde_mm256_mask_adds_epu8(src, k, a, b);

    simde_test_x86_write_u8x32(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask32(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x32(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x32(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_maskz_adds_epu8(SIMDE_MUNIT_TEST_ARGS) {
#if 1
  const struct {
    simde__mmask32 k;
    uint8_t a[32];
    uint8_t b[32];
    uint8_t r[32];
  } test_vec[8] = {
    { UINT32_C(4271688964),
      { UINT8_C(238), UINT8_C(234), UINT8_C(150), UINT8_C( 25), UINT8_C( 22), UINT8_C(208), UINT8_C(249), UINT8_C(  4),
        UINT8_C(223), UINT8_C( 54), UINT8_C(162), UINT8_C(132), UINT8_C(215), UINT8_C( 20), UINT8_C( 72), UINT8_C( 75),
        UINT8_C(140), UINT8_C(192), UINT8_C(126), UINT8_C(216), UINT8_C( 45), UINT8_C(  4), UINT8_C( 41), UINT8_C(249),
        UINT8_C(197), UINT8_C( 86), UINT8_C( 27), UINT8_C(201), UINT8_C( 35), UINT8_C(184), UINT8_C(200), UINT8_C( 17) },
      { UINT8_C(162), UINT8_C( 94), UINT8_C( 43), UINT8_C(184), UINT8_C( 46), UINT8_C( 36), UINT8_C(188), UINT8_C( 14),
        UINT8_C( 90), UINT8_C( 94), UINT8_C(146), UINT8_C( 49), UINT8_C(114), UINT8_C(219), UINT8_C(125), UINT8_C(254),
        UINT8_C(155), UINT8_C(251), UINT8_C(215), UINT8_C(200),    UINT8_MAX, UINT8_C(  0), UINT8_C(193), UINT8_C(196),
        UINT8_C( 87), UINT8_C(221), UINT8_C(142), UINT8_C(122), UINT8_C(149), UINT8_C( 86), UINT8_C(140), UINT8_C( 55) },
      { UINT8_C(  0), UINT8_C(  0), UINT8_C(193), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
           UINT8_MAX, UINT8_C(  0),    UINT8_MAX, UINT8_C(181), UINT8_C(  0), UINT8_C(  0), UINT8_C(197),    UINT8_MAX,
        UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,
        UINT8_C(  0),    UINT8_MAX, UINT8_C(169),    UINT8_MAX, UINT8_C(184),    UINT8_MAX,    UINT8_MAX, UINT8_C( 72) } },
    { UINT32_C(3807360948),
      { UINT8_C(219), UINT8_C(171), UINT8_C(240), UINT8_C( 54), UINT8_C(  9), UINT8_C(131), UINT8_C(103), UINT8_C(123),
        UINT8_C( 94), UINT8_C(228), UINT8_C(121), UINT8_C(249), UINT8_C(224), UINT8_C( 80), UINT8_C(194), UINT8_C(223),
        UINT8_C( 81), UINT8_C(131), UINT8_C(164), UINT8_C(168), UINT8_C( 96), UINT8_C( 50), UINT8_C( 34), UINT8_C(245),
        UINT8_C(136), UINT8_C(174), UINT8_C( 44), UINT8_C( 60), UINT8_C(101), UINT8_C( 27), UINT8_C( 30), UINT8_C( 65) },
      { UINT8_C(198), UINT8_C( 15), UINT8_C(119), UINT8_C(207), UINT8_C(146), UINT8_C(222), UINT8_C( 74), UINT8_C(240),
        UINT8_C(195), UINT8_C(196), UINT8_C(233), UINT8_C(163), UINT8_C( 20), UINT8_C(171), UINT8_C(130), UINT8_C(101),
        UINT8_C( 47), UINT8_C( 38), UINT8_C( 13), UINT8_C(143), UINT8_C( 88), UINT8_C( 48), UINT8_C(133), UINT8_C(224),
        UINT8_C(222), UINT8_C(177), UINT8_C( 28), UINT8_C( 68), UINT8_C(205), UINT8_C( 59), UINT8_C(133), UINT8_C(147) },
      { UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(155),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,
           UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(244), UINT8_C(251), UINT8_C(  0),    UINT8_MAX,
        UINT8_C(128), UINT8_C(169), UINT8_C(177),    UINT8_MAX, UINT8_C(  0), UINT8_C( 98), UINT8_C(167),    UINT8_MAX,
        UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 86), UINT8_C(163), UINT8_C(212) } },
    { UINT32_C(3697540170),
      { UINT8_C(218), UINT8_C(173), UINT8_C(204), UINT8_C(157), UINT8_C(113), UINT8_C(181), UINT8_C( 64), UINT8_C(134),
        UINT8_C( 97), UINT8_C(195), UINT8_C(235), UINT8_C(144), UINT8_C(233), UINT8_C(249), UINT8_C( 31), UINT8_C( 66),
        UINT8_C( 41), UINT8_C(164), UINT8_C( 34), UINT8_C(  7), UINT8_C( 86), UINT8_C( 63), UINT8_C( 75), UINT8_C( 35),
        UINT8_C(122), UINT8_C(208), UINT8_C(182), UINT8_C(196), UINT8_C(204), UINT8_C( 25), UINT8_C(160), UINT8_C(167) },
      { UINT8_C(199), UINT8_C(108), UINT8_C( 68), UINT8_C( 56), UINT8_C( 33), UINT8_C(133), UINT8_C(190), UINT8_C(130),
        UINT8_C( 72), UINT8_C(170), UINT8_C( 18), UINT8_C( 49), UINT8_C(163), UINT8_C( 50), UINT8_C(115), UINT8_C(204),
        UINT8_C(214), UINT8_C(150), UINT8_C(211), UINT8_C( 44), UINT8_C(213), UINT8_C( 31), UINT8_C( 79), UINT8_C( 79),
        UINT8_C(239), UINT8_C(  6), UINT8_C( 19), UINT8_C(188), UINT8_C( 31), UINT8_C(179), UINT8_C( 99), UINT8_C(230) },
      { UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(213), UINT8_C(  0), UINT8_C(  0), UINT8_C(254), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(253), UINT8_C(193),    UINT8_MAX,    UINT8_MAX, UINT8_C(146),    UINT8_MAX,
           UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 94), UINT8_C(154), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(201),    UINT8_MAX, UINT8_C(235), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX } },
    { UINT32_C(1075816223),
      { UINT8_C( 44), UINT8_C(221), UINT8_C(195), UINT8_C(116), UINT8_C(135), UINT8_C(213), UINT8_C(166), UINT8_C( 42),
        UINT8_C(  7), UINT8_C( 25), UINT8_C(246), UINT8_C(222), UINT8_C(175), UINT8_C(202), UINT8_C( 10), UINT8_C(132),
        UINT8_C(233), UINT8_C( 90), UINT8_C(211), UINT8_C(216), UINT8_C( 96), UINT8_C(230), UINT8_C(148), UINT8_C(127),
        UINT8_C(153), UINT8_C(247), UINT8_C(102), UINT8_C(184), UINT8_C(159), UINT8_C(133), UINT8_C(249), UINT8_C(203) },
      { UINT8_C( 98), UINT8_C(188), UINT8_C( 64), UINT8_C(234), UINT8_C(145), UINT8_C(230), UINT8_C( 20), UINT8_C(153),
           UINT8_MAX, UINT8_C( 11), UINT8_C(119), UINT8_C(175), UINT8_C(213), UINT8_C(129), UINT8_C( 51), UINT8_C(190),
        UINT8_C(219), UINT8_C(  7), UINT8_C(150), UINT8_C( 59), UINT8_C(237), UINT8_C( 43), UINT8_C(187), UINT8_C(135),
        UINT8_C( 34), UINT8_C( 33), UINT8_C( 63), UINT8_C(193), UINT8_C(166), UINT8_C( 56), UINT8_C(141), UINT8_C(  8) },
      { UINT8_C(142),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
           UINT8_MAX, UINT8_C( 36),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,
           UINT8_MAX, UINT8_C( 97),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0) } },
    { UINT32_C(2264059380),
      { UINT8_C(179), UINT8_C(  7), UINT8_C( 31), UINT8_C(178), UINT8_C( 18), UINT8_C(150), UINT8_C( 97), UINT8_C(231),
        UINT8_C( 23), UINT8_C(149), UINT8_C(165), UINT8_C(243), UINT8_C(156), UINT8_C( 59), UINT8_C( 46), UINT8_C(137),
        UINT8_C(102), UINT8_C(233), UINT8_C( 16), UINT8_C(137), UINT8_C( 10), UINT8_C( 80), UINT8_C( 74), UINT8_C(176),
        UINT8_C(136), UINT8_C(215), UINT8_C(185), UINT8_C(125), UINT8_C(164), UINT8_C(171), UINT8_C(  3), UINT8_C( 87) },
      { UINT8_C(178), UINT8_C( 34), UINT8_C( 10), UINT8_C(196), UINT8_C(184), UINT8_C(107), UINT8_C(171), UINT8_C(207),
        UINT8_C(  0), UINT8_C( 80), UINT8_C(194), UINT8_C(156), UINT8_C(140), UINT8_C(241), UINT8_C( 38), UINT8_C(242),
        UINT8_C(218), UINT8_C( 54), UINT8_C(123), UINT8_C(229), UINT8_C(134), UINT8_C(198), UINT8_C(149), UINT8_C( 15),
        UINT8_C(157), UINT8_C( 78), UINT8_C(140), UINT8_C( 66), UINT8_C(250), UINT8_C(143), UINT8_C(153), UINT8_C(172) },
      { UINT8_C(  0), UINT8_C(  0), UINT8_C( 41), UINT8_C(  0), UINT8_C(202),    UINT8_MAX,    UINT8_MAX,    UINT8_MAX,
        UINT8_C( 23), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C( 84),    UINT8_MAX,
        UINT8_C(  0),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(144),    UINT8_MAX, UINT8_C(223), UINT8_C(191),
        UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX } },
    { UINT32_C(1769055153),
      { UINT8_C( 15), UINT8_C( 28), UINT8_C( 56), UINT8_C( 15), UINT8_C(109), UINT8_C(251), UINT8_C(172), UINT8_C(249),
        UINT8_C(236), UINT8_C(210), UINT8_C(235), UINT8_C(198), UINT8_C(  8), UINT8_C(103), UINT8_C(171), UINT8_C(143),
        UINT8_C( 45), UINT8_C( 65), UINT8_C(158), UINT8_C(202), UINT8_C(143), UINT8_C( 42), UINT8_C( 12), UINT8_C(137),
        UINT8_C(185), UINT8_C(166), UINT8_C( 54), UINT8_C(106), UINT8_C( 73), UINT8_C(167), UINT8_C(211), UINT8_C( 88) },
      { UINT8_C(195), UINT8_C( 11), UINT8_C(104), UINT8_C( 48), UINT8_C(  6), UINT8_C( 20), UINT8_C( 41), UINT8_C(242),
        UINT8_C(230), UINT8_C( 21), UINT8_C(185), UINT8_C(238), UINT8_C(124), UINT8_C(100), UINT8_C(125), UINT8_C(169),
        UINT8_C(165), UINT8_C( 27), UINT8_C(115), UINT8_C( 53), UINT8_C( 69), UINT8_C(128), UINT8_C(190), UINT8_C(254),
        UINT8_C( 38), UINT8_C(244), UINT8_C(104), UINT8_C(111), UINT8_C(155), UINT8_C( 59), UINT8_C(200), UINT8_C( 95) },
      { UINT8_C(210), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(115),    UINT8_MAX, UINT8_C(  0),    UINT8_MAX,
           UINT8_MAX, UINT8_C(231), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(203), UINT8_C(  0),    UINT8_MAX,
        UINT8_C(210), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(212), UINT8_C(170), UINT8_C(202), UINT8_C(  0),
        UINT8_C(223), UINT8_C(  0), UINT8_C(  0), UINT8_C(217), UINT8_C(  0), UINT8_C(226),    UINT8_MAX, UINT8_C(  0) } },
    { UINT32_C(1301229639),
      { UINT8_C( 68), UINT8_C(185), UINT8_C( 64), UINT8_C( 42), UINT8_C(206), UINT8_C(249), UINT8_C( 24), UINT8_C( 74),
        UINT8_C( 93), UINT8_C(150), UINT8_C(243), UINT8_C(  3), UINT8_C(177), UINT8_C(102), UINT8_C( 56), UINT8_C(247),
        UINT8_C(230), UINT8_C(246), UINT8_C(245), UINT8_C( 12), UINT8_C(235), UINT8_C( 94), UINT8_C(124), UINT8_C(134),
        UINT8_C(153), UINT8_C( 68), UINT8_C(229), UINT8_C(224), UINT8_C(116), UINT8_C(117), UINT8_C( 46), UINT8_C(184) },
      { UINT8_C( 46), UINT8_C(110), UINT8_C(226), UINT8_C(252), UINT8_C(103), UINT8_C(250), UINT8_C( 70), UINT8_C(196),
        UINT8_C(144), UINT8_C( 57), UINT8_C(199), UINT8_C( 66), UINT8_C(159),    UINT8_MAX, UINT8_C( 57), UINT8_C(134),
        UINT8_C(246), UINT8_C( 46), UINT8_C(146), UINT8_C(225), UINT8_C(140), UINT8_C( 14), UINT8_C(103), UINT8_C( 38),
        UINT8_C( 82), UINT8_C( 77), UINT8_C(  6), UINT8_C(198), UINT8_C(194), UINT8_C( 52), UINT8_C(126), UINT8_C(240) },
      { UINT8_C(114),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C( 94), UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX, UINT8_C(  0), UINT8_C(  0),
           UINT8_MAX,    UINT8_MAX,    UINT8_MAX, UINT8_C(237), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(172),
        UINT8_C(235), UINT8_C(  0), UINT8_C(235),    UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(172), UINT8_C(  0) } },
    { UINT32_C(3339780172),
      { UINT8_C( 20), UINT8_C(214), UINT8_C( 48), UINT8_C(172), UINT8_C(224), UINT8_C(109), UINT8_C(207), UINT8_C(185),
        UINT8_C(208), UINT8_C(144), UINT8_C(223), UINT8_C(195), UINT8_C( 50), UINT8_C(132), UINT8_C(240), UINT8_C(113),
        UINT8_C(234), UINT8_C(208), UINT8_C(179), UINT8_C(  6), UINT8_C( 23), UINT8_C( 91), UINT8_C(148), UINT8_C(115),
        UINT8_C(192), UINT8_C(132), UINT8_C(126), UINT8_C( 12), UINT8_C(132), UINT8_C(143), UINT8_C(211), UINT8_C(153) },
      { UINT8_C(101), UINT8_C(  3), UINT8_C( 69), UINT8_C( 69), UINT8_C(113), UINT8_C( 20),    UINT8_MAX, UINT8_C( 65),
        UINT8_C(164), UINT8_C(222), UINT8_C(  5), UINT8_C(214), UINT8_C( 98), UINT8_C(245), UINT8_C( 72), UINT8_C( 77),
        UINT8_C(197), UINT8_C(251), UINT8_C( 83), UINT8_C(220), UINT8_C( 87), UINT8_C(231), UINT8_C( 79), UINT8_C( 23),
        UINT8_C(108), UINT8_C(205), UINT8_C( 35), UINT8_C(240), UINT8_C( 93), UINT8_C(246), UINT8_C(137), UINT8_C(194) },
      { UINT8_C(  0), UINT8_C(  0), UINT8_C(117), UINT8_C(241), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX, UINT8_C(  0),
        UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
           UINT8_MAX, UINT8_C(  0), UINT8_C(  0), UINT8_C(  0), UINT8_C(110), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),
           UINT8_MAX,    UINT8_MAX, UINT8_C(161), UINT8_C(  0), UINT8_C(  0), UINT8_C(  0),    UINT8_MAX,    UINT8_MAX } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i a = simde_x_mm256_loadu_epu8(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epu8(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_adds_epu8(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i8x32(r, simde_x_mm256_loadu_epu8(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask32 k = simde_test_x86_random_mmask32();
    simde__m256i a = simde_test_x86_random_u8x32();
    simde__m256i b = simde_test_x86_random_u8x32();
    simde__m256i r = simde_mm256_maskz_adds_epu8(k, a, b);

    simde_test_x86_write_mmask32(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_u8x32(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x32(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u8x32(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_mask_adds_epu16(SIMDE_MUNIT_TEST_ARGS) {
#if 1
  const struct {
    uint16_t src[16];
    simde__mmask16 k;
    uint16_t a[16];
    uint16_t b[16];
    uint16_t r[16];
  } test_vec[8] = {
    { { UINT16_C(32755), UINT16_C(52649), UINT16_C(46880), UINT16_C(36148), UINT16_C(63868), UINT16_C(49171), UINT16_C(24581), UINT16_C(52542),
        UINT16_C(57555), UINT16_C(  961), UINT16_C(59196), UINT16_C(24514), UINT16_C(20648), UINT16_C(60484), UINT16_C(51900), UINT16_C(44997) },
      UINT16_C(28233),
      { UINT16_C(27005), UINT16_C(45350), UINT16_C(41718), UINT16_C( 2475), UINT16_C(45155), UINT16_C(41322), UINT16_C(15742), UINT16_C(16257),
        UINT16_C(48448), UINT16_C(  551), UINT16_C(53020), UINT16_C(24914), UINT16_C( 4027), UINT16_C(32811), UINT16_C(29886), UINT16_C(15343) },
      { UINT16_C( 5597), UINT16_C(54253), UINT16_C(39095), UINT16_C( 6877), UINT16_C(18248), UINT16_C(50875), UINT16_C(15748), UINT16_C(50182),
        UINT16_C(11770), UINT16_C( 6086), UINT16_C( 6652), UINT16_C(46968), UINT16_C(41768), UINT16_C(58935), UINT16_C( 9751), UINT16_C(62498) },
      { UINT16_C(32602), UINT16_C(52649), UINT16_C(46880), UINT16_C( 9352), UINT16_C(63868), UINT16_C(49171), UINT16_C(31490), UINT16_C(52542),
        UINT16_C(57555), UINT16_C( 6637), UINT16_C(59672),      UINT16_MAX, UINT16_C(20648),      UINT16_MAX, UINT16_C(39637), UINT16_C(44997) } },
    { { UINT16_C( 3899), UINT16_C(62407), UINT16_C(42151), UINT16_C(61197), UINT16_C(51691), UINT16_C(28598), UINT16_C(48134), UINT16_C(   51),
        UINT16_C(64233), UINT16_C(58647), UINT16_C(36627), UINT16_C(15260), UINT16_C(54066), UINT16_C(18721), UINT16_C(17402), UINT16_C(13629) },
      UINT16_C( 1362),
      { UINT16_C(63784), UINT16_C(13993), UINT16_C(38377), UINT16_C(40959), UINT16_C( 1284), UINT16_C(14427), UINT16_C(17413), UINT16_C( 7474),
        UINT16_C(17705), UINT16_C(50604), UINT16_C(57216), UINT16_C(41368), UINT16_C(37416), UINT16_C(26341), UINT16_C(14280), UINT16_C(61547) },
      { UINT16_C( 5169), UINT16_C( 6694), UINT16_C( 9641), UINT16_C(44729), UINT16_C( 5162), UINT16_C(12518), UINT16_C( 6232), UINT16_C(33101),
        UINT16_C(63837), UINT16_C(56646), UINT16_C(57048), UINT16_C(  382), UINT16_C(25457), UINT16_C(14695), UINT16_C(53915), UINT16_C(52265) },
      { UINT16_C( 3899), UINT16_C(20687), UINT16_C(42151), UINT16_C(61197), UINT16_C( 6446), UINT16_C(28598), UINT16_C(23645), UINT16_C(   51),
             UINT16_MAX, UINT16_C(58647),      UINT16_MAX, UINT16_C(15260), UINT16_C(54066), UINT16_C(18721), UINT16_C(17402), UINT16_C(13629) } },
    { { UINT16_C(20710), UINT16_C(37094), UINT16_C(40821), UINT16_C(41022), UINT16_C( 9395), UINT16_C( 3024), UINT16_C( 7484), UINT16_C(39308),
        UINT16_C(53782), UINT16_C(61302), UINT16_C(62640), UINT16_C( 8688), UINT16_C(22360), UINT16_C(62298), UINT16_C(33833), UINT16_C( 4031) },
      UINT16_C(42452),
      { UINT16_C(18847), UINT16_C(56644), UINT16_C(63465), UINT16_C(47361), UINT16_C(15618), UINT16_C(36566), UINT16_C(60886), UINT16_C(19552),
        UINT16_C( 4316), UINT16_C(52289), UINT16_C(39218), UINT16_C(35875), UINT16_C(19596), UINT16_C(19216), UINT16_C(58459), UINT16_C(64496) },
      { UINT16_C(13358), UINT16_C( 6104), UINT16_C(55851), UINT16_C(11729), UINT16_C(42775), UINT16_C(61115), UINT16_C( 7060), UINT16_C(28730),
        UINT16_C(31531), UINT16_C(23868), UINT16_C(24340), UINT16_C(41194), UINT16_C(64171), UINT16_C( 2027), UINT16_C(56287), UINT16_C( 3330) },
      { UINT16_C(20710), UINT16_C(37094),      UINT16_MAX, UINT16_C(41022), UINT16_C(58393), UINT16_C( 3024),      UINT16_MAX, UINT16_C(48282),
        UINT16_C(35847), UINT16_C(61302), UINT16_C(63558), UINT16_C( 8688), UINT16_C(22360), UINT16_C(21243), UINT16_C(33833),      UINT16_MAX } },
    { { UINT16_C(55823), UINT16_C(14884), UINT16_C(62900), UINT16_C(52327), UINT16_C( 8861), UINT16_C(12730), UINT16_C(62525), UINT16_C(27042),
        UINT16_C(56944), UINT16_C(33990), UINT16_C(45118), UINT16_C(59685), UINT16_C( 4267), UINT16_C(35568), UINT16_C(62188), UINT16_C(64407) },
      UINT16_C(48077),
      { UINT16_C(33078), UINT16_C(40369), UINT16_C(20045), UINT16_C( 1984), UINT16_C(64895), UINT16_C( 8700), UINT16_C(27750), UINT16_C(11520),
        UINT16_C(16112), UINT16_C( 5597), UINT16_C(34855), UINT16_C( 6182), UINT16_C( 4626), UINT16_C(43274), UINT16_C(55053), UINT16_C(17253) },
      { UINT16_C( 5721), UINT16_C(42721), UINT16_C(41316), UINT16_C(58286), UINT16_C(43678), UINT16_C( 1285), UINT16_C( 1302), UINT16_C( 1586),
        UINT16_C( 3907), UINT16_C(27164), UINT16_C(17048), UINT16_C(43650), UINT16_C(36180), UINT16_C(24916), UINT16_C(47460), UINT16_C(48549) },
      { UINT16_C(38799), UINT16_C(14884), UINT16_C(61361), UINT16_C(60270), UINT16_C( 8861), UINT16_C(12730), UINT16_C(29052), UINT16_C(13106),
        UINT16_C(20019), UINT16_C(32761), UINT16_C(45118), UINT16_C(49832), UINT16_C(40806),      UINT16_MAX, UINT16_C(62188),      UINT16_MAX } },
    { { UINT16_C(34511), UINT16_C(13156), UINT16_C( 4647), UINT16_C(50454), UINT16_C( 7100), UINT16_C(53962), UINT16_C(64544), UINT16_C(25560),
        UINT16_C(62476), UINT16_C(42190), UINT16_C(20534), UINT16_C(35406), UINT16_C(41693), UINT16_C(17132), UINT16_C(37211), UINT16_C(11007) },
      UINT16_C(25367),
      { UINT16_C(15965), UINT16_C(29813), UINT16_C(12547), UINT16_C(52879), UINT16_C(45059), UINT16_C(56522), UINT16_C(54803), UINT16_C(57808),
        UINT16_C( 1914), UINT16_C(51506), UINT16_C( 3985), UINT16_C(32107), UINT16_C(51025), UINT16_C(20750), UINT16_C( 9713), UINT16_C(20404) },
      { UINT16_C(10851), UINT16_C(26563), UINT16_C(21083), UINT16_C(24373), UINT16_C(65282), UINT16_C( 5691), UINT16_C( 3030), UINT16_C(20727),
        UINT16_C(10514), UINT16_C(42009), UINT16_C(34105), UINT16_C(35361), UINT16_C(12364), UINT16_C(15835), UINT16_C(36949), UINT16_C(47500) },
      { UINT16_C(26816), UINT16_C(56376), UINT16_C(33630), UINT16_C(50454),      UINT16_MAX, UINT16_C(53962), UINT16_C(64544), UINT16_C(25560),
        UINT16_C(12428),      UINT16_MAX, UINT16_C(20534), UINT16_C(35406), UINT16_C(41693), UINT16_C(36585), UINT16_C(46662), UINT16_C(11007) } },
    { { UINT16_C(20410), UINT16_C( 5408), UINT16_C(21922), UINT16_C(42100), UINT16_C(44884), UINT16_C(10938), UINT16_C(45755), UINT16_C(52603),
        UINT16_C(38107), UINT16_C( 5233), UINT16_C(37657), UINT16_C(26015), UINT16_C(31427), UINT16_C( 6307), UINT16_C(12042), UINT16_C(50385) },
      UINT16_C(61823),
      { UINT16_C( 8666), UINT16_C(20038), UINT16_C(39877), UINT16_C(33022), UINT16_C(47557), UINT16_C(16434), UINT16_C( 3462), UINT16_C(63701),
        UINT16_C(60962), UINT16_C(49547), UINT16_C(20052), UINT16_C(63291), UINT16_C(18022), UINT16_C(14374), UINT16_C(42250), UINT16_C(58409) },
      { UINT16_C(28870), UINT16_C(35891), UINT16_C(12555), UINT16_C(53260), UINT16_C(16106), UINT16_C(28689), UINT16_C(58955), UINT16_C(28008),
        UINT16_C(62420), UINT16_C(10286), UINT16_C(27201), UINT16_C(43039), UINT16_C(18096), UINT16_C(47840), UINT16_C( 2539), UINT16_C(45727) },
      { UINT16_C(37536), UINT16_C(55929), UINT16_C(52432),      UINT16_MAX, UINT16_C(63663), UINT16_C(45123), UINT16_C(62417), UINT16_C(52603),
             UINT16_MAX, UINT16_C( 5233), UINT16_C(37657), UINT16_C(26015), UINT16_C(36118), UINT16_C(62214), UINT16_C(44789),      UINT16_MAX } },
    { { UINT16_C(53881), UINT16_C(33854), UINT16_C(18947), UINT16_C(60757), UINT16_C(26248), UINT16_C(54109), UINT16_C(50764), UINT16_C( 8257),
        UINT16_C(28601), UINT16_C(64329), UINT16_C(26841), UINT16_C(35235), UINT16_C(33710), UINT16_C(39492), UINT16_C(58252), UINT16_C( 1612) },
      UINT16_C(35509),
      { UINT16_C(47242), UINT16_C(57300), UINT16_C(23717), UINT16_C(  581), UINT16_C(37167), UINT16_C(28872), UINT16_C(33458), UINT16_C(64480),
        UINT16_C(47485), UINT16_C( 8291), UINT16_C( 4675), UINT16_C(34723), UINT16_C(12204), UINT16_C(63594), UINT16_C( 7989), UINT16_C(49282) },
      { UINT16_C(22231), UINT16_C(31903), UINT16_C(58802), UINT16_C(57726), UINT16_C(18294), UINT16_C(10322), UINT16_C(13001), UINT16_C(17955),
        UINT16_C(34795), UINT16_C(11878), UINT16_C( 2457), UINT16_C(17845), UINT16_C( 7992), UINT16_C(28221), UINT16_C(48958), UINT16_C( 5422) },
      {      UINT16_MAX, UINT16_C(33854),      UINT16_MAX, UINT16_C(60757), UINT16_C(55461), UINT16_C(39194), UINT16_C(50764),      UINT16_MAX,
        UINT16_C(28601), UINT16_C(20169), UINT16_C(26841), UINT16_C(52568), UINT16_C(33710), UINT16_C(39492), UINT16_C(58252), UINT16_C(54704) } },
    { { UINT16_C(52501), UINT16_C(51089), UINT16_C( 4274), UINT16_C(10664), UINT16_C(64087), UINT16_C( 8273), UINT16_C(29996), UINT16_C( 6246),
        UINT16_C(52476), UINT16_C(38214), UINT16_C(64725), UINT16_C( 3546), UINT16_C( 5915), UINT16_C(23163), UINT16_C(43478), UINT16_C(60271) },
      UINT16_C(  375),
      { UINT16_C(10674), UINT16_C(23057), UINT16_C(26706), UINT16_C(42069), UINT16_C(33160), UINT16_C(60953), UINT16_C( 5529), UINT16_C(57530),
        UINT16_C(36778), UINT16_C(34012), UINT16_C(63388), UINT16_C( 6299), UINT16_C(29009), UINT16_C(49601), UINT16_C(14428), UINT16_C( 3778) },
      { UINT16_C(54114), UINT16_C(46184), UINT16_C(48443), UINT16_C(50008), UINT16_C(28991), UINT16_C(55473), UINT16_C(27526), UINT16_C(12472),
        UINT16_C(38138), UINT16_C(38580), UINT16_C(20364), UINT16_C(56750), UINT16_C(28864), UINT16_C( 7326), UINT16_C(24744), UINT16_C( 2602) },
      { UINT16_C(64788),      UINT16_MAX,      UINT16_MAX, UINT16_C(10664), UINT16_C(62151),      UINT16_MAX, UINT16_C(33055), UINT16_C( 6246),
             UINT16_MAX, UINT16_C(38214), UINT16_C(64725), UINT16_C( 3546), UINT16_C( 5915), UINT16_C(23163), UINT16_C(43478), UINT16_C(60271) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i src = simde_x_mm256_loadu_epu16(test_vec[i].src);
    simde__m256i a = simde_x_mm256_loadu_epu16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epu16(test_vec[i].b);
    simde__m256i r = simde_mm256_mask_adds_epu16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epu16(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i src = simde_test_x86_random_u16x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m256i a = simde_test_x86_random_u16x16();
    simde__m256i b = simde_test_x86_random_u16x16();
    simde__m256i r = simde_mm256_mask_adds_epu16(src, k, a, b);

    simde_test_x86_write_u16x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm256_maskz_adds_epu16(SIMDE_MUNIT_TEST_ARGS) {
#if 1
  const struct {
    simde__mmask16 k;
    uint16_t a[16];
    uint16_t b[16];
    uint16_t r[16];
  } test_vec[8] = {
    { UINT16_C(60695),
      { UINT16_C(44628), UINT16_C(  295), UINT16_C( 9939), UINT16_C( 7562), UINT16_C(12606), UINT16_C(25978), UINT16_C(39895), UINT16_C(47597),
        UINT16_C(32514), UINT16_C( 6307), UINT16_C(34673), UINT16_C( 8403), UINT16_C(29795), UINT16_C(31625), UINT16_C(24929), UINT16_C(46409) },
      { UINT16_C(28687), UINT16_C(58038), UINT16_C(16534), UINT16_C(54528), UINT16_C(31345), UINT16_C(18746), UINT16_C(10005), UINT16_C( 5890),
        UINT16_C(42406), UINT16_C( 6191), UINT16_C(  557), UINT16_C(36920), UINT16_C(49782), UINT16_C(55051), UINT16_C(21539), UINT16_C(12940) },
      {      UINT16_MAX, UINT16_C(58333), UINT16_C(26473), UINT16_C(    0), UINT16_C(43951), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),
             UINT16_MAX, UINT16_C(    0), UINT16_C(35230), UINT16_C(45323), UINT16_C(    0),      UINT16_MAX, UINT16_C(46468), UINT16_C(59349) } },
    { UINT16_C(17092),
      { UINT16_C( 5251), UINT16_C(62512), UINT16_C(27278), UINT16_C(41789), UINT16_C(16274), UINT16_C(14523), UINT16_C(60133), UINT16_C( 4688),
        UINT16_C(35309), UINT16_C(25506), UINT16_C(44619), UINT16_C(28219), UINT16_C(50946), UINT16_C(51104), UINT16_C(46090), UINT16_C(36130) },
      { UINT16_C(21193), UINT16_C(22401), UINT16_C(49084), UINT16_C(20219), UINT16_C(46846), UINT16_C(58247), UINT16_C(55200), UINT16_C(36341),
        UINT16_C(39008), UINT16_C(44017), UINT16_C(11334), UINT16_C(18457), UINT16_C(47603), UINT16_C(64783), UINT16_C(12654), UINT16_C(14218) },
      { UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(41029),
        UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(58744), UINT16_C(    0) } },
    { UINT16_C( 3203),
      { UINT16_C(35275), UINT16_C(51598), UINT16_C( 5439), UINT16_C(57517), UINT16_C(41709), UINT16_C(19821), UINT16_C(24122), UINT16_C(33017),
        UINT16_C( 4746), UINT16_C(32457), UINT16_C(55500), UINT16_C(14971), UINT16_C( 1546), UINT16_C(36209), UINT16_C(65298), UINT16_C(56781) },
      { UINT16_C(23689), UINT16_C(51366), UINT16_C(21361), UINT16_C(24232), UINT16_C( 5878), UINT16_C(12460), UINT16_C(42356), UINT16_C(65457),
        UINT16_C(31415), UINT16_C(33661), UINT16_C(63570), UINT16_C(23741), UINT16_C(12030), UINT16_C( 4330), UINT16_C(46894), UINT16_C(47085) },
      { UINT16_C(58964),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX,
        UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(38712), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { UINT16_C(37907),
      { UINT16_C(10471), UINT16_C(56803), UINT16_C(36670), UINT16_C(45582), UINT16_C(48948), UINT16_C(60593), UINT16_C(11833), UINT16_C(35695),
        UINT16_C(11559), UINT16_C( 9704), UINT16_C(53851), UINT16_C(35126), UINT16_C( 9097), UINT16_C(40256), UINT16_C(49335), UINT16_C(40738) },
      { UINT16_C( 1512), UINT16_C( 9852), UINT16_C(35477), UINT16_C(51672), UINT16_C(35401), UINT16_C(33461), UINT16_C( 9656), UINT16_C(57102),
        UINT16_C(63058), UINT16_C(44293), UINT16_C(15304), UINT16_C(20791), UINT16_C(30558), UINT16_C( 5870), UINT16_C( 4151), UINT16_C( 8117) },
      { UINT16_C(11983),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(39655), UINT16_C(    0), UINT16_C(    0), UINT16_C(48855) } },
    { UINT16_C(12566),
      { UINT16_C( 7868), UINT16_C( 1396), UINT16_C(10920), UINT16_C(24712), UINT16_C(38479), UINT16_C(41280), UINT16_C(17804), UINT16_C(21582),
        UINT16_C(34176), UINT16_C(56997), UINT16_C(38141), UINT16_C(13556), UINT16_C(43428), UINT16_C(47700), UINT16_C(39387), UINT16_C(38757) },
      { UINT16_C(55991), UINT16_C(24476), UINT16_C( 9220), UINT16_C(21440), UINT16_C(  186), UINT16_C(18164), UINT16_C(16965), UINT16_C(50586),
        UINT16_C(16584), UINT16_C(50595), UINT16_C(39124), UINT16_C(30969), UINT16_C(19777), UINT16_C( 7219), UINT16_C(39143), UINT16_C(40627) },
      { UINT16_C(    0), UINT16_C(25872), UINT16_C(20140), UINT16_C(    0), UINT16_C(38665), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),
        UINT16_C(50760), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(63205), UINT16_C(54919), UINT16_C(    0), UINT16_C(    0) } },
    { UINT16_C(20594),
      { UINT16_C(48756), UINT16_C(12233), UINT16_C(48574), UINT16_C(  885), UINT16_C( 4096), UINT16_C(51400), UINT16_C(27472), UINT16_C( 9357),
        UINT16_C(34307), UINT16_C(17820), UINT16_C(53204), UINT16_C(47969), UINT16_C( 5480), UINT16_C(55897), UINT16_C(22373), UINT16_C(55633) },
      { UINT16_C( 6677), UINT16_C(54024), UINT16_C(32472), UINT16_C(55510), UINT16_C(40590), UINT16_C(56992), UINT16_C(11530), UINT16_C( 3330),
        UINT16_C(40627), UINT16_C(34642), UINT16_C(46190), UINT16_C(54850), UINT16_C(40137), UINT16_C(11952), UINT16_C(  499), UINT16_C( 2311) },
      { UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(44686),      UINT16_MAX, UINT16_C(39002), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(45617), UINT16_C(    0), UINT16_C(22872), UINT16_C(    0) } },
    { UINT16_C( 4124),
      { UINT16_C(45966), UINT16_C( 7372), UINT16_C(27729), UINT16_C(23546), UINT16_C(64665), UINT16_C(19561), UINT16_C(48026), UINT16_C( 2260),
        UINT16_C( 5743), UINT16_C(14558), UINT16_C(36786), UINT16_C(42598), UINT16_C(28304), UINT16_C(44207), UINT16_C(35710), UINT16_C( 3232) },
      { UINT16_C(27710), UINT16_C(36904), UINT16_C( 8920), UINT16_C(29163), UINT16_C(21534), UINT16_C(47294), UINT16_C(37392), UINT16_C(32705),
        UINT16_C(40872), UINT16_C(23480), UINT16_C( 7726), UINT16_C(48897), UINT16_C(45196), UINT16_C( 2667), UINT16_C( 3131), UINT16_C(31254) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(36649), UINT16_C(52709),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { UINT16_C(15992),
      { UINT16_C(62816), UINT16_C(32450), UINT16_C(32842), UINT16_C(23095), UINT16_C(63506), UINT16_C(48089), UINT16_C(37271), UINT16_C(50710),
        UINT16_C( 6064), UINT16_C(15493), UINT16_C(61639), UINT16_C(  583), UINT16_C(24060), UINT16_C(30076), UINT16_C(34460), UINT16_C(64710) },
      { UINT16_C(34940), UINT16_C(50811), UINT16_C(45577), UINT16_C( 6944), UINT16_C(63914), UINT16_C(16854), UINT16_C(60555), UINT16_C(15111),
        UINT16_C(35843), UINT16_C(51831), UINT16_C(48765), UINT16_C(31181), UINT16_C(18716), UINT16_C(47342), UINT16_C(46288), UINT16_C(19636) },
      { UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(30039),      UINT16_MAX, UINT16_C(64943),      UINT16_MAX, UINT16_C(    0),
        UINT16_C(    0),      UINT16_MAX,      UINT16_MAX, UINT16_C(31764), UINT16_C(42776),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i a = simde_x_mm256_loadu_epu16(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epu16(test_vec[i].b);
    simde__m256i r = simde_mm256_maskz_adds_epu16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x16(r, simde_x_mm256_loadu_epu16(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask32 k = simde_test_x86_random_mmask32();
    simde__m256i a = simde_test_x86_random_u16x16();
    simde__m256i b = simde_test_x86_random_u16x16();
    simde__m256i r = simde_mm256_maskz_adds_epu16(k, a, b);

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_u16x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_u16x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_adds_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi8(INT8_C( -30), INT8_C( -16), INT8_C( -64), INT8_C( 113),
                           INT8_C(  51), INT8_C(-115), INT8_C(  19), INT8_C( -80),
                           INT8_C(-110), INT8_C( -62), INT8_C( -91), INT8_C(   8),
                           INT8_C(  11), INT8_C( -49), INT8_C( -43), INT8_C(-101),
                           INT8_C(  16), INT8_C(  61), INT8_C(  -2), INT8_C(-100),
                           INT8_C(   1), INT8_C( -80), INT8_C( 127), INT8_C(-105),
                           INT8_C( -74), INT8_C(-126), INT8_C( 122), INT8_C(  30),
                           INT8_C( -73), INT8_C(  48), INT8_C( -33), INT8_C(   2),
                           INT8_C( -82), INT8_C( -10), INT8_C(  33), INT8_C(  20),
                           INT8_C(-120), INT8_C(-120), INT8_C( -46), INT8_C( -54),
                           INT8_C( -43), INT8_C(  63), INT8_C(  81), INT8_C( 103),
                           INT8_C( 118), INT8_C( -84), INT8_C(  76), INT8_C(  47),
                           INT8_C( 101), INT8_C( -83), INT8_C(  15), INT8_C(  23),
                           INT8_C(  36), INT8_C(-116), INT8_C( -88), INT8_C( -45),
                           INT8_C(  -6), INT8_C(  86), INT8_C(  20), INT8_C(  39),
                           INT8_C( 117), INT8_C(  80), INT8_C(  97), INT8_C(  41)),
      simde_mm512_set_epi8(INT8_C( 122), INT8_C(  43), INT8_C(  38), INT8_C(  97),
                           INT8_C(  38), INT8_C(  68), INT8_C(  88), INT8_C(  56),
                           INT8_C( -39), INT8_C(  61), INT8_C( -57), INT8_C( -66),
                           INT8_C(-121), INT8_C(  29), INT8_C( 110), INT8_C( 127),
                           INT8_C(-115), INT8_C(  16), INT8_C( -46), INT8_C(  65),
                           INT8_C( -36), INT8_C(  26), INT8_C(  12), INT8_C( -79),
                           INT8_C( -66), INT8_C( -28), INT8_C( -67), INT8_C( -79),
                           INT8_C( -60), INT8_C( -13), INT8_C(  74), INT8_C(  25),
                           INT8_C(  46), INT8_C(   8), INT8_C(  70), INT8_C( 106),
                           INT8_C(-119), INT8_C(   0), INT8_C( -96), INT8_C( -51),
                           INT8_C( -50), INT8_C(  93), INT8_C(  92), INT8_C(  65),
                           INT8_C( -96), INT8_C(-126), INT8_C(   4), INT8_C(   9),
                           INT8_C(  26), INT8_C(-119), INT8_C(  93), INT8_C(  15),
                           INT8_C(-115), INT8_C(  77), INT8_C( 125), INT8_C( -62),
                           INT8_C(  77), INT8_C( -22), INT8_C( 101), INT8_C(  83),
                           INT8_C(  45), INT8_C( -68), INT8_C(  97), INT8_C( -19)),
      simde_mm512_set_epi8(INT8_C(  92), INT8_C(  27), INT8_C( -26), INT8_C( 127),
                           INT8_C(  89), INT8_C( -47), INT8_C( 107), INT8_C( -24),
                           INT8_C(-128), INT8_C(  -1), INT8_C(-128), INT8_C( -58),
                           INT8_C(-110), INT8_C( -20), INT8_C(  67), INT8_C(  26),
                           INT8_C( -99), INT8_C(  77), INT8_C( -48), INT8_C( -35),
                           INT8_C( -35), INT8_C( -54), INT8_C( 127), INT8_C(-128),
                           INT8_C(-128), INT8_C(-128), INT8_C(  55), INT8_C( -49),
                           INT8_C(-128), INT8_C(  35), INT8_C(  41), INT8_C(  27),
                           INT8_C( -36), INT8_C(  -2), INT8_C( 103), INT8_C( 126),
                           INT8_C(-128), INT8_C(-120), INT8_C(-128), INT8_C(-105),
                           INT8_C( -93), INT8_C( 127), INT8_C( 127), INT8_C( 127),
                           INT8_C(  22), INT8_C(-128), INT8_C(  80), INT8_C(  56),
                           INT8_C( 127), INT8_C(-128), INT8_C( 108), INT8_C(  38),
                           INT8_C( -79), INT8_C( -39), INT8_C(  37), INT8_C(-107),
                           INT8_C(  71), INT8_C(  64), INT8_C( 121), INT8_C( 122),
                           INT8_C( 127), INT8_C(  12), INT8_C( 127), INT8_C(  22)) },
    { simde_mm512_set_epi8(INT8_C(  60), INT8_C(  96), INT8_C(-106), INT8_C(-115),
                           INT8_C(  31), INT8_C(-108), INT8_C(  15), INT8_C(  63),
                           INT8_C( -77), INT8_C( 125), INT8_C(  41), INT8_C( -80),
                           INT8_C(  36), INT8_C(-118), INT8_C(-116), INT8_C( -63),
                           INT8_C( -44), INT8_C(  70), INT8_C(  99), INT8_C(  48),
                           INT8_C(  67), INT8_C(  43), INT8_C(  83), INT8_C(  53),
                           INT8_C(  60), INT8_C(  57), INT8_C(-118), INT8_C(  55),
                           INT8_C( -62), INT8_C( -42), INT8_C(  19), INT8_C(  52),
                           INT8_C( -37), INT8_C( -32), INT8_C(  33), INT8_C(-114),
                           INT8_C( -96), INT8_C(  53), INT8_C(  18), INT8_C( -37),
                           INT8_C(   6), INT8_C( -87), INT8_C( -11), INT8_C( -10),
                           INT8_C(  -4), INT8_C(  84), INT8_C( 104), INT8_C(  55),
                           INT8_C(  92), INT8_C( 117), INT8_C(  16), INT8_C( 115),
                           INT8_C(  83), INT8_C(  80), INT8_C(-125), INT8_C(  64),
                           INT8_C(  74), INT8_C(  -9), INT8_C( -62), INT8_C(  67),
                           INT8_C( 102), INT8_C(  66), INT8_C( -86), INT8_C(-112)),
      simde_mm512_set_epi8(INT8_C( -97), INT8_C(  52), INT8_C( -91), INT8_C(   7),
                           INT8_C(  -1), INT8_C( -19), INT8_C(  38), INT8_C(  15),
                           INT8_C(  76), INT8_C(  63), INT8_C(  10), INT8_C(  18),
                           INT8_C(  54), INT8_C(  73), INT8_C(  15), INT8_C( -54),
                           INT8_C(-114), INT8_C(  63), INT8_C(  44), INT8_C(   4),
                           INT8_C(  41), INT8_C(  90), INT8_C( -78), INT8_C( -67),
                           INT8_C(  95), INT8_C( -14), INT8_C( -10), INT8_C( -44),
                           INT8_C(  53), INT8_C( 102), INT8_C( 127), INT8_C( 120),
                           INT8_C( -89), INT8_C(  10), INT8_C( -36), INT8_C( -50),
                           INT8_C( 118), INT8_C(-106), INT8_C( -52), INT8_C(  76),
                           INT8_C(  66), INT8_C( -36), INT8_C( -79), INT8_C(  91),
                           INT8_C(  14), INT8_C( 103), INT8_C( -73), INT8_C(-123),
                           INT8_C(  68), INT8_C( -54), INT8_C(-103), INT8_C(  38),
                           INT8_C(  58), INT8_C(   0), INT8_C( -62), INT8_C( -76),
                           INT8_C(  93), INT8_C( 119), INT8_C(  14), INT8_C(  84),
                           INT8_C( -83), INT8_C(  76), INT8_C( -47), INT8_C( -53)),
      simde_mm512_set_epi8(INT8_C( -37), INT8_C( 127), INT8_C(-128), INT8_C(-108),
                           INT8_C(  30), INT8_C(-127), INT8_C(  53), INT8_C(  78),
                           INT8_C(  -1), INT8_C( 127), INT8_C(  51), INT8_C( -62),
                           INT8_C(  90), INT8_C( -45), INT8_C(-101), INT8_C(-117),
                           INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C(  52),
                           INT8_C( 108), INT8_C( 127), INT8_C(   5), INT8_C( -14),
                           INT8_C( 127), INT8_C(  43), INT8_C(-128), INT8_C(  11),
                           INT8_C(  -9), INT8_C(  60), INT8_C( 127), INT8_C( 127),
                           INT8_C(-126), INT8_C( -22), INT8_C(  -3), INT8_C(-128),
                           INT8_C(  22), INT8_C( -53), INT8_C( -34), INT8_C(  39),
                           INT8_C(  72), INT8_C(-123), INT8_C( -90), INT8_C(  81),
                           INT8_C(  10), INT8_C( 127), INT8_C(  31), INT8_C( -68),
                           INT8_C( 127), INT8_C(  63), INT8_C( -87), INT8_C( 127),
                           INT8_C( 127), INT8_C(  80), INT8_C(-128), INT8_C( -12),
                           INT8_C( 127), INT8_C( 110), INT8_C( -48), INT8_C( 127),
                           INT8_C(  19), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi8(INT8_C( -80), INT8_C( 122), INT8_C(-100), INT8_C( 123),
                           INT8_C(  82), INT8_C(-107), INT8_C(  95), INT8_C( -80),
                           INT8_C(-115), INT8_C(-109), INT8_C(  60), INT8_C(  72),
                           INT8_C(  57), INT8_C(  99), INT8_C( -30), INT8_C(-115),
                           INT8_C(  69), INT8_C(  18), INT8_C(  -5), INT8_C(  -6),
                           INT8_C( -90), INT8_C( -20), INT8_C(  17), INT8_C( 121),
                           INT8_C(-100), INT8_C( -55), INT8_C(  90), INT8_C( -42),
                           INT8_C(-113), INT8_C( 119), INT8_C(  44), INT8_C(-123),
                           INT8_C( -39), INT8_C( 114), INT8_C(  37), INT8_C( -84),
                           INT8_C( -88), INT8_C(  15), INT8_C(  23), INT8_C( 114),
                           INT8_C(  63), INT8_C( -61), INT8_C(  22), INT8_C(   8),
                           INT8_C(  54), INT8_C(  19), INT8_C( -34), INT8_C(  18),
                           INT8_C( -36), INT8_C( 104), INT8_C(   1), INT8_C(   5),
                           INT8_C(-116), INT8_C(  79), INT8_C( -20), INT8_C(  -4),
                           INT8_C(  95), INT8_C( 103), INT8_C(   3), INT8_C(  64),
                           INT8_C(  34), INT8_C( -49), INT8_C( -27), INT8_C(  41)),
      simde_mm512_set_epi8(INT8_C(  97), INT8_C(-116), INT8_C( -44), INT8_C( -44),
                           INT8_C(  67), INT8_C( -66), INT8_C(  77), INT8_C(  60),
                           INT8_C(  74), INT8_C( -19), INT8_C(-123), INT8_C( -81),
                           INT8_C( -74), INT8_C( -21), INT8_C( -32), INT8_C(-122),
                           INT8_C(-110), INT8_C( 116), INT8_C( 115), INT8_C(  99),
                           INT8_C(  29), INT8_C(  50), INT8_C(-119), INT8_C(-126),
                           INT8_C( -53), INT8_C(  68), INT8_C( -37), INT8_C(  84),
                           INT8_C(  23), INT8_C( -10), INT8_C(  15), INT8_C(  37),
                           INT8_C( -93), INT8_C(  57), INT8_C(  28), INT8_C(-116),
                           INT8_C(  56), INT8_C(-124), INT8_C( -93), INT8_C( -67),
                           INT8_C(  27), INT8_C(  -1), INT8_C( -70), INT8_C( -30),
                           INT8_C( -39), INT8_C(  59), INT8_C(  76), INT8_C( 108),
                           INT8_C(  13), INT8_C( -29), INT8_C(  60), INT8_C( 106),
                           INT8_C( 106), INT8_C( -77), INT8_C(-102), INT8_C(   3),
                           INT8_C( -18), INT8_C(  23), INT8_C( 113), INT8_C( -97),
                           INT8_C( 109), INT8_C(  81), INT8_C(  42), INT8_C(  93)),
      simde_mm512_set_epi8(INT8_C(  17), INT8_C(   6), INT8_C(-128), INT8_C(  79),
                           INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( -20),
                           INT8_C( -41), INT8_C(-128), INT8_C( -63), INT8_C(  -9),
                           INT8_C( -17), INT8_C(  78), INT8_C( -62), INT8_C(-128),
                           INT8_C( -41), INT8_C( 127), INT8_C( 110), INT8_C(  93),
                           INT8_C( -61), INT8_C(  30), INT8_C(-102), INT8_C(  -5),
                           INT8_C(-128), INT8_C(  13), INT8_C(  53), INT8_C(  42),
                           INT8_C( -90), INT8_C( 109), INT8_C(  59), INT8_C( -86),
                           INT8_C(-128), INT8_C( 127), INT8_C(  65), INT8_C(-128),
                           INT8_C( -32), INT8_C(-109), INT8_C( -70), INT8_C(  47),
                           INT8_C(  90), INT8_C( -62), INT8_C( -48), INT8_C( -22),
                           INT8_C(  15), INT8_C(  78), INT8_C(  42), INT8_C( 126),
                           INT8_C( -23), INT8_C(  75), INT8_C(  61), INT8_C( 111),
                           INT8_C( -10), INT8_C(   2), INT8_C(-122), INT8_C(  -1),
                           INT8_C(  77), INT8_C( 126), INT8_C( 116), INT8_C( -33),
                           INT8_C( 127), INT8_C(  32), INT8_C(  15), INT8_C( 127)) },
    { simde_mm512_set_epi8(INT8_C( -94), INT8_C(  96), INT8_C( -77), INT8_C( -55),
                           INT8_C(  18), INT8_C(  12), INT8_C(  78), INT8_C( -17),
                           INT8_C( -94), INT8_C(  83), INT8_C( -20), INT8_C(-101),
                           INT8_C( -62), INT8_C( -62), INT8_C( -77), INT8_C(  21),
                           INT8_C(  21), INT8_C( -49), INT8_C(-127), INT8_C( -47),
                           INT8_C( -26), INT8_C(  62), INT8_C( -51), INT8_C( -34),
                           INT8_C(   9), INT8_C( -56), INT8_C(  63), INT8_C(   4),
                           INT8_C(  64), INT8_C(  46), INT8_C(  68), INT8_C( -18),
                           INT8_C(-110), INT8_C( 103), INT8_C(-101), INT8_C(  45),
                           INT8_C( -81), INT8_C( 125), INT8_C(   8), INT8_C(  72),
                           INT8_C( -48), INT8_C( -28), INT8_C( 102), INT8_C( -90),
                           INT8_C(-101), INT8_C( -40), INT8_C( -94), INT8_C(-127),
                           INT8_C( -36), INT8_C( -32), INT8_C(  34), INT8_C( -39),
                           INT8_C( -64), INT8_C( 111), INT8_C(  36), INT8_C(  91),
                           INT8_C(  90), INT8_C( -10), INT8_C(  88), INT8_C(-117),
                           INT8_C( 117), INT8_C(-111), INT8_C(-128), INT8_C( -35)),
      simde_mm512_set_epi8(INT8_C(  65), INT8_C(  -2), INT8_C(   4), INT8_C( -77),
                           INT8_C(  33), INT8_C(-119), INT8_C( -36), INT8_C(  95),
                           INT8_C( -54), INT8_C(  57), INT8_C( -15), INT8_C(  20),
                           INT8_C( 119), INT8_C( -56), INT8_C( -57), INT8_C(  22),
                           INT8_C( -23), INT8_C( -56), INT8_C( -77), INT8_C( 114),
                           INT8_C( -25), INT8_C(  85), INT8_C(  65), INT8_C( 124),
                           INT8_C(  80), INT8_C(  94), INT8_C(  -6), INT8_C(-128),
                           INT8_C(  25), INT8_C( -80), INT8_C(  31), INT8_C( -23),
                           INT8_C( 123), INT8_C(-123), INT8_C( -56), INT8_C(  34),
                           INT8_C(  72), INT8_C( -39), INT8_C( -17), INT8_C(  39),
                           INT8_C(  29), INT8_C( -88), INT8_C(  65), INT8_C(  81),
                           INT8_C(-120), INT8_C(  80), INT8_C( -76), INT8_C( -59),
                           INT8_C(  48), INT8_C( -90), INT8_C( -85), INT8_C(  82),
                           INT8_C( 115), INT8_C(-114), INT8_C(  84), INT8_C( -40),
                           INT8_C( -69), INT8_C( -11), INT8_C(-118), INT8_C(  39),
                           INT8_C( -17), INT8_C( -23), INT8_C(   4), INT8_C( -37)),
      simde_mm512_set_epi8(INT8_C( -29), INT8_C(  94), INT8_C( -73), INT8_C(-128),
                           INT8_C(  51), INT8_C(-107), INT8_C(  42), INT8_C(  78),
                           INT8_C(-128), INT8_C( 127), INT8_C( -35), INT8_C( -81),
                           INT8_C(  57), INT8_C(-118), INT8_C(-128), INT8_C(  43),
                           INT8_C(  -2), INT8_C(-105), INT8_C(-128), INT8_C(  67),
                           INT8_C( -51), INT8_C( 127), INT8_C(  14), INT8_C(  90),
                           INT8_C(  89), INT8_C(  38), INT8_C(  57), INT8_C(-124),
                           INT8_C(  89), INT8_C( -34), INT8_C(  99), INT8_C( -41),
                           INT8_C(  13), INT8_C( -20), INT8_C(-128), INT8_C(  79),
                           INT8_C(  -9), INT8_C(  86), INT8_C(  -9), INT8_C( 111),
                           INT8_C( -19), INT8_C(-116), INT8_C( 127), INT8_C(  -9),
                           INT8_C(-128), INT8_C(  40), INT8_C(-128), INT8_C(-128),
                           INT8_C(  12), INT8_C(-122), INT8_C( -51), INT8_C(  43),
                           INT8_C(  51), INT8_C(  -3), INT8_C( 120), INT8_C(  51),
                           INT8_C(  21), INT8_C( -21), INT8_C( -30), INT8_C( -78),
                           INT8_C( 100), INT8_C(-128), INT8_C(-124), INT8_C( -72)) },
    { simde_mm512_set_epi8(INT8_C(  31), INT8_C( -81), INT8_C( 115), INT8_C(  18),
                           INT8_C(  20), INT8_C( -33), INT8_C(-125), INT8_C( -45),
                           INT8_C(  54), INT8_C(  82), INT8_C(  61), INT8_C(  81),
                           INT8_C( 123), INT8_C(  18), INT8_C(  89), INT8_C(  34),
                           INT8_C(  10), INT8_C( -54), INT8_C( -24), INT8_C(  54),
                           INT8_C(  49), INT8_C( -70), INT8_C(   7), INT8_C( -12),
                           INT8_C( 110), INT8_C( -16), INT8_C( -10), INT8_C(   4),
                           INT8_C(   3), INT8_C( -17), INT8_C( -46), INT8_C( -29),
                           INT8_C(  26), INT8_C( -38), INT8_C(-107), INT8_C( -25),
                           INT8_C(  61), INT8_C(  -4), INT8_C( -21), INT8_C( -11),
                           INT8_C( -29), INT8_C(  62), INT8_C( 120), INT8_C(  84),
                           INT8_C( -48), INT8_C( -42), INT8_C( -15), INT8_C( -64),
                           INT8_C( -90), INT8_C( 111), INT8_C(-123), INT8_C(  61),
                           INT8_C(  22), INT8_C( -48), INT8_C(-107), INT8_C( 122),
                           INT8_C(  27), INT8_C( 119), INT8_C( -63), INT8_C( -61),
                           INT8_C( -69), INT8_C(  46), INT8_C(  34), INT8_C(-101)),
      simde_mm512_set_epi8(INT8_C(  44), INT8_C( 101), INT8_C( -48), INT8_C( -49),
                           INT8_C(  63), INT8_C(  93), INT8_C( -48), INT8_C(-122),
                           INT8_C( -67), INT8_C(  27), INT8_C(   1), INT8_C( -55),
                           INT8_C( -79), INT8_C( -44), INT8_C(-115), INT8_C(-126),
                           INT8_C( -42), INT8_C(  29), INT8_C(  66), INT8_C( -83),
                           INT8_C( 108), INT8_C(  -1), INT8_C( -47), INT8_C(-104),
                           INT8_C(  62), INT8_C(  29), INT8_C(   4), INT8_C( -47),
                           INT8_C(  98), INT8_C(-109), INT8_C( -23), INT8_C(  69),
                           INT8_C(-110), INT8_C( -47), INT8_C(  48), INT8_C(-101),
                           INT8_C( -80), INT8_C(-110), INT8_C( -55), INT8_C(  25),
                           INT8_C(  39), INT8_C(  58), INT8_C(   4), INT8_C(-108),
                           INT8_C( -23), INT8_C( -79), INT8_C( 124), INT8_C(  36),
                           INT8_C( 113), INT8_C( 110), INT8_C( 105), INT8_C( 113),
                           INT8_C( -54), INT8_C( -63), INT8_C(  50), INT8_C( -99),
                           INT8_C(  -8), INT8_C( -76), INT8_C( 102), INT8_C(  51),
                           INT8_C(  58), INT8_C(  16), INT8_C( -94), INT8_C( -93)),
      simde_mm512_set_epi8(INT8_C(  75), INT8_C(  20), INT8_C(  67), INT8_C( -31),
                           INT8_C(  83), INT8_C(  60), INT8_C(-128), INT8_C(-128),
                           INT8_C( -13), INT8_C( 109), INT8_C(  62), INT8_C(  26),
                           INT8_C(  44), INT8_C( -26), INT8_C( -26), INT8_C( -92),
                           INT8_C( -32), INT8_C( -25), INT8_C(  42), INT8_C( -29),
                           INT8_C( 127), INT8_C( -71), INT8_C( -40), INT8_C(-116),
                           INT8_C( 127), INT8_C(  13), INT8_C(  -6), INT8_C( -43),
                           INT8_C( 101), INT8_C(-126), INT8_C( -69), INT8_C(  40),
                           INT8_C( -84), INT8_C( -85), INT8_C( -59), INT8_C(-126),
                           INT8_C( -19), INT8_C(-114), INT8_C( -76), INT8_C(  14),
                           INT8_C(  10), INT8_C( 120), INT8_C( 124), INT8_C( -24),
                           INT8_C( -71), INT8_C(-121), INT8_C( 109), INT8_C( -28),
                           INT8_C(  23), INT8_C( 127), INT8_C( -18), INT8_C( 127),
                           INT8_C( -32), INT8_C(-111), INT8_C( -57), INT8_C(  23),
                           INT8_C(  19), INT8_C(  43), INT8_C(  39), INT8_C( -10),
                           INT8_C( -11), INT8_C(  62), INT8_C( -60), INT8_C(-128)) },
    { simde_mm512_set_epi8(INT8_C(  64), INT8_C(  55), INT8_C(  97), INT8_C( -61),
                           INT8_C(  89), INT8_C( 120), INT8_C(  49), INT8_C( 124),
                           INT8_C(  16), INT8_C( 113), INT8_C(  48), INT8_C(  70),
                           INT8_C(  60), INT8_C( -11), INT8_C(  98), INT8_C(  70),
                           INT8_C( 111), INT8_C(  55), INT8_C( -50), INT8_C( -21),
                           INT8_C(-102), INT8_C(   3), INT8_C(  75), INT8_C(  -9),
                           INT8_C(  34), INT8_C(  96), INT8_C(-104), INT8_C(  37),
                           INT8_C(  74), INT8_C(  22), INT8_C(  13), INT8_C(  82),
                           INT8_C(-108), INT8_C(-112), INT8_C(  49), INT8_C( -58),
                           INT8_C( 116), INT8_C(  58), INT8_C( -29), INT8_C(  60),
                           INT8_C( 125), INT8_C( 105), INT8_C( -20), INT8_C(  -4),
                           INT8_C(-106), INT8_C(-111), INT8_C( -29), INT8_C( -32),
                           INT8_C( -43), INT8_C(-103), INT8_C(  84), INT8_C(  40),
                           INT8_C(  63), INT8_C(  88), INT8_C( -74), INT8_C( -49),
                           INT8_C(  10), INT8_C(  50), INT8_C(  38), INT8_C( 108),
                           INT8_C(  42), INT8_C( -40), INT8_C( -81), INT8_C(  31)),
      simde_mm512_set_epi8(INT8_C(-107), INT8_C( 105), INT8_C( -15), INT8_C( -49),
                           INT8_C(  74), INT8_C( -31), INT8_C( -49), INT8_C( 112),
                           INT8_C(-103), INT8_C(  69), INT8_C( -55), INT8_C(-115),
                           INT8_C( -48), INT8_C(  96), INT8_C(  -8), INT8_C( -35),
                           INT8_C( -24), INT8_C( 101), INT8_C(  70), INT8_C( -69),
                           INT8_C(-105), INT8_C( -28), INT8_C( -38), INT8_C( -85),
                           INT8_C( -94), INT8_C(-122), INT8_C( -80), INT8_C(  95),
                           INT8_C(  -1), INT8_C(  87), INT8_C(  21), INT8_C(   6),
                           INT8_C(-103), INT8_C(  38), INT8_C(  15), INT8_C( -98),
                           INT8_C( -27), INT8_C(  57), INT8_C( -76), INT8_C(-112),
                           INT8_C(  77), INT8_C(  -3), INT8_C(  57), INT8_C(   3),
                           INT8_C( 123), INT8_C( -18), INT8_C( -90), INT8_C(-119),
                           INT8_C( 112), INT8_C(  61), INT8_C( -48), INT8_C( -16),
                           INT8_C( -36), INT8_C(-125), INT8_C( -10), INT8_C(  77),
                           INT8_C( 106), INT8_C(  72), INT8_C(  62), INT8_C( -24),
                           INT8_C( -65), INT8_C( 102), INT8_C(-122), INT8_C(  19)),
      simde_mm512_set_epi8(INT8_C( -43), INT8_C( 127), INT8_C(  82), INT8_C(-110),
                           INT8_C( 127), INT8_C(  89), INT8_C(   0), INT8_C( 127),
                           INT8_C( -87), INT8_C( 127), INT8_C(  -7), INT8_C( -45),
                           INT8_C(  12), INT8_C(  85), INT8_C(  90), INT8_C(  35),
                           INT8_C(  87), INT8_C( 127), INT8_C(  20), INT8_C( -90),
                           INT8_C(-128), INT8_C( -25), INT8_C(  37), INT8_C( -94),
                           INT8_C( -60), INT8_C( -26), INT8_C(-128), INT8_C( 127),
                           INT8_C(  73), INT8_C( 109), INT8_C(  34), INT8_C(  88),
                           INT8_C(-128), INT8_C( -74), INT8_C(  64), INT8_C(-128),
                           INT8_C(  89), INT8_C( 115), INT8_C(-105), INT8_C( -52),
                           INT8_C( 127), INT8_C( 102), INT8_C(  37), INT8_C(  -1),
                           INT8_C(  17), INT8_C(-128), INT8_C(-119), INT8_C(-128),
                           INT8_C(  69), INT8_C( -42), INT8_C(  36), INT8_C(  24),
                           INT8_C(  27), INT8_C( -37), INT8_C( -84), INT8_C(  28),
                           INT8_C( 116), INT8_C( 122), INT8_C( 100), INT8_C(  84),
                           INT8_C( -23), INT8_C(  62), INT8_C(-128), INT8_C(  50)) },
    { simde_mm512_set_epi8(INT8_C(  90), INT8_C(-125), INT8_C(  -5), INT8_C( -69),
                           INT8_C( 103), INT8_C( -40), INT8_C(  35), INT8_C( -38),
                           INT8_C( 107), INT8_C(-117), INT8_C( -80), INT8_C( 123),
                           INT8_C(  57), INT8_C( -70), INT8_C( -34), INT8_C(  38),
                           INT8_C( 114), INT8_C(-126), INT8_C(  20), INT8_C( -93),
                           INT8_C(  17), INT8_C(-128), INT8_C(   2), INT8_C(  15),
                           INT8_C(-122), INT8_C( -10), INT8_C(  -2), INT8_C(  84),
                           INT8_C(  93), INT8_C(  89), INT8_C(  57), INT8_C( -50),
                           INT8_C(   4), INT8_C( -63), INT8_C( 118), INT8_C(  19),
                           INT8_C( -68), INT8_C( -29), INT8_C( -16), INT8_C( -30),
                           INT8_C( -76), INT8_C(-110), INT8_C( 121), INT8_C(  87),
                           INT8_C(-117), INT8_C(-113), INT8_C( -94), INT8_C(  61),
                           INT8_C( -29), INT8_C(  43), INT8_C( -69), INT8_C( 123),
                           INT8_C(  79), INT8_C(-101), INT8_C(  40), INT8_C(  64),
                           INT8_C(  34), INT8_C(-102), INT8_C(  77), INT8_C( 114),
                           INT8_C( -49), INT8_C( -31), INT8_C( 103), INT8_C(  81)),
      simde_mm512_set_epi8(INT8_C(  34), INT8_C(-125), INT8_C( -96), INT8_C(-111),
                           INT8_C(  23), INT8_C(  58), INT8_C(  -5), INT8_C( -22),
                           INT8_C( -80), INT8_C( -90), INT8_C( -33), INT8_C( -65),
                           INT8_C( -34), INT8_C( 100), INT8_C( -10), INT8_C( -57),
                           INT8_C(  37), INT8_C(  -9), INT8_C( -29), INT8_C(-114),
                           INT8_C(-102), INT8_C(  53), INT8_C( 107), INT8_C( 124),
                           INT8_C( -12), INT8_C( -57), INT8_C( -57), INT8_C( -37),
                           INT8_C(  50), INT8_C(  22), INT8_C( -71), INT8_C(  -2),
                           INT8_C( -31), INT8_C(  60), INT8_C(  62), INT8_C(  19),
                           INT8_C( 117), INT8_C(   1), INT8_C( -26), INT8_C(-100),
                           INT8_C(  29), INT8_C(  -6), INT8_C(   7), INT8_C(-100),
                           INT8_C(-120), INT8_C( -33), INT8_C(  17), INT8_C(-103),
                           INT8_C(  10), INT8_C( -45), INT8_C( -76), INT8_C(  10),
                           INT8_C( -10), INT8_C(  15), INT8_C(  36), INT8_C(   2),
                           INT8_C(  74), INT8_C(  44), INT8_C( -16), INT8_C( 122),
                           INT8_C(  -4), INT8_C(  15), INT8_C( -27), INT8_C( -15)),
      simde_mm512_set_epi8(INT8_C( 124), INT8_C(-128), INT8_C(-101), INT8_C(-128),
                           INT8_C( 126), INT8_C(  18), INT8_C(  30), INT8_C( -60),
                           INT8_C(  27), INT8_C(-128), INT8_C(-113), INT8_C(  58),
                           INT8_C(  23), INT8_C(  30), INT8_C( -44), INT8_C( -19),
                           INT8_C( 127), INT8_C(-128), INT8_C(  -9), INT8_C(-128),
                           INT8_C( -85), INT8_C( -75), INT8_C( 109), INT8_C( 127),
                           INT8_C(-128), INT8_C( -67), INT8_C( -59), INT8_C(  47),
                           INT8_C( 127), INT8_C( 111), INT8_C( -14), INT8_C( -52),
                           INT8_C( -27), INT8_C(  -3), INT8_C( 127), INT8_C(  38),
                           INT8_C(  49), INT8_C( -28), INT8_C( -42), INT8_C(-128),
                           INT8_C( -47), INT8_C(-116), INT8_C( 127), INT8_C( -13),
                           INT8_C(-128), INT8_C(-128), INT8_C( -77), INT8_C( -42),
                           INT8_C( -19), INT8_C(  -2), INT8_C(-128), INT8_C( 127),
                           INT8_C(  69), INT8_C( -86), INT8_C(  76), INT8_C(  66),
                           INT8_C( 108), INT8_C( -58), INT8_C(  61), INT8_C( 127),
                           INT8_C( -53), INT8_C( -16), INT8_C(  76), INT8_C(  66)) },
    { simde_mm512_set_epi8(INT8_C( -52), INT8_C(-111), INT8_C(  67), INT8_C( -14),
                           INT8_C(  11), INT8_C( -63), INT8_C( -35), INT8_C( -86),
                           INT8_C(-114), INT8_C( -97), INT8_C( 124), INT8_C( -30),
                           INT8_C(  -1), INT8_C(  55), INT8_C(  84), INT8_C(  12),
                           INT8_C(  86), INT8_C( -43), INT8_C(  81), INT8_C( -30),
                           INT8_C( -25), INT8_C(  74), INT8_C(-105), INT8_C(  11),
                           INT8_C( -97), INT8_C(   0), INT8_C(  98), INT8_C( 106),
                           INT8_C( -83), INT8_C( -89), INT8_C(  16), INT8_C( -42),
                           INT8_C(  81), INT8_C(  35), INT8_C(-127), INT8_C(-127),
                           INT8_C( -38), INT8_C(  51), INT8_C( -75), INT8_C(  24),
                           INT8_C(  86), INT8_C( -37), INT8_C(  97), INT8_C( -82),
                           INT8_C( 109), INT8_C(  21), INT8_C( 127), INT8_C(  16),
                           INT8_C( 101), INT8_C( -69), INT8_C(  95), INT8_C(-121),
                           INT8_C( -66), INT8_C(  39), INT8_C( 104), INT8_C(-109),
                           INT8_C(  -1), INT8_C(  35), INT8_C(  68), INT8_C(  93),
                           INT8_C( -25), INT8_C( 120), INT8_C( -19), INT8_C(  84)),
      simde_mm512_set_epi8(INT8_C(-111), INT8_C(-108), INT8_C( 102), INT8_C( -40),
                           INT8_C(-124), INT8_C( -61), INT8_C(  80), INT8_C( 102),
                           INT8_C( -35), INT8_C(   3), INT8_C( -76), INT8_C(  98),
                           INT8_C(   9), INT8_C( -39), INT8_C( -24), INT8_C( -92),
                           INT8_C( -73), INT8_C( -19), INT8_C(-103), INT8_C( -41),
                           INT8_C( -32), INT8_C(  -8), INT8_C( 126), INT8_C(  75),
                           INT8_C( -64), INT8_C( -26), INT8_C( -15), INT8_C( -35),
                           INT8_C( -20), INT8_C(  -6), INT8_C(  86), INT8_C( -56),
                           INT8_C( -56), INT8_C(  67), INT8_C(  59), INT8_C( -73),
                           INT8_C(  35), INT8_C( -67), INT8_C(  92), INT8_C(  34),
                           INT8_C(  16), INT8_C( -12), INT8_C(  44), INT8_C( -59),
                           INT8_C( -82), INT8_C(-102), INT8_C(  93), INT8_C( -43),
                           INT8_C(  79), INT8_C( -19), INT8_C( -45), INT8_C( -87),
                           INT8_C( -50), INT8_C( -23), INT8_C( -17), INT8_C( 110),
                           INT8_C( 102), INT8_C( -59), INT8_C(-117), INT8_C(  84),
                           INT8_C( -75), INT8_C(  87), INT8_C(  69), INT8_C(  -6)),
      simde_mm512_set_epi8(INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C( -54),
                           INT8_C(-113), INT8_C(-124), INT8_C(  45), INT8_C(  16),
                           INT8_C(-128), INT8_C( -94), INT8_C(  48), INT8_C(  68),
                           INT8_C(   8), INT8_C(  16), INT8_C(  60), INT8_C( -80),
                           INT8_C(  13), INT8_C( -62), INT8_C( -22), INT8_C( -71),
                           INT8_C( -57), INT8_C(  66), INT8_C(  21), INT8_C(  86),
                           INT8_C(-128), INT8_C( -26), INT8_C(  83), INT8_C(  71),
                           INT8_C(-103), INT8_C( -95), INT8_C( 102), INT8_C( -98),
                           INT8_C(  25), INT8_C( 102), INT8_C( -68), INT8_C(-128),
                           INT8_C(  -3), INT8_C( -16), INT8_C(  17), INT8_C(  58),
                           INT8_C( 102), INT8_C( -49), INT8_C( 127), INT8_C(-128),
                           INT8_C(  27), INT8_C( -81), INT8_C( 127), INT8_C( -27),
                           INT8_C( 127), INT8_C( -88), INT8_C(  50), INT8_C(-128),
                           INT8_C(-116), INT8_C(  16), INT8_C(  87), INT8_C(   1),
                           INT8_C( 101), INT8_C( -24), INT8_C( -49), INT8_C( 127),
                           INT8_C(-100), INT8_C( 127), INT8_C(  50), INT8_C(  78)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_adds_epi8(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_adds_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask64 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi8(INT8_C(  92), INT8_C( 116), INT8_C( -78), INT8_C( -19),
                           INT8_C( -73), INT8_C(  22), INT8_C( -66), INT8_C( -29),
                           INT8_C(  55), INT8_C(  78), INT8_C( -45), INT8_C(-119),
                           INT8_C(  30), INT8_C( -29), INT8_C( -23), INT8_C(  89),
                           INT8_C(  15), INT8_C( 119), INT8_C( -88), INT8_C( 109),
                           INT8_C( -16), INT8_C( -10), INT8_C(   0), INT8_C( -18),
                           INT8_C(   1), INT8_C( 116), INT8_C(  62), INT8_C( -73),
                           INT8_C(-122), INT8_C(   0), INT8_C( 111), INT8_C(  95),
                           INT8_C( -20), INT8_C(   3), INT8_C( 122), INT8_C(-121),
                           INT8_C(  13), INT8_C(  25), INT8_C( -35), INT8_C(-127),
                           INT8_C(  82), INT8_C(  90), INT8_C(  53), INT8_C( 123),
                           INT8_C(  73), INT8_C( 108), INT8_C( -18), INT8_C(  15),
                           INT8_C(  81), INT8_C( -27), INT8_C( 102), INT8_C( 118),
                           INT8_C(  39), INT8_C( -77), INT8_C(  45), INT8_C(  81),
                           INT8_C( -17), INT8_C( -28), INT8_C(  67), INT8_C(-118),
                           INT8_C(  79), INT8_C(-113), INT8_C(-122), INT8_C( 124)),
      UINT64_C(10224647434006242820),
      simde_mm512_set_epi8(INT8_C(  68), INT8_C(-108), INT8_C(-121), INT8_C(-115),
                           INT8_C( -59), INT8_C( -78), INT8_C(-111), INT8_C(  56),
                           INT8_C(   9), INT8_C(-121), INT8_C( 120), INT8_C( -36),
                           INT8_C(  36), INT8_C(-103), INT8_C(-104), INT8_C(  25),
                           INT8_C(-111), INT8_C( -63), INT8_C( -35), INT8_C(-120),
                           INT8_C(  28), INT8_C( -44), INT8_C( -26), INT8_C( -86),
                           INT8_C( -13), INT8_C(  53), INT8_C( -88), INT8_C(-107),
                           INT8_C(  68), INT8_C(  42), INT8_C(-118), INT8_C( 111),
                           INT8_C(  54), INT8_C( -58), INT8_C( -13), INT8_C(  27),
                           INT8_C(  23), INT8_C(  41), INT8_C(-119), INT8_C(  44),
                           INT8_C(   7), INT8_C(-120), INT8_C(  32), INT8_C( -43),
                           INT8_C( 114), INT8_C( -72), INT8_C(  73), INT8_C( -96),
                           INT8_C(  96), INT8_C( 110), INT8_C( -81), INT8_C( -76),
                           INT8_C( 103), INT8_C(-100), INT8_C( -22), INT8_C(  18),
                           INT8_C( 115), INT8_C(  54), INT8_C( -40), INT8_C( 125),
                           INT8_C( 110), INT8_C(  31), INT8_C(  51), INT8_C(-104)),
      simde_mm512_set_epi8(INT8_C( -14), INT8_C(  85), INT8_C( -80), INT8_C(  80),
                           INT8_C(  48), INT8_C(  93), INT8_C(  79), INT8_C( 127),
                           INT8_C(  16), INT8_C(  41), INT8_C(  54), INT8_C(-116),
                           INT8_C(  17), INT8_C(  42), INT8_C( -86), INT8_C(  38),
                           INT8_C(-118), INT8_C( -56), INT8_C(  60), INT8_C(  19),
                           INT8_C(-100), INT8_C(-107), INT8_C( 105), INT8_C( -76),
                           INT8_C(  10), INT8_C(  -9), INT8_C( -12), INT8_C( -56),
                           INT8_C( -71), INT8_C(  96), INT8_C(   6), INT8_C(  24),
                           INT8_C(  69), INT8_C(  73), INT8_C( -27), INT8_C(   3),
                           INT8_C( -95), INT8_C(  50), INT8_C( -67), INT8_C( -39),
                           INT8_C(  97), INT8_C(  62), INT8_C( 101), INT8_C(  56),
                           INT8_C( -79), INT8_C(  83), INT8_C( -99), INT8_C( -56),
                           INT8_C( -17), INT8_C( -56), INT8_C(  -8), INT8_C( -16),
                           INT8_C(  84), INT8_C( -95), INT8_C( -73), INT8_C(  54),
                           INT8_C( 125), INT8_C(  85), INT8_C(  78), INT8_C( -65),
                           INT8_C(   4), INT8_C( 113), INT8_C( -16), INT8_C( -53)),
      simde_mm512_set_epi8(INT8_C(  54), INT8_C( 116), INT8_C( -78), INT8_C( -19),
                           INT8_C( -11), INT8_C(  15), INT8_C( -66), INT8_C( 127),
                           INT8_C(  25), INT8_C( -80), INT8_C( 127), INT8_C(-119),
                           INT8_C(  30), INT8_C( -61), INT8_C( -23), INT8_C(  63),
                           INT8_C(  15), INT8_C( 119), INT8_C(  25), INT8_C(-101),
                           INT8_C( -72), INT8_C(-128), INT8_C(  79), INT8_C( -18),
                           INT8_C(  -3), INT8_C( 116), INT8_C(-100), INT8_C( -73),
                           INT8_C(  -3), INT8_C(   0), INT8_C(-112), INT8_C( 127),
                           INT8_C( -20), INT8_C(   3), INT8_C( -40), INT8_C(  30),
                           INT8_C(  13), INT8_C(  25), INT8_C( -35), INT8_C(   5),
                           INT8_C( 104), INT8_C(  90), INT8_C(  53), INT8_C( 123),
                           INT8_C(  35), INT8_C(  11), INT8_C( -18), INT8_C(-128),
                           INT8_C(  81), INT8_C( -27), INT8_C( -89), INT8_C( 118),
                           INT8_C(  39), INT8_C(-128), INT8_C( -95), INT8_C(  81),
                           INT8_C( -17), INT8_C( -28), INT8_C(  67), INT8_C(-118),
                           INT8_C(  79), INT8_C( 127), INT8_C(-122), INT8_C( 124)) },
    { simde_mm512_set_epi8(INT8_C(-123), INT8_C(-100), INT8_C( -39), INT8_C( -44),
                           INT8_C(  22), INT8_C( 112), INT8_C(  16), INT8_C(  15),
                           INT8_C(-104), INT8_C(-102), INT8_C( 102), INT8_C( -95),
                           INT8_C(  69), INT8_C(  13), INT8_C( -21), INT8_C(  45),
                           INT8_C(  62), INT8_C(-120), INT8_C(-111), INT8_C(  32),
                           INT8_C(-107), INT8_C( -30), INT8_C(  99), INT8_C( -64),
                           INT8_C(   8), INT8_C( -42), INT8_C(  81), INT8_C( -34),
                           INT8_C( -46), INT8_C(  26), INT8_C(  31), INT8_C(  -2),
                           INT8_C(  68), INT8_C(  -7), INT8_C( -71), INT8_C(  46),
                           INT8_C( -21), INT8_C( -73), INT8_C(  21), INT8_C(  83),
                           INT8_C(-108), INT8_C( -97), INT8_C( -69), INT8_C(  73),
                           INT8_C(  57), INT8_C( -37), INT8_C(  21), INT8_C(  82),
                           INT8_C(-119), INT8_C(-126), INT8_C( 126), INT8_C(  91),
                           INT8_C( 115), INT8_C(  31), INT8_C( -79), INT8_C(  28),
                           INT8_C(-106), INT8_C( -18), INT8_C(  65), INT8_C(-104),
                           INT8_C(  81), INT8_C(  38), INT8_C( -84), INT8_C(  -2)),
      UINT64_C( 4597426592773770833),
      simde_mm512_set_epi8(INT8_C(  74), INT8_C(  72), INT8_C(  -4), INT8_C(  28),
                           INT8_C(-115), INT8_C(  93), INT8_C( 102), INT8_C(  44),
                           INT8_C(-103), INT8_C( -29), INT8_C( -50), INT8_C(  48),
                           INT8_C( -96), INT8_C( -50), INT8_C(  46), INT8_C( -65),
                           INT8_C(   4), INT8_C(  43), INT8_C( -75), INT8_C(  97),
                           INT8_C( -26), INT8_C(-103), INT8_C(  71), INT8_C(-107),
                           INT8_C(  91), INT8_C(  45), INT8_C( -11), INT8_C(  47),
                           INT8_C(  29), INT8_C(  25), INT8_C(  26), INT8_C(  -9),
                           INT8_C(  10), INT8_C(  36), INT8_C( -79), INT8_C( -53),
                           INT8_C(  41), INT8_C(   1), INT8_C( -23), INT8_C( -63),
                           INT8_C(-127), INT8_C(  68), INT8_C(  48), INT8_C(  36),
                           INT8_C(  89), INT8_C(-112), INT8_C( -31), INT8_C( 120),
                           INT8_C(  35), INT8_C(  62), INT8_C( -21), INT8_C(-114),
                           INT8_C(-104), INT8_C(  57), INT8_C(  42), INT8_C(-111),
                           INT8_C(  94), INT8_C( -63), INT8_C(  -9), INT8_C(  64),
                           INT8_C( -65), INT8_C(  -2), INT8_C( 110), INT8_C(  -8)),
      simde_mm512_set_epi8(INT8_C(  44), INT8_C( -92), INT8_C( -31), INT8_C(  26),
                           INT8_C( -99), INT8_C( -53), INT8_C( 117), INT8_C(  18),
                           INT8_C( -63), INT8_C(  45), INT8_C(  12), INT8_C(  24),
                           INT8_C(-108), INT8_C(  18), INT8_C( -60), INT8_C(  28),
                           INT8_C(  50), INT8_C( -11), INT8_C( -68), INT8_C( -31),
                           INT8_C( 105), INT8_C(-106), INT8_C(  98), INT8_C(  51),
                           INT8_C(  58), INT8_C( 103), INT8_C( 111), INT8_C(-127),
                           INT8_C(  68), INT8_C( -56), INT8_C( 124), INT8_C(-119),
                           INT8_C(  74), INT8_C( -62), INT8_C(-116), INT8_C(  37),
                           INT8_C( -12), INT8_C( 114), INT8_C(   0), INT8_C(  61),
                           INT8_C( 103), INT8_C(  -4), INT8_C(-105), INT8_C( -68),
                           INT8_C(  39), INT8_C(-100), INT8_C( -93), INT8_C(  11),
                           INT8_C( -80), INT8_C( -19), INT8_C( -22), INT8_C( -39),
                           INT8_C( 127), INT8_C( -38), INT8_C(-125), INT8_C(-111),
                           INT8_C(  84), INT8_C( -96), INT8_C(  87), INT8_C( -22),
                           INT8_C(  -5), INT8_C(  -3), INT8_C(-127), INT8_C(  41)),
      simde_mm512_set_epi8(INT8_C(-123), INT8_C(-100), INT8_C( -35), INT8_C(  54),
                           INT8_C(-128), INT8_C(  40), INT8_C( 127), INT8_C(  62),
                           INT8_C(-128), INT8_C(  16), INT8_C( 102), INT8_C( -95),
                           INT8_C(-128), INT8_C( -32), INT8_C( -21), INT8_C( -37),
                           INT8_C(  62), INT8_C(  32), INT8_C(-111), INT8_C(  66),
                           INT8_C(-107), INT8_C(-128), INT8_C( 127), INT8_C( -56),
                           INT8_C(   8), INT8_C( -42), INT8_C( 100), INT8_C( -34),
                           INT8_C( -46), INT8_C(  26), INT8_C(  31), INT8_C(  -2),
                           INT8_C(  84), INT8_C(  -7), INT8_C(-128), INT8_C( -16),
                           INT8_C(  29), INT8_C( 115), INT8_C( -23), INT8_C(  -2),
                           INT8_C(-108), INT8_C( -97), INT8_C( -57), INT8_C( -32),
                           INT8_C(  57), INT8_C(-128), INT8_C(-124), INT8_C( 127),
                           INT8_C(-119), INT8_C(-126), INT8_C( 126), INT8_C(  91),
                           INT8_C(  23), INT8_C(  19), INT8_C( -83), INT8_C(  28),
                           INT8_C(-106), INT8_C(-128), INT8_C(  65), INT8_C(  42),
                           INT8_C(  81), INT8_C(  38), INT8_C( -84), INT8_C(  33)) },
    { simde_mm512_set_epi8(INT8_C( -30), INT8_C( -37), INT8_C(  51), INT8_C( -36),
                           INT8_C(   8), INT8_C(  52), INT8_C(  97), INT8_C( 123),
                           INT8_C( -49), INT8_C(-124), INT8_C(  95), INT8_C( -83),
                           INT8_C(  70), INT8_C( -50), INT8_C( -61), INT8_C(  25),
                           INT8_C( -97), INT8_C(  28), INT8_C( -58), INT8_C(  11),
                           INT8_C( -14), INT8_C( 126), INT8_C(  81), INT8_C(  45),
                           INT8_C( -23), INT8_C( 120), INT8_C( -83), INT8_C( -16),
                           INT8_C(   7), INT8_C(  51), INT8_C( -57), INT8_C( -50),
                           INT8_C( -21), INT8_C(  98), INT8_C(  88), INT8_C(   0),
                           INT8_C( -66), INT8_C(   3), INT8_C( 124), INT8_C(-113),
                           INT8_C(  50), INT8_C(  88), INT8_C( -85), INT8_C( -93),
                           INT8_C( -44), INT8_C( -13), INT8_C( -94), INT8_C(  17),
                           INT8_C(  -2), INT8_C(  79), INT8_C(-116), INT8_C(  43),
                           INT8_C( -77), INT8_C(-125), INT8_C( -23), INT8_C(-120),
                           INT8_C(  96), INT8_C( -64), INT8_C( -23), INT8_C( -46),
                           INT8_C( -29), INT8_C( -71), INT8_C(  71), INT8_C(  90)),
      UINT64_C(12627002542648829104),
      simde_mm512_set_epi8(INT8_C(  56), INT8_C( -45), INT8_C(-108), INT8_C( -19),
                           INT8_C(-124), INT8_C( -27), INT8_C(  22), INT8_C( 126),
                           INT8_C(-106), INT8_C( -68), INT8_C( -60), INT8_C(   8),
                           INT8_C(  60), INT8_C(  93), INT8_C( -33), INT8_C( -27),
                           INT8_C(  -7), INT8_C(  27), INT8_C(-122), INT8_C( -38),
                           INT8_C(  23), INT8_C(   6), INT8_C(  45), INT8_C( -21),
                           INT8_C( -23), INT8_C(-101), INT8_C( 116), INT8_C( 127),
                           INT8_C(  96), INT8_C(  40), INT8_C( -97), INT8_C(  40),
                           INT8_C(  86), INT8_C( -44), INT8_C(  70), INT8_C( -71),
                           INT8_C(  62), INT8_C( -21), INT8_C(  66), INT8_C(  68),
                           INT8_C( -87), INT8_C( -61), INT8_C(  48), INT8_C( -70),
                           INT8_C(  18), INT8_C( -78), INT8_C( -98), INT8_C( 117),
                           INT8_C(  74), INT8_C(  32), INT8_C(  93), INT8_C( 125),
                           INT8_C( -47), INT8_C( -60), INT8_C( -86), INT8_C( 117),
                           INT8_C( 122), INT8_C( -54), INT8_C(  50), INT8_C( 123),
                           INT8_C( -31), INT8_C( -74), INT8_C( -64), INT8_C(  54)),
      simde_mm512_set_epi8(INT8_C( 115), INT8_C(  51), INT8_C( -91), INT8_C(  56),
                           INT8_C(  64), INT8_C( -39), INT8_C(-119), INT8_C( -28),
                           INT8_C( -54), INT8_C(  28), INT8_C(  54), INT8_C(  -8),
                           INT8_C( -54), INT8_C(-128), INT8_C( -28), INT8_C( -71),
                           INT8_C( 107), INT8_C( -66), INT8_C(-114), INT8_C( -88),
                           INT8_C(  34), INT8_C( -83), INT8_C( -21), INT8_C( -64),
                           INT8_C( 121), INT8_C( -20), INT8_C( -89), INT8_C( -94),
                           INT8_C( 112), INT8_C( -27), INT8_C(  81), INT8_C( -54),
                           INT8_C( -64), INT8_C(-114), INT8_C(  48), INT8_C( -89),
                           INT8_C( -61), INT8_C(  26), INT8_C(  43), INT8_C(  29),
                           INT8_C(   0), INT8_C(  64), INT8_C( 123), INT8_C( -67),
                           INT8_C(  15), INT8_C( 120), INT8_C(  36), INT8_C(  40),
                           INT8_C( 106), INT8_C(-118), INT8_C(-108), INT8_C( -58),
                           INT8_C(  26), INT8_C(-111), INT8_C(  63), INT8_C( -98),
                           INT8_C( -13), INT8_C( -12), INT8_C(-124), INT8_C(  96),
                           INT8_C( -13), INT8_C( -98), INT8_C(  99), INT8_C( -13)),
      simde_mm512_set_epi8(INT8_C( 127), INT8_C( -37), INT8_C(-128), INT8_C( -36),
                           INT8_C( -60), INT8_C( -66), INT8_C( -97), INT8_C(  98),
                           INT8_C( -49), INT8_C(-124), INT8_C(  -6), INT8_C(   0),
                           INT8_C(   6), INT8_C( -35), INT8_C( -61), INT8_C(  25),
                           INT8_C( -97), INT8_C(  28), INT8_C( -58), INT8_C(-126),
                           INT8_C(  57), INT8_C( -77), INT8_C(  24), INT8_C( -85),
                           INT8_C(  98), INT8_C(-121), INT8_C(  27), INT8_C( -16),
                           INT8_C( 127), INT8_C(  51), INT8_C( -57), INT8_C( -14),
                           INT8_C( -21), INT8_C(-128), INT8_C( 118), INT8_C(   0),
                           INT8_C(   1), INT8_C(   5), INT8_C( 124), INT8_C(-113),
                           INT8_C( -87), INT8_C(  88), INT8_C( -85), INT8_C( -93),
                           INT8_C(  33), INT8_C( -13), INT8_C( -94), INT8_C( 127),
                           INT8_C( 127), INT8_C(  79), INT8_C( -15), INT8_C(  43),
                           INT8_C( -77), INT8_C(-128), INT8_C( -23), INT8_C(-120),
                           INT8_C( 109), INT8_C( -64), INT8_C( -74), INT8_C( 127),
                           INT8_C( -29), INT8_C( -71), INT8_C(  71), INT8_C(  90)) },
    { simde_mm512_set_epi8(INT8_C(  72), INT8_C(  45), INT8_C( 120), INT8_C(  -5),
                           INT8_C(-109), INT8_C(  62), INT8_C(  17), INT8_C(  31),
                           INT8_C( -30), INT8_C( -58), INT8_C(  56), INT8_C(  21),
                           INT8_C(  72), INT8_C( -74), INT8_C( -40), INT8_C( 120),
                           INT8_C(  95), INT8_C( 108), INT8_C(  32), INT8_C(  64),
                           INT8_C(-128), INT8_C( 102), INT8_C( -21), INT8_C(  28),
                           INT8_C( 105), INT8_C(  52), INT8_C(  85), INT8_C(-104),
                           INT8_C(  57), INT8_C( -31), INT8_C( -38), INT8_C(-124),
                           INT8_C(-107), INT8_C(  -2), INT8_C(  55), INT8_C(  46),
                           INT8_C( -71), INT8_C(  77), INT8_C(  18), INT8_C(  70),
                           INT8_C(  89), INT8_C( 125), INT8_C( -42), INT8_C(-125),
                           INT8_C( 121), INT8_C( -11), INT8_C( -69), INT8_C( -59),
                           INT8_C( -53), INT8_C(  34), INT8_C(   9), INT8_C(  64),
                           INT8_C( -61), INT8_C( -25), INT8_C(-115), INT8_C( 100),
                           INT8_C(  65), INT8_C(   8), INT8_C(  69), INT8_C(  -8),
                           INT8_C( -15), INT8_C( -51), INT8_C(   1), INT8_C(  90)),
      UINT64_C(14515151237088493607),
      simde_mm512_set_epi8(INT8_C( -26), INT8_C(  -9), INT8_C(  66), INT8_C(   1),
                           INT8_C( -13), INT8_C(  60), INT8_C(-119), INT8_C( -83),
                           INT8_C(-122), INT8_C( -64), INT8_C( -83), INT8_C( -74),
                           INT8_C( 119), INT8_C(  -8), INT8_C(  12), INT8_C( 113),
                           INT8_C( -12), INT8_C( -84), INT8_C(   6), INT8_C(  69),
                           INT8_C(   2), INT8_C( -75), INT8_C( -34), INT8_C(-126),
                           INT8_C(   3), INT8_C(-128), INT8_C(  -9), INT8_C(  24),
                           INT8_C(  11), INT8_C( -94), INT8_C( -32), INT8_C( 110),
                           INT8_C(  33), INT8_C( -24), INT8_C( 125), INT8_C(  35),
                           INT8_C(-103), INT8_C( -48), INT8_C( -22), INT8_C(  38),
                           INT8_C( -81), INT8_C(   9), INT8_C( -11), INT8_C(-124),
                           INT8_C(  71), INT8_C(  31), INT8_C( -42), INT8_C(  93),
                           INT8_C(  67), INT8_C(  45), INT8_C(  51), INT8_C( -92),
                           INT8_C( 126), INT8_C( 108), INT8_C(-123), INT8_C( -71),
                           INT8_C( 113), INT8_C(  32), INT8_C(  71), INT8_C(  55),
                           INT8_C( -26), INT8_C(  82), INT8_C( -81), INT8_C( -20)),
      simde_mm512_set_epi8(INT8_C(-125), INT8_C( 121), INT8_C(-128), INT8_C( 103),
                           INT8_C(   0), INT8_C( 101), INT8_C( -41), INT8_C(  89),
                           INT8_C( -83), INT8_C( -65), INT8_C(   9), INT8_C(  -7),
                           INT8_C( -63), INT8_C(  13), INT8_C( 105), INT8_C(  92),
                           INT8_C( -18), INT8_C( -21), INT8_C(-102), INT8_C(-114),
                           INT8_C(  74), INT8_C( 121), INT8_C( -45), INT8_C(  52),
                           INT8_C( -63), INT8_C( -93), INT8_C(  98), INT8_C( 106),
                           INT8_C(-109), INT8_C( -47), INT8_C(  37), INT8_C(  70),
                           INT8_C( 100), INT8_C( 121), INT8_C(  18), INT8_C(  28),
                           INT8_C(-117), INT8_C( 107), INT8_C(   3), INT8_C( -62),
                           INT8_C(  42), INT8_C(  72), INT8_C(  91), INT8_C(  86),
                           INT8_C( -72), INT8_C(   9), INT8_C( -80), INT8_C( 118),
                           INT8_C( 122), INT8_C(-108), INT8_C( -70), INT8_C( -63),
                           INT8_C(  56), INT8_C(  71), INT8_C( -14), INT8_C(  49),
                           INT8_C( -73), INT8_C(  53), INT8_C( -29), INT8_C(   3),
                           INT8_C( -73), INT8_C(  43), INT8_C( -22), INT8_C(  85)),
      simde_mm512_set_epi8(INT8_C(-128), INT8_C( 112), INT8_C( 120), INT8_C(  -5),
                           INT8_C( -13), INT8_C(  62), INT8_C(  17), INT8_C(   6),
                           INT8_C( -30), INT8_C(-128), INT8_C( -74), INT8_C( -81),
                           INT8_C(  72), INT8_C( -74), INT8_C( -40), INT8_C( 120),
                           INT8_C(  95), INT8_C( 108), INT8_C( -96), INT8_C(  64),
                           INT8_C(  76), INT8_C(  46), INT8_C( -21), INT8_C( -74),
                           INT8_C( 105), INT8_C(  52), INT8_C(  89), INT8_C(-104),
                           INT8_C(  57), INT8_C(-128), INT8_C( -38), INT8_C( 127),
                           INT8_C(-107), INT8_C(  97), INT8_C(  55), INT8_C(  46),
                           INT8_C( -71), INT8_C(  77), INT8_C( -19), INT8_C( -24),
                           INT8_C( -39), INT8_C( 125), INT8_C(  80), INT8_C( -38),
                           INT8_C(  -1), INT8_C( -11), INT8_C(-122), INT8_C( 127),
                           INT8_C( -53), INT8_C( -63), INT8_C(   9), INT8_C(  64),
                           INT8_C( -61), INT8_C( -25), INT8_C(-115), INT8_C( 100),
                           INT8_C(  65), INT8_C(   8), INT8_C(  42), INT8_C(  -8),
                           INT8_C( -15), INT8_C( 125), INT8_C(-103), INT8_C(  65)) },
    { simde_mm512_set_epi8(INT8_C( -47), INT8_C(  84), INT8_C(-126), INT8_C( -64),
                           INT8_C(  14), INT8_C(  11), INT8_C(  37), INT8_C( -23),
                           INT8_C(  67), INT8_C( 124), INT8_C(  58), INT8_C( -94),
                           INT8_C(  30), INT8_C( -33), INT8_C(  70), INT8_C( -24),
                           INT8_C(  38), INT8_C( -97), INT8_C( -56), INT8_C( -60),
                           INT8_C( -59), INT8_C(  65), INT8_C( -74), INT8_C(  45),
                           INT8_C( -11), INT8_C(  55), INT8_C( -82), INT8_C(  12),
                           INT8_C( 106), INT8_C(  22), INT8_C(-124), INT8_C(  -4),
                           INT8_C(   2), INT8_C( -81), INT8_C(  14), INT8_C(  90),
                           INT8_C(-100), INT8_C(-122), INT8_C( -35), INT8_C(  81),
                           INT8_C( -14), INT8_C( -42), INT8_C( 125), INT8_C(-125),
                           INT8_C( -57), INT8_C(  90), INT8_C(  -9), INT8_C(  63),
                           INT8_C(  53), INT8_C(  77), INT8_C(  63), INT8_C( -84),
                           INT8_C(  27), INT8_C(  22), INT8_C(   3), INT8_C( -37),
                           INT8_C(  65), INT8_C( 118), INT8_C(-126), INT8_C(  97),
                           INT8_C( 109), INT8_C(   7), INT8_C(-114), INT8_C( -75)),
      UINT64_C( 8707623543556880126),
      simde_mm512_set_epi8(INT8_C(  84), INT8_C( -71), INT8_C(   8), INT8_C(  12),
                           INT8_C( -11), INT8_C( -76), INT8_C(  62), INT8_C(  93),
                           INT8_C( -75), INT8_C( -77), INT8_C( -84), INT8_C(-108),
                           INT8_C( -35), INT8_C(  14), INT8_C( -60), INT8_C(  18),
                           INT8_C(  23), INT8_C( -60), INT8_C( -63), INT8_C(-114),
                           INT8_C( -55), INT8_C(  75), INT8_C( -99), INT8_C( -55),
                           INT8_C(  58), INT8_C(  76), INT8_C(-102), INT8_C(-118),
                           INT8_C( -30), INT8_C(  39), INT8_C( 119), INT8_C(  85),
                           INT8_C(  -8), INT8_C( -72), INT8_C( -60), INT8_C( -94),
                           INT8_C(-112), INT8_C( 119), INT8_C( 124), INT8_C(  76),
                           INT8_C( -42), INT8_C(-124), INT8_C(  54), INT8_C(  74),
                           INT8_C( -92), INT8_C(  99), INT8_C(  79), INT8_C(  -3),
                           INT8_C(  61), INT8_C( -89), INT8_C(  84), INT8_C( -94),
                           INT8_C(  31), INT8_C(-116), INT8_C( -67), INT8_C(-102),
                           INT8_C( -72), INT8_C( -91), INT8_C(-105), INT8_C(-108),
                           INT8_C( -44), INT8_C(  74), INT8_C( -28), INT8_C( 124)),
      simde_mm512_set_epi8(INT8_C(   1), INT8_C(  75), INT8_C(  21), INT8_C( -36),
                           INT8_C(-126), INT8_C( 122), INT8_C(  71), INT8_C(  76),
                           INT8_C(  28), INT8_C( -56), INT8_C(  32), INT8_C( 101),
                           INT8_C(-107), INT8_C(-111), INT8_C( -88), INT8_C( -19),
                           INT8_C( -77), INT8_C(  19), INT8_C( -21), INT8_C(-111),
                           INT8_C( -68), INT8_C(  82), INT8_C(-118), INT8_C( -76),
                           INT8_C(  47), INT8_C( 127), INT8_C(  62), INT8_C( -16),
                           INT8_C(  10), INT8_C( -14), INT8_C(-100), INT8_C(  86),
                           INT8_C(  29), INT8_C( 107), INT8_C(  56), INT8_C(  21),
                           INT8_C(  24), INT8_C(  68), INT8_C( -96), INT8_C(  64),
                           INT8_C(  48), INT8_C(  13), INT8_C( -83), INT8_C(   4),
                           INT8_C(  -3), INT8_C( -64), INT8_C(  17), INT8_C(-115),
                           INT8_C(  21), INT8_C( 108), INT8_C( 125), INT8_C( -60),
                           INT8_C( -72), INT8_C(  74), INT8_C(  -5), INT8_C( -58),
                           INT8_C( -41), INT8_C(  22), INT8_C(-115), INT8_C( 102),
                           INT8_C(  59), INT8_C( -80), INT8_C( -15), INT8_C( -63)),
      simde_mm512_set_epi8(INT8_C( -47), INT8_C(   4), INT8_C(  29), INT8_C( -24),
                           INT8_C(-128), INT8_C(  11), INT8_C(  37), INT8_C( -23),
                           INT8_C( -47), INT8_C(-128), INT8_C(  58), INT8_C(  -7),
                           INT8_C(  30), INT8_C( -97), INT8_C(-128), INT8_C(  -1),
                           INT8_C( -54), INT8_C( -97), INT8_C( -84), INT8_C(-128),
                           INT8_C( -59), INT8_C(  65), INT8_C( -74), INT8_C(-128),
                           INT8_C( -11), INT8_C( 127), INT8_C( -40), INT8_C(-128),
                           INT8_C( -20), INT8_C(  22), INT8_C(  19), INT8_C(  -4),
                           INT8_C(   2), INT8_C(  35), INT8_C(  14), INT8_C( -73),
                           INT8_C(-100), INT8_C( 127), INT8_C(  28), INT8_C( 127),
                           INT8_C(   6), INT8_C( -42), INT8_C( 125), INT8_C(-125),
                           INT8_C( -95), INT8_C(  90), INT8_C(  -9), INT8_C(-118),
                           INT8_C(  53), INT8_C(  77), INT8_C( 127), INT8_C(-128),
                           INT8_C(  27), INT8_C( -42), INT8_C( -72), INT8_C( -37),
                           INT8_C(-113), INT8_C( -69), INT8_C(-128), INT8_C(  -6),
                           INT8_C(  15), INT8_C(  -6), INT8_C( -43), INT8_C( -75)) },
    { simde_mm512_set_epi8(INT8_C( -64), INT8_C( -53), INT8_C( -42), INT8_C( 126),
                           INT8_C(  67), INT8_C(  50), INT8_C( -18), INT8_C(  76),
                           INT8_C( -19), INT8_C( 123), INT8_C( -87), INT8_C( 106),
                           INT8_C( -74), INT8_C(  44), INT8_C( 117), INT8_C( 103),
                           INT8_C(  81), INT8_C( 122), INT8_C(  56), INT8_C( -10),
                           INT8_C(  67), INT8_C(  79), INT8_C(  83), INT8_C( -38),
                           INT8_C( -13), INT8_C(  43), INT8_C(  27), INT8_C( -97),
                           INT8_C( 102), INT8_C( 126), INT8_C(  38), INT8_C( -62),
                           INT8_C( -24), INT8_C( 117), INT8_C( -38), INT8_C( -93),
                           INT8_C( -58), INT8_C(-124), INT8_C( -75), INT8_C(  10),
                           INT8_C(  18), INT8_C( -74), INT8_C(  14), INT8_C(  36),
                           INT8_C(  -7), INT8_C( 113), INT8_C(  40), INT8_C(  48),
                           INT8_C(-107), INT8_C( -34), INT8_C( -75), INT8_C(  85),
                           INT8_C( -35), INT8_C(-116), INT8_C(  65), INT8_C( -21),
                           INT8_C(  15), INT8_C(   3), INT8_C(  45), INT8_C(  21),
                           INT8_C(  72), INT8_C(  93), INT8_C( 108), INT8_C( 125)),
      UINT64_C(12576710173448868104),
      simde_mm512_set_epi8(INT8_C(  90), INT8_C( -38), INT8_C( -98), INT8_C( -70),
                           INT8_C(-108), INT8_C(  20), INT8_C(  43), INT8_C(-128),
                           INT8_C(  77), INT8_C( 108), INT8_C(  53), INT8_C(  82),
                           INT8_C( -50), INT8_C(  52), INT8_C(  56), INT8_C(  58),
                           INT8_C(-120), INT8_C( -43), INT8_C( 114), INT8_C(  93),
                           INT8_C( -44), INT8_C( -15), INT8_C(  38), INT8_C( -17),
                           INT8_C(-110), INT8_C(-123), INT8_C( -39), INT8_C( 114),
                           INT8_C(  51), INT8_C(-115), INT8_C( -74), INT8_C(  43),
                           INT8_C(  41), INT8_C( -36), INT8_C(  19), INT8_C(  69),
                           INT8_C(  60), INT8_C( -53), INT8_C( 112), INT8_C( 108),
                           INT8_C(   8), INT8_C(  46), INT8_C( -35), INT8_C(  26),
                           INT8_C(  11), INT8_C(  42), INT8_C(  47), INT8_C(  59),
                           INT8_C( -57), INT8_C(  94), INT8_C(-125), INT8_C(-124),
                           INT8_C(  36), INT8_C(  57), INT8_C(  68), INT8_C( -52),
                           INT8_C(  39), INT8_C(  50), INT8_C( -48), INT8_C(  94),
                           INT8_C(  53), INT8_C(  11), INT8_C(  29), INT8_C(  65)),
      simde_mm512_set_epi8(INT8_C(-107), INT8_C( 120), INT8_C( -58), INT8_C( 107),
                           INT8_C( -32), INT8_C( -32), INT8_C(  88), INT8_C( -43),
                           INT8_C(  31), INT8_C( -32), INT8_C( -64), INT8_C(  27),
                           INT8_C(  82), INT8_C( -90), INT8_C( -54), INT8_C( -84),
                           INT8_C( -30), INT8_C(  63), INT8_C(  24), INT8_C(  81),
                           INT8_C(  -8), INT8_C(   9), INT8_C( -35), INT8_C(-101),
                           INT8_C(  83), INT8_C( 107), INT8_C( -47), INT8_C( -56),
                           INT8_C(  57), INT8_C( -88), INT8_C(-115), INT8_C( -20),
                           INT8_C(  58), INT8_C(  75), INT8_C(  56), INT8_C(  93),
                           INT8_C(  49), INT8_C(  43), INT8_C( 108), INT8_C( 118),
                           INT8_C( -79), INT8_C( 112), INT8_C(  44), INT8_C(-112),
                           INT8_C( -52), INT8_C(  10), INT8_C(  28), INT8_C( -86),
                           INT8_C(  65), INT8_C(  62), INT8_C(  86), INT8_C(-107),
                           INT8_C(  24), INT8_C( -55), INT8_C(  54), INT8_C(-110),
                           INT8_C( -33), INT8_C( 110), INT8_C(-116), INT8_C( -39),
                           INT8_C(  39), INT8_C(-112), INT8_C(  64), INT8_C(  43)),
      simde_mm512_set_epi8(INT8_C( -17), INT8_C( -53), INT8_C(-128), INT8_C( 126),
                           INT8_C(-128), INT8_C( -12), INT8_C( 127), INT8_C(  76),
                           INT8_C( 108), INT8_C( 123), INT8_C( -87), INT8_C( 106),
                           INT8_C(  32), INT8_C(  44), INT8_C( 117), INT8_C( -26),
                           INT8_C(  81), INT8_C(  20), INT8_C( 127), INT8_C( 127),
                           INT8_C(  67), INT8_C(  79), INT8_C(   3), INT8_C(-118),
                           INT8_C( -13), INT8_C( -16), INT8_C(  27), INT8_C( -97),
                           INT8_C( 102), INT8_C(-128), INT8_C(  38), INT8_C( -62),
                           INT8_C(  99), INT8_C( 117), INT8_C(  75), INT8_C( 127),
                           INT8_C( 109), INT8_C(-124), INT8_C( 127), INT8_C( 127),
                           INT8_C(  18), INT8_C( 127), INT8_C(   9), INT8_C( -86),
                           INT8_C(  -7), INT8_C(  52), INT8_C(  75), INT8_C(  48),
                           INT8_C(-107), INT8_C( -34), INT8_C( -75), INT8_C(-128),
                           INT8_C( -35), INT8_C(-116), INT8_C(  65), INT8_C(-128),
                           INT8_C(  15), INT8_C(   3), INT8_C(  45), INT8_C(  21),
                           INT8_C(  92), INT8_C(  93), INT8_C( 108), INT8_C( 125)) },
    { simde_mm512_set_epi8(INT8_C(-103), INT8_C(-124), INT8_C(  13), INT8_C( -80),
                           INT8_C( -32), INT8_C( 123), INT8_C( 115), INT8_C( -49),
                           INT8_C( 113), INT8_C( -51), INT8_C( 119), INT8_C( -67),
                           INT8_C( -21), INT8_C(  19), INT8_C( -12), INT8_C(  29),
                           INT8_C(  91), INT8_C( -64), INT8_C(  88), INT8_C(-105),
                           INT8_C(  24), INT8_C(  31), INT8_C(  77), INT8_C(  92),
                           INT8_C( -85), INT8_C(  48), INT8_C(-113), INT8_C(-114),
                           INT8_C(  16), INT8_C(  18), INT8_C( -44), INT8_C( -66),
                           INT8_C(  25), INT8_C(-105), INT8_C(-100), INT8_C( -91),
                           INT8_C(  26), INT8_C(-123), INT8_C( -33), INT8_C( -37),
                           INT8_C(  63), INT8_C(  13), INT8_C(-124), INT8_C(  41),
                           INT8_C(-104), INT8_C( -59), INT8_C(  -2), INT8_C( -54),
                           INT8_C(  73), INT8_C(   3), INT8_C( -18), INT8_C(  10),
                           INT8_C(  19), INT8_C(  58), INT8_C( -12), INT8_C( -75),
                           INT8_C( -88), INT8_C(  59), INT8_C( 104), INT8_C( -40),
                           INT8_C( -82), INT8_C(  42), INT8_C( -73), INT8_C( -94)),
      UINT64_C( 2633789449456316803),
      simde_mm512_set_epi8(INT8_C( 118), INT8_C(  44), INT8_C( -14), INT8_C(  84),
                           INT8_C( -69), INT8_C( -48), INT8_C(  40), INT8_C(  86),
                           INT8_C(  -1), INT8_C( 121), INT8_C( -40), INT8_C(  44),
                           INT8_C(  -1), INT8_C(  38), INT8_C( -44), INT8_C(  38),
                           INT8_C(  93), INT8_C(-107), INT8_C(  -4), INT8_C( -61),
                           INT8_C( -52), INT8_C(  22), INT8_C( -54), INT8_C(-125),
                           INT8_C( -29), INT8_C(  25), INT8_C( -77), INT8_C( -68),
                           INT8_C(   9), INT8_C( -35), INT8_C( -11), INT8_C(  53),
                           INT8_C(  28), INT8_C(  61), INT8_C( -35), INT8_C(-106),
                           INT8_C( -46), INT8_C( 121), INT8_C(-102), INT8_C( 121),
                           INT8_C( -54), INT8_C( -60), INT8_C(   7), INT8_C(   2),
                           INT8_C(-119), INT8_C( 111), INT8_C( -20), INT8_C( -58),
                           INT8_C( -98), INT8_C(  83), INT8_C(  32), INT8_C( -49),
                           INT8_C( -11), INT8_C(  48), INT8_C(  92), INT8_C(  47),
                           INT8_C( -38), INT8_C(  11), INT8_C(  26), INT8_C(  90),
                           INT8_C(  -5), INT8_C(  73), INT8_C(  45), INT8_C(  -2)),
      simde_mm512_set_epi8(INT8_C( -31), INT8_C( 126), INT8_C(-118), INT8_C( -57),
                           INT8_C(-126), INT8_C(  29), INT8_C( 101), INT8_C( -15),
                           INT8_C( -61), INT8_C( 118), INT8_C( 102), INT8_C(  12),
                           INT8_C( -59), INT8_C( -41), INT8_C( -60), INT8_C(  46),
                           INT8_C(  63), INT8_C(  78), INT8_C(  87), INT8_C(  18),
                           INT8_C(  18), INT8_C(  32), INT8_C( -26), INT8_C(  -6),
                           INT8_C( -20), INT8_C(-123), INT8_C(  99), INT8_C(  65),
                           INT8_C(  13), INT8_C(  25), INT8_C( 108), INT8_C(-121),
                           INT8_C(  42), INT8_C(   0), INT8_C( 104), INT8_C( -17),
                           INT8_C( 111), INT8_C(-110), INT8_C( -39), INT8_C( 125),
                           INT8_C( -50), INT8_C( -47), INT8_C(  30), INT8_C(  92),
                           INT8_C(  19), INT8_C(  -8), INT8_C(-120), INT8_C( 127),
                           INT8_C( -49), INT8_C(  23), INT8_C(  16), INT8_C( -64),
                           INT8_C( -79), INT8_C( 116), INT8_C(  -5), INT8_C( -50),
                           INT8_C(  32), INT8_C(  22), INT8_C( -42), INT8_C(  -3),
                           INT8_C(  30), INT8_C(  64), INT8_C(  96), INT8_C( -66)),
      simde_mm512_set_epi8(INT8_C(-103), INT8_C(-124), INT8_C(-128), INT8_C( -80),
                           INT8_C( -32), INT8_C( -19), INT8_C( 115), INT8_C( -49),
                           INT8_C( -62), INT8_C( -51), INT8_C( 119), INT8_C( -67),
                           INT8_C( -60), INT8_C(  -3), INT8_C( -12), INT8_C(  84),
                           INT8_C(  91), INT8_C( -64), INT8_C(  88), INT8_C( -43),
                           INT8_C( -34), INT8_C(  31), INT8_C(  77), INT8_C(-128),
                           INT8_C( -49), INT8_C(  48), INT8_C(-113), INT8_C(-114),
                           INT8_C(  22), INT8_C( -10), INT8_C( -44), INT8_C( -66),
                           INT8_C(  70), INT8_C(  61), INT8_C(-100), INT8_C( -91),
                           INT8_C(  26), INT8_C(  11), INT8_C( -33), INT8_C( -37),
                           INT8_C(  63), INT8_C(  13), INT8_C(-124), INT8_C(  41),
                           INT8_C(-104), INT8_C( -59), INT8_C(-128), INT8_C(  69),
                           INT8_C(-128), INT8_C( 106), INT8_C(  48), INT8_C(-113),
                           INT8_C(  19), INT8_C(  58), INT8_C( -12), INT8_C(  -3),
                           INT8_C(  -6), INT8_C(  59), INT8_C( 104), INT8_C( -40),
                           INT8_C( -82), INT8_C(  42), INT8_C( 127), INT8_C( -68)) },
    { simde_mm512_set_epi8(INT8_C( 117), INT8_C(  69), INT8_C( 121), INT8_C( -45),
                           INT8_C(  30), INT8_C( -73), INT8_C(   2), INT8_C( -40),
                           INT8_C(  95), INT8_C(  -3), INT8_C(  16), INT8_C( -78),
                           INT8_C(-128), INT8_C( -41), INT8_C( -66), INT8_C(  66),
                           INT8_C(  93), INT8_C( -32), INT8_C( -50), INT8_C(  -2),
                           INT8_C( -61), INT8_C(  29), INT8_C( -88), INT8_C(-118),
                           INT8_C( -27), INT8_C(  42), INT8_C(  78), INT8_C( -46),
                           INT8_C( -79), INT8_C(  38), INT8_C( -75), INT8_C(  14),
                           INT8_C(-118), INT8_C(-114), INT8_C(-120), INT8_C(  42),
                           INT8_C(   4), INT8_C(  79), INT8_C( -84), INT8_C(-110),
                           INT8_C(   0), INT8_C(  85), INT8_C( -20), INT8_C(  61),
                           INT8_C(  40), INT8_C( -75), INT8_C(  69), INT8_C( -99),
                           INT8_C(  73), INT8_C( -34), INT8_C( -14), INT8_C(  72),
                           INT8_C( 101), INT8_C(  96), INT8_C(-114), INT8_C(-124),
                           INT8_C( -87), INT8_C(  43), INT8_C(-118), INT8_C( 114),
                           INT8_C( -88), INT8_C(   8), INT8_C( -52), INT8_C(  75)),
      UINT64_C(14890918166471265655),
      simde_mm512_set_epi8(INT8_C( -56), INT8_C( -70), INT8_C(  66), INT8_C(-111),
                           INT8_C( -69), INT8_C(  15), INT8_C( -82), INT8_C( -12),
                           INT8_C( -89), INT8_C(  37), INT8_C(  80), INT8_C( 120),
                           INT8_C( -83), INT8_C(-120), INT8_C(  95), INT8_C(  21),
                           INT8_C(  91), INT8_C( -97), INT8_C( -72), INT8_C(  42),
                           INT8_C(  22), INT8_C( -70), INT8_C(  71), INT8_C( -78),
                           INT8_C(  -5), INT8_C(  52), INT8_C( -22), INT8_C( -34),
                           INT8_C(  16), INT8_C(  92), INT8_C(  91), INT8_C( -72),
                           INT8_C(   3), INT8_C( -31), INT8_C( -95), INT8_C( -56),
                           INT8_C( -50), INT8_C(  68), INT8_C( -24), INT8_C( -50),
                           INT8_C(  94), INT8_C(  67), INT8_C( 108), INT8_C(-118),
                           INT8_C( -65), INT8_C(  31), INT8_C(  70), INT8_C( 108),
                           INT8_C( -66), INT8_C( 107), INT8_C( -85), INT8_C(  38),
                           INT8_C(   2), INT8_C(  32), INT8_C(  56), INT8_C(  66),
                           INT8_C(  -2), INT8_C( -74), INT8_C( 112), INT8_C(  10),
                           INT8_C(  64), INT8_C( 100), INT8_C( -55), INT8_C(  83)),
      simde_mm512_set_epi8(INT8_C( -69), INT8_C( -45), INT8_C(  23), INT8_C(  70),
                           INT8_C(  22), INT8_C( 113), INT8_C(  75), INT8_C(-117),
                           INT8_C(  69), INT8_C( -74), INT8_C(  40), INT8_C( -39),
                           INT8_C( -51), INT8_C( -91), INT8_C(  25), INT8_C(  -4),
                           INT8_C( -12), INT8_C( -13), INT8_C( -77), INT8_C(  75),
                           INT8_C(-116), INT8_C( -36), INT8_C(  38), INT8_C( -52),
                           INT8_C(  13), INT8_C( -52), INT8_C(-100), INT8_C( -46),
                           INT8_C(  13), INT8_C(  83), INT8_C( -94), INT8_C( 102),
                           INT8_C( 116), INT8_C( -60), INT8_C(  44), INT8_C(  -7),
                           INT8_C( 104), INT8_C( -50), INT8_C( -30), INT8_C(  17),
                           INT8_C(  13), INT8_C(-107), INT8_C( 111), INT8_C( -13),
                           INT8_C(  17), INT8_C( -45), INT8_C( -31), INT8_C(  76),
                           INT8_C(-102), INT8_C( -71), INT8_C( 116), INT8_C(  72),
                           INT8_C( -11), INT8_C( -97), INT8_C(  93), INT8_C(  48),
                           INT8_C( -75), INT8_C( -90), INT8_C( -10), INT8_C( -78),
                           INT8_C(  -8), INT8_C( -10), INT8_C(  48), INT8_C( -11)),
      simde_mm512_set_epi8(INT8_C(-125), INT8_C(-115), INT8_C( 121), INT8_C( -45),
                           INT8_C( -47), INT8_C( 127), INT8_C(  -7), INT8_C( -40),
                           INT8_C( -20), INT8_C(  -3), INT8_C( 120), INT8_C( -78),
                           INT8_C(-128), INT8_C(-128), INT8_C( 120), INT8_C(  17),
                           INT8_C(  93), INT8_C( -32), INT8_C(-128), INT8_C(  -2),
                           INT8_C( -94), INT8_C(  29), INT8_C( 109), INT8_C(-128),
                           INT8_C( -27), INT8_C(  42), INT8_C(-122), INT8_C( -46),
                           INT8_C(  29), INT8_C( 127), INT8_C( -75), INT8_C(  30),
                           INT8_C(-118), INT8_C( -91), INT8_C(-120), INT8_C(  42),
                           INT8_C(  54), INT8_C(  79), INT8_C( -54), INT8_C( -33),
                           INT8_C( 107), INT8_C( -40), INT8_C( 127), INT8_C(  61),
                           INT8_C( -48), INT8_C( -14), INT8_C(  39), INT8_C( -99),
                           INT8_C(  73), INT8_C(  36), INT8_C( -14), INT8_C(  72),
                           INT8_C(  -9), INT8_C( -65), INT8_C(-114), INT8_C( 114),
                           INT8_C( -87), INT8_C(-128), INT8_C( 102), INT8_C( -68),
                           INT8_C( -88), INT8_C(  90), INT8_C(  -7), INT8_C(  72)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_adds_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_adds_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask64 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT64_C(17286015531074160252),
      simde_mm512_set_epi8(INT8_C(-115), INT8_C( -27), INT8_C(  62), INT8_C( -85),
                           INT8_C(  49), INT8_C(-115), INT8_C(  38), INT8_C(   4),
                           INT8_C(  92), INT8_C( 116), INT8_C( -78), INT8_C( -19),
                           INT8_C( -73), INT8_C(  22), INT8_C( -66), INT8_C( -29),
                           INT8_C(  55), INT8_C(  78), INT8_C( -45), INT8_C(-119),
                           INT8_C(  30), INT8_C( -29), INT8_C( -23), INT8_C(  89),
                           INT8_C(  15), INT8_C( 119), INT8_C( -88), INT8_C( 109),
                           INT8_C( -16), INT8_C( -10), INT8_C(   0), INT8_C( -18),
                           INT8_C(   1), INT8_C( 116), INT8_C(  62), INT8_C( -73),
                           INT8_C(-122), INT8_C(   0), INT8_C( 111), INT8_C(  95),
                           INT8_C( -20), INT8_C(   3), INT8_C( 122), INT8_C(-121),
                           INT8_C(  13), INT8_C(  25), INT8_C( -35), INT8_C(-127),
                           INT8_C(  82), INT8_C(  90), INT8_C(  53), INT8_C( 123),
                           INT8_C(  73), INT8_C( 108), INT8_C( -18), INT8_C(  15),
                           INT8_C(  81), INT8_C( -27), INT8_C( 102), INT8_C( 118),
                           INT8_C(  39), INT8_C( -77), INT8_C(  45), INT8_C(  81)),
      simde_mm512_set_epi8(INT8_C(  68), INT8_C(-108), INT8_C(-121), INT8_C(-115),
                           INT8_C( -59), INT8_C( -78), INT8_C(-111), INT8_C(  56),
                           INT8_C(   9), INT8_C(-121), INT8_C( 120), INT8_C( -36),
                           INT8_C(  36), INT8_C(-103), INT8_C(-104), INT8_C(  25),
                           INT8_C(-111), INT8_C( -63), INT8_C( -35), INT8_C(-120),
                           INT8_C(  28), INT8_C( -44), INT8_C( -26), INT8_C( -86),
                           INT8_C( -13), INT8_C(  53), INT8_C( -88), INT8_C(-107),
                           INT8_C(  68), INT8_C(  42), INT8_C(-118), INT8_C( 111),
                           INT8_C(  54), INT8_C( -58), INT8_C( -13), INT8_C(  27),
                           INT8_C(  23), INT8_C(  41), INT8_C(-119), INT8_C(  44),
                           INT8_C(   7), INT8_C(-120), INT8_C(  32), INT8_C( -43),
                           INT8_C( 114), INT8_C( -72), INT8_C(  73), INT8_C( -96),
                           INT8_C(  96), INT8_C( 110), INT8_C( -81), INT8_C( -76),
                           INT8_C( 103), INT8_C(-100), INT8_C( -22), INT8_C(  18),
                           INT8_C( 115), INT8_C(  54), INT8_C( -40), INT8_C( 125),
                           INT8_C( 110), INT8_C(  31), INT8_C(  51), INT8_C(-104)),
      simde_mm512_set_epi8(INT8_C( -47), INT8_C(-128), INT8_C( -59), INT8_C(   0),
                           INT8_C( -10), INT8_C(-128), INT8_C( -73), INT8_C(  60),
                           INT8_C( 101), INT8_C(  -5), INT8_C(  42), INT8_C(   0),
                           INT8_C(   0), INT8_C( -81), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  15), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -49), INT8_C(   3),
                           INT8_C(   2), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  52), INT8_C(   0), INT8_C(-118), INT8_C(   0),
                           INT8_C(   0), INT8_C(  58), INT8_C(   0), INT8_C(   0),
                           INT8_C( -99), INT8_C(  41), INT8_C(  -8), INT8_C( 127),
                           INT8_C( -13), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( 127), INT8_C( -47), INT8_C(  38), INT8_C(-128),
                           INT8_C( 127), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   8), INT8_C( -40), INT8_C(   0),
                           INT8_C(   0), INT8_C(  27), INT8_C(  62), INT8_C( 127),
                           INT8_C( 127), INT8_C( -46), INT8_C(   0), INT8_C(   0)) },
    { UINT64_C( 9031211210081693899),
      simde_mm512_set_epi8(INT8_C(-106), INT8_C( -18), INT8_C(  65), INT8_C(-104),
                           INT8_C(  81), INT8_C(  38), INT8_C( -84), INT8_C(  -2),
                           INT8_C( -14), INT8_C(  85), INT8_C( -80), INT8_C(  80),
                           INT8_C(  48), INT8_C(  93), INT8_C(  79), INT8_C( 127),
                           INT8_C(  16), INT8_C(  41), INT8_C(  54), INT8_C(-116),
                           INT8_C(  17), INT8_C(  42), INT8_C( -86), INT8_C(  38),
                           INT8_C(-118), INT8_C( -56), INT8_C(  60), INT8_C(  19),
                           INT8_C(-100), INT8_C(-107), INT8_C( 105), INT8_C( -76),
                           INT8_C(  10), INT8_C(  -9), INT8_C( -12), INT8_C( -56),
                           INT8_C( -71), INT8_C(  96), INT8_C(   6), INT8_C(  24),
                           INT8_C(  69), INT8_C(  73), INT8_C( -27), INT8_C(   3),
                           INT8_C( -95), INT8_C(  50), INT8_C( -67), INT8_C( -39),
                           INT8_C(  97), INT8_C(  62), INT8_C( 101), INT8_C(  56),
                           INT8_C( -79), INT8_C(  83), INT8_C( -99), INT8_C( -56),
                           INT8_C( -17), INT8_C( -56), INT8_C(  -8), INT8_C( -16),
                           INT8_C(  84), INT8_C( -95), INT8_C( -73), INT8_C(  54)),
      simde_mm512_set_epi8(INT8_C(  63), INT8_C( -51), INT8_C(  87), INT8_C(  32),
                           INT8_C( -65), INT8_C(  55), INT8_C(  14), INT8_C(  81),
                           INT8_C(-123), INT8_C(-100), INT8_C( -39), INT8_C( -44),
                           INT8_C(  22), INT8_C( 112), INT8_C(  16), INT8_C(  15),
                           INT8_C(-104), INT8_C(-102), INT8_C( 102), INT8_C( -95),
                           INT8_C(  69), INT8_C(  13), INT8_C( -21), INT8_C(  45),
                           INT8_C(  62), INT8_C(-120), INT8_C(-111), INT8_C(  32),
                           INT8_C(-107), INT8_C( -30), INT8_C(  99), INT8_C( -64),
                           INT8_C(   8), INT8_C( -42), INT8_C(  81), INT8_C( -34),
                           INT8_C( -46), INT8_C(  26), INT8_C(  31), INT8_C(  -2),
                           INT8_C(  68), INT8_C(  -7), INT8_C( -71), INT8_C(  46),
                           INT8_C( -21), INT8_C( -73), INT8_C(  21), INT8_C(  83),
                           INT8_C(-108), INT8_C( -97), INT8_C( -69), INT8_C(  73),
                           INT8_C(  57), INT8_C( -37), INT8_C(  21), INT8_C(  82),
                           INT8_C(-119), INT8_C(-126), INT8_C( 126), INT8_C(  91),
                           INT8_C( 115), INT8_C(  31), INT8_C( -79), INT8_C(  28)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C( -69), INT8_C( 127), INT8_C( -72),
                           INT8_C(  16), INT8_C(  93), INT8_C(   0), INT8_C(  79),
                           INT8_C(   0), INT8_C( -15), INT8_C(   0), INT8_C(  36),
                           INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C( 127),
                           INT8_C(   0), INT8_C( -61), INT8_C(   0), INT8_C(   0),
                           INT8_C(  86), INT8_C(  55), INT8_C(-107), INT8_C(   0),
                           INT8_C( -56), INT8_C(   0), INT8_C( -51), INT8_C(  51),
                           INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( 122), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  66), INT8_C( -98), INT8_C(  49),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  44),
                           INT8_C( -11), INT8_C( -35), INT8_C(  32), INT8_C( 127),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(-128), INT8_C(-128), INT8_C(   0), INT8_C(   0),
                           INT8_C( 127), INT8_C(   0), INT8_C(-128), INT8_C(  82)) },
    { UINT64_C( 6828010367541473016),
      simde_mm512_set_epi8(INT8_C(  84), INT8_C( -96), INT8_C(  87), INT8_C( -22),
                           INT8_C(  -5), INT8_C(  -3), INT8_C(-127), INT8_C(  41),
                           INT8_C(  74), INT8_C(  72), INT8_C(  -4), INT8_C(  28),
                           INT8_C(-115), INT8_C(  93), INT8_C( 102), INT8_C(  44),
                           INT8_C(-103), INT8_C( -29), INT8_C( -50), INT8_C(  48),
                           INT8_C( -96), INT8_C( -50), INT8_C(  46), INT8_C( -65),
                           INT8_C(   4), INT8_C(  43), INT8_C( -75), INT8_C(  97),
                           INT8_C( -26), INT8_C(-103), INT8_C(  71), INT8_C(-107),
                           INT8_C(  91), INT8_C(  45), INT8_C( -11), INT8_C(  47),
                           INT8_C(  29), INT8_C(  25), INT8_C(  26), INT8_C(  -9),
                           INT8_C(  10), INT8_C(  36), INT8_C( -79), INT8_C( -53),
                           INT8_C(  41), INT8_C(   1), INT8_C( -23), INT8_C( -63),
                           INT8_C(-127), INT8_C(  68), INT8_C(  48), INT8_C(  36),
                           INT8_C(  89), INT8_C(-112), INT8_C( -31), INT8_C( 120),
                           INT8_C(  35), INT8_C(  62), INT8_C( -21), INT8_C(-114),
                           INT8_C(-104), INT8_C(  57), INT8_C(  42), INT8_C(-111)),
      simde_mm512_set_epi8(INT8_C(  96), INT8_C( -64), INT8_C( -23), INT8_C( -46),
                           INT8_C( -29), INT8_C( -71), INT8_C(  71), INT8_C(  90),
                           INT8_C(  44), INT8_C( -92), INT8_C( -31), INT8_C(  26),
                           INT8_C( -99), INT8_C( -53), INT8_C( 117), INT8_C(  18),
                           INT8_C( -63), INT8_C(  45), INT8_C(  12), INT8_C(  24),
                           INT8_C(-108), INT8_C(  18), INT8_C( -60), INT8_C(  28),
                           INT8_C(  50), INT8_C( -11), INT8_C( -68), INT8_C( -31),
                           INT8_C( 105), INT8_C(-106), INT8_C(  98), INT8_C(  51),
                           INT8_C(  58), INT8_C( 103), INT8_C( 111), INT8_C(-127),
                           INT8_C(  68), INT8_C( -56), INT8_C( 124), INT8_C(-119),
                           INT8_C(  74), INT8_C( -62), INT8_C(-116), INT8_C(  37),
                           INT8_C( -12), INT8_C( 114), INT8_C(   0), INT8_C(  61),
                           INT8_C( 103), INT8_C(  -4), INT8_C(-105), INT8_C( -68),
                           INT8_C(  39), INT8_C(-100), INT8_C( -93), INT8_C(  11),
                           INT8_C( -80), INT8_C( -19), INT8_C( -22), INT8_C( -39),
                           INT8_C( 127), INT8_C( -38), INT8_C(-125), INT8_C(-111)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(-128), INT8_C(   0), INT8_C( -68),
                           INT8_C( -34), INT8_C( -74), INT8_C( -56), INT8_C(   0),
                           INT8_C( 118), INT8_C( -20), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  62),
                           INT8_C(-128), INT8_C(  16), INT8_C( -38), INT8_C(  72),
                           INT8_C(   0), INT8_C( -32), INT8_C( -14), INT8_C( -37),
                           INT8_C(   0), INT8_C(  32), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( 127), INT8_C(   0), INT8_C( 100), INT8_C( -80),
                           INT8_C(  97), INT8_C( -31), INT8_C( 127), INT8_C(-128),
                           INT8_C(  84), INT8_C( -26), INT8_C(-128), INT8_C( -16),
                           INT8_C(  29), INT8_C( 115), INT8_C( -23), INT8_C(   0),
                           INT8_C(   0), INT8_C(  64), INT8_C( -57), INT8_C(   0),
                           INT8_C( 127), INT8_C(-128), INT8_C(-124), INT8_C(   0),
                           INT8_C( -45), INT8_C(  43), INT8_C( -43), INT8_C(-128),
                           INT8_C(  23), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT64_C(18325019528117086600),
      simde_mm512_set_epi8(INT8_C( 122), INT8_C( -54), INT8_C(  50), INT8_C( 123),
                           INT8_C( -31), INT8_C( -74), INT8_C( -64), INT8_C(  54),
                           INT8_C( -81), INT8_C(  60), INT8_C(  31), INT8_C( -23),
                           INT8_C( 108), INT8_C(-119), INT8_C( -92), INT8_C( -80),
                           INT8_C( -30), INT8_C( -37), INT8_C(  51), INT8_C( -36),
                           INT8_C(   8), INT8_C(  52), INT8_C(  97), INT8_C( 123),
                           INT8_C( -49), INT8_C(-124), INT8_C(  95), INT8_C( -83),
                           INT8_C(  70), INT8_C( -50), INT8_C( -61), INT8_C(  25),
                           INT8_C( -97), INT8_C(  28), INT8_C( -58), INT8_C(  11),
                           INT8_C( -14), INT8_C( 126), INT8_C(  81), INT8_C(  45),
                           INT8_C( -23), INT8_C( 120), INT8_C( -83), INT8_C( -16),
                           INT8_C(   7), INT8_C(  51), INT8_C( -57), INT8_C( -50),
                           INT8_C( -21), INT8_C(  98), INT8_C(  88), INT8_C(   0),
                           INT8_C( -66), INT8_C(   3), INT8_C( 124), INT8_C(-113),
                           INT8_C(  50), INT8_C(  88), INT8_C( -85), INT8_C( -93),
                           INT8_C( -44), INT8_C( -13), INT8_C( -94), INT8_C(  17)),
      simde_mm512_set_epi8(INT8_C( -13), INT8_C( -12), INT8_C(-124), INT8_C(  96),
                           INT8_C( -13), INT8_C( -98), INT8_C(  99), INT8_C( -13),
                           INT8_C(  56), INT8_C( -45), INT8_C(-108), INT8_C( -19),
                           INT8_C(-124), INT8_C( -27), INT8_C(  22), INT8_C( 126),
                           INT8_C(-106), INT8_C( -68), INT8_C( -60), INT8_C(   8),
                           INT8_C(  60), INT8_C(  93), INT8_C( -33), INT8_C( -27),
                           INT8_C(  -7), INT8_C(  27), INT8_C(-122), INT8_C( -38),
                           INT8_C(  23), INT8_C(   6), INT8_C(  45), INT8_C( -21),
                           INT8_C( -23), INT8_C(-101), INT8_C( 116), INT8_C( 127),
                           INT8_C(  96), INT8_C(  40), INT8_C( -97), INT8_C(  40),
                           INT8_C(  86), INT8_C( -44), INT8_C(  70), INT8_C( -71),
                           INT8_C(  62), INT8_C( -21), INT8_C(  66), INT8_C(  68),
                           INT8_C( -87), INT8_C( -61), INT8_C(  48), INT8_C( -70),
                           INT8_C(  18), INT8_C( -78), INT8_C( -98), INT8_C( 117),
                           INT8_C(  74), INT8_C(  32), INT8_C(  93), INT8_C( 125),
                           INT8_C( -47), INT8_C( -60), INT8_C( -86), INT8_C( 117)),
      simde_mm512_set_epi8(INT8_C( 109), INT8_C( -66), INT8_C( -74), INT8_C( 127),
                           INT8_C( -44), INT8_C(-128), INT8_C(  35), INT8_C(   0),
                           INT8_C(   0), INT8_C(  15), INT8_C(   0), INT8_C(   0),
                           INT8_C( -16), INT8_C(-128), INT8_C( -70), INT8_C(  46),
                           INT8_C(-128), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  68), INT8_C( 127), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -27), INT8_C(   0),
                           INT8_C(  93), INT8_C(   0), INT8_C( -16), INT8_C(   4),
                           INT8_C(-120), INT8_C(   0), INT8_C(  58), INT8_C( 127),
                           INT8_C(   0), INT8_C(   0), INT8_C( -16), INT8_C(  85),
                           INT8_C(  63), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   9), INT8_C(  18),
                           INT8_C(-108), INT8_C(  37), INT8_C( 127), INT8_C(   0),
                           INT8_C( -48), INT8_C(   0), INT8_C(   0), INT8_C(   4),
                           INT8_C( 124), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( -91), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT64_C( 7677112093376593822),
      simde_mm512_set_epi8(INT8_C( -53), INT8_C(  34), INT8_C(   9), INT8_C(  64),
                           INT8_C( -61), INT8_C( -25), INT8_C(-115), INT8_C( 100),
                           INT8_C(  65), INT8_C(   8), INT8_C(  69), INT8_C(  -8),
                           INT8_C( -15), INT8_C( -51), INT8_C(   1), INT8_C(  90),
                           INT8_C( 115), INT8_C(  51), INT8_C( -91), INT8_C(  56),
                           INT8_C(  64), INT8_C( -39), INT8_C(-119), INT8_C( -28),
                           INT8_C( -54), INT8_C(  28), INT8_C(  54), INT8_C(  -8),
                           INT8_C( -54), INT8_C(-128), INT8_C( -28), INT8_C( -71),
                           INT8_C( 107), INT8_C( -66), INT8_C(-114), INT8_C( -88),
                           INT8_C(  34), INT8_C( -83), INT8_C( -21), INT8_C( -64),
                           INT8_C( 121), INT8_C( -20), INT8_C( -89), INT8_C( -94),
                           INT8_C( 112), INT8_C( -27), INT8_C(  81), INT8_C( -54),
                           INT8_C( -64), INT8_C(-114), INT8_C(  48), INT8_C( -89),
                           INT8_C( -61), INT8_C(  26), INT8_C(  43), INT8_C(  29),
                           INT8_C(   0), INT8_C(  64), INT8_C( 123), INT8_C( -67),
                           INT8_C(  15), INT8_C( 120), INT8_C(  36), INT8_C(  40)),
      simde_mm512_set_epi8(INT8_C( 113), INT8_C(  32), INT8_C(  71), INT8_C(  55),
                           INT8_C( -26), INT8_C(  82), INT8_C( -81), INT8_C( -20),
                           INT8_C( -55), INT8_C( 112), INT8_C(  45), INT8_C(  37),
                           INT8_C(  67), INT8_C( -69), INT8_C(  64), INT8_C(  39),
                           INT8_C(  72), INT8_C(  45), INT8_C( 120), INT8_C(  -5),
                           INT8_C(-109), INT8_C(  62), INT8_C(  17), INT8_C(  31),
                           INT8_C( -30), INT8_C( -58), INT8_C(  56), INT8_C(  21),
                           INT8_C(  72), INT8_C( -74), INT8_C( -40), INT8_C( 120),
                           INT8_C(  95), INT8_C( 108), INT8_C(  32), INT8_C(  64),
                           INT8_C(-128), INT8_C( 102), INT8_C( -21), INT8_C(  28),
                           INT8_C( 105), INT8_C(  52), INT8_C(  85), INT8_C(-104),
                           INT8_C(  57), INT8_C( -31), INT8_C( -38), INT8_C(-124),
                           INT8_C(-107), INT8_C(  -2), INT8_C(  55), INT8_C(  46),
                           INT8_C( -71), INT8_C(  77), INT8_C(  18), INT8_C(  70),
                           INT8_C(  89), INT8_C( 125), INT8_C( -42), INT8_C(-125),
                           INT8_C( 121), INT8_C( -11), INT8_C( -69), INT8_C( -59)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(  66), INT8_C(  80), INT8_C(   0),
                           INT8_C( -87), INT8_C(   0), INT8_C(-128), INT8_C(   0),
                           INT8_C(  10), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  52), INT8_C(   0), INT8_C(  65), INT8_C(   0),
                           INT8_C( 127), INT8_C(   0), INT8_C(   0), INT8_C(  51),
                           INT8_C(   0), INT8_C(  23), INT8_C(   0), INT8_C(   0),
                           INT8_C( -84), INT8_C( -30), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(-128), INT8_C( -68), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -24),
                           INT8_C( -94), INT8_C(   0), INT8_C( -42), INT8_C(   0),
                           INT8_C( 127), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                           INT8_C(   0), INT8_C(   0), INT8_C( 103), INT8_C( -43),
                           INT8_C(-128), INT8_C( 103), INT8_C(  61), INT8_C(  99),
                           INT8_C(  89), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                           INT8_C( 127), INT8_C( 109), INT8_C( -33), INT8_C(   0)) },
    { UINT64_C( 4840581956081845689),
      simde_mm512_set_epi8(INT8_C( 122), INT8_C(-108), INT8_C( -70), INT8_C( -63),
                           INT8_C(  56), INT8_C(  71), INT8_C( -14), INT8_C(  49),
                           INT8_C( -73), INT8_C(  53), INT8_C( -29), INT8_C(   3),
                           INT8_C( -73), INT8_C(  43), INT8_C( -22), INT8_C(  85),
                           INT8_C( -26), INT8_C(  -9), INT8_C(  66), INT8_C(   1),
                           INT8_C( -13), INT8_C(  60), INT8_C(-119), INT8_C( -83),
                           INT8_C(-122), INT8_C( -64), INT8_C( -83), INT8_C( -74),
                           INT8_C( 119), INT8_C(  -8), INT8_C(  12), INT8_C( 113),
                           INT8_C( -12), INT8_C( -84), INT8_C(   6), INT8_C(  69),
                           INT8_C(   2), INT8_C( -75), INT8_C( -34), INT8_C(-126),
                           INT8_C(   3), INT8_C(-128), INT8_C(  -9), INT8_C(  24),
                           INT8_C(  11), INT8_C( -94), INT8_C( -32), INT8_C( 110),
                           INT8_C(  33), INT8_C( -24), INT8_C( 125), INT8_C(  35),
                           INT8_C(-103), INT8_C( -48), INT8_C( -22), INT8_C(  38),
                           INT8_C( -81), INT8_C(   9), INT8_C( -11), INT8_C(-124),
                           INT8_C(  71), INT8_C(  31), INT8_C( -42), INT8_C(  93)),
      simde_mm512_set_epi8(INT8_C(  53), INT8_C(  77), INT8_C(  63), INT8_C( -84),
                           INT8_C(  27), INT8_C(  22), INT8_C(   3), INT8_C( -37),
                           INT8_C(  65), INT8_C( 118), INT8_C(-126), INT8_C(  97),
                           INT8_C( 109), INT8_C(   7), INT8_C(-114), INT8_C( -75),
                           INT8_C(-125), INT8_C( 121), INT8_C(-128), INT8_C( 103),
                           INT8_C(   0), INT8_C( 101), INT8_C( -41), INT8_C(  89),
                           INT8_C( -83), INT8_C( -65), INT8_C(   9), INT8_C(  -7),
                           INT8_C( -63), INT8_C(  13), INT8_C( 105), INT8_C(  92),
                           INT8_C( -18), INT8_C( -21), INT8_C(-102), INT8_C(-114),
                           INT8_C(  74), INT8_C( 121), INT8_C( -45), INT8_C(  52),
                           INT8_C( -63), INT8_C( -93), INT8_C(  98), INT8_C( 106),
                           INT8_C(-109), INT8_C( -47), INT8_C(  37), INT8_C(  70),
                           INT8_C( 100), INT8_C( 121), INT8_C(  18), INT8_C(  28),
                           INT8_C(-117), INT8_C( 107), INT8_C(   3), INT8_C( -62),
                           INT8_C(  42), INT8_C(  72), INT8_C(  91), INT8_C(  86),
                           INT8_C( -72), INT8_C(   9), INT8_C( -80), INT8_C( 118)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C( -31), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -11), INT8_C(  12),
                           INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C(   0),
                           INT8_C(  36), INT8_C(  50), INT8_C(   0), INT8_C(  10),
                           INT8_C(   0), INT8_C(   0), INT8_C( -62), INT8_C( 104),
                           INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C(   6),
                           INT8_C(-128), INT8_C(   0), INT8_C( -74), INT8_C(   0),
                           INT8_C(   0), INT8_C(   5), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(-105), INT8_C( -96), INT8_C( -45),
                           INT8_C(  76), INT8_C(  46), INT8_C( -79), INT8_C(   0),
                           INT8_C(   0), INT8_C(-128), INT8_C(  89), INT8_C(   0),
                           INT8_C( -98), INT8_C(-128), INT8_C(   0), INT8_C(   0),
                           INT8_C( 127), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  59), INT8_C(   0), INT8_C( -24),
                           INT8_C( -39), INT8_C(   0), INT8_C(  80), INT8_C( -38),
                           INT8_C(  -1), INT8_C(   0), INT8_C(   0), INT8_C( 127)) },
    { UINT64_C(17498311407133456191),
      simde_mm512_set_epi8(INT8_C(  61), INT8_C( -89), INT8_C(  84), INT8_C( -94),
                           INT8_C(  31), INT8_C(-116), INT8_C( -67), INT8_C(-102),
                           INT8_C( -72), INT8_C( -91), INT8_C(-105), INT8_C(-108),
                           INT8_C( -44), INT8_C(  74), INT8_C( -28), INT8_C( 124),
                           INT8_C( 120), INT8_C( -41), INT8_C( -79), INT8_C( 122),
                           INT8_C(  87), INT8_C(-119), INT8_C(  54), INT8_C(  -2),
                           INT8_C( -47), INT8_C(  84), INT8_C(-126), INT8_C( -64),
                           INT8_C(  14), INT8_C(  11), INT8_C(  37), INT8_C( -23),
                           INT8_C(  67), INT8_C( 124), INT8_C(  58), INT8_C( -94),
                           INT8_C(  30), INT8_C( -33), INT8_C(  70), INT8_C( -24),
                           INT8_C(  38), INT8_C( -97), INT8_C( -56), INT8_C( -60),
                           INT8_C( -59), INT8_C(  65), INT8_C( -74), INT8_C(  45),
                           INT8_C( -11), INT8_C(  55), INT8_C( -82), INT8_C(  12),
                           INT8_C( 106), INT8_C(  22), INT8_C(-124), INT8_C(  -4),
                           INT8_C(   2), INT8_C( -81), INT8_C(  14), INT8_C(  90),
                           INT8_C(-100), INT8_C(-122), INT8_C( -35), INT8_C(  81)),
      simde_mm512_set_epi8(INT8_C(  21), INT8_C( 108), INT8_C( 125), INT8_C( -60),
                           INT8_C( -72), INT8_C(  74), INT8_C(  -5), INT8_C( -58),
                           INT8_C( -41), INT8_C(  22), INT8_C(-115), INT8_C( 102),
                           INT8_C(  59), INT8_C( -80), INT8_C( -15), INT8_C( -63),
                           INT8_C(  84), INT8_C( -71), INT8_C(   8), INT8_C(  12),
                           INT8_C( -11), INT8_C( -76), INT8_C(  62), INT8_C(  93),
                           INT8_C( -75), INT8_C( -77), INT8_C( -84), INT8_C(-108),
                           INT8_C( -35), INT8_C(  14), INT8_C( -60), INT8_C(  18),
                           INT8_C(  23), INT8_C( -60), INT8_C( -63), INT8_C(-114),
                           INT8_C( -55), INT8_C(  75), INT8_C( -99), INT8_C( -55),
                           INT8_C(  58), INT8_C(  76), INT8_C(-102), INT8_C(-118),
                           INT8_C( -30), INT8_C(  39), INT8_C( 119), INT8_C(  85),
                           INT8_C(  -8), INT8_C( -72), INT8_C( -60), INT8_C( -94),
                           INT8_C(-112), INT8_C( 119), INT8_C( 124), INT8_C(  76),
                           INT8_C( -42), INT8_C(-124), INT8_C(  54), INT8_C(  74),
                           INT8_C( -92), INT8_C(  99), INT8_C(  79), INT8_C(  -3)),
      simde_mm512_set_epi8(INT8_C(  82), INT8_C(  19), INT8_C( 127), INT8_C(-128),
                           INT8_C(   0), INT8_C(   0), INT8_C( -72), INT8_C(   0),
                           INT8_C(-113), INT8_C( -69), INT8_C(   0), INT8_C(  -6),
                           INT8_C(   0), INT8_C(  -6), INT8_C( -43), INT8_C(   0),
                           INT8_C(   0), INT8_C(-112), INT8_C( -71), INT8_C( 127),
                           INT8_C(  76), INT8_C(-128), INT8_C(   0), INT8_C(  91),
                           INT8_C(-122), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -23), INT8_C(  -5),
                           INT8_C(  90), INT8_C(  64), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  42), INT8_C( -29), INT8_C( -79),
                           INT8_C(   0), INT8_C( -21), INT8_C(   0), INT8_C(-128),
                           INT8_C( -89), INT8_C(   0), INT8_C(  45), INT8_C(   0),
                           INT8_C( -19), INT8_C( -17), INT8_C(-128), INT8_C( -82),
                           INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C(  72),
                           INT8_C(   0), INT8_C(   0), INT8_C(  68), INT8_C( 127),
                           INT8_C(-128), INT8_C( -23), INT8_C(  44), INT8_C(  78)) },
    { UINT64_C( 3462613925466476941),
      simde_mm512_set_epi8(INT8_C(  18), INT8_C( -74), INT8_C(  14), INT8_C(  36),
                           INT8_C(  -7), INT8_C( 113), INT8_C(  40), INT8_C(  48),
                           INT8_C(-107), INT8_C( -34), INT8_C( -75), INT8_C(  85),
                           INT8_C( -35), INT8_C(-116), INT8_C(  65), INT8_C( -21),
                           INT8_C(  15), INT8_C(   3), INT8_C(  45), INT8_C(  21),
                           INT8_C(  72), INT8_C(  93), INT8_C( 108), INT8_C( 125),
                           INT8_C(   1), INT8_C(  75), INT8_C(  21), INT8_C( -36),
                           INT8_C(-126), INT8_C( 122), INT8_C(  71), INT8_C(  76),
                           INT8_C(  28), INT8_C( -56), INT8_C(  32), INT8_C( 101),
                           INT8_C(-107), INT8_C(-111), INT8_C( -88), INT8_C( -19),
                           INT8_C( -77), INT8_C(  19), INT8_C( -21), INT8_C(-111),
                           INT8_C( -68), INT8_C(  82), INT8_C(-118), INT8_C( -76),
                           INT8_C(  47), INT8_C( 127), INT8_C(  62), INT8_C( -16),
                           INT8_C(  10), INT8_C( -14), INT8_C(-100), INT8_C(  86),
                           INT8_C(  29), INT8_C( 107), INT8_C(  56), INT8_C(  21),
                           INT8_C(  24), INT8_C(  68), INT8_C( -96), INT8_C(  64)),
      simde_mm512_set_epi8(INT8_C( -57), INT8_C(  94), INT8_C(-125), INT8_C(-124),
                           INT8_C(  36), INT8_C(  57), INT8_C(  68), INT8_C( -52),
                           INT8_C(  39), INT8_C(  50), INT8_C( -48), INT8_C(  94),
                           INT8_C(  53), INT8_C(  11), INT8_C(  29), INT8_C(  65),
                           INT8_C( -82), INT8_C(-119), INT8_C( 115), INT8_C(  68),
                           INT8_C( -69), INT8_C( 118), INT8_C(  17), INT8_C(   8),
                           INT8_C( -64), INT8_C( -53), INT8_C( -42), INT8_C( 126),
                           INT8_C(  67), INT8_C(  50), INT8_C( -18), INT8_C(  76),
                           INT8_C( -19), INT8_C( 123), INT8_C( -87), INT8_C( 106),
                           INT8_C( -74), INT8_C(  44), INT8_C( 117), INT8_C( 103),
                           INT8_C(  81), INT8_C( 122), INT8_C(  56), INT8_C( -10),
                           INT8_C(  67), INT8_C(  79), INT8_C(  83), INT8_C( -38),
                           INT8_C( -13), INT8_C(  43), INT8_C(  27), INT8_C( -97),
                           INT8_C( 102), INT8_C( 126), INT8_C(  38), INT8_C( -62),
                           INT8_C( -24), INT8_C( 117), INT8_C( -38), INT8_C( -93),
                           INT8_C( -58), INT8_C(-124), INT8_C( -75), INT8_C(  10)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(-111), INT8_C( -88),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  18), INT8_C(-105), INT8_C(   0), INT8_C(  44),
                           INT8_C( -67), INT8_C(   0), INT8_C( 127), INT8_C(   0),
                           INT8_C(   3), INT8_C( 127), INT8_C(   0), INT8_C( 127),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C(   0),
                           INT8_C(   9), INT8_C(  67), INT8_C( -55), INT8_C( 127),
                           INT8_C(-128), INT8_C( -67), INT8_C(   0), INT8_C(  84),
                           INT8_C(   4), INT8_C( 127), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(-113),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  24),
                           INT8_C(   5), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( -34), INT8_C( -56), INT8_C(   0), INT8_C(  74)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_adds_epi8(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_adds_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi16(INT16_C(  5952), INT16_C(-21138), INT16_C( 23605), INT16_C( -3799),
                            INT16_C(  1987), INT16_C( 29254), INT16_C( -4184), INT16_C( 13346),
                            INT16_C( 26203), INT16_C(-24373), INT16_C( 15487), INT16_C( 15569),
                            INT16_C( 13821), INT16_C( -3315), INT16_C( 30667), INT16_C(  9811),
                            INT16_C( 22769), INT16_C( 21964), INT16_C(  -315), INT16_C(-15723),
                            INT16_C( -9658), INT16_C(-27088), INT16_C( 18289), INT16_C( 13213),
                            INT16_C(-12712), INT16_C( 23293), INT16_C( 12497), INT16_C( 16576),
                            INT16_C(-19358), INT16_C(-11263), INT16_C( -3745), INT16_C(  7925)),
      simde_mm512_set_epi16(INT16_C(-22600), INT16_C(  9825), INT16_C( 27347), INT16_C( -4649),
                            INT16_C(  4812), INT16_C( 27489), INT16_C( 14729), INT16_C(-20872),
                            INT16_C(-24487), INT16_C( 23053), INT16_C( 28966), INT16_C(-23890),
                            INT16_C( -9737), INT16_C(-24021), INT16_C( 12009), INT16_C(-16892),
                            INT16_C(-11018), INT16_C( -7664), INT16_C( 22598), INT16_C(  9112),
                            INT16_C(  -468), INT16_C(-16246), INT16_C( 11633), INT16_C(  3402),
                            INT16_C( 12940), INT16_C(-18453), INT16_C(-10463), INT16_C(-24372),
                            INT16_C(  9722), INT16_C(  4912), INT16_C( 14086), INT16_C( -2976)),
      simde_mm512_set_epi16(INT16_C(-16648), INT16_C(-11313), INT16_C( 32767), INT16_C( -8448),
                            INT16_C(  6799), INT16_C( 32767), INT16_C( 10545), INT16_C( -7526),
                            INT16_C(  1716), INT16_C( -1320), INT16_C( 32767), INT16_C( -8321),
                            INT16_C(  4084), INT16_C(-27336), INT16_C( 32767), INT16_C( -7081),
                            INT16_C( 11751), INT16_C( 14300), INT16_C( 22283), INT16_C( -6611),
                            INT16_C(-10126), INT16_C(-32768), INT16_C( 29922), INT16_C( 16615),
                            INT16_C(   228), INT16_C(  4840), INT16_C(  2034), INT16_C( -7796),
                            INT16_C( -9636), INT16_C( -6351), INT16_C( 10341), INT16_C(  4949)) },
    { simde_mm512_set_epi16(INT16_C(-21870), INT16_C( -1966), INT16_C(-26101), INT16_C(  8905),
                            INT16_C(-13069), INT16_C( 13555), INT16_C(-23554), INT16_C(-12947),
                            INT16_C( 31779), INT16_C( 21013), INT16_C( 32688), INT16_C( 12512),
                            INT16_C(-26721), INT16_C(  7598), INT16_C( 21566), INT16_C(-11203),
                            INT16_C( 27021), INT16_C( -8539), INT16_C(  -532), INT16_C(-17200),
                            INT16_C(-18609), INT16_C(-30604), INT16_C(  6829), INT16_C(-13538),
                            INT16_C( 21529), INT16_C( 29340), INT16_C( 23176), INT16_C( 10860),
                            INT16_C( 31408), INT16_C(  3656), INT16_C(-30572), INT16_C( -4393)),
      simde_mm512_set_epi16(INT16_C( 29137), INT16_C( 29112), INT16_C(-21893), INT16_C( 14830),
                            INT16_C( 28350), INT16_C( 21222), INT16_C( 26922), INT16_C( 26197),
                            INT16_C(-25881), INT16_C( 28163), INT16_C(-19268), INT16_C( 10158),
                            INT16_C(-29245), INT16_C( 16103), INT16_C(-23323), INT16_C( 12478),
                            INT16_C( -6515), INT16_C( 24007), INT16_C(-28366), INT16_C(-32677),
                            INT16_C(-27350), INT16_C(-26916), INT16_C(-11979), INT16_C( 13820),
                            INT16_C(-22393), INT16_C(-20903), INT16_C( -9892), INT16_C(-22447),
                            INT16_C(-14286), INT16_C(  3532), INT16_C( 19664), INT16_C(-26842)),
      simde_mm512_set_epi16(INT16_C(  7267), INT16_C( 27146), INT16_C(-32768), INT16_C( 23735),
                            INT16_C( 15281), INT16_C( 32767), INT16_C(  3368), INT16_C( 13250),
                            INT16_C(  5898), INT16_C( 32767), INT16_C( 13420), INT16_C( 22670),
                            INT16_C(-32768), INT16_C( 23701), INT16_C( -1757), INT16_C(  1275),
                            INT16_C( 20506), INT16_C( 15468), INT16_C(-28898), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( -5150), INT16_C(   282),
                            INT16_C(  -864), INT16_C(  8437), INT16_C( 13284), INT16_C(-11587),
                            INT16_C( 17122), INT16_C(  7188), INT16_C(-10908), INT16_C(-31235)) },
    { simde_mm512_set_epi16(INT16_C(-22689), INT16_C( 25207), INT16_C( -9300), INT16_C( 28977),
                            INT16_C(  2270), INT16_C(-32394), INT16_C( 26182), INT16_C( 23367),
                            INT16_C( 31661), INT16_C(  9636), INT16_C(-18580), INT16_C(-28875),
                            INT16_C(-25943), INT16_C( 14067), INT16_C( 10599), INT16_C( 30831),
                            INT16_C(-23557), INT16_C( 11377), INT16_C( -4608), INT16_C( 14955),
                            INT16_C( 11781), INT16_C(-12602), INT16_C( -3223), INT16_C( 11440),
                            INT16_C( -2896), INT16_C(-24304), INT16_C( 29156), INT16_C( -4347),
                            INT16_C( -2421), INT16_C( 31033), INT16_C( -3132), INT16_C(-19324)),
      simde_mm512_set_epi16(INT16_C( -2724), INT16_C(  4704), INT16_C( 23772), INT16_C(-20368),
                            INT16_C(-30374), INT16_C( 15838), INT16_C( 19402), INT16_C( 16976),
                            INT16_C(-28476), INT16_C( -2715), INT16_C(-27679), INT16_C( -8608),
                            INT16_C(-20120), INT16_C( 29206), INT16_C( -1643), INT16_C(-29856),
                            INT16_C( 18656), INT16_C( -9660), INT16_C( -4660), INT16_C(-30272),
                            INT16_C(   610), INT16_C( -3787), INT16_C( 22038), INT16_C(-29446),
                            INT16_C(-26504), INT16_C(-28984), INT16_C(-21830), INT16_C(-27507),
                            INT16_C( 24148), INT16_C(-12945), INT16_C( 24529), INT16_C(  7913)),
      simde_mm512_set_epi16(INT16_C(-25413), INT16_C( 29911), INT16_C( 14472), INT16_C(  8609),
                            INT16_C(-28104), INT16_C(-16556), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(  3185), INT16_C(  6921), INT16_C(-32768), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(  8956), INT16_C(   975),
                            INT16_C( -4901), INT16_C(  1717), INT16_C( -9268), INT16_C(-15317),
                            INT16_C( 12391), INT16_C(-16389), INT16_C( 18815), INT16_C(-18006),
                            INT16_C(-29400), INT16_C(-32768), INT16_C(  7326), INT16_C(-31854),
                            INT16_C( 21727), INT16_C( 18088), INT16_C( 21397), INT16_C(-11411)) },
    { simde_mm512_set_epi16(INT16_C(-18069), INT16_C( 21685), INT16_C( -5808), INT16_C( 24672),
                            INT16_C( -8190), INT16_C( -2830), INT16_C(-23214), INT16_C(  1083),
                            INT16_C(  6699), INT16_C(   871), INT16_C(-14072), INT16_C( -2337),
                            INT16_C( 13356), INT16_C(  7260), INT16_C(-23788), INT16_C(-19723),
                            INT16_C( 13480), INT16_C(-15429), INT16_C(-20153), INT16_C( 18680),
                            INT16_C(-26180), INT16_C( 27923), INT16_C(-14099), INT16_C( 25829),
                            INT16_C(-27481), INT16_C( 21639), INT16_C( 16654), INT16_C( 19972),
                            INT16_C( 15776), INT16_C( 23126), INT16_C( 30799), INT16_C(-14043)),
      simde_mm512_set_epi16(INT16_C( 16111), INT16_C(-26103), INT16_C( -8800), INT16_C(-24747),
                            INT16_C( 27941), INT16_C(   603), INT16_C( -8112), INT16_C(-28325),
                            INT16_C( -6110), INT16_C(-15530), INT16_C(  -938), INT16_C(-26515),
                            INT16_C( -2492), INT16_C( 16124), INT16_C(-27181), INT16_C( -3573),
                            INT16_C( 11652), INT16_C( 11646), INT16_C(  6964), INT16_C( 27675),
                            INT16_C( 13103), INT16_C( -2742), INT16_C( 27906), INT16_C( -1077),
                            INT16_C(-30636), INT16_C(-23050), INT16_C(  8363), INT16_C( 31350),
                            INT16_C( -4750), INT16_C(  -220), INT16_C( -8954), INT16_C( 15617)),
      simde_mm512_set_epi16(INT16_C( -1958), INT16_C( -4418), INT16_C(-14608), INT16_C(   -75),
                            INT16_C( 19751), INT16_C( -2227), INT16_C(-31326), INT16_C(-27242),
                            INT16_C(   589), INT16_C(-14659), INT16_C(-15010), INT16_C(-28852),
                            INT16_C( 10864), INT16_C( 23384), INT16_C(-32768), INT16_C(-23296),
                            INT16_C( 25132), INT16_C( -3783), INT16_C(-13189), INT16_C( 32767),
                            INT16_C(-13077), INT16_C( 25181), INT16_C( 13807), INT16_C( 24752),
                            INT16_C(-32768), INT16_C( -1411), INT16_C( 25017), INT16_C( 32767),
                            INT16_C( 11026), INT16_C( 22906), INT16_C( 21845), INT16_C(  1574)) },
    { simde_mm512_set_epi16(INT16_C( 25180), INT16_C(-32332), INT16_C(-18604), INT16_C(-25763),
                            INT16_C( 28237), INT16_C(-22027), INT16_C( 18301), INT16_C( 27599),
                            INT16_C( 11815), INT16_C(-14839), INT16_C(-13552), INT16_C( -6300),
                            INT16_C(-30211), INT16_C(-25552), INT16_C(  7375), INT16_C(  5267),
                            INT16_C(-14761), INT16_C(-28375), INT16_C(-32756), INT16_C( 13128),
                            INT16_C( 17683), INT16_C( -4430), INT16_C( -1805), INT16_C(-17763),
                            INT16_C(   206), INT16_C( 15851), INT16_C(  1078), INT16_C( 27402),
                            INT16_C(-15177), INT16_C(  5068), INT16_C(-15079), INT16_C(-13584)),
      simde_mm512_set_epi16(INT16_C( -3533), INT16_C( -3885), INT16_C(   789), INT16_C( -3755),
                            INT16_C(  3668), INT16_C( 19955), INT16_C(-22356), INT16_C(  8403),
                            INT16_C(-16920), INT16_C( 15195), INT16_C( -1853), INT16_C(-12121),
                            INT16_C(  5095), INT16_C( -9909), INT16_C(   574), INT16_C( 31911),
                            INT16_C( -1319), INT16_C( -2880), INT16_C(-31000), INT16_C(-30829),
                            INT16_C(-12917), INT16_C( 21541), INT16_C( 27413), INT16_C(-28040),
                            INT16_C(-19236), INT16_C(-24178), INT16_C(-29369), INT16_C( -3607),
                            INT16_C( -2666), INT16_C(-13470), INT16_C(  4416), INT16_C( 23850)),
      simde_mm512_set_epi16(INT16_C( 21647), INT16_C(-32768), INT16_C(-17815), INT16_C(-29518),
                            INT16_C( 31905), INT16_C( -2072), INT16_C( -4055), INT16_C( 32767),
                            INT16_C( -5105), INT16_C(   356), INT16_C(-15405), INT16_C(-18421),
                            INT16_C(-25116), INT16_C(-32768), INT16_C(  7949), INT16_C( 32767),
                            INT16_C(-16080), INT16_C(-31255), INT16_C(-32768), INT16_C(-17701),
                            INT16_C(  4766), INT16_C( 17111), INT16_C( 25608), INT16_C(-32768),
                            INT16_C(-19030), INT16_C( -8327), INT16_C(-28291), INT16_C( 23795),
                            INT16_C(-17843), INT16_C( -8402), INT16_C(-10663), INT16_C( 10266)) },
    { simde_mm512_set_epi16(INT16_C( 11424), INT16_C( 29699), INT16_C( 29618), INT16_C(-29898),
                            INT16_C(-23079), INT16_C( -6822), INT16_C( -7340), INT16_C(-12296),
                            INT16_C( 17833), INT16_C( -2043), INT16_C(-29766), INT16_C( 30776),
                            INT16_C(-18161), INT16_C(-32768), INT16_C(-31882), INT16_C(  3163),
                            INT16_C( 20808), INT16_C(-24577), INT16_C(  7685), INT16_C(  6279),
                            INT16_C(-27880), INT16_C( 27718), INT16_C(-18275), INT16_C( 24470),
                            INT16_C(  8834), INT16_C( 20594), INT16_C( 23052), INT16_C(-18860),
                            INT16_C(  7784), INT16_C(-16668), INT16_C(-30440), INT16_C(  6387)),
      simde_mm512_set_epi16(INT16_C( 15054), INT16_C(-23448), INT16_C(   866), INT16_C(-23509),
                            INT16_C(    -2), INT16_C(  6267), INT16_C(-15891), INT16_C(-24685),
                            INT16_C( 17511), INT16_C(  7218), INT16_C(-13659), INT16_C(-27162),
                            INT16_C( 10378), INT16_C(-16488), INT16_C(-11578), INT16_C(-27832),
                            INT16_C(-17566), INT16_C(-29377), INT16_C( -2221), INT16_C( 18656),
                            INT16_C(  7516), INT16_C(-30776), INT16_C(-19010), INT16_C( 26269),
                            INT16_C(-18451), INT16_C( 31467), INT16_C(-22066), INT16_C( 30155),
                            INT16_C( -3394), INT16_C( -6131), INT16_C( 13533), INT16_C( 21770)),
      simde_mm512_set_epi16(INT16_C( 26478), INT16_C(  6251), INT16_C( 30484), INT16_C(-32768),
                            INT16_C(-23081), INT16_C(  -555), INT16_C(-23231), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(  5175), INT16_C(-32768), INT16_C(  3614),
                            INT16_C( -7783), INT16_C(-32768), INT16_C(-32768), INT16_C(-24669),
                            INT16_C(  3242), INT16_C(-32768), INT16_C(  5464), INT16_C( 24935),
                            INT16_C(-20364), INT16_C( -3058), INT16_C(-32768), INT16_C( 32767),
                            INT16_C( -9617), INT16_C( 32767), INT16_C(   986), INT16_C( 11295),
                            INT16_C(  4390), INT16_C(-22799), INT16_C(-16907), INT16_C( 28157)) },
    { simde_mm512_set_epi16(INT16_C( 11108), INT16_C(  5360), INT16_C(  8246), INT16_C(  2034),
                            INT16_C(  1485), INT16_C( 28421), INT16_C( 17428), INT16_C(-13900),
                            INT16_C(  4889), INT16_C( 17148), INT16_C( 13806), INT16_C(-26968),
                            INT16_C( -7783), INT16_C( 11345), INT16_C( 17472), INT16_C( 26483),
                            INT16_C(-10340), INT16_C( -7816), INT16_C( 22407), INT16_C( 32028),
                            INT16_C( 18301), INT16_C(-30148), INT16_C( -4736), INT16_C(-10574),
                            INT16_C( 10686), INT16_C( -5463), INT16_C( 19858), INT16_C( -4320),
                            INT16_C(-28154), INT16_C(  8984), INT16_C( 32741), INT16_C( 29052)),
      simde_mm512_set_epi16(INT16_C(-22772), INT16_C(-14596), INT16_C( -9458), INT16_C(  2054),
                            INT16_C(-24090), INT16_C(  1808), INT16_C(  9897), INT16_C( 20003),
                            INT16_C( 21444), INT16_C(  8548), INT16_C( -1773), INT16_C(  8905),
                            INT16_C( 14811), INT16_C(  5752), INT16_C(-17876), INT16_C( -7032),
                            INT16_C(-19669), INT16_C(-30631), INT16_C(-19220), INT16_C( 18906),
                            INT16_C( 15935), INT16_C(  2767), INT16_C(-30937), INT16_C( 13780),
                            INT16_C( 28270), INT16_C( 29263), INT16_C( 11106), INT16_C( 19546),
                            INT16_C( 16947), INT16_C(-29470), INT16_C( -1882), INT16_C(-19990)),
      simde_mm512_set_epi16(INT16_C(-11664), INT16_C( -9236), INT16_C( -1212), INT16_C(  4088),
                            INT16_C(-22605), INT16_C( 30229), INT16_C( 27325), INT16_C(  6103),
                            INT16_C( 26333), INT16_C( 25696), INT16_C( 12033), INT16_C(-18063),
                            INT16_C(  7028), INT16_C( 17097), INT16_C(  -404), INT16_C( 19451),
                            INT16_C(-30009), INT16_C(-32768), INT16_C(  3187), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-27381), INT16_C(-32768), INT16_C(  3206),
                            INT16_C( 32767), INT16_C( 23800), INT16_C( 30964), INT16_C( 15226),
                            INT16_C(-11207), INT16_C(-20486), INT16_C( 30859), INT16_C(  9062)) },
    { simde_mm512_set_epi16(INT16_C(-30561), INT16_C( -5745), INT16_C( 32288), INT16_C( 11324),
                            INT16_C(-18220), INT16_C(-24874), INT16_C(-10154), INT16_C( 15837),
                            INT16_C( 22748), INT16_C(  6197), INT16_C( 23429), INT16_C(-23495),
                            INT16_C( -9809), INT16_C(-19367), INT16_C(-10631), INT16_C( -3845),
                            INT16_C( -5637), INT16_C(-22436), INT16_C(  6126), INT16_C( 16975),
                            INT16_C(  5830), INT16_C( 26963), INT16_C(-12238), INT16_C( 18631),
                            INT16_C(-23641), INT16_C(-17151), INT16_C(-13103), INT16_C(  5981),
                            INT16_C( 25710), INT16_C( 32081), INT16_C(-26911), INT16_C(-20494)),
      simde_mm512_set_epi16(INT16_C( 30869), INT16_C(-18069), INT16_C(  6380), INT16_C(-23760),
                            INT16_C( -6614), INT16_C( 32014), INT16_C( -2152), INT16_C( 25253),
                            INT16_C( -8172), INT16_C(-22416), INT16_C(  2949), INT16_C(  5712),
                            INT16_C( 30074), INT16_C( 20521), INT16_C( 28471), INT16_C( -3867),
                            INT16_C( -3892), INT16_C( 21304), INT16_C(  2611), INT16_C(  5827),
                            INT16_C(-30009), INT16_C( 31479), INT16_C(-19390), INT16_C(  1047),
                            INT16_C(-19967), INT16_C( 11404), INT16_C( 25909), INT16_C(  6741),
                            INT16_C(  1933), INT16_C(-26599), INT16_C(-25469), INT16_C(-30351)),
      simde_mm512_set_epi16(INT16_C(   308), INT16_C(-23814), INT16_C( 32767), INT16_C(-12436),
                            INT16_C(-24834), INT16_C(  7140), INT16_C(-12306), INT16_C( 32767),
                            INT16_C( 14576), INT16_C(-16219), INT16_C( 26378), INT16_C(-17783),
                            INT16_C( 20265), INT16_C(  1154), INT16_C( 17840), INT16_C( -7712),
                            INT16_C( -9529), INT16_C( -1132), INT16_C(  8737), INT16_C( 22802),
                            INT16_C(-24179), INT16_C( 32767), INT16_C(-31628), INT16_C( 19678),
                            INT16_C(-32768), INT16_C( -5747), INT16_C( 12806), INT16_C( 12722),
                            INT16_C( 27643), INT16_C(  5482), INT16_C(-32768), INT16_C(-32768)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_adds_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_adds_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t src[32];
    const simde__mmask32 k;
    const int16_t a[32];
    const int16_t b[32];
    const int16_t r[32];
  } test_vec[] = {
    { {  INT16_C(    72), -INT16_C(  5287), -INT16_C(  4542),  INT16_C(  2288), -INT16_C(  8745),  INT16_C( 27056), -INT16_C( 11265), -INT16_C( 27298),
        -INT16_C( 32448), -INT16_C( 11055), -INT16_C( 28733),  INT16_C(  7301),  INT16_C( 28274), -INT16_C( 27095), -INT16_C(  1294),  INT16_C( 15311),
         INT16_C( 10490),  INT16_C( 15398),  INT16_C(  5654), -INT16_C(  4795), -INT16_C(  2572), -INT16_C(  3241), -INT16_C( 19000),  INT16_C(  2185),
         INT16_C( 23095), -INT16_C(  1315),  INT16_C( 25321),  INT16_C( 23574),  INT16_C( 16336), -INT16_C( 15374), -INT16_C( 16071),  INT16_C( 13566) },
      UINT32_C(   7349481),
      { -INT16_C( 19142),  INT16_C( 12013),  INT16_C( 17579),  INT16_C( 29474), -INT16_C( 21510),  INT16_C( 12668),  INT16_C( 22789), -INT16_C(  4565),
         INT16_C( 17083), -INT16_C( 29878),  INT16_C( 15489), -INT16_C( 17586),  INT16_C( 19709), -INT16_C(  6161),  INT16_C( 24432), -INT16_C( 21529),
        -INT16_C( 11243), -INT16_C( 16167), -INT16_C(  1255),  INT16_C(  4915), -INT16_C( 20570), -INT16_C( 21692),  INT16_C( 28424), -INT16_C( 15462),
        -INT16_C(  6991),  INT16_C( 13135), -INT16_C( 25311),  INT16_C(  7918), -INT16_C(  8726),  INT16_C( 23045), -INT16_C(  5060),  INT16_C( 20741) },
      { -INT16_C(  8255), -INT16_C(  9711),  INT16_C( 17882), -INT16_C( 32275),  INT16_C( 12788), -INT16_C(   724), -INT16_C( 14688),  INT16_C( 21184),
         INT16_C(  4011), -INT16_C( 13179),  INT16_C( 29613), -INT16_C( 26646), -INT16_C(  4016), -INT16_C( 29455), -INT16_C(  2084), -INT16_C( 25122),
        -INT16_C(  4138), -INT16_C( 20361),  INT16_C( 25652),  INT16_C( 10545),  INT16_C( 24213),  INT16_C( 13862), -INT16_C(  6620), -INT16_C( 12408),
         INT16_C(  3574), -INT16_C( 23653), -INT16_C( 31104), -INT16_C( 12230),  INT16_C( 11126),  INT16_C( 21084),  INT16_C( 14882), -INT16_C(  1808) },
      { -INT16_C( 27397), -INT16_C(  5287), -INT16_C(  4542), -INT16_C(  2801), -INT16_C(  8745),  INT16_C( 11944),  INT16_C(  8101),  INT16_C( 16619),
        -INT16_C( 32448), -INT16_C( 11055),       INT16_MAX,  INT16_C(  7301),  INT16_C( 28274),       INT16_MIN, -INT16_C(  1294),  INT16_C( 15311),
         INT16_C( 10490),  INT16_C( 15398),  INT16_C(  5654), -INT16_C(  4795),  INT16_C(  3643), -INT16_C(  7830),  INT16_C( 21804),  INT16_C(  2185),
         INT16_C( 23095), -INT16_C(  1315),  INT16_C( 25321),  INT16_C( 23574),  INT16_C( 16336), -INT16_C( 15374), -INT16_C( 16071),  INT16_C( 13566) } },
    { {  INT16_C( 26410),  INT16_C( 24233), -INT16_C(  9524),  INT16_C( 24967), -INT16_C( 21192),  INT16_C( 23959),  INT16_C(  8084), -INT16_C( 30164),
        -INT16_C( 14292), -INT16_C( 21459),  INT16_C( 26446), -INT16_C( 15236), -INT16_C(  9838), -INT16_C( 19178),  INT16_C(  1555),  INT16_C( 15789),
         INT16_C( 22126),  INT16_C( 15004),  INT16_C(  9009),  INT16_C( 27035),  INT16_C( 13265),  INT16_C( 26054), -INT16_C(  3246),  INT16_C( 32751),
         INT16_C(  7355),  INT16_C(  2347), -INT16_C( 22397),  INT16_C(  5581), -INT16_C(  7295), -INT16_C( 27446),  INT16_C( 30954),  INT16_C( 22738) },
      UINT32_C(4287786702),
      {  INT16_C( 11665),  INT16_C( 25193),  INT16_C( 12128), -INT16_C( 19513), -INT16_C( 18910), -INT16_C(  8910),  INT16_C( 24018),  INT16_C( 21990),
        -INT16_C( 19707), -INT16_C( 31125),  INT16_C( 13719), -INT16_C( 32485), -INT16_C(  4691),  INT16_C( 31961),  INT16_C( 27483), -INT16_C(  4997),
        -INT16_C(  7016), -INT16_C(  1713),  INT16_C(  5652),  INT16_C( 13996), -INT16_C(  8499), -INT16_C( 24812), -INT16_C(  1477),  INT16_C( 16885),
         INT16_C( 24750),  INT16_C( 17863), -INT16_C(  7531),  INT16_C( 17350), -INT16_C( 24625),  INT16_C( 10943),  INT16_C( 14858), -INT16_C( 24041) },
      {  INT16_C( 26143),  INT16_C( 13211),  INT16_C( 18300),  INT16_C( 18793),  INT16_C( 32037),  INT16_C( 25065), -INT16_C(  8584),  INT16_C(  9890),
         INT16_C( 26942), -INT16_C( 11413),  INT16_C( 12620),  INT16_C(  6934), -INT16_C( 10800), -INT16_C(  9658),  INT16_C( 23824),  INT16_C( 12156),
         INT16_C(  6339),  INT16_C( 16226), -INT16_C( 13473), -INT16_C( 31351),  INT16_C( 29257), -INT16_C( 15898), -INT16_C( 30640), -INT16_C( 28953),
         INT16_C( 21233),  INT16_C( 15713),  INT16_C( 30851),  INT16_C( 21337), -INT16_C( 24755),  INT16_C( 23853), -INT16_C( 22020), -INT16_C( 16500) },
      {  INT16_C( 26410),       INT16_MAX,  INT16_C( 30428), -INT16_C(   720), -INT16_C( 21192),  INT16_C( 23959),  INT16_C( 15434),  INT16_C( 31880),
        -INT16_C( 14292),       INT16_MIN,  INT16_C( 26339), -INT16_C( 25551), -INT16_C(  9838),  INT16_C( 22303),       INT16_MAX,  INT16_C( 15789),
         INT16_C( 22126),  INT16_C( 14513),  INT16_C(  9009),  INT16_C( 27035),  INT16_C( 20758),  INT16_C( 26054), -INT16_C(  3246), -INT16_C( 12068),
              INT16_MAX,       INT16_MAX,  INT16_C( 23320),       INT16_MAX,       INT16_MIN,       INT16_MAX, -INT16_C(  7162),       INT16_MIN } },
    { { -INT16_C(  4415),  INT16_C(  8702), -INT16_C( 30790),  INT16_C(   934), -INT16_C( 29447),  INT16_C( 18884), -INT16_C( 21740),  INT16_C(  1495),
         INT16_C( 14845), -INT16_C( 32701), -INT16_C( 25423), -INT16_C(   301),  INT16_C(    59),  INT16_C( 14172), -INT16_C(  5975),  INT16_C( 27638),
        -INT16_C(  2857), -INT16_C( 28276),  INT16_C( 12924),  INT16_C( 30100),  INT16_C( 22718), -INT16_C( 11585), -INT16_C( 27133),  INT16_C(   215),
         INT16_C(  6863), -INT16_C( 32640),  INT16_C( 21430), -INT16_C(  3713), -INT16_C(  9389), -INT16_C(   984),  INT16_C(  7875), -INT16_C( 26009) },
      UINT32_C(2402022163),
      { -INT16_C( 16603), -INT16_C(  7420), -INT16_C( 15593),  INT16_C(  6837), -INT16_C( 29350),  INT16_C( 10522), -INT16_C( 25945),  INT16_C( 24234),
         INT16_C( 10733),  INT16_C( 16463),  INT16_C( 30724), -INT16_C( 14531), -INT16_C( 23402), -INT16_C( 22174), -INT16_C( 29288), -INT16_C( 17096),
         INT16_C( 15693),  INT16_C( 25761),  INT16_C( 22016),  INT16_C( 23167), -INT16_C( 26141), -INT16_C( 29820),  INT16_C( 11828),  INT16_C(  8681),
         INT16_C( 14423),  INT16_C( 23394), -INT16_C( 24656),  INT16_C( 18210), -INT16_C( 31677), -INT16_C(  9232),  INT16_C( 10514),  INT16_C( 24473) },
      {  INT16_C( 14950),  INT16_C( 26307),  INT16_C( 17040),  INT16_C( 29889),  INT16_C( 17884),  INT16_C(  4351), -INT16_C(  6029), -INT16_C( 13775),
        -INT16_C( 27872), -INT16_C( 11995),  INT16_C( 18226),  INT16_C( 30232),  INT16_C(  2252), -INT16_C(  8623), -INT16_C(  5583), -INT16_C( 26819),
         INT16_C(    36), -INT16_C( 18946), -INT16_C( 16573),  INT16_C(  7977),  INT16_C( 10244),  INT16_C( 30511),  INT16_C( 24592),  INT16_C( 12353),
         INT16_C( 26356),  INT16_C(  9729),  INT16_C(  6573),  INT16_C( 31132), -INT16_C(  4574),  INT16_C( 21335), -INT16_C( 27432), -INT16_C(   533) },
      { -INT16_C(  1653),  INT16_C( 18887), -INT16_C( 30790),  INT16_C(   934), -INT16_C( 11466),  INT16_C( 18884), -INT16_C( 21740),  INT16_C(  1495),
        -INT16_C( 17139),  INT16_C(  4468), -INT16_C( 25423), -INT16_C(   301), -INT16_C( 21150), -INT16_C( 30797),       INT16_MIN,       INT16_MIN,
         INT16_C( 15729),  INT16_C(  6815),  INT16_C( 12924),  INT16_C( 31144),  INT16_C( 22718),  INT16_C(   691), -INT16_C( 27133),  INT16_C(   215),
              INT16_MAX,       INT16_MAX, -INT16_C( 18083),       INT16_MAX, -INT16_C(  9389), -INT16_C(   984),  INT16_C(  7875),  INT16_C( 23940) } },
    { { -INT16_C(  5739), -INT16_C( 10062), -INT16_C(  9304), -INT16_C( 21257),  INT16_C(  9731),  INT16_C(  4899),  INT16_C( 25734),  INT16_C( 31299),
         INT16_C( 17866),  INT16_C( 30625),  INT16_C( 15710), -INT16_C( 32527),  INT16_C( 18475),  INT16_C(  1236), -INT16_C( 16419),  INT16_C( 29185),
        -INT16_C( 19544),  INT16_C( 20554),  INT16_C( 16782), -INT16_C( 28164),  INT16_C(  8039), -INT16_C(  4700), -INT16_C(  6269),  INT16_C( 19816),
         INT16_C(  2348), -INT16_C( 29756), -INT16_C( 19130),  INT16_C( 29195), -INT16_C(  8194), -INT16_C(  9354),  INT16_C( 30622),  INT16_C( 17997) },
      UINT32_C(3096876842),
      { -INT16_C( 27944),  INT16_C( 16201), -INT16_C(  4687),  INT16_C( 13356), -INT16_C( 27436),  INT16_C(   385),  INT16_C( 18077), -INT16_C(  7028),
        -INT16_C( 26629), -INT16_C(  1706), -INT16_C( 13193),  INT16_C(  5588),  INT16_C(  8515),  INT16_C( 27996), -INT16_C(  3400), -INT16_C( 28635),
         INT16_C( 28293),  INT16_C( 14031), -INT16_C(   933),  INT16_C( 12139), -INT16_C(  4976),  INT16_C( 11824), -INT16_C( 17358),  INT16_C( 11794),
         INT16_C( 26708), -INT16_C( 13529), -INT16_C(   972),  INT16_C( 30688),  INT16_C( 15389), -INT16_C( 10524),  INT16_C(  2351), -INT16_C( 19354) },
      {  INT16_C( 13943), -INT16_C( 11542),  INT16_C( 21810), -INT16_C( 15871),  INT16_C( 12866),  INT16_C( 29936),  INT16_C(   750),  INT16_C( 17058),
        -INT16_C( 13718), -INT16_C( 25075), -INT16_C(  4410), -INT16_C(  7403), -INT16_C(  1750),  INT16_C( 22969),  INT16_C(  8194),  INT16_C( 30989),
        -INT16_C(  1962), -INT16_C( 30645),  INT16_C( 19789), -INT16_C( 28854),  INT16_C( 15231),  INT16_C( 27908), -INT16_C( 22979), -INT16_C( 22352),
        -INT16_C( 17040),  INT16_C( 13894),  INT16_C( 23723), -INT16_C( 10726), -INT16_C( 11435),  INT16_C( 22575),  INT16_C( 15859),  INT16_C( 18897) },
      { -INT16_C(  5739),  INT16_C(  4659), -INT16_C(  9304), -INT16_C(  2515),  INT16_C(  9731),  INT16_C( 30321),  INT16_C( 25734),  INT16_C( 31299),
              INT16_MIN, -INT16_C( 26781), -INT16_C( 17603), -INT16_C( 32527),  INT16_C(  6765),  INT16_C(  1236), -INT16_C( 16419),  INT16_C(  2354),
        -INT16_C( 19544), -INT16_C( 16614),  INT16_C( 18856), -INT16_C( 28164),  INT16_C( 10255), -INT16_C(  4700), -INT16_C(  6269), -INT16_C( 10558),
         INT16_C(  2348), -INT16_C( 29756), -INT16_C( 19130),  INT16_C( 19962),  INT16_C(  3954),  INT16_C( 12051),  INT16_C( 30622), -INT16_C(   457) } },
    { {  INT16_C(  7477), -INT16_C( 32047),  INT16_C(  7274), -INT16_C(  5870),  INT16_C(  5719), -INT16_C( 27562),  INT16_C(  1724),  INT16_C( 11580),
        -INT16_C( 31804),  INT16_C( 28515),  INT16_C( 32223),  INT16_C( 13381),  INT16_C( 30033),  INT16_C( 17548),  INT16_C( 24242), -INT16_C(  6258),
         INT16_C( 24443), -INT16_C(  6807),  INT16_C( 31611), -INT16_C( 11570),  INT16_C(  9361),  INT16_C( 20071), -INT16_C( 23765), -INT16_C(  4229),
        -INT16_C(  8666),  INT16_C(  1374), -INT16_C( 23460), -INT16_C( 21190), -INT16_C( 14823), -INT16_C( 13327),  INT16_C( 32548), -INT16_C( 24654) },
      UINT32_C(1518607327),
      {  INT16_C( 21143),  INT16_C( 10285), -INT16_C( 27529), -INT16_C( 23946), -INT16_C(  3785),  INT16_C( 24209), -INT16_C(  4144),  INT16_C( 11363),
        -INT16_C( 25197), -INT16_C( 21287), -INT16_C( 13724), -INT16_C( 30601),  INT16_C( 10570),  INT16_C( 10536), -INT16_C( 21435), -INT16_C(  9085),
        -INT16_C( 20225),  INT16_C( 30212),  INT16_C( 31556),  INT16_C( 31768), -INT16_C( 22164),  INT16_C( 15578),  INT16_C( 15768),  INT16_C( 11368),
         INT16_C( 16859),  INT16_C( 16344),  INT16_C( 20492),  INT16_C( 22215), -INT16_C(  4231), -INT16_C( 16769),  INT16_C(   668), -INT16_C( 25702) },
      { -INT16_C( 24653), -INT16_C(  2287),  INT16_C( 10522), -INT16_C( 31117),  INT16_C( 19922),  INT16_C( 27331),  INT16_C( 11147),  INT16_C( 26262),
         INT16_C( 28525),  INT16_C( 31141),  INT16_C( 27839),  INT16_C( 14543),  INT16_C( 20060), -INT16_C(  1801), -INT16_C( 28336),  INT16_C(   915),
        -INT16_C( 23504),  INT16_C( 19195),  INT16_C( 28365), -INT16_C( 24623), -INT16_C( 27460),  INT16_C( 18185), -INT16_C( 24385),  INT16_C( 11437),
         INT16_C( 21007), -INT16_C( 12635),  INT16_C( 29886),  INT16_C(  6662), -INT16_C(   574),  INT16_C(  4882), -INT16_C( 23153), -INT16_C( 16618) },
      { -INT16_C(  3510),  INT16_C(  7998), -INT16_C( 17007),       INT16_MIN,  INT16_C( 16137), -INT16_C( 27562),  INT16_C(  7003),       INT16_MAX,
         INT16_C(  3328),  INT16_C(  9854),  INT16_C( 32223), -INT16_C( 16058),  INT16_C( 30630),  INT16_C( 17548),  INT16_C( 24242), -INT16_C(  6258),
         INT16_C( 24443), -INT16_C(  6807),       INT16_MAX, -INT16_C( 11570),  INT16_C(  9361),  INT16_C( 20071), -INT16_C( 23765),  INT16_C( 22805),
        -INT16_C(  8666),  INT16_C(  3709), -INT16_C( 23460),  INT16_C( 28877), -INT16_C(  4805), -INT16_C( 13327), -INT16_C( 22485), -INT16_C( 24654) } },
    { {  INT16_C(  4425),  INT16_C(  5642), -INT16_C(  9344),  INT16_C( 15541), -INT16_C( 16529),  INT16_C( 11907),  INT16_C( 12383),  INT16_C( 28251),
         INT16_C(   130),  INT16_C( 16444),  INT16_C( 17013),  INT16_C( 14171),  INT16_C( 27968), -INT16_C( 12470),  INT16_C( 24851),  INT16_C( 23694),
        -INT16_C( 26510), -INT16_C(  3469),  INT16_C( 10355), -INT16_C(  7634), -INT16_C( 19993),  INT16_C( 17937),  INT16_C( 27873),  INT16_C( 25524),
        -INT16_C(  3988), -INT16_C(  7772), -INT16_C(   205),  INT16_C( 29465),  INT16_C( 25452),  INT16_C( 32578), -INT16_C( 12092),  INT16_C( 14300) },
      UINT32_C(3693694825),
      {  INT16_C( 22647),  INT16_C( 24511), -INT16_C( 12279), -INT16_C(  5211),  INT16_C( 23100), -INT16_C( 22450), -INT16_C(  3510),  INT16_C( 32138),
        -INT16_C( 23567),  INT16_C( 24304),  INT16_C( 12806), -INT16_C( 13347), -INT16_C( 18173),  INT16_C( 27650),  INT16_C( 11016), -INT16_C( 32696),
         INT16_C(  1923), -INT16_C( 29217), -INT16_C( 31529),  INT16_C(  4984), -INT16_C( 14626),  INT16_C( 10684),  INT16_C( 18105), -INT16_C( 21850),
        -INT16_C( 26647), -INT16_C(  4344), -INT16_C(  6455), -INT16_C( 13126), -INT16_C( 17249), -INT16_C( 22472), -INT16_C( 32280),  INT16_C( 27432) },
      {  INT16_C(  1928),  INT16_C( 24824),  INT16_C( 28811),  INT16_C( 27251),  INT16_C( 12087), -INT16_C(  3949),  INT16_C( 14709),  INT16_C( 24218),
        -INT16_C( 23600), -INT16_C( 26034),  INT16_C(  2185),  INT16_C( 10342), -INT16_C( 24635), -INT16_C( 21040), -INT16_C(  2016), -INT16_C( 22504),
         INT16_C(  4607), -INT16_C( 29944),  INT16_C( 31873), -INT16_C( 18187), -INT16_C( 30549),  INT16_C(  8616),  INT16_C( 17345), -INT16_C( 28033),
        -INT16_C( 12826),  INT16_C( 28460), -INT16_C( 27946), -INT16_C( 25705),  INT16_C( 26673),  INT16_C( 20808),  INT16_C( 24672),  INT16_C( 24826) },
      {  INT16_C( 24575),  INT16_C(  5642), -INT16_C(  9344),  INT16_C( 22040), -INT16_C( 16529), -INT16_C( 26399),  INT16_C( 11199),  INT16_C( 28251),
              INT16_MIN, -INT16_C(  1730),  INT16_C( 14991), -INT16_C(  3005),  INT16_C( 27968), -INT16_C( 12470),  INT16_C(  9000),  INT16_C( 23694),
         INT16_C(  6530), -INT16_C(  3469),  INT16_C( 10355), -INT16_C( 13203), -INT16_C( 19993),  INT16_C( 19300),  INT16_C( 27873),  INT16_C( 25524),
        -INT16_C(  3988), -INT16_C(  7772),       INT16_MIN,       INT16_MIN,  INT16_C(  9424),  INT16_C( 32578), -INT16_C(  7608),       INT16_MAX } },
    { {  INT16_C(   625), -INT16_C(  3093), -INT16_C(  8066),  INT16_C( 10923),  INT16_C( 21608),  INT16_C( 10571), -INT16_C( 13673),  INT16_C( 32187),
        -INT16_C(  6248),  INT16_C( 28396), -INT16_C( 31878), -INT16_C( 21751),  INT16_C( 20971),  INT16_C( 19709), -INT16_C(  2127),  INT16_C(  9132),
        -INT16_C( 26631),  INT16_C( 30742), -INT16_C( 16009), -INT16_C(  8286), -INT16_C(  4843), -INT16_C( 21496), -INT16_C( 15177),  INT16_C( 20265),
         INT16_C(  5547),  INT16_C(  9661), -INT16_C( 14695), -INT16_C( 31535), -INT16_C( 12777), -INT16_C( 13872),  INT16_C( 31941), -INT16_C( 16660) },
      UINT32_C(2318795283),
      { -INT16_C( 10045), -INT16_C(  9879),  INT16_C( 29381),  INT16_C( 32133), -INT16_C( 20682), -INT16_C(  7732), -INT16_C( 30012),  INT16_C( 23815),
        -INT16_C( 10160),  INT16_C( 26850), -INT16_C( 19802),  INT16_C( 27441),  INT16_C(  7471),  INT16_C( 16937),  INT16_C( 24607), -INT16_C(  7475),
         INT16_C( 13880), -INT16_C(   325),  INT16_C( 16808), -INT16_C(  8581),  INT16_C( 18416), -INT16_C( 19264), -INT16_C( 14383),  INT16_C(  8722),
        -INT16_C(  2913),  INT16_C( 17802), -INT16_C( 17498), -INT16_C( 10832), -INT16_C(  9768), -INT16_C(  2280), -INT16_C(  6855),  INT16_C( 29401) },
      { -INT16_C( 27365), -INT16_C( 15248), -INT16_C(  5162), -INT16_C( 14686),  INT16_C( 25138),  INT16_C(  1146), -INT16_C( 29655), -INT16_C( 14298),
        -INT16_C( 20352),  INT16_C(  9997), -INT16_C( 17045),  INT16_C( 17404),  INT16_C(  5271), -INT16_C( 12230),  INT16_C(  5113),  INT16_C(  5442),
        -INT16_C( 19800),  INT16_C( 32473),  INT16_C( 31645), -INT16_C( 12220), -INT16_C( 16418),  INT16_C(  2004), -INT16_C(  1461), -INT16_C( 13104),
        -INT16_C(  8790),  INT16_C(  5619), -INT16_C(  4197),  INT16_C( 12888), -INT16_C( 28156), -INT16_C(   766),  INT16_C( 17829),  INT16_C( 19986) },
      {       INT16_MIN, -INT16_C( 25127), -INT16_C(  8066),  INT16_C( 10923),  INT16_C(  4456),  INT16_C( 10571), -INT16_C( 13673),  INT16_C( 32187),
        -INT16_C(  6248),       INT16_MAX, -INT16_C( 31878), -INT16_C( 21751),  INT16_C( 20971),  INT16_C( 19709), -INT16_C(  2127),  INT16_C(  9132),
        -INT16_C( 26631),  INT16_C( 32148),       INT16_MAX, -INT16_C(  8286),  INT16_C(  1998), -INT16_C( 17260), -INT16_C( 15177),  INT16_C( 20265),
         INT16_C(  5547),  INT16_C( 23421), -INT16_C( 14695),  INT16_C(  2056), -INT16_C( 12777), -INT16_C( 13872),  INT16_C( 31941),       INT16_MAX } },
    { { -INT16_C(  5129), -INT16_C( 27188),  INT16_C(  4455),  INT16_C( 17765),  INT16_C( 14800),  INT16_C(  6988),  INT16_C(  7219), -INT16_C(  8729),
        -INT16_C(  9478), -INT16_C( 27150),  INT16_C( 19146), -INT16_C( 12601), -INT16_C( 13860), -INT16_C( 32309), -INT16_C(  8690),  INT16_C(  1743),
        -INT16_C( 25399),  INT16_C( 12443),  INT16_C(   173),  INT16_C( 32117), -INT16_C( 15815),  INT16_C( 27800), -INT16_C( 32546), -INT16_C( 10167),
         INT16_C( 15194),  INT16_C(  9325),  INT16_C( 13445),  INT16_C( 25074), -INT16_C( 16642),  INT16_C(  3298), -INT16_C( 19812),  INT16_C( 25874) },
      UINT32_C(4220955982),
      {  INT16_C(  2989), -INT16_C(  6536),  INT16_C(  4301), -INT16_C( 21422), -INT16_C( 25712), -INT16_C(  5244), -INT16_C(  3370),  INT16_C( 23311),
         INT16_C(   550),  INT16_C(  9404), -INT16_C( 24640),  INT16_C( 23601),  INT16_C( 17233), -INT16_C( 24639),  INT16_C( 22513), -INT16_C( 24934),
         INT16_C(  4707),  INT16_C( 12421), -INT16_C( 10462), -INT16_C( 19492),  INT16_C( 24947),  INT16_C( 18846), -INT16_C( 21165),  INT16_C( 31141),
         INT16_C( 25007),  INT16_C( 28574), -INT16_C( 12544),  INT16_C( 20939), -INT16_C( 29422),  INT16_C(  1008), -INT16_C( 29980),  INT16_C( 18338) },
      {  INT16_C( 10140), -INT16_C( 16520),  INT16_C( 21758),  INT16_C( 29042),  INT16_C(  4277),  INT16_C(  2235),  INT16_C( 24765),  INT16_C( 28034),
         INT16_C(  8385), -INT16_C( 15652), -INT16_C( 22289),  INT16_C(   275),  INT16_C(  1077),  INT16_C(  6405), -INT16_C( 22642),  INT16_C( 11105),
        -INT16_C(  9778), -INT16_C( 13078),  INT16_C( 23597), -INT16_C(  7362), -INT16_C(  1684),  INT16_C( 10731),  INT16_C( 27993),  INT16_C(  6806),
         INT16_C( 29581),  INT16_C( 31964), -INT16_C(  4069),  INT16_C( 20606), -INT16_C( 31756), -INT16_C( 32151), -INT16_C( 13782), -INT16_C(  1875) },
      { -INT16_C(  5129), -INT16_C( 23056),  INT16_C( 26059),  INT16_C(  7620),  INT16_C( 14800),  INT16_C(  6988),  INT16_C( 21395), -INT16_C(  8729),
         INT16_C(  8935), -INT16_C( 27150),       INT16_MIN,  INT16_C( 23876), -INT16_C( 13860), -INT16_C( 18234), -INT16_C(  8690), -INT16_C( 13829),
        -INT16_C( 25399), -INT16_C(   657),  INT16_C( 13135),  INT16_C( 32117),  INT16_C( 23263),  INT16_C( 27800), -INT16_C( 32546),       INT16_MAX,
              INT16_MAX,       INT16_MAX,  INT16_C( 13445),       INT16_MAX,       INT16_MIN, -INT16_C( 31143),       INT16_MIN,  INT16_C( 16463) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi16(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_adds_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_adds_epi16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask32 k;
    const int16_t a[32];
    const int16_t b[32];
    const int16_t r[32];
  } test_vec[] = {
    { UINT32_C(1382488990),
      {  INT16_C( 13382), -INT16_C(  1131), -INT16_C(  8449),  INT16_C(  8490),  INT16_C( 27735),  INT16_C( 28052), -INT16_C( 21201),  INT16_C( 29768),
         INT16_C( 19795), -INT16_C( 30934), -INT16_C( 32642),  INT16_C( 22005),  INT16_C(  9346),  INT16_C(  8543), -INT16_C( 14529), -INT16_C( 31373),
         INT16_C(  2299), -INT16_C(  1408), -INT16_C( 21530),  INT16_C( 15643), -INT16_C( 20713),  INT16_C( 18091), -INT16_C(  3236), -INT16_C( 20549),
        -INT16_C(  6847), -INT16_C( 16586),  INT16_C( 11109), -INT16_C(  6124),  INT16_C( 29776), -INT16_C( 28919),  INT16_C( 31803),  INT16_C( 13845) },
      { -INT16_C( 27260),  INT16_C( 27184),  INT16_C( 19264),  INT16_C( 22695),  INT16_C( 21242),  INT16_C( 22174),  INT16_C( 22854), -INT16_C( 30971),
         INT16_C( 15423), -INT16_C( 23482),  INT16_C( 23399), -INT16_C( 18548), -INT16_C( 27185),  INT16_C(  2631),  INT16_C( 23569), -INT16_C( 27328),
         INT16_C( 28913),  INT16_C( 13055), -INT16_C( 22597), -INT16_C( 19062),  INT16_C( 10489),  INT16_C( 16139),  INT16_C(  4226), -INT16_C( 15930),
         INT16_C(  3404), -INT16_C( 19355), -INT16_C(  3480),  INT16_C( 14187), -INT16_C( 19833), -INT16_C( 26303), -INT16_C( 32498),  INT16_C(    46) },
      {  INT16_C(     0),  INT16_C( 26053),  INT16_C( 10815),  INT16_C( 31185),       INT16_MAX,  INT16_C(     0),  INT16_C(     0), -INT16_C(  1203),
              INT16_MAX,       INT16_MIN,  INT16_C(     0),  INT16_C(  3457), -INT16_C( 17839),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
         INT16_C( 31212),  INT16_C( 11647),       INT16_MIN,  INT16_C(     0),  INT16_C(     0),       INT16_MAX,  INT16_C(   990),  INT16_C(     0),
         INT16_C(     0),       INT16_MIN,  INT16_C(     0),  INT16_C(     0),  INT16_C(  9943),  INT16_C(     0), -INT16_C(   695),  INT16_C(     0) } },
    { UINT32_C(2888969969),
      { -INT16_C( 17195), -INT16_C( 12703),  INT16_C( 27876),  INT16_C( 26126), -INT16_C( 11140), -INT16_C( 14041), -INT16_C( 29215),  INT16_C( 18813),
        -INT16_C(  6017),  INT16_C(  1664), -INT16_C( 15973), -INT16_C( 22113), -INT16_C( 12734),  INT16_C( 13225), -INT16_C(  9220), -INT16_C( 11809),
         INT16_C( 16535),  INT16_C( 31903), -INT16_C( 21076),  INT16_C( 10722),  INT16_C(  2690),  INT16_C( 25586),  INT16_C( 28567),  INT16_C(  5805),
         INT16_C( 11607), -INT16_C(  3556), -INT16_C( 17169),  INT16_C( 12700),  INT16_C( 17802), -INT16_C( 31131),  INT16_C( 17441), -INT16_C( 18345) },
      { -INT16_C(  2427),  INT16_C( 12596),  INT16_C(  6052),  INT16_C(  9818),  INT16_C( 19489), -INT16_C( 18295),  INT16_C( 14011),  INT16_C(  5070),
        -INT16_C(  5532),  INT16_C( 21253), -INT16_C( 24154),  INT16_C( 12420), -INT16_C(  5657),  INT16_C(  2230),  INT16_C(  3374), -INT16_C( 19520),
        -INT16_C(  2812), -INT16_C( 22300),  INT16_C( 16140),  INT16_C( 11726),  INT16_C( 22411),  INT16_C( 18405), -INT16_C( 19570), -INT16_C(  3494),
         INT16_C( 24477),  INT16_C( 17477), -INT16_C( 14079), -INT16_C(  6028),  INT16_C( 11187), -INT16_C(  7696), -INT16_C( 20424),  INT16_C( 15508) },
      { -INT16_C( 19622),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(  8349), -INT16_C( 32336), -INT16_C( 15204),  INT16_C( 23883),
         INT16_C(     0),  INT16_C( 22917),       INT16_MIN, -INT16_C(  9693),  INT16_C(     0),  INT16_C( 15455),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(  9603),  INT16_C(     0),  INT16_C(     0),  INT16_C( 25101),       INT16_MAX,  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0), -INT16_C( 31248),  INT16_C(  6672),  INT16_C(     0),       INT16_MIN,  INT16_C(     0), -INT16_C(  2837) } },
    { UINT32_C(2984540325),
      { -INT16_C( 19785),  INT16_C( 17374), -INT16_C( 15606), -INT16_C( 26486), -INT16_C(  7050),  INT16_C(  5258), -INT16_C( 12477),  INT16_C( 17496),
        -INT16_C( 13160),  INT16_C( 19244),  INT16_C(  7415),  INT16_C( 12332), -INT16_C( 16179),  INT16_C( 29292),  INT16_C( 20793), -INT16_C(  4060),
         INT16_C(   515),  INT16_C(  3379), -INT16_C( 16954),  INT16_C( 15525),  INT16_C( 12193), -INT16_C(  6832), -INT16_C( 22274), -INT16_C( 26839),
         INT16_C( 22133),  INT16_C( 27874),  INT16_C(  3954),  INT16_C( 16284),  INT16_C(  2511),  INT16_C(  2226), -INT16_C( 10662),  INT16_C( 24057) },
      {  INT16_C( 11480), -INT16_C( 24981),  INT16_C(  4330), -INT16_C( 29733),  INT16_C( 11072),  INT16_C( 15984), -INT16_C( 25900),  INT16_C( 18901),
        -INT16_C( 18192),  INT16_C( 25269),  INT16_C( 21191), -INT16_C( 26974),  INT16_C( 21595), -INT16_C( 19041), -INT16_C( 26582),  INT16_C(   530),
         INT16_C( 32196), -INT16_C( 20831),  INT16_C( 31886), -INT16_C( 12742), -INT16_C( 21849),  INT16_C( 31500), -INT16_C(  7612),  INT16_C( 13508),
         INT16_C( 31386),  INT16_C( 24983),  INT16_C( 14796),  INT16_C( 10231), -INT16_C( 26995), -INT16_C( 18468), -INT16_C(  4562), -INT16_C(  3143) },
      { -INT16_C(  8305),  INT16_C(     0), -INT16_C( 11276),  INT16_C(     0),  INT16_C(     0),  INT16_C( 21242),  INT16_C(     0),       INT16_MAX,
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C( 14642),  INT16_C(  5416),  INT16_C( 10251), -INT16_C(  5789),  INT16_C(     0),
         INT16_C(     0),  INT16_C(     0),  INT16_C( 14932),  INT16_C(     0),  INT16_C(     0),  INT16_C( 24668), -INT16_C( 29886), -INT16_C( 13331),
              INT16_MAX,  INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C( 24484), -INT16_C( 16242),  INT16_C(     0),  INT16_C( 20914) } },
    { UINT32_C(4204878444),
      { -INT16_C(  9258),  INT16_C( 32456), -INT16_C( 11130), -INT16_C( 13575), -INT16_C( 16714),  INT16_C( 20735), -INT16_C( 27080),  INT16_C(  1201),
        -INT16_C( 22065),  INT16_C( 23595),  INT16_C(  1855),  INT16_C( 28179), -INT16_C( 13067),  INT16_C( 24929),  INT16_C(   551), -INT16_C(   677),
         INT16_C(  9182),  INT16_C( 25723),  INT16_C( 30200), -INT16_C( 20946),  INT16_C( 11571),  INT16_C( 27647), -INT16_C( 20285), -INT16_C( 28049),
        -INT16_C( 26023), -INT16_C( 26130),  INT16_C(   417), -INT16_C( 27129),  INT16_C( 26830), -INT16_C(  2568),  INT16_C( 21354),  INT16_C( 18674) },
      {  INT16_C( 28279),  INT16_C( 28588), -INT16_C(  9245),  INT16_C(  5661),  INT16_C(  7176), -INT16_C( 13183), -INT16_C(  3891),  INT16_C(  9822),
         INT16_C( 19850),  INT16_C( 11199), -INT16_C( 14770),  INT16_C(  7361), -INT16_C( 18130), -INT16_C( 26351),  INT16_C(  1037), -INT16_C( 31519),
        -INT16_C( 29070),  INT16_C( 22003),  INT16_C(  4201),  INT16_C( 29035), -INT16_C(  5075), -INT16_C(  1475), -INT16_C( 25380),  INT16_C( 26144),
        -INT16_C(  7959),  INT16_C( 14225),  INT16_C( 21158), -INT16_C( 10924),  INT16_C( 25868),  INT16_C(  6510),  INT16_C( 20329), -INT16_C(  9315) },
      {  INT16_C(     0),  INT16_C(     0), -INT16_C( 20375), -INT16_C(  7914),  INT16_C(     0),  INT16_C(  7552), -INT16_C( 30971),  INT16_C(     0),
         INT16_C(     0),       INT16_MAX,  INT16_C(     0),       INT16_MAX, -INT16_C( 31197),  INT16_C(     0),  INT16_C(  1588),  INT16_C(     0),
        -INT16_C( 19888),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C( 26172),  INT16_C(     0), -INT16_C(  1905),
         INT16_C(     0), -INT16_C( 11905),  INT16_C(     0),       INT16_MIN,       INT16_MAX,  INT16_C(  3942),       INT16_MAX,  INT16_C(  9359) } },
    { UINT32_C(1177587933),
      { -INT16_C( 25696), -INT16_C( 12872), -INT16_C(  2681),  INT16_C( 25543), -INT16_C(  5999),  INT16_C( 31433),  INT16_C( 23240),  INT16_C( 28338),
         INT16_C(  1709), -INT16_C( 18109), -INT16_C( 20117), -INT16_C( 10798),  INT16_C( 28417), -INT16_C(  8528), -INT16_C(  7681), -INT16_C( 24795),
        -INT16_C(  8836),  INT16_C(  1133),  INT16_C( 13522),  INT16_C( 25703),  INT16_C( 12572), -INT16_C(  6946), -INT16_C( 28533),  INT16_C( 14419),
        -INT16_C( 26986),  INT16_C(   753), -INT16_C( 15544),  INT16_C( 18903), -INT16_C( 30926),  INT16_C( 12583),  INT16_C( 19560), -INT16_C(  6703) },
      {  INT16_C( 15913), -INT16_C(   791),  INT16_C( 20594), -INT16_C( 28832),  INT16_C( 16001),  INT16_C(  3443), -INT16_C( 14641),  INT16_C( 25925),
         INT16_C( 14173), -INT16_C( 23193),  INT16_C( 16122),  INT16_C( 11758),  INT16_C(  5574),  INT16_C( 11870),  INT16_C( 12130), -INT16_C( 29933),
        -INT16_C(   915), -INT16_C(  8057), -INT16_C(  6323), -INT16_C( 12689), -INT16_C(  7642), -INT16_C(  2597),  INT16_C(  8617),  INT16_C(  1626),
        -INT16_C( 15784),  INT16_C( 21163), -INT16_C( 26368), -INT16_C( 14721), -INT16_C(  8530),  INT16_C(  4341),  INT16_C(  2061),  INT16_C( 31644) },
      { -INT16_C(  9783),  INT16_C(     0),  INT16_C( 17913), -INT16_C(  3289),  INT16_C( 10002),  INT16_C(     0),  INT16_C(  8599),       INT16_MAX,
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),       INT16_MAX,  INT16_C(     0),  INT16_C(     0),       INT16_MIN,
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(  4930), -INT16_C(  9543),  INT16_C(     0),  INT16_C(     0),
         INT16_C(     0),  INT16_C( 21916),       INT16_MIN,  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C( 21621),  INT16_C(     0) } },
    { UINT32_C(1381704453),
      { -INT16_C( 13813),  INT16_C( 12576), -INT16_C(   852),  INT16_C( 21798), -INT16_C( 32739),  INT16_C( 30043),  INT16_C(  1602),  INT16_C( 17351),
         INT16_C( 18335),  INT16_C( 19977), -INT16_C(   475),  INT16_C( 12894), -INT16_C(  1529),  INT16_C(  3245),  INT16_C(  2078),  INT16_C( 10590),
         INT16_C( 32466),  INT16_C( 32602), -INT16_C( 32646), -INT16_C( 26668),  INT16_C( 12288),  INT16_C( 17164), -INT16_C( 11210), -INT16_C( 10618),
        -INT16_C( 28901),  INT16_C( 16420), -INT16_C( 32114), -INT16_C( 27278),  INT16_C(  8317), -INT16_C( 25695), -INT16_C(   216), -INT16_C(  1084) },
      {  INT16_C(  7805), -INT16_C(  1926),  INT16_C( 20126), -INT16_C( 24945), -INT16_C( 25474), -INT16_C( 18975),  INT16_C( 26480), -INT16_C( 29813),
        -INT16_C( 20489), -INT16_C( 31285),  INT16_C( 15665), -INT16_C( 20966), -INT16_C( 17571), -INT16_C( 31159),  INT16_C(  3514),  INT16_C( 14209),
        -INT16_C(  1237), -INT16_C( 14033), -INT16_C( 16567), -INT16_C( 14232),  INT16_C( 18779), -INT16_C( 13443),  INT16_C(  2225), -INT16_C( 22442),
         INT16_C(  8631), -INT16_C(  6099),  INT16_C( 18270), -INT16_C( 17257), -INT16_C(  8190), -INT16_C( 17342), -INT16_C( 15378),  INT16_C(  6643) },
      { -INT16_C(  6008),  INT16_C(     0),  INT16_C( 19274),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),
        -INT16_C(  2154), -INT16_C( 11308),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C( 27914),  INT16_C(     0),  INT16_C(     0),
         INT16_C( 31229),  INT16_C( 18569),  INT16_C(     0),       INT16_MIN,  INT16_C( 31067),  INT16_C(     0), -INT16_C(  8985),  INT16_C(     0),
         INT16_C(     0),  INT16_C( 10321),  INT16_C(     0),  INT16_C(     0),  INT16_C(   127),  INT16_C(     0), -INT16_C( 15594),  INT16_C(     0) } },
    { UINT32_C( 132326334),
      {  INT16_C( 19426),  INT16_C( 15823),  INT16_C( 19604),  INT16_C( 17672),  INT16_C( 24148),  INT16_C(  3053),  INT16_C(  6783), -INT16_C(  8716),
        -INT16_C( 29855),  INT16_C( 25497), -INT16_C(  9365),  INT16_C( 22815),  INT16_C(  5022),  INT16_C( 23667),  INT16_C( 22070),  INT16_C(  6244),
         INT16_C( 13217),  INT16_C( 13653),  INT16_C( 23936), -INT16_C( 11141),  INT16_C( 26811),  INT16_C( 15072), -INT16_C( 11133), -INT16_C(  7145),
        -INT16_C( 20129), -INT16_C( 13752),  INT16_C( 26508),  INT16_C( 11044), -INT16_C( 26758), -INT16_C( 20345), -INT16_C(  5139), -INT16_C( 28984) },
      {  INT16_C(  7455), -INT16_C( 24637),  INT16_C( 15994),  INT16_C( 13683),  INT16_C( 21415),  INT16_C( 10863), -INT16_C( 30937), -INT16_C( 31218),
         INT16_C( 22072), -INT16_C( 15279),  INT16_C( 30142),  INT16_C( 14575),  INT16_C( 30476), -INT16_C(  1559), -INT16_C( 20126), -INT16_C( 32377),
         INT16_C( 19151),  INT16_C( 18720), -INT16_C( 27511),  INT16_C( 12415), -INT16_C(  4377),  INT16_C(  3930),  INT16_C( 26741), -INT16_C( 21099),
        -INT16_C(  6465),  INT16_C( 32114),  INT16_C( 24923),  INT16_C( 26549), -INT16_C( 24872),  INT16_C( 15200), -INT16_C(  6320),  INT16_C(  8124) },
      {  INT16_C(     0), -INT16_C(  8814),       INT16_MAX,  INT16_C( 31355),       INT16_MAX,  INT16_C( 13916),  INT16_C(     0),       INT16_MIN,
        -INT16_C(  7783),  INT16_C( 10218),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C( 22108),  INT16_C(     0),  INT16_C(     0),
         INT16_C( 32368),  INT16_C( 32373),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C( 19002),  INT16_C( 15608), -INT16_C( 28244),
        -INT16_C( 26594),  INT16_C( 18362),       INT16_MAX,  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { UINT32_C(3144211762),
      { -INT16_C(  6287),  INT16_C( 22763),  INT16_C( 17878),  INT16_C( 19303), -INT16_C(   595),  INT16_C( 27897),  INT16_C( 27619),  INT16_C( 16361),
        -INT16_C( 24628), -INT16_C( 23130),  INT16_C(  1853), -INT16_C( 29216), -INT16_C( 25362),  INT16_C(  8364),  INT16_C(  5497), -INT16_C(  5413),
        -INT16_C( 14596), -INT16_C( 11709), -INT16_C( 22005), -INT16_C( 18146),  INT16_C(  6055), -INT16_C( 29915),  INT16_C(  3970),  INT16_C( 20170),
         INT16_C( 28846), -INT16_C(  5133), -INT16_C( 11401),  INT16_C( 26233),  INT16_C(  9584), -INT16_C(  5754),  INT16_C( 25146),  INT16_C( 14292) },
      {  INT16_C(  5928),  INT16_C( 13321),  INT16_C( 10177),  INT16_C( 27117),  INT16_C(  4670), -INT16_C( 16140), -INT16_C( 16863), -INT16_C( 12529),
         INT16_C(   558), -INT16_C( 22853),  INT16_C( 13526),  INT16_C( 17932), -INT16_C( 28071), -INT16_C( 27601),  INT16_C(  1012),  INT16_C(  7627),
        -INT16_C( 11238), -INT16_C(  9135),  INT16_C( 16124),  INT16_C( 14917),  INT16_C( 14672),  INT16_C( 29435),  INT16_C(  2807),  INT16_C(  9537),
        -INT16_C(  1012), -INT16_C(  7477), -INT16_C( 10448), -INT16_C( 30168),  INT16_C( 22634),  INT16_C( 24094), -INT16_C(  5797),  INT16_C( 30331) },
      {  INT16_C(     0),       INT16_MAX,  INT16_C(     0),  INT16_C(     0),  INT16_C(  4075),  INT16_C( 11757),  INT16_C(     0),  INT16_C(     0),
        -INT16_C( 24070),  INT16_C(     0),  INT16_C( 15379), -INT16_C( 11284),       INT16_MIN,  INT16_C(     0),  INT16_C(  6509),  INT16_C(  2214),
         INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C(  3229),  INT16_C(     0), -INT16_C(   480),  INT16_C(  6777),  INT16_C(     0),
         INT16_C( 27834), -INT16_C( 12610),  INT16_C(     0), -INT16_C(  3935),  INT16_C( 32218),  INT16_C( 18340),  INT16_C(     0),       INT16_MAX } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_adds_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_adds_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu8(UINT8_C( 52), UINT8_C( 29), UINT8_C( 31), UINT8_C(206),
                             UINT8_C( 40), UINT8_C(160), UINT8_C(244), UINT8_C( 85),
                             UINT8_C( 47), UINT8_C(153), UINT8_C(218), UINT8_C(226),
                             UINT8_C( 99), UINT8_C(129), UINT8_C( 68), UINT8_C(218),
                             UINT8_C(122), UINT8_C( 56), UINT8_C(240), UINT8_C( 10),
                             UINT8_C(228), UINT8_C( 40), UINT8_C( 70), UINT8_C( 14),
                             UINT8_C( 67), UINT8_C( 37), UINT8_C( 87), UINT8_C(131),
                             UINT8_C( 29), UINT8_C(175), UINT8_C(167), UINT8_C(223),
                             UINT8_C(170), UINT8_C( 76), UINT8_C( 39), UINT8_C(254),
                             UINT8_C(222), UINT8_C(105), UINT8_C(193), UINT8_C(144),
                             UINT8_C(182), UINT8_C(106), UINT8_C(199), UINT8_C(161),
                             UINT8_C(173), UINT8_C( 65), UINT8_C( 54), UINT8_C(148),
                             UINT8_C(222), UINT8_C(  9), UINT8_C(237), UINT8_C( 76),
                             UINT8_C(149), UINT8_C(196), UINT8_C(194), UINT8_C(114),
                             UINT8_C(167), UINT8_C( 47), UINT8_C(174), UINT8_C(135),
                             UINT8_C(170), UINT8_C(  1), UINT8_C( 40), UINT8_C( 68)),
      simde_x_mm512_set_epu8(UINT8_C(183), UINT8_C(129), UINT8_C(214), UINT8_C( 34),
                             UINT8_C(187), UINT8_C(142), UINT8_C(238), UINT8_C(248),
                             UINT8_C(175), UINT8_C(127), UINT8_C(231), UINT8_C(164),
                             UINT8_C( 16), UINT8_C(128), UINT8_C( 32), UINT8_C( 28),
                             UINT8_C(111), UINT8_C( 40), UINT8_C(  8), UINT8_C(160),
                             UINT8_C( 37), UINT8_C(193), UINT8_C(195), UINT8_C(135),
                             UINT8_C(102), UINT8_C( 87), UINT8_C(146), UINT8_C(206),
                             UINT8_C( 94), UINT8_C(109), UINT8_C(112), UINT8_C( 83),
                             UINT8_C( 63), UINT8_C(215), UINT8_C( 46), UINT8_C( 55),
                             UINT8_C( 78), UINT8_C(211), UINT8_C(175), UINT8_C(210),
                             UINT8_C( 60), UINT8_C(205), UINT8_C(118), UINT8_C(142),
                             UINT8_C(109), UINT8_C( 48), UINT8_C( 54), UINT8_C( 54),
                             UINT8_C( 99), UINT8_C(248), UINT8_C(120), UINT8_C(165),
                             UINT8_C(189), UINT8_C(132), UINT8_C(130), UINT8_C( 16),
                             UINT8_C(  6), UINT8_C(104), UINT8_C(176), UINT8_C(111),
                             UINT8_C(254), UINT8_C(193), UINT8_C(106), UINT8_C(  5)),
      simde_x_mm512_set_epu8(UINT8_C(235), UINT8_C(158), UINT8_C(245), UINT8_C(240),
                             UINT8_C(227), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(222), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(115), UINT8_C(255), UINT8_C(100), UINT8_C(246),
                             UINT8_C(233), UINT8_C( 96), UINT8_C(248), UINT8_C(170),
                             UINT8_C(255), UINT8_C(233), UINT8_C(255), UINT8_C(149),
                             UINT8_C(169), UINT8_C(124), UINT8_C(233), UINT8_C(255),
                             UINT8_C(123), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(233), UINT8_C(255), UINT8_C( 85), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(242), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(113), UINT8_C(108), UINT8_C(202),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(241),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(130),
                             UINT8_C(173), UINT8_C(151), UINT8_C(255), UINT8_C(246),
                             UINT8_C(255), UINT8_C(194), UINT8_C(146), UINT8_C( 73)) },
    { simde_x_mm512_set_epu8(UINT8_C( 38), UINT8_C(254), UINT8_C(204), UINT8_C( 72),
                             UINT8_C(177), UINT8_C( 63), UINT8_C( 67), UINT8_C( 50),
                             UINT8_C( 71), UINT8_C(226), UINT8_C(146), UINT8_C(  4),
                             UINT8_C(134), UINT8_C( 91), UINT8_C(195), UINT8_C(148),
                             UINT8_C( 28), UINT8_C(158), UINT8_C(223), UINT8_C(181),
                             UINT8_C(218), UINT8_C(114), UINT8_C(200), UINT8_C(243),
                             UINT8_C(255), UINT8_C( 48), UINT8_C(233), UINT8_C( 82),
                             UINT8_C( 22), UINT8_C( 26), UINT8_C(128), UINT8_C(166),
                             UINT8_C( 79), UINT8_C(206), UINT8_C(109), UINT8_C(113),
                             UINT8_C( 86), UINT8_C(112), UINT8_C(215), UINT8_C(187),
                             UINT8_C(174), UINT8_C(250), UINT8_C(225), UINT8_C(206),
                             UINT8_C(242), UINT8_C(141), UINT8_C(197), UINT8_C(179),
                             UINT8_C( 88), UINT8_C(237), UINT8_C(191), UINT8_C(149),
                             UINT8_C(130), UINT8_C( 63), UINT8_C( 54), UINT8_C(129),
                             UINT8_C(126), UINT8_C( 18), UINT8_C( 43), UINT8_C(251),
                             UINT8_C(214), UINT8_C(155), UINT8_C( 61), UINT8_C(138)),
      simde_x_mm512_set_epu8(UINT8_C(142), UINT8_C( 33), UINT8_C( 51), UINT8_C(244),
                             UINT8_C( 33), UINT8_C(106), UINT8_C( 75), UINT8_C(210),
                             UINT8_C(121), UINT8_C( 52), UINT8_C(233), UINT8_C(109),
                             UINT8_C(205), UINT8_C(175), UINT8_C(230), UINT8_C( 70),
                             UINT8_C( 28), UINT8_C(216), UINT8_C( 76), UINT8_C(137),
                             UINT8_C(129), UINT8_C(  5), UINT8_C(149), UINT8_C(132),
                             UINT8_C(159), UINT8_C( 55), UINT8_C( 47), UINT8_C(212),
                             UINT8_C(202), UINT8_C( 26), UINT8_C(247), UINT8_C( 19),
                             UINT8_C(109), UINT8_C( 59), UINT8_C( 49), UINT8_C(175),
                             UINT8_C( 15), UINT8_C(240), UINT8_C(155), UINT8_C( 20),
                             UINT8_C( 85), UINT8_C( 57), UINT8_C( 91), UINT8_C(177),
                             UINT8_C(179), UINT8_C(200), UINT8_C( 26), UINT8_C( 99),
                             UINT8_C(  9), UINT8_C(177), UINT8_C(151), UINT8_C(246),
                             UINT8_C(116), UINT8_C(162), UINT8_C(144), UINT8_C( 44),
                             UINT8_C(240), UINT8_C(226), UINT8_C(215), UINT8_C(206),
                             UINT8_C(114), UINT8_C(250), UINT8_C(170), UINT8_C(220)),
      simde_x_mm512_set_epu8(UINT8_C(180), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(210), UINT8_C(169), UINT8_C(142), UINT8_C(255),
                             UINT8_C(192), UINT8_C(255), UINT8_C(255), UINT8_C(113),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(218),
                             UINT8_C( 56), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(119), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(103), UINT8_C(255), UINT8_C(255),
                             UINT8_C(224), UINT8_C( 52), UINT8_C(255), UINT8_C(185),
                             UINT8_C(188), UINT8_C(255), UINT8_C(158), UINT8_C(255),
                             UINT8_C(101), UINT8_C(255), UINT8_C(255), UINT8_C(207),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(223), UINT8_C(255),
                             UINT8_C( 97), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(246), UINT8_C(225), UINT8_C(198), UINT8_C(173),
                             UINT8_C(255), UINT8_C(244), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(231), UINT8_C(255)) },
    { simde_x_mm512_set_epu8(UINT8_C(103), UINT8_C(253), UINT8_C( 31), UINT8_C(185),
                             UINT8_C(101), UINT8_C(201), UINT8_C( 20), UINT8_C( 77),
                             UINT8_C(110), UINT8_C(116), UINT8_C(226), UINT8_C( 37),
                             UINT8_C(155), UINT8_C( 44), UINT8_C(198), UINT8_C(182),
                             UINT8_C(132), UINT8_C( 61), UINT8_C(104), UINT8_C(181),
                             UINT8_C( 77), UINT8_C(161), UINT8_C(154), UINT8_C(195),
                             UINT8_C(148), UINT8_C(239), UINT8_C(178), UINT8_C(165),
                             UINT8_C(  6), UINT8_C(251), UINT8_C( 97), UINT8_C(247),
                             UINT8_C(  8), UINT8_C( 52), UINT8_C( 96), UINT8_C( 34),
                             UINT8_C(233), UINT8_C(  1), UINT8_C(  7), UINT8_C(254),
                             UINT8_C(  3), UINT8_C(  8), UINT8_C(168), UINT8_C(221),
                             UINT8_C(238), UINT8_C(144), UINT8_C( 18), UINT8_C(245),
                             UINT8_C(216), UINT8_C(239), UINT8_C(128), UINT8_C(174),
                             UINT8_C(108), UINT8_C(236), UINT8_C(204), UINT8_C(151),
                             UINT8_C( 83), UINT8_C(224), UINT8_C(253), UINT8_C( 94),
                             UINT8_C( 40), UINT8_C( 10), UINT8_C(195), UINT8_C( 72)),
      simde_x_mm512_set_epu8(UINT8_C(196), UINT8_C(220), UINT8_C(209), UINT8_C(146),
                             UINT8_C( 66), UINT8_C(240), UINT8_C(160), UINT8_C( 58),
                             UINT8_C(206), UINT8_C(157), UINT8_C(225), UINT8_C(238),
                             UINT8_C(235), UINT8_C( 53), UINT8_C( 91), UINT8_C( 97),
                             UINT8_C(175), UINT8_C( 22), UINT8_C( 33), UINT8_C(101),
                             UINT8_C(189), UINT8_C(131), UINT8_C(219), UINT8_C( 30),
                             UINT8_C( 36), UINT8_C( 11), UINT8_C(180), UINT8_C( 46),
                             UINT8_C(126), UINT8_C(239), UINT8_C(135), UINT8_C( 44),
                             UINT8_C(198), UINT8_C(149), UINT8_C(235), UINT8_C( 72),
                             UINT8_C( 18), UINT8_C(214), UINT8_C( 92), UINT8_C(199),
                             UINT8_C( 79), UINT8_C( 28), UINT8_C(156), UINT8_C( 37),
                             UINT8_C(212), UINT8_C(167), UINT8_C( 62), UINT8_C(201),
                             UINT8_C(139), UINT8_C(215), UINT8_C( 68), UINT8_C(117),
                             UINT8_C( 83), UINT8_C(105), UINT8_C(232), UINT8_C(251),
                             UINT8_C(  6), UINT8_C(165), UINT8_C(199), UINT8_C(183),
                             UINT8_C(254), UINT8_C( 55), UINT8_C(167), UINT8_C(238)),
      simde_x_mm512_set_epu8(UINT8_C(255), UINT8_C(255), UINT8_C(240), UINT8_C(255),
                             UINT8_C(167), UINT8_C(255), UINT8_C(180), UINT8_C(135),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C( 97), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C( 83), UINT8_C(137), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(225),
                             UINT8_C(184), UINT8_C(250), UINT8_C(255), UINT8_C(211),
                             UINT8_C(132), UINT8_C(255), UINT8_C(232), UINT8_C(255),
                             UINT8_C(206), UINT8_C(201), UINT8_C(255), UINT8_C(106),
                             UINT8_C(251), UINT8_C(215), UINT8_C( 99), UINT8_C(255),
                             UINT8_C( 82), UINT8_C( 36), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C( 80), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(196), UINT8_C(255),
                             UINT8_C(191), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C( 89), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C( 65), UINT8_C(255), UINT8_C(255)) },
    { simde_x_mm512_set_epu8(UINT8_C(  7), UINT8_C( 99), UINT8_C(210), UINT8_C(110),
                             UINT8_C(202), UINT8_C( 10), UINT8_C(247), UINT8_C( 68),
                             UINT8_C(177), UINT8_C(186), UINT8_C(247), UINT8_C( 36),
                             UINT8_C(212), UINT8_C(146), UINT8_C(251), UINT8_C( 66),
                             UINT8_C(  0), UINT8_C(  8), UINT8_C(116), UINT8_C(102),
                             UINT8_C(175), UINT8_C(223), UINT8_C( 15), UINT8_C(232),
                             UINT8_C(127), UINT8_C( 63), UINT8_C( 84), UINT8_C(165),
                             UINT8_C( 18), UINT8_C(250), UINT8_C( 71), UINT8_C( 18),
                             UINT8_C(132), UINT8_C(155), UINT8_C(102), UINT8_C( 18),
                             UINT8_C(177), UINT8_C(146), UINT8_C(169), UINT8_C(139),
                             UINT8_C( 21), UINT8_C( 79), UINT8_C(  4), UINT8_C(129),
                             UINT8_C(168), UINT8_C( 42), UINT8_C(  9), UINT8_C(184),
                             UINT8_C( 46), UINT8_C( 53), UINT8_C(243), UINT8_C(140),
                             UINT8_C(158), UINT8_C(112), UINT8_C( 35), UINT8_C( 49),
                             UINT8_C(253), UINT8_C(233), UINT8_C( 33), UINT8_C(178),
                             UINT8_C(117), UINT8_C(161), UINT8_C(168), UINT8_C( 39)),
      simde_x_mm512_set_epu8(UINT8_C( 75), UINT8_C(218), UINT8_C(162), UINT8_C( 30),
                             UINT8_C(119), UINT8_C(231), UINT8_C(193), UINT8_C(230),
                             UINT8_C(151), UINT8_C( 49), UINT8_C(234), UINT8_C( 78),
                             UINT8_C( 76), UINT8_C( 37), UINT8_C(174), UINT8_C(105),
                             UINT8_C(224), UINT8_C(210), UINT8_C(188), UINT8_C(142),
                             UINT8_C(116), UINT8_C(129), UINT8_C( 93), UINT8_C(124),
                             UINT8_C(248), UINT8_C(145), UINT8_C(230), UINT8_C( 41),
                             UINT8_C( 66), UINT8_C(115), UINT8_C(180), UINT8_C( 47),
                             UINT8_C( 55), UINT8_C(104), UINT8_C(123), UINT8_C(150),
                             UINT8_C(165), UINT8_C( 22), UINT8_C(  0), UINT8_C(101),
                             UINT8_C( 55), UINT8_C( 23), UINT8_C( 90), UINT8_C( 64),
                             UINT8_C( 52), UINT8_C(254), UINT8_C(116), UINT8_C(243),
                             UINT8_C(  4), UINT8_C( 84), UINT8_C( 82), UINT8_C(103),
                             UINT8_C( 43), UINT8_C(237), UINT8_C(  7), UINT8_C( 76),
                             UINT8_C( 88), UINT8_C(212), UINT8_C( 63), UINT8_C(146),
                             UINT8_C(224), UINT8_C(223), UINT8_C(207), UINT8_C(220)),
      simde_x_mm512_set_epu8(UINT8_C( 82), UINT8_C(255), UINT8_C(255), UINT8_C(140),
                             UINT8_C(255), UINT8_C(241), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(235), UINT8_C(255), UINT8_C(114),
                             UINT8_C(255), UINT8_C(183), UINT8_C(255), UINT8_C(171),
                             UINT8_C(224), UINT8_C(218), UINT8_C(255), UINT8_C(244),
                             UINT8_C(255), UINT8_C(255), UINT8_C(108), UINT8_C(255),
                             UINT8_C(255), UINT8_C(208), UINT8_C(255), UINT8_C(206),
                             UINT8_C( 84), UINT8_C(255), UINT8_C(251), UINT8_C( 65),
                             UINT8_C(187), UINT8_C(255), UINT8_C(225), UINT8_C(168),
                             UINT8_C(255), UINT8_C(168), UINT8_C(169), UINT8_C(240),
                             UINT8_C( 76), UINT8_C(102), UINT8_C( 94), UINT8_C(193),
                             UINT8_C(220), UINT8_C(255), UINT8_C(125), UINT8_C(255),
                             UINT8_C( 50), UINT8_C(137), UINT8_C(255), UINT8_C(243),
                             UINT8_C(201), UINT8_C(255), UINT8_C( 42), UINT8_C(125),
                             UINT8_C(255), UINT8_C(255), UINT8_C( 96), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255)) },
    { simde_x_mm512_set_epu8(UINT8_C(233), UINT8_C(112), UINT8_C( 62), UINT8_C( 65),
                             UINT8_C( 37), UINT8_C(247), UINT8_C(105), UINT8_C( 54),
                             UINT8_C(214), UINT8_C(255), UINT8_C(145), UINT8_C( 18),
                             UINT8_C(160), UINT8_C( 92), UINT8_C( 29), UINT8_C(253),
                             UINT8_C(230), UINT8_C( 83), UINT8_C(176), UINT8_C( 43),
                             UINT8_C( 78), UINT8_C( 31), UINT8_C(113), UINT8_C(239),
                             UINT8_C( 38), UINT8_C(192), UINT8_C(127), UINT8_C( 44),
                             UINT8_C(218), UINT8_C(134), UINT8_C(224), UINT8_C( 35),
                             UINT8_C(155), UINT8_C(171), UINT8_C( 44), UINT8_C( 53),
                             UINT8_C(101), UINT8_C(234), UINT8_C(121), UINT8_C( 95),
                             UINT8_C(232), UINT8_C(104), UINT8_C( 20), UINT8_C( 52),
                             UINT8_C( 59), UINT8_C( 62), UINT8_C( 55), UINT8_C( 43),
                             UINT8_C(246), UINT8_C( 36), UINT8_C( 25), UINT8_C( 98),
                             UINT8_C(155), UINT8_C(231), UINT8_C( 43), UINT8_C(203),
                             UINT8_C( 77), UINT8_C( 43), UINT8_C( 80), UINT8_C( 85),
                             UINT8_C(227), UINT8_C( 34), UINT8_C( 77), UINT8_C(179)),
      simde_x_mm512_set_epu8(UINT8_C( 68), UINT8_C(183), UINT8_C(190), UINT8_C(248),
                             UINT8_C( 54), UINT8_C(194), UINT8_C(  0), UINT8_C(190),
                             UINT8_C(102), UINT8_C(185), UINT8_C(  5), UINT8_C(238),
                             UINT8_C(188), UINT8_C(104), UINT8_C(139), UINT8_C(163),
                             UINT8_C(242), UINT8_C( 94), UINT8_C(181), UINT8_C(214),
                             UINT8_C(170), UINT8_C(182), UINT8_C(139), UINT8_C( 96),
                             UINT8_C( 80), UINT8_C( 38), UINT8_C(151), UINT8_C(132),
                             UINT8_C(243), UINT8_C( 37), UINT8_C(120), UINT8_C( 30),
                             UINT8_C(115), UINT8_C( 34), UINT8_C(222), UINT8_C(121),
                             UINT8_C(112), UINT8_C(205), UINT8_C(161), UINT8_C( 52),
                             UINT8_C(142), UINT8_C(118), UINT8_C(195), UINT8_C(200),
                             UINT8_C( 59), UINT8_C(  6), UINT8_C(115), UINT8_C(109),
                             UINT8_C( 93), UINT8_C(174), UINT8_C(200), UINT8_C(190),
                             UINT8_C(151), UINT8_C(126), UINT8_C(210), UINT8_C(200),
                             UINT8_C( 79), UINT8_C( 43), UINT8_C( 20), UINT8_C(239),
                             UINT8_C(149), UINT8_C(119), UINT8_C( 96), UINT8_C(120)),
      simde_x_mm512_set_epu8(UINT8_C(255), UINT8_C(255), UINT8_C(252), UINT8_C(255),
                             UINT8_C( 91), UINT8_C(255), UINT8_C(105), UINT8_C(244),
                             UINT8_C(255), UINT8_C(255), UINT8_C(150), UINT8_C(255),
                             UINT8_C(255), UINT8_C(196), UINT8_C(168), UINT8_C(255),
                             UINT8_C(255), UINT8_C(177), UINT8_C(255), UINT8_C(255),
                             UINT8_C(248), UINT8_C(213), UINT8_C(252), UINT8_C(255),
                             UINT8_C(118), UINT8_C(230), UINT8_C(255), UINT8_C(176),
                             UINT8_C(255), UINT8_C(171), UINT8_C(255), UINT8_C( 65),
                             UINT8_C(255), UINT8_C(205), UINT8_C(255), UINT8_C(174),
                             UINT8_C(213), UINT8_C(255), UINT8_C(255), UINT8_C(147),
                             UINT8_C(255), UINT8_C(222), UINT8_C(215), UINT8_C(252),
                             UINT8_C(118), UINT8_C( 68), UINT8_C(170), UINT8_C(152),
                             UINT8_C(255), UINT8_C(210), UINT8_C(225), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(253), UINT8_C(255),
                             UINT8_C(156), UINT8_C( 86), UINT8_C(100), UINT8_C(255),
                             UINT8_C(255), UINT8_C(153), UINT8_C(173), UINT8_C(255)) },
    { simde_x_mm512_set_epu8(UINT8_C(150), UINT8_C(150), UINT8_C(146), UINT8_C(157),
                             UINT8_C(103), UINT8_C( 31), UINT8_C(236), UINT8_C(192),
                             UINT8_C(155), UINT8_C( 46), UINT8_C(170), UINT8_C(176),
                             UINT8_C(189), UINT8_C( 74), UINT8_C(246), UINT8_C(  0),
                             UINT8_C( 85), UINT8_C( 30), UINT8_C(190), UINT8_C( 12),
                             UINT8_C(117), UINT8_C( 74), UINT8_C(194), UINT8_C( 97),
                             UINT8_C(132), UINT8_C( 63), UINT8_C(227), UINT8_C( 21),
                             UINT8_C(244), UINT8_C(229), UINT8_C(176), UINT8_C(122),
                             UINT8_C(159), UINT8_C( 35), UINT8_C(115), UINT8_C(240),
                             UINT8_C( 88), UINT8_C(151), UINT8_C(101), UINT8_C(194),
                             UINT8_C( 49), UINT8_C(128), UINT8_C( 87), UINT8_C(152),
                             UINT8_C(173), UINT8_C(242), UINT8_C(150), UINT8_C( 26),
                             UINT8_C(183), UINT8_C( 56), UINT8_C(181), UINT8_C(193),
                             UINT8_C(231), UINT8_C(135), UINT8_C(190), UINT8_C( 42),
                             UINT8_C( 33), UINT8_C(174), UINT8_C( 57), UINT8_C(232),
                             UINT8_C(176), UINT8_C(125), UINT8_C( 87), UINT8_C( 79)),
      simde_x_mm512_set_epu8(UINT8_C( 68), UINT8_C(  1), UINT8_C(208), UINT8_C( 74),
                             UINT8_C(102), UINT8_C(168), UINT8_C( 41), UINT8_C( 41),
                             UINT8_C( 83), UINT8_C( 95), UINT8_C(152), UINT8_C(131),
                             UINT8_C(230), UINT8_C( 27), UINT8_C(126), UINT8_C(230),
                             UINT8_C(128), UINT8_C(133), UINT8_C(107), UINT8_C(197),
                             UINT8_C( 23), UINT8_C(  2), UINT8_C(108), UINT8_C(207),
                             UINT8_C( 82), UINT8_C(175), UINT8_C(247), UINT8_C(180),
                             UINT8_C( 57), UINT8_C( 31), UINT8_C(124), UINT8_C(203),
                             UINT8_C(246), UINT8_C(195), UINT8_C(173), UINT8_C(161),
                             UINT8_C(132), UINT8_C(162), UINT8_C( 13), UINT8_C(205),
                             UINT8_C(225), UINT8_C( 25), UINT8_C( 39), UINT8_C(181),
                             UINT8_C(149), UINT8_C(149), UINT8_C(138), UINT8_C(158),
                             UINT8_C(215), UINT8_C(147), UINT8_C(  0), UINT8_C( 87),
                             UINT8_C(244), UINT8_C( 10), UINT8_C(  1), UINT8_C( 19),
                             UINT8_C(173), UINT8_C(221), UINT8_C(118), UINT8_C( 51),
                             UINT8_C( 98), UINT8_C(183), UINT8_C(125), UINT8_C( 92)),
      simde_x_mm512_set_epu8(UINT8_C(218), UINT8_C(151), UINT8_C(255), UINT8_C(231),
                             UINT8_C(205), UINT8_C(199), UINT8_C(255), UINT8_C(233),
                             UINT8_C(238), UINT8_C(141), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(101), UINT8_C(255), UINT8_C(230),
                             UINT8_C(213), UINT8_C(163), UINT8_C(255), UINT8_C(209),
                             UINT8_C(140), UINT8_C( 76), UINT8_C(255), UINT8_C(255),
                             UINT8_C(214), UINT8_C(238), UINT8_C(255), UINT8_C(201),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(230), UINT8_C(255), UINT8_C(255),
                             UINT8_C(220), UINT8_C(255), UINT8_C(114), UINT8_C(255),
                             UINT8_C(255), UINT8_C(153), UINT8_C(126), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(184),
                             UINT8_C(255), UINT8_C(203), UINT8_C(181), UINT8_C(255),
                             UINT8_C(255), UINT8_C(145), UINT8_C(191), UINT8_C( 61),
                             UINT8_C(206), UINT8_C(255), UINT8_C(175), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(212), UINT8_C(171)) },
    { simde_x_mm512_set_epu8(UINT8_C(166), UINT8_C(112), UINT8_C( 16), UINT8_C( 20),
                             UINT8_C(189), UINT8_C( 23), UINT8_C( 28), UINT8_C(226),
                             UINT8_C( 92), UINT8_C(105), UINT8_C(249), UINT8_C( 71),
                             UINT8_C( 22), UINT8_C(153), UINT8_C( 38), UINT8_C( 58),
                             UINT8_C( 86), UINT8_C(191), UINT8_C( 97), UINT8_C(186),
                             UINT8_C(117), UINT8_C( 93), UINT8_C( 63), UINT8_C(142),
                             UINT8_C( 82), UINT8_C(123), UINT8_C(210), UINT8_C(159),
                             UINT8_C(175), UINT8_C(187), UINT8_C( 33), UINT8_C(134),
                             UINT8_C(156), UINT8_C(243), UINT8_C(191), UINT8_C(139),
                             UINT8_C(173), UINT8_C( 70), UINT8_C(113), UINT8_C( 30),
                             UINT8_C(184), UINT8_C( 82), UINT8_C(234), UINT8_C(177),
                             UINT8_C(234), UINT8_C(131), UINT8_C(180), UINT8_C( 12),
                             UINT8_C( 47), UINT8_C(213), UINT8_C( 54), UINT8_C(222),
                             UINT8_C(114), UINT8_C(251), UINT8_C(236), UINT8_C( 57),
                             UINT8_C(205), UINT8_C(187), UINT8_C( 48), UINT8_C(112),
                             UINT8_C( 34), UINT8_C( 49), UINT8_C( 95), UINT8_C(198)),
      simde_x_mm512_set_epu8(UINT8_C(116), UINT8_C(101), UINT8_C(212), UINT8_C(199),
                             UINT8_C( 88), UINT8_C(165), UINT8_C(253), UINT8_C( 88),
                             UINT8_C(138), UINT8_C( 27), UINT8_C(153), UINT8_C(155),
                             UINT8_C(248), UINT8_C(175), UINT8_C(152), UINT8_C(246),
                             UINT8_C( 17), UINT8_C(224), UINT8_C( 49), UINT8_C(127),
                             UINT8_C(200), UINT8_C( 46), UINT8_C( 58), UINT8_C(159),
                             UINT8_C(177), UINT8_C( 24), UINT8_C(139), UINT8_C(240),
                             UINT8_C( 45), UINT8_C(115), UINT8_C(179), UINT8_C(230),
                             UINT8_C(249), UINT8_C(217), UINT8_C(202), UINT8_C( 21),
                             UINT8_C(134), UINT8_C(111), UINT8_C( 79), UINT8_C(153),
                             UINT8_C(130), UINT8_C( 83), UINT8_C(196), UINT8_C(101),
                             UINT8_C(151), UINT8_C(117), UINT8_C(152), UINT8_C(155),
                             UINT8_C(116), UINT8_C(201), UINT8_C(241), UINT8_C( 71),
                             UINT8_C( 24), UINT8_C(105), UINT8_C(212), UINT8_C(191),
                             UINT8_C(  1), UINT8_C(252), UINT8_C( 15), UINT8_C( 49),
                             UINT8_C(105), UINT8_C(159), UINT8_C(109), UINT8_C( 31)),
      simde_x_mm512_set_epu8(UINT8_C(255), UINT8_C(213), UINT8_C(228), UINT8_C(219),
                             UINT8_C(255), UINT8_C(188), UINT8_C(255), UINT8_C(255),
                             UINT8_C(230), UINT8_C(132), UINT8_C(255), UINT8_C(226),
                             UINT8_C(255), UINT8_C(255), UINT8_C(190), UINT8_C(255),
                             UINT8_C(103), UINT8_C(255), UINT8_C(146), UINT8_C(255),
                             UINT8_C(255), UINT8_C(139), UINT8_C(121), UINT8_C(255),
                             UINT8_C(255), UINT8_C(147), UINT8_C(255), UINT8_C(255),
                             UINT8_C(220), UINT8_C(255), UINT8_C(212), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(160),
                             UINT8_C(255), UINT8_C(181), UINT8_C(192), UINT8_C(183),
                             UINT8_C(255), UINT8_C(165), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(248), UINT8_C(255), UINT8_C(167),
                             UINT8_C(163), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(138), UINT8_C(255), UINT8_C(255), UINT8_C(248),
                             UINT8_C(206), UINT8_C(255), UINT8_C( 63), UINT8_C(161),
                             UINT8_C(139), UINT8_C(208), UINT8_C(204), UINT8_C(229)) },
    { simde_x_mm512_set_epu8(UINT8_C(239), UINT8_C(174), UINT8_C( 81), UINT8_C( 89),
                             UINT8_C( 63), UINT8_C(160), UINT8_C(219), UINT8_C(204),
                             UINT8_C(161), UINT8_C(  4), UINT8_C(134), UINT8_C(137),
                             UINT8_C( 48), UINT8_C( 33), UINT8_C(178), UINT8_C(122),
                             UINT8_C( 99), UINT8_C(167), UINT8_C( 95), UINT8_C(109),
                             UINT8_C( 38), UINT8_C(191), UINT8_C(  3), UINT8_C( 48),
                             UINT8_C(143), UINT8_C( 51), UINT8_C( 81), UINT8_C( 76),
                             UINT8_C(251), UINT8_C(204), UINT8_C(198), UINT8_C(  1),
                             UINT8_C(175), UINT8_C(123), UINT8_C( 24), UINT8_C(237),
                             UINT8_C(190), UINT8_C( 62), UINT8_C( 87), UINT8_C(255),
                             UINT8_C(119), UINT8_C(202), UINT8_C(  4), UINT8_C( 40),
                             UINT8_C(152), UINT8_C(128), UINT8_C(116), UINT8_C( 72),
                             UINT8_C(245), UINT8_C(148), UINT8_C( 39), UINT8_C(106),
                             UINT8_C(195), UINT8_C(186), UINT8_C(241), UINT8_C(178),
                             UINT8_C( 68), UINT8_C(210), UINT8_C( 61), UINT8_C(218),
                             UINT8_C(116), UINT8_C( 98), UINT8_C( 41), UINT8_C( 22)),
      simde_x_mm512_set_epu8(UINT8_C(  3), UINT8_C(245), UINT8_C(  8), UINT8_C(179),
                             UINT8_C(242), UINT8_C( 19), UINT8_C(107), UINT8_C(170),
                             UINT8_C(178), UINT8_C(175), UINT8_C(174), UINT8_C(170),
                             UINT8_C( 72), UINT8_C(195), UINT8_C(199), UINT8_C( 66),
                             UINT8_C(182), UINT8_C(183), UINT8_C( 58), UINT8_C(245),
                             UINT8_C( 62), UINT8_C( 31), UINT8_C( 93), UINT8_C( 74),
                             UINT8_C(114), UINT8_C( 92), UINT8_C(199), UINT8_C( 53),
                             UINT8_C( 94), UINT8_C(185), UINT8_C(107), UINT8_C(124),
                             UINT8_C(160), UINT8_C(193), UINT8_C( 17), UINT8_C( 65),
                             UINT8_C(242), UINT8_C( 55), UINT8_C(163), UINT8_C(102),
                             UINT8_C( 59), UINT8_C(173), UINT8_C(244), UINT8_C(232),
                             UINT8_C(182), UINT8_C(139), UINT8_C(254), UINT8_C( 81),
                             UINT8_C(228), UINT8_C(175), UINT8_C(104), UINT8_C(167),
                             UINT8_C( 38), UINT8_C(114), UINT8_C(184), UINT8_C(118),
                             UINT8_C(167), UINT8_C( 58), UINT8_C(153), UINT8_C(103),
                             UINT8_C(204), UINT8_C(126), UINT8_C(111), UINT8_C( 63)),
      simde_x_mm512_set_epu8(UINT8_C(242), UINT8_C(255), UINT8_C( 89), UINT8_C(255),
                             UINT8_C(255), UINT8_C(179), UINT8_C(255), UINT8_C(255),
                             UINT8_C(255), UINT8_C(179), UINT8_C(255), UINT8_C(255),
                             UINT8_C(120), UINT8_C(228), UINT8_C(255), UINT8_C(188),
                             UINT8_C(255), UINT8_C(255), UINT8_C(153), UINT8_C(255),
                             UINT8_C(100), UINT8_C(222), UINT8_C( 96), UINT8_C(122),
                             UINT8_C(255), UINT8_C(143), UINT8_C(255), UINT8_C(129),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(125),
                             UINT8_C(255), UINT8_C(255), UINT8_C( 41), UINT8_C(255),
                             UINT8_C(255), UINT8_C(117), UINT8_C(250), UINT8_C(255),
                             UINT8_C(178), UINT8_C(255), UINT8_C(248), UINT8_C(255),
                             UINT8_C(255), UINT8_C(255), UINT8_C(255), UINT8_C(153),
                             UINT8_C(255), UINT8_C(255), UINT8_C(143), UINT8_C(255),
                             UINT8_C(233), UINT8_C(255), UINT8_C(255), UINT8_C(255),
                             UINT8_C(235), UINT8_C(255), UINT8_C(214), UINT8_C(255),
                             UINT8_C(255), UINT8_C(224), UINT8_C(152), UINT8_C( 85)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_adds_epu8(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_adds_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask64 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu8(UINT8_C(  92), UINT8_C( 116), UINT8_C( 178), UINT8_C( 237),
                             UINT8_C( 183), UINT8_C(  22), UINT8_C( 190), UINT8_C( 227),
                             UINT8_C(  55), UINT8_C(  78), UINT8_C( 211), UINT8_C( 137),
                             UINT8_C(  30), UINT8_C( 227), UINT8_C( 233), UINT8_C(  89),
                             UINT8_C(  15), UINT8_C( 119), UINT8_C( 168), UINT8_C( 109),
                             UINT8_C( 240), UINT8_C( 246), UINT8_C(   0), UINT8_C( 238),
                             UINT8_C(   1), UINT8_C( 116), UINT8_C(  62), UINT8_C( 183),
                             UINT8_C( 134), UINT8_C(   0), UINT8_C( 111), UINT8_C(  95),
                             UINT8_C( 236), UINT8_C(   3), UINT8_C( 122), UINT8_C( 135),
                             UINT8_C(  13), UINT8_C(  25), UINT8_C( 221), UINT8_C( 129),
                             UINT8_C(  82), UINT8_C(  90), UINT8_C(  53), UINT8_C( 123),
                             UINT8_C(  73), UINT8_C( 108), UINT8_C( 238), UINT8_C(  15),
                             UINT8_C(  81), UINT8_C( 229), UINT8_C( 102), UINT8_C( 118),
                             UINT8_C(  39), UINT8_C( 179), UINT8_C(  45), UINT8_C(  81),
                             UINT8_C( 239), UINT8_C( 228), UINT8_C(  67), UINT8_C( 138),
                             UINT8_C(  79), UINT8_C( 143), UINT8_C( 134), UINT8_C( 124)),
      UINT64_C(10224647434006242820),
      simde_x_mm512_set_epu8(UINT8_C(  68), UINT8_C( 148), UINT8_C( 135), UINT8_C( 141),
                             UINT8_C( 197), UINT8_C( 178), UINT8_C( 145), UINT8_C(  56),
                             UINT8_C(   9), UINT8_C( 135), UINT8_C( 120), UINT8_C( 220),
                             UINT8_C(  36), UINT8_C( 153), UINT8_C( 152), UINT8_C(  25),
                             UINT8_C( 145), UINT8_C( 193), UINT8_C( 221), UINT8_C( 136),
                             UINT8_C(  28), UINT8_C( 212), UINT8_C( 230), UINT8_C( 170),
                             UINT8_C( 243), UINT8_C(  53), UINT8_C( 168), UINT8_C( 149),
                             UINT8_C(  68), UINT8_C(  42), UINT8_C( 138), UINT8_C( 111),
                             UINT8_C(  54), UINT8_C( 198), UINT8_C( 243), UINT8_C(  27),
                             UINT8_C(  23), UINT8_C(  41), UINT8_C( 137), UINT8_C(  44),
                             UINT8_C(   7), UINT8_C( 136), UINT8_C(  32), UINT8_C( 213),
                             UINT8_C( 114), UINT8_C( 184), UINT8_C(  73), UINT8_C( 160),
                             UINT8_C(  96), UINT8_C( 110), UINT8_C( 175), UINT8_C( 180),
                             UINT8_C( 103), UINT8_C( 156), UINT8_C( 234), UINT8_C(  18),
                             UINT8_C( 115), UINT8_C(  54), UINT8_C( 216), UINT8_C( 125),
                             UINT8_C( 110), UINT8_C(  31), UINT8_C(  51), UINT8_C( 152)),
      simde_x_mm512_set_epu8(UINT8_C( 242), UINT8_C(  85), UINT8_C( 176), UINT8_C(  80),
                             UINT8_C(  48), UINT8_C(  93), UINT8_C(  79), UINT8_C( 127),
                             UINT8_C(  16), UINT8_C(  41), UINT8_C(  54), UINT8_C( 140),
                             UINT8_C(  17), UINT8_C(  42), UINT8_C( 170), UINT8_C(  38),
                             UINT8_C( 138), UINT8_C( 200), UINT8_C(  60), UINT8_C(  19),
                             UINT8_C( 156), UINT8_C( 149), UINT8_C( 105), UINT8_C( 180),
                             UINT8_C(  10), UINT8_C( 247), UINT8_C( 244), UINT8_C( 200),
                             UINT8_C( 185), UINT8_C(  96), UINT8_C(   6), UINT8_C(  24),
                             UINT8_C(  69), UINT8_C(  73), UINT8_C( 229), UINT8_C(   3),
                             UINT8_C( 161), UINT8_C(  50), UINT8_C( 189), UINT8_C( 217),
                             UINT8_C(  97), UINT8_C(  62), UINT8_C( 101), UINT8_C(  56),
                             UINT8_C( 177), UINT8_C(  83), UINT8_C( 157), UINT8_C( 200),
                             UINT8_C( 239), UINT8_C( 200), UINT8_C( 248), UINT8_C( 240),
                             UINT8_C(  84), UINT8_C( 161), UINT8_C( 183), UINT8_C(  54),
                             UINT8_C( 125), UINT8_C(  85), UINT8_C(  78), UINT8_C( 191),
                             UINT8_C(   4), UINT8_C( 113), UINT8_C( 240), UINT8_C( 203)),
      simde_x_mm512_set_epu8(UINT8_C( 255), UINT8_C( 116), UINT8_C( 178), UINT8_C( 237),
                             UINT8_C( 245), UINT8_C( 255), UINT8_C( 190), UINT8_C( 183),
                             UINT8_C(  25), UINT8_C( 176), UINT8_C( 174), UINT8_C( 137),
                             UINT8_C(  30), UINT8_C( 195), UINT8_C( 233), UINT8_C(  63),
                             UINT8_C(  15), UINT8_C( 119), UINT8_C( 255), UINT8_C( 155),
                             UINT8_C( 184), UINT8_C( 255), UINT8_C( 255), UINT8_C( 238),
                             UINT8_C( 253), UINT8_C( 116), UINT8_C( 255), UINT8_C( 183),
                             UINT8_C( 253), UINT8_C(   0), UINT8_C( 144), UINT8_C( 135),
                             UINT8_C( 236), UINT8_C(   3), UINT8_C( 255), UINT8_C(  30),
                             UINT8_C(  13), UINT8_C(  25), UINT8_C( 221), UINT8_C( 255),
                             UINT8_C( 104), UINT8_C(  90), UINT8_C(  53), UINT8_C( 123),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 238), UINT8_C( 255),
                             UINT8_C(  81), UINT8_C( 229), UINT8_C( 255), UINT8_C( 118),
                             UINT8_C(  39), UINT8_C( 255), UINT8_C( 255), UINT8_C(  81),
                             UINT8_C( 239), UINT8_C( 228), UINT8_C(  67), UINT8_C( 138),
                             UINT8_C(  79), UINT8_C( 144), UINT8_C( 134), UINT8_C( 124)) },
    { simde_x_mm512_set_epu8(UINT8_C( 133), UINT8_C( 156), UINT8_C( 217), UINT8_C( 212),
                             UINT8_C(  22), UINT8_C( 112), UINT8_C(  16), UINT8_C(  15),
                             UINT8_C( 152), UINT8_C( 154), UINT8_C( 102), UINT8_C( 161),
                             UINT8_C(  69), UINT8_C(  13), UINT8_C( 235), UINT8_C(  45),
                             UINT8_C(  62), UINT8_C( 136), UINT8_C( 145), UINT8_C(  32),
                             UINT8_C( 149), UINT8_C( 226), UINT8_C(  99), UINT8_C( 192),
                             UINT8_C(   8), UINT8_C( 214), UINT8_C(  81), UINT8_C( 222),
                             UINT8_C( 210), UINT8_C(  26), UINT8_C(  31), UINT8_C( 254),
                             UINT8_C(  68), UINT8_C( 249), UINT8_C( 185), UINT8_C(  46),
                             UINT8_C( 235), UINT8_C( 183), UINT8_C(  21), UINT8_C(  83),
                             UINT8_C( 148), UINT8_C( 159), UINT8_C( 187), UINT8_C(  73),
                             UINT8_C(  57), UINT8_C( 219), UINT8_C(  21), UINT8_C(  82),
                             UINT8_C( 137), UINT8_C( 130), UINT8_C( 126), UINT8_C(  91),
                             UINT8_C( 115), UINT8_C(  31), UINT8_C( 177), UINT8_C(  28),
                             UINT8_C( 150), UINT8_C( 238), UINT8_C(  65), UINT8_C( 152),
                             UINT8_C(  81), UINT8_C(  38), UINT8_C( 172), UINT8_C( 254)),
      UINT64_C( 4597426592773770833),
      simde_x_mm512_set_epu8(UINT8_C(  74), UINT8_C(  72), UINT8_C( 252), UINT8_C(  28),
                             UINT8_C( 141), UINT8_C(  93), UINT8_C( 102), UINT8_C(  44),
                             UINT8_C( 153), UINT8_C( 227), UINT8_C( 206), UINT8_C(  48),
                             UINT8_C( 160), UINT8_C( 206), UINT8_C(  46), UINT8_C( 191),
                             UINT8_C(   4), UINT8_C(  43), UINT8_C( 181), UINT8_C(  97),
                             UINT8_C( 230), UINT8_C( 153), UINT8_C(  71), UINT8_C( 149),
                             UINT8_C(  91), UINT8_C(  45), UINT8_C( 245), UINT8_C(  47),
                             UINT8_C(  29), UINT8_C(  25), UINT8_C(  26), UINT8_C( 247),
                             UINT8_C(  10), UINT8_C(  36), UINT8_C( 177), UINT8_C( 203),
                             UINT8_C(  41), UINT8_C(   1), UINT8_C( 233), UINT8_C( 193),
                             UINT8_C( 129), UINT8_C(  68), UINT8_C(  48), UINT8_C(  36),
                             UINT8_C(  89), UINT8_C( 144), UINT8_C( 225), UINT8_C( 120),
                             UINT8_C(  35), UINT8_C(  62), UINT8_C( 235), UINT8_C( 142),
                             UINT8_C( 152), UINT8_C(  57), UINT8_C(  42), UINT8_C( 145),
                             UINT8_C(  94), UINT8_C( 193), UINT8_C( 247), UINT8_C(  64),
                             UINT8_C( 191), UINT8_C( 254), UINT8_C( 110), UINT8_C( 248)),
      simde_x_mm512_set_epu8(UINT8_C(  44), UINT8_C( 164), UINT8_C( 225), UINT8_C(  26),
                             UINT8_C( 157), UINT8_C( 203), UINT8_C( 117), UINT8_C(  18),
                             UINT8_C( 193), UINT8_C(  45), UINT8_C(  12), UINT8_C(  24),
                             UINT8_C( 148), UINT8_C(  18), UINT8_C( 196), UINT8_C(  28),
                             UINT8_C(  50), UINT8_C( 245), UINT8_C( 188), UINT8_C( 225),
                             UINT8_C( 105), UINT8_C( 150), UINT8_C(  98), UINT8_C(  51),
                             UINT8_C(  58), UINT8_C( 103), UINT8_C( 111), UINT8_C( 129),
                             UINT8_C(  68), UINT8_C( 200), UINT8_C( 124), UINT8_C( 137),
                             UINT8_C(  74), UINT8_C( 194), UINT8_C( 140), UINT8_C(  37),
                             UINT8_C( 244), UINT8_C( 114), UINT8_C(   0), UINT8_C(  61),
                             UINT8_C( 103), UINT8_C( 252), UINT8_C( 151), UINT8_C( 188),
                             UINT8_C(  39), UINT8_C( 156), UINT8_C( 163), UINT8_C(  11),
                             UINT8_C( 176), UINT8_C( 237), UINT8_C( 234), UINT8_C( 217),
                             UINT8_C( 127), UINT8_C( 218), UINT8_C( 131), UINT8_C( 145),
                             UINT8_C(  84), UINT8_C( 160), UINT8_C(  87), UINT8_C( 234),
                             UINT8_C( 251), UINT8_C( 253), UINT8_C( 129), UINT8_C(  41)),
      simde_x_mm512_set_epu8(UINT8_C( 133), UINT8_C( 156), UINT8_C( 255), UINT8_C(  54),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 219), UINT8_C(  62),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 102), UINT8_C( 161),
                             UINT8_C( 255), UINT8_C( 224), UINT8_C( 235), UINT8_C( 219),
                             UINT8_C(  62), UINT8_C( 255), UINT8_C( 145), UINT8_C( 255),
                             UINT8_C( 149), UINT8_C( 255), UINT8_C( 169), UINT8_C( 200),
                             UINT8_C(   8), UINT8_C( 214), UINT8_C( 255), UINT8_C( 222),
                             UINT8_C( 210), UINT8_C(  26), UINT8_C(  31), UINT8_C( 254),
                             UINT8_C(  84), UINT8_C( 249), UINT8_C( 255), UINT8_C( 240),
                             UINT8_C( 255), UINT8_C( 115), UINT8_C( 233), UINT8_C( 254),
                             UINT8_C( 148), UINT8_C( 159), UINT8_C( 199), UINT8_C( 224),
                             UINT8_C(  57), UINT8_C( 255), UINT8_C( 255), UINT8_C( 131),
                             UINT8_C( 137), UINT8_C( 130), UINT8_C( 126), UINT8_C(  91),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 173), UINT8_C(  28),
                             UINT8_C( 150), UINT8_C( 255), UINT8_C(  65), UINT8_C( 255),
                             UINT8_C(  81), UINT8_C(  38), UINT8_C( 172), UINT8_C( 255)) },
    { simde_x_mm512_set_epu8(UINT8_C( 226), UINT8_C( 219), UINT8_C(  51), UINT8_C( 220),
                             UINT8_C(   8), UINT8_C(  52), UINT8_C(  97), UINT8_C( 123),
                             UINT8_C( 207), UINT8_C( 132), UINT8_C(  95), UINT8_C( 173),
                             UINT8_C(  70), UINT8_C( 206), UINT8_C( 195), UINT8_C(  25),
                             UINT8_C( 159), UINT8_C(  28), UINT8_C( 198), UINT8_C(  11),
                             UINT8_C( 242), UINT8_C( 126), UINT8_C(  81), UINT8_C(  45),
                             UINT8_C( 233), UINT8_C( 120), UINT8_C( 173), UINT8_C( 240),
                             UINT8_C(   7), UINT8_C(  51), UINT8_C( 199), UINT8_C( 206),
                             UINT8_C( 235), UINT8_C(  98), UINT8_C(  88), UINT8_C(   0),
                             UINT8_C( 190), UINT8_C(   3), UINT8_C( 124), UINT8_C( 143),
                             UINT8_C(  50), UINT8_C(  88), UINT8_C( 171), UINT8_C( 163),
                             UINT8_C( 212), UINT8_C( 243), UINT8_C( 162), UINT8_C(  17),
                             UINT8_C( 254), UINT8_C(  79), UINT8_C( 140), UINT8_C(  43),
                             UINT8_C( 179), UINT8_C( 131), UINT8_C( 233), UINT8_C( 136),
                             UINT8_C(  96), UINT8_C( 192), UINT8_C( 233), UINT8_C( 210),
                             UINT8_C( 227), UINT8_C( 185), UINT8_C(  71), UINT8_C(  90)),
      UINT64_C(12627002542648829104),
      simde_x_mm512_set_epu8(UINT8_C(  56), UINT8_C( 211), UINT8_C( 148), UINT8_C( 237),
                             UINT8_C( 132), UINT8_C( 229), UINT8_C(  22), UINT8_C( 126),
                             UINT8_C( 150), UINT8_C( 188), UINT8_C( 196), UINT8_C(   8),
                             UINT8_C(  60), UINT8_C(  93), UINT8_C( 223), UINT8_C( 229),
                             UINT8_C( 249), UINT8_C(  27), UINT8_C( 134), UINT8_C( 218),
                             UINT8_C(  23), UINT8_C(   6), UINT8_C(  45), UINT8_C( 235),
                             UINT8_C( 233), UINT8_C( 155), UINT8_C( 116), UINT8_C( 127),
                             UINT8_C(  96), UINT8_C(  40), UINT8_C( 159), UINT8_C(  40),
                             UINT8_C(  86), UINT8_C( 212), UINT8_C(  70), UINT8_C( 185),
                             UINT8_C(  62), UINT8_C( 235), UINT8_C(  66), UINT8_C(  68),
                             UINT8_C( 169), UINT8_C( 195), UINT8_C(  48), UINT8_C( 186),
                             UINT8_C(  18), UINT8_C( 178), UINT8_C( 158), UINT8_C( 117),
                             UINT8_C(  74), UINT8_C(  32), UINT8_C(  93), UINT8_C( 125),
                             UINT8_C( 209), UINT8_C( 196), UINT8_C( 170), UINT8_C( 117),
                             UINT8_C( 122), UINT8_C( 202), UINT8_C(  50), UINT8_C( 123),
                             UINT8_C( 225), UINT8_C( 182), UINT8_C( 192), UINT8_C(  54)),
      simde_x_mm512_set_epu8(UINT8_C( 115), UINT8_C(  51), UINT8_C( 165), UINT8_C(  56),
                             UINT8_C(  64), UINT8_C( 217), UINT8_C( 137), UINT8_C( 228),
                             UINT8_C( 202), UINT8_C(  28), UINT8_C(  54), UINT8_C( 248),
                             UINT8_C( 202), UINT8_C( 128), UINT8_C( 228), UINT8_C( 185),
                             UINT8_C( 107), UINT8_C( 190), UINT8_C( 142), UINT8_C( 168),
                             UINT8_C(  34), UINT8_C( 173), UINT8_C( 235), UINT8_C( 192),
                             UINT8_C( 121), UINT8_C( 236), UINT8_C( 167), UINT8_C( 162),
                             UINT8_C( 112), UINT8_C( 229), UINT8_C(  81), UINT8_C( 202),
                             UINT8_C( 192), UINT8_C( 142), UINT8_C(  48), UINT8_C( 167),
                             UINT8_C( 195), UINT8_C(  26), UINT8_C(  43), UINT8_C(  29),
                             UINT8_C(   0), UINT8_C(  64), UINT8_C( 123), UINT8_C( 189),
                             UINT8_C(  15), UINT8_C( 120), UINT8_C(  36), UINT8_C(  40),
                             UINT8_C( 106), UINT8_C( 138), UINT8_C( 148), UINT8_C( 198),
                             UINT8_C(  26), UINT8_C( 145), UINT8_C(  63), UINT8_C( 158),
                             UINT8_C( 243), UINT8_C( 244), UINT8_C( 132), UINT8_C(  96),
                             UINT8_C( 243), UINT8_C( 158), UINT8_C(  99), UINT8_C( 243)),
      simde_x_mm512_set_epu8(UINT8_C( 171), UINT8_C( 219), UINT8_C( 255), UINT8_C( 220),
                             UINT8_C( 196), UINT8_C( 255), UINT8_C( 159), UINT8_C( 255),
                             UINT8_C( 207), UINT8_C( 132), UINT8_C( 250), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C( 221), UINT8_C( 195), UINT8_C(  25),
                             UINT8_C( 159), UINT8_C(  28), UINT8_C( 198), UINT8_C( 255),
                             UINT8_C(  57), UINT8_C( 179), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 255), UINT8_C( 240),
                             UINT8_C( 208), UINT8_C(  51), UINT8_C( 199), UINT8_C( 242),
                             UINT8_C( 235), UINT8_C( 255), UINT8_C( 118), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 124), UINT8_C( 143),
                             UINT8_C( 169), UINT8_C(  88), UINT8_C( 171), UINT8_C( 163),
                             UINT8_C(  33), UINT8_C( 243), UINT8_C( 162), UINT8_C( 157),
                             UINT8_C( 180), UINT8_C(  79), UINT8_C( 241), UINT8_C(  43),
                             UINT8_C( 179), UINT8_C( 255), UINT8_C( 233), UINT8_C( 136),
                             UINT8_C( 255), UINT8_C( 192), UINT8_C( 182), UINT8_C( 219),
                             UINT8_C( 227), UINT8_C( 185), UINT8_C(  71), UINT8_C(  90)) },
    { simde_x_mm512_set_epu8(UINT8_C(  72), UINT8_C(  45), UINT8_C( 120), UINT8_C( 251),
                             UINT8_C( 147), UINT8_C(  62), UINT8_C(  17), UINT8_C(  31),
                             UINT8_C( 226), UINT8_C( 198), UINT8_C(  56), UINT8_C(  21),
                             UINT8_C(  72), UINT8_C( 182), UINT8_C( 216), UINT8_C( 120),
                             UINT8_C(  95), UINT8_C( 108), UINT8_C(  32), UINT8_C(  64),
                             UINT8_C( 128), UINT8_C( 102), UINT8_C( 235), UINT8_C(  28),
                             UINT8_C( 105), UINT8_C(  52), UINT8_C(  85), UINT8_C( 152),
                             UINT8_C(  57), UINT8_C( 225), UINT8_C( 218), UINT8_C( 132),
                             UINT8_C( 149), UINT8_C( 254), UINT8_C(  55), UINT8_C(  46),
                             UINT8_C( 185), UINT8_C(  77), UINT8_C(  18), UINT8_C(  70),
                             UINT8_C(  89), UINT8_C( 125), UINT8_C( 214), UINT8_C( 131),
                             UINT8_C( 121), UINT8_C( 245), UINT8_C( 187), UINT8_C( 197),
                             UINT8_C( 203), UINT8_C(  34), UINT8_C(   9), UINT8_C(  64),
                             UINT8_C( 195), UINT8_C( 231), UINT8_C( 141), UINT8_C( 100),
                             UINT8_C(  65), UINT8_C(   8), UINT8_C(  69), UINT8_C( 248),
                             UINT8_C( 241), UINT8_C( 205), UINT8_C(   1), UINT8_C(  90)),
      UINT64_C(14515151237088493607),
      simde_x_mm512_set_epu8(UINT8_C( 230), UINT8_C( 247), UINT8_C(  66), UINT8_C(   1),
                             UINT8_C( 243), UINT8_C(  60), UINT8_C( 137), UINT8_C( 173),
                             UINT8_C( 134), UINT8_C( 192), UINT8_C( 173), UINT8_C( 182),
                             UINT8_C( 119), UINT8_C( 248), UINT8_C(  12), UINT8_C( 113),
                             UINT8_C( 244), UINT8_C( 172), UINT8_C(   6), UINT8_C(  69),
                             UINT8_C(   2), UINT8_C( 181), UINT8_C( 222), UINT8_C( 130),
                             UINT8_C(   3), UINT8_C( 128), UINT8_C( 247), UINT8_C(  24),
                             UINT8_C(  11), UINT8_C( 162), UINT8_C( 224), UINT8_C( 110),
                             UINT8_C(  33), UINT8_C( 232), UINT8_C( 125), UINT8_C(  35),
                             UINT8_C( 153), UINT8_C( 208), UINT8_C( 234), UINT8_C(  38),
                             UINT8_C( 175), UINT8_C(   9), UINT8_C( 245), UINT8_C( 132),
                             UINT8_C(  71), UINT8_C(  31), UINT8_C( 214), UINT8_C(  93),
                             UINT8_C(  67), UINT8_C(  45), UINT8_C(  51), UINT8_C( 164),
                             UINT8_C( 126), UINT8_C( 108), UINT8_C( 133), UINT8_C( 185),
                             UINT8_C( 113), UINT8_C(  32), UINT8_C(  71), UINT8_C(  55),
                             UINT8_C( 230), UINT8_C(  82), UINT8_C( 175), UINT8_C( 236)),
      simde_x_mm512_set_epu8(UINT8_C( 131), UINT8_C( 121), UINT8_C( 128), UINT8_C( 103),
                             UINT8_C(   0), UINT8_C( 101), UINT8_C( 215), UINT8_C(  89),
                             UINT8_C( 173), UINT8_C( 191), UINT8_C(   9), UINT8_C( 249),
                             UINT8_C( 193), UINT8_C(  13), UINT8_C( 105), UINT8_C(  92),
                             UINT8_C( 238), UINT8_C( 235), UINT8_C( 154), UINT8_C( 142),
                             UINT8_C(  74), UINT8_C( 121), UINT8_C( 211), UINT8_C(  52),
                             UINT8_C( 193), UINT8_C( 163), UINT8_C(  98), UINT8_C( 106),
                             UINT8_C( 147), UINT8_C( 209), UINT8_C(  37), UINT8_C(  70),
                             UINT8_C( 100), UINT8_C( 121), UINT8_C(  18), UINT8_C(  28),
                             UINT8_C( 139), UINT8_C( 107), UINT8_C(   3), UINT8_C( 194),
                             UINT8_C(  42), UINT8_C(  72), UINT8_C(  91), UINT8_C(  86),
                             UINT8_C( 184), UINT8_C(   9), UINT8_C( 176), UINT8_C( 118),
                             UINT8_C( 122), UINT8_C( 148), UINT8_C( 186), UINT8_C( 193),
                             UINT8_C(  56), UINT8_C(  71), UINT8_C( 242), UINT8_C(  49),
                             UINT8_C( 183), UINT8_C(  53), UINT8_C( 227), UINT8_C(   3),
                             UINT8_C( 183), UINT8_C(  43), UINT8_C( 234), UINT8_C(  85)),
      simde_x_mm512_set_epu8(UINT8_C( 255), UINT8_C( 255), UINT8_C( 120), UINT8_C( 251),
                             UINT8_C( 243), UINT8_C(  62), UINT8_C(  17), UINT8_C( 255),
                             UINT8_C( 226), UINT8_C( 255), UINT8_C( 182), UINT8_C( 255),
                             UINT8_C(  72), UINT8_C( 182), UINT8_C( 216), UINT8_C( 120),
                             UINT8_C(  95), UINT8_C( 108), UINT8_C( 160), UINT8_C(  64),
                             UINT8_C(  76), UINT8_C( 255), UINT8_C( 235), UINT8_C( 182),
                             UINT8_C( 105), UINT8_C(  52), UINT8_C( 255), UINT8_C( 152),
                             UINT8_C(  57), UINT8_C( 255), UINT8_C( 218), UINT8_C( 180),
                             UINT8_C( 149), UINT8_C( 255), UINT8_C(  55), UINT8_C(  46),
                             UINT8_C( 185), UINT8_C(  77), UINT8_C( 237), UINT8_C( 232),
                             UINT8_C( 217), UINT8_C( 125), UINT8_C( 255), UINT8_C( 218),
                             UINT8_C( 255), UINT8_C( 245), UINT8_C( 255), UINT8_C( 211),
                             UINT8_C( 203), UINT8_C( 193), UINT8_C(   9), UINT8_C(  64),
                             UINT8_C( 195), UINT8_C( 231), UINT8_C( 141), UINT8_C( 100),
                             UINT8_C(  65), UINT8_C(   8), UINT8_C( 255), UINT8_C( 248),
                             UINT8_C( 241), UINT8_C( 125), UINT8_C( 255), UINT8_C( 255)) },
    { simde_x_mm512_set_epu8(UINT8_C( 209), UINT8_C(  84), UINT8_C( 130), UINT8_C( 192),
                             UINT8_C(  14), UINT8_C(  11), UINT8_C(  37), UINT8_C( 233),
                             UINT8_C(  67), UINT8_C( 124), UINT8_C(  58), UINT8_C( 162),
                             UINT8_C(  30), UINT8_C( 223), UINT8_C(  70), UINT8_C( 232),
                             UINT8_C(  38), UINT8_C( 159), UINT8_C( 200), UINT8_C( 196),
                             UINT8_C( 197), UINT8_C(  65), UINT8_C( 182), UINT8_C(  45),
                             UINT8_C( 245), UINT8_C(  55), UINT8_C( 174), UINT8_C(  12),
                             UINT8_C( 106), UINT8_C(  22), UINT8_C( 132), UINT8_C( 252),
                             UINT8_C(   2), UINT8_C( 175), UINT8_C(  14), UINT8_C(  90),
                             UINT8_C( 156), UINT8_C( 134), UINT8_C( 221), UINT8_C(  81),
                             UINT8_C( 242), UINT8_C( 214), UINT8_C( 125), UINT8_C( 131),
                             UINT8_C( 199), UINT8_C(  90), UINT8_C( 247), UINT8_C(  63),
                             UINT8_C(  53), UINT8_C(  77), UINT8_C(  63), UINT8_C( 172),
                             UINT8_C(  27), UINT8_C(  22), UINT8_C(   3), UINT8_C( 219),
                             UINT8_C(  65), UINT8_C( 118), UINT8_C( 130), UINT8_C(  97),
                             UINT8_C( 109), UINT8_C(   7), UINT8_C( 142), UINT8_C( 181)),
      UINT64_C( 8707623543556880126),
      simde_x_mm512_set_epu8(UINT8_C(  84), UINT8_C( 185), UINT8_C(   8), UINT8_C(  12),
                             UINT8_C( 245), UINT8_C( 180), UINT8_C(  62), UINT8_C(  93),
                             UINT8_C( 181), UINT8_C( 179), UINT8_C( 172), UINT8_C( 148),
                             UINT8_C( 221), UINT8_C(  14), UINT8_C( 196), UINT8_C(  18),
                             UINT8_C(  23), UINT8_C( 196), UINT8_C( 193), UINT8_C( 142),
                             UINT8_C( 201), UINT8_C(  75), UINT8_C( 157), UINT8_C( 201),
                             UINT8_C(  58), UINT8_C(  76), UINT8_C( 154), UINT8_C( 138),
                             UINT8_C( 226), UINT8_C(  39), UINT8_C( 119), UINT8_C(  85),
                             UINT8_C( 248), UINT8_C( 184), UINT8_C( 196), UINT8_C( 162),
                             UINT8_C( 144), UINT8_C( 119), UINT8_C( 124), UINT8_C(  76),
                             UINT8_C( 214), UINT8_C( 132), UINT8_C(  54), UINT8_C(  74),
                             UINT8_C( 164), UINT8_C(  99), UINT8_C(  79), UINT8_C( 253),
                             UINT8_C(  61), UINT8_C( 167), UINT8_C(  84), UINT8_C( 162),
                             UINT8_C(  31), UINT8_C( 140), UINT8_C( 189), UINT8_C( 154),
                             UINT8_C( 184), UINT8_C( 165), UINT8_C( 151), UINT8_C( 148),
                             UINT8_C( 212), UINT8_C(  74), UINT8_C( 228), UINT8_C( 124)),
      simde_x_mm512_set_epu8(UINT8_C(   1), UINT8_C(  75), UINT8_C(  21), UINT8_C( 220),
                             UINT8_C( 130), UINT8_C( 122), UINT8_C(  71), UINT8_C(  76),
                             UINT8_C(  28), UINT8_C( 200), UINT8_C(  32), UINT8_C( 101),
                             UINT8_C( 149), UINT8_C( 145), UINT8_C( 168), UINT8_C( 237),
                             UINT8_C( 179), UINT8_C(  19), UINT8_C( 235), UINT8_C( 145),
                             UINT8_C( 188), UINT8_C(  82), UINT8_C( 138), UINT8_C( 180),
                             UINT8_C(  47), UINT8_C( 127), UINT8_C(  62), UINT8_C( 240),
                             UINT8_C(  10), UINT8_C( 242), UINT8_C( 156), UINT8_C(  86),
                             UINT8_C(  29), UINT8_C( 107), UINT8_C(  56), UINT8_C(  21),
                             UINT8_C(  24), UINT8_C(  68), UINT8_C( 160), UINT8_C(  64),
                             UINT8_C(  48), UINT8_C(  13), UINT8_C( 173), UINT8_C(   4),
                             UINT8_C( 253), UINT8_C( 192), UINT8_C(  17), UINT8_C( 141),
                             UINT8_C(  21), UINT8_C( 108), UINT8_C( 125), UINT8_C( 196),
                             UINT8_C( 184), UINT8_C(  74), UINT8_C( 251), UINT8_C( 198),
                             UINT8_C( 215), UINT8_C(  22), UINT8_C( 141), UINT8_C( 102),
                             UINT8_C(  59), UINT8_C( 176), UINT8_C( 241), UINT8_C( 193)),
      simde_x_mm512_set_epu8(UINT8_C( 209), UINT8_C( 255), UINT8_C(  29), UINT8_C( 232),
                             UINT8_C( 255), UINT8_C(  11), UINT8_C(  37), UINT8_C( 233),
                             UINT8_C( 209), UINT8_C( 255), UINT8_C(  58), UINT8_C( 249),
                             UINT8_C(  30), UINT8_C( 159), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 202), UINT8_C( 159), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 197), UINT8_C(  65), UINT8_C( 182), UINT8_C( 255),
                             UINT8_C( 245), UINT8_C( 203), UINT8_C( 216), UINT8_C( 255),
                             UINT8_C( 236), UINT8_C(  22), UINT8_C( 255), UINT8_C( 252),
                             UINT8_C(   2), UINT8_C( 255), UINT8_C(  14), UINT8_C( 183),
                             UINT8_C( 156), UINT8_C( 187), UINT8_C( 255), UINT8_C( 140),
                             UINT8_C( 255), UINT8_C( 214), UINT8_C( 125), UINT8_C( 131),
                             UINT8_C( 255), UINT8_C(  90), UINT8_C( 247), UINT8_C( 255),
                             UINT8_C(  53), UINT8_C(  77), UINT8_C( 209), UINT8_C( 255),
                             UINT8_C(  27), UINT8_C( 214), UINT8_C( 255), UINT8_C( 219),
                             UINT8_C( 255), UINT8_C( 187), UINT8_C( 255), UINT8_C( 250),
                             UINT8_C( 255), UINT8_C( 250), UINT8_C( 255), UINT8_C( 181)) },
    { simde_x_mm512_set_epu8(UINT8_C( 192), UINT8_C( 203), UINT8_C( 214), UINT8_C( 126),
                             UINT8_C(  67), UINT8_C(  50), UINT8_C( 238), UINT8_C(  76),
                             UINT8_C( 237), UINT8_C( 123), UINT8_C( 169), UINT8_C( 106),
                             UINT8_C( 182), UINT8_C(  44), UINT8_C( 117), UINT8_C( 103),
                             UINT8_C(  81), UINT8_C( 122), UINT8_C(  56), UINT8_C( 246),
                             UINT8_C(  67), UINT8_C(  79), UINT8_C(  83), UINT8_C( 218),
                             UINT8_C( 243), UINT8_C(  43), UINT8_C(  27), UINT8_C( 159),
                             UINT8_C( 102), UINT8_C( 126), UINT8_C(  38), UINT8_C( 194),
                             UINT8_C( 232), UINT8_C( 117), UINT8_C( 218), UINT8_C( 163),
                             UINT8_C( 198), UINT8_C( 132), UINT8_C( 181), UINT8_C(  10),
                             UINT8_C(  18), UINT8_C( 182), UINT8_C(  14), UINT8_C(  36),
                             UINT8_C( 249), UINT8_C( 113), UINT8_C(  40), UINT8_C(  48),
                             UINT8_C( 149), UINT8_C( 222), UINT8_C( 181), UINT8_C(  85),
                             UINT8_C( 221), UINT8_C( 140), UINT8_C(  65), UINT8_C( 235),
                             UINT8_C(  15), UINT8_C(   3), UINT8_C(  45), UINT8_C(  21),
                             UINT8_C(  72), UINT8_C(  93), UINT8_C( 108), UINT8_C( 125)),
      UINT64_C(12576710173448868104),
      simde_x_mm512_set_epu8(UINT8_C(  90), UINT8_C( 218), UINT8_C( 158), UINT8_C( 186),
                             UINT8_C( 148), UINT8_C(  20), UINT8_C(  43), UINT8_C( 128),
                             UINT8_C(  77), UINT8_C( 108), UINT8_C(  53), UINT8_C(  82),
                             UINT8_C( 206), UINT8_C(  52), UINT8_C(  56), UINT8_C(  58),
                             UINT8_C( 136), UINT8_C( 213), UINT8_C( 114), UINT8_C(  93),
                             UINT8_C( 212), UINT8_C( 241), UINT8_C(  38), UINT8_C( 239),
                             UINT8_C( 146), UINT8_C( 133), UINT8_C( 217), UINT8_C( 114),
                             UINT8_C(  51), UINT8_C( 141), UINT8_C( 182), UINT8_C(  43),
                             UINT8_C(  41), UINT8_C( 220), UINT8_C(  19), UINT8_C(  69),
                             UINT8_C(  60), UINT8_C( 203), UINT8_C( 112), UINT8_C( 108),
                             UINT8_C(   8), UINT8_C(  46), UINT8_C( 221), UINT8_C(  26),
                             UINT8_C(  11), UINT8_C(  42), UINT8_C(  47), UINT8_C(  59),
                             UINT8_C( 199), UINT8_C(  94), UINT8_C( 131), UINT8_C( 132),
                             UINT8_C(  36), UINT8_C(  57), UINT8_C(  68), UINT8_C( 204),
                             UINT8_C(  39), UINT8_C(  50), UINT8_C( 208), UINT8_C(  94),
                             UINT8_C(  53), UINT8_C(  11), UINT8_C(  29), UINT8_C(  65)),
      simde_x_mm512_set_epu8(UINT8_C( 149), UINT8_C( 120), UINT8_C( 198), UINT8_C( 107),
                             UINT8_C( 224), UINT8_C( 224), UINT8_C(  88), UINT8_C( 213),
                             UINT8_C(  31), UINT8_C( 224), UINT8_C( 192), UINT8_C(  27),
                             UINT8_C(  82), UINT8_C( 166), UINT8_C( 202), UINT8_C( 172),
                             UINT8_C( 226), UINT8_C(  63), UINT8_C(  24), UINT8_C(  81),
                             UINT8_C( 248), UINT8_C(   9), UINT8_C( 221), UINT8_C( 155),
                             UINT8_C(  83), UINT8_C( 107), UINT8_C( 209), UINT8_C( 200),
                             UINT8_C(  57), UINT8_C( 168), UINT8_C( 141), UINT8_C( 236),
                             UINT8_C(  58), UINT8_C(  75), UINT8_C(  56), UINT8_C(  93),
                             UINT8_C(  49), UINT8_C(  43), UINT8_C( 108), UINT8_C( 118),
                             UINT8_C( 177), UINT8_C( 112), UINT8_C(  44), UINT8_C( 144),
                             UINT8_C( 204), UINT8_C(  10), UINT8_C(  28), UINT8_C( 170),
                             UINT8_C(  65), UINT8_C(  62), UINT8_C(  86), UINT8_C( 149),
                             UINT8_C(  24), UINT8_C( 201), UINT8_C(  54), UINT8_C( 146),
                             UINT8_C( 223), UINT8_C( 110), UINT8_C( 140), UINT8_C( 217),
                             UINT8_C(  39), UINT8_C( 144), UINT8_C(  64), UINT8_C(  43)),
      simde_x_mm512_set_epu8(UINT8_C( 239), UINT8_C( 203), UINT8_C( 255), UINT8_C( 126),
                             UINT8_C( 255), UINT8_C( 244), UINT8_C( 131), UINT8_C(  76),
                             UINT8_C( 108), UINT8_C( 123), UINT8_C( 169), UINT8_C( 106),
                             UINT8_C( 255), UINT8_C(  44), UINT8_C( 117), UINT8_C( 230),
                             UINT8_C(  81), UINT8_C( 255), UINT8_C( 138), UINT8_C( 174),
                             UINT8_C(  67), UINT8_C(  79), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 243), UINT8_C( 240), UINT8_C(  27), UINT8_C( 159),
                             UINT8_C( 102), UINT8_C( 255), UINT8_C(  38), UINT8_C( 194),
                             UINT8_C(  99), UINT8_C( 117), UINT8_C(  75), UINT8_C( 162),
                             UINT8_C( 109), UINT8_C( 132), UINT8_C( 220), UINT8_C( 226),
                             UINT8_C(  18), UINT8_C( 158), UINT8_C( 255), UINT8_C( 170),
                             UINT8_C( 249), UINT8_C(  52), UINT8_C(  75), UINT8_C(  48),
                             UINT8_C( 149), UINT8_C( 222), UINT8_C( 181), UINT8_C( 255),
                             UINT8_C( 221), UINT8_C( 140), UINT8_C(  65), UINT8_C( 255),
                             UINT8_C(  15), UINT8_C(   3), UINT8_C(  45), UINT8_C(  21),
                             UINT8_C(  92), UINT8_C(  93), UINT8_C( 108), UINT8_C( 125)) },
    { simde_x_mm512_set_epu8(UINT8_C( 153), UINT8_C( 132), UINT8_C(  13), UINT8_C( 176),
                             UINT8_C( 224), UINT8_C( 123), UINT8_C( 115), UINT8_C( 207),
                             UINT8_C( 113), UINT8_C( 205), UINT8_C( 119), UINT8_C( 189),
                             UINT8_C( 235), UINT8_C(  19), UINT8_C( 244), UINT8_C(  29),
                             UINT8_C(  91), UINT8_C( 192), UINT8_C(  88), UINT8_C( 151),
                             UINT8_C(  24), UINT8_C(  31), UINT8_C(  77), UINT8_C(  92),
                             UINT8_C( 171), UINT8_C(  48), UINT8_C( 143), UINT8_C( 142),
                             UINT8_C(  16), UINT8_C(  18), UINT8_C( 212), UINT8_C( 190),
                             UINT8_C(  25), UINT8_C( 151), UINT8_C( 156), UINT8_C( 165),
                             UINT8_C(  26), UINT8_C( 133), UINT8_C( 223), UINT8_C( 219),
                             UINT8_C(  63), UINT8_C(  13), UINT8_C( 132), UINT8_C(  41),
                             UINT8_C( 152), UINT8_C( 197), UINT8_C( 254), UINT8_C( 202),
                             UINT8_C(  73), UINT8_C(   3), UINT8_C( 238), UINT8_C(  10),
                             UINT8_C(  19), UINT8_C(  58), UINT8_C( 244), UINT8_C( 181),
                             UINT8_C( 168), UINT8_C(  59), UINT8_C( 104), UINT8_C( 216),
                             UINT8_C( 174), UINT8_C(  42), UINT8_C( 183), UINT8_C( 162)),
      UINT64_C( 2633789449456316803),
      simde_x_mm512_set_epu8(UINT8_C( 118), UINT8_C(  44), UINT8_C( 242), UINT8_C(  84),
                             UINT8_C( 187), UINT8_C( 208), UINT8_C(  40), UINT8_C(  86),
                             UINT8_C( 255), UINT8_C( 121), UINT8_C( 216), UINT8_C(  44),
                             UINT8_C( 255), UINT8_C(  38), UINT8_C( 212), UINT8_C(  38),
                             UINT8_C(  93), UINT8_C( 149), UINT8_C( 252), UINT8_C( 195),
                             UINT8_C( 204), UINT8_C(  22), UINT8_C( 202), UINT8_C( 131),
                             UINT8_C( 227), UINT8_C(  25), UINT8_C( 179), UINT8_C( 188),
                             UINT8_C(   9), UINT8_C( 221), UINT8_C( 245), UINT8_C(  53),
                             UINT8_C(  28), UINT8_C(  61), UINT8_C( 221), UINT8_C( 150),
                             UINT8_C( 210), UINT8_C( 121), UINT8_C( 154), UINT8_C( 121),
                             UINT8_C( 202), UINT8_C( 196), UINT8_C(   7), UINT8_C(   2),
                             UINT8_C( 137), UINT8_C( 111), UINT8_C( 236), UINT8_C( 198),
                             UINT8_C( 158), UINT8_C(  83), UINT8_C(  32), UINT8_C( 207),
                             UINT8_C( 245), UINT8_C(  48), UINT8_C(  92), UINT8_C(  47),
                             UINT8_C( 218), UINT8_C(  11), UINT8_C(  26), UINT8_C(  90),
                             UINT8_C( 251), UINT8_C(  73), UINT8_C(  45), UINT8_C( 254)),
      simde_x_mm512_set_epu8(UINT8_C( 225), UINT8_C( 126), UINT8_C( 138), UINT8_C( 199),
                             UINT8_C( 130), UINT8_C(  29), UINT8_C( 101), UINT8_C( 241),
                             UINT8_C( 195), UINT8_C( 118), UINT8_C( 102), UINT8_C(  12),
                             UINT8_C( 197), UINT8_C( 215), UINT8_C( 196), UINT8_C(  46),
                             UINT8_C(  63), UINT8_C(  78), UINT8_C(  87), UINT8_C(  18),
                             UINT8_C(  18), UINT8_C(  32), UINT8_C( 230), UINT8_C( 250),
                             UINT8_C( 236), UINT8_C( 133), UINT8_C(  99), UINT8_C(  65),
                             UINT8_C(  13), UINT8_C(  25), UINT8_C( 108), UINT8_C( 135),
                             UINT8_C(  42), UINT8_C(   0), UINT8_C( 104), UINT8_C( 239),
                             UINT8_C( 111), UINT8_C( 146), UINT8_C( 217), UINT8_C( 125),
                             UINT8_C( 206), UINT8_C( 209), UINT8_C(  30), UINT8_C(  92),
                             UINT8_C(  19), UINT8_C( 248), UINT8_C( 136), UINT8_C( 127),
                             UINT8_C( 207), UINT8_C(  23), UINT8_C(  16), UINT8_C( 192),
                             UINT8_C( 177), UINT8_C( 116), UINT8_C( 251), UINT8_C( 206),
                             UINT8_C(  32), UINT8_C(  22), UINT8_C( 214), UINT8_C( 253),
                             UINT8_C(  30), UINT8_C(  64), UINT8_C(  96), UINT8_C( 190)),
      simde_x_mm512_set_epu8(UINT8_C( 153), UINT8_C( 132), UINT8_C( 255), UINT8_C( 176),
                             UINT8_C( 224), UINT8_C( 237), UINT8_C( 115), UINT8_C( 207),
                             UINT8_C( 255), UINT8_C( 205), UINT8_C( 119), UINT8_C( 189),
                             UINT8_C( 255), UINT8_C( 253), UINT8_C( 244), UINT8_C(  84),
                             UINT8_C(  91), UINT8_C( 192), UINT8_C(  88), UINT8_C( 213),
                             UINT8_C( 222), UINT8_C(  31), UINT8_C(  77), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(  48), UINT8_C( 143), UINT8_C( 142),
                             UINT8_C(  22), UINT8_C( 246), UINT8_C( 212), UINT8_C( 190),
                             UINT8_C(  70), UINT8_C(  61), UINT8_C( 156), UINT8_C( 165),
                             UINT8_C(  26), UINT8_C( 255), UINT8_C( 223), UINT8_C( 219),
                             UINT8_C(  63), UINT8_C(  13), UINT8_C( 132), UINT8_C(  41),
                             UINT8_C( 152), UINT8_C( 197), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C( 106), UINT8_C(  48), UINT8_C( 255),
                             UINT8_C(  19), UINT8_C(  58), UINT8_C( 244), UINT8_C( 253),
                             UINT8_C( 250), UINT8_C(  59), UINT8_C( 104), UINT8_C( 216),
                             UINT8_C( 174), UINT8_C(  42), UINT8_C( 141), UINT8_C( 255)) },
    { simde_x_mm512_set_epu8(UINT8_C( 117), UINT8_C(  69), UINT8_C( 121), UINT8_C( 211),
                             UINT8_C(  30), UINT8_C( 183), UINT8_C(   2), UINT8_C( 216),
                             UINT8_C(  95), UINT8_C( 253), UINT8_C(  16), UINT8_C( 178),
                             UINT8_C( 128), UINT8_C( 215), UINT8_C( 190), UINT8_C(  66),
                             UINT8_C(  93), UINT8_C( 224), UINT8_C( 206), UINT8_C( 254),
                             UINT8_C( 195), UINT8_C(  29), UINT8_C( 168), UINT8_C( 138),
                             UINT8_C( 229), UINT8_C(  42), UINT8_C(  78), UINT8_C( 210),
                             UINT8_C( 177), UINT8_C(  38), UINT8_C( 181), UINT8_C(  14),
                             UINT8_C( 138), UINT8_C( 142), UINT8_C( 136), UINT8_C(  42),
                             UINT8_C(   4), UINT8_C(  79), UINT8_C( 172), UINT8_C( 146),
                             UINT8_C(   0), UINT8_C(  85), UINT8_C( 236), UINT8_C(  61),
                             UINT8_C(  40), UINT8_C( 181), UINT8_C(  69), UINT8_C( 157),
                             UINT8_C(  73), UINT8_C( 222), UINT8_C( 242), UINT8_C(  72),
                             UINT8_C( 101), UINT8_C(  96), UINT8_C( 142), UINT8_C( 132),
                             UINT8_C( 169), UINT8_C(  43), UINT8_C( 138), UINT8_C( 114),
                             UINT8_C( 168), UINT8_C(   8), UINT8_C( 204), UINT8_C(  75)),
      UINT64_C(14890918166471265655),
      simde_x_mm512_set_epu8(UINT8_C( 200), UINT8_C( 186), UINT8_C(  66), UINT8_C( 145),
                             UINT8_C( 187), UINT8_C(  15), UINT8_C( 174), UINT8_C( 244),
                             UINT8_C( 167), UINT8_C(  37), UINT8_C(  80), UINT8_C( 120),
                             UINT8_C( 173), UINT8_C( 136), UINT8_C(  95), UINT8_C(  21),
                             UINT8_C(  91), UINT8_C( 159), UINT8_C( 184), UINT8_C(  42),
                             UINT8_C(  22), UINT8_C( 186), UINT8_C(  71), UINT8_C( 178),
                             UINT8_C( 251), UINT8_C(  52), UINT8_C( 234), UINT8_C( 222),
                             UINT8_C(  16), UINT8_C(  92), UINT8_C(  91), UINT8_C( 184),
                             UINT8_C(   3), UINT8_C( 225), UINT8_C( 161), UINT8_C( 200),
                             UINT8_C( 206), UINT8_C(  68), UINT8_C( 232), UINT8_C( 206),
                             UINT8_C(  94), UINT8_C(  67), UINT8_C( 108), UINT8_C( 138),
                             UINT8_C( 191), UINT8_C(  31), UINT8_C(  70), UINT8_C( 108),
                             UINT8_C( 190), UINT8_C( 107), UINT8_C( 171), UINT8_C(  38),
                             UINT8_C(   2), UINT8_C(  32), UINT8_C(  56), UINT8_C(  66),
                             UINT8_C( 254), UINT8_C( 182), UINT8_C( 112), UINT8_C(  10),
                             UINT8_C(  64), UINT8_C( 100), UINT8_C( 201), UINT8_C(  83)),
      simde_x_mm512_set_epu8(UINT8_C( 187), UINT8_C( 211), UINT8_C(  23), UINT8_C(  70),
                             UINT8_C(  22), UINT8_C( 113), UINT8_C(  75), UINT8_C( 139),
                             UINT8_C(  69), UINT8_C( 182), UINT8_C(  40), UINT8_C( 217),
                             UINT8_C( 205), UINT8_C( 165), UINT8_C(  25), UINT8_C( 252),
                             UINT8_C( 244), UINT8_C( 243), UINT8_C( 179), UINT8_C(  75),
                             UINT8_C( 140), UINT8_C( 220), UINT8_C(  38), UINT8_C( 204),
                             UINT8_C(  13), UINT8_C( 204), UINT8_C( 156), UINT8_C( 210),
                             UINT8_C(  13), UINT8_C(  83), UINT8_C( 162), UINT8_C( 102),
                             UINT8_C( 116), UINT8_C( 196), UINT8_C(  44), UINT8_C( 249),
                             UINT8_C( 104), UINT8_C( 206), UINT8_C( 226), UINT8_C(  17),
                             UINT8_C(  13), UINT8_C( 149), UINT8_C( 111), UINT8_C( 243),
                             UINT8_C(  17), UINT8_C( 211), UINT8_C( 225), UINT8_C(  76),
                             UINT8_C( 154), UINT8_C( 185), UINT8_C( 116), UINT8_C(  72),
                             UINT8_C( 245), UINT8_C( 159), UINT8_C(  93), UINT8_C(  48),
                             UINT8_C( 181), UINT8_C( 166), UINT8_C( 246), UINT8_C( 178),
                             UINT8_C( 248), UINT8_C( 246), UINT8_C(  48), UINT8_C( 245)),
      simde_x_mm512_set_epu8(UINT8_C( 255), UINT8_C( 255), UINT8_C( 121), UINT8_C( 211),
                             UINT8_C( 209), UINT8_C( 128), UINT8_C( 249), UINT8_C( 216),
                             UINT8_C( 236), UINT8_C( 253), UINT8_C( 120), UINT8_C( 178),
                             UINT8_C( 128), UINT8_C( 255), UINT8_C( 120), UINT8_C( 255),
                             UINT8_C(  93), UINT8_C( 224), UINT8_C( 255), UINT8_C( 254),
                             UINT8_C( 162), UINT8_C(  29), UINT8_C( 109), UINT8_C( 255),
                             UINT8_C( 229), UINT8_C(  42), UINT8_C( 255), UINT8_C( 210),
                             UINT8_C(  29), UINT8_C( 175), UINT8_C( 181), UINT8_C( 255),
                             UINT8_C( 138), UINT8_C( 255), UINT8_C( 136), UINT8_C(  42),
                             UINT8_C( 255), UINT8_C(  79), UINT8_C( 255), UINT8_C( 223),
                             UINT8_C( 107), UINT8_C( 216), UINT8_C( 219), UINT8_C(  61),
                             UINT8_C( 208), UINT8_C( 242), UINT8_C( 255), UINT8_C( 157),
                             UINT8_C(  73), UINT8_C( 255), UINT8_C( 242), UINT8_C(  72),
                             UINT8_C( 247), UINT8_C( 191), UINT8_C( 142), UINT8_C( 114),
                             UINT8_C( 169), UINT8_C( 255), UINT8_C( 255), UINT8_C( 188),
                             UINT8_C( 168), UINT8_C( 255), UINT8_C( 249), UINT8_C( 255)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_adds_epu8(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_adds_epu8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask64 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT64_C(17286015531074160252),
      simde_x_mm512_set_epu8(UINT8_C( 141), UINT8_C( 229), UINT8_C(  62), UINT8_C( 171),
                             UINT8_C(  49), UINT8_C( 141), UINT8_C(  38), UINT8_C(   4),
                             UINT8_C(  92), UINT8_C( 116), UINT8_C( 178), UINT8_C( 237),
                             UINT8_C( 183), UINT8_C(  22), UINT8_C( 190), UINT8_C( 227),
                             UINT8_C(  55), UINT8_C(  78), UINT8_C( 211), UINT8_C( 137),
                             UINT8_C(  30), UINT8_C( 227), UINT8_C( 233), UINT8_C(  89),
                             UINT8_C(  15), UINT8_C( 119), UINT8_C( 168), UINT8_C( 109),
                             UINT8_C( 240), UINT8_C( 246), UINT8_C(   0), UINT8_C( 238),
                             UINT8_C(   1), UINT8_C( 116), UINT8_C(  62), UINT8_C( 183),
                             UINT8_C( 134), UINT8_C(   0), UINT8_C( 111), UINT8_C(  95),
                             UINT8_C( 236), UINT8_C(   3), UINT8_C( 122), UINT8_C( 135),
                             UINT8_C(  13), UINT8_C(  25), UINT8_C( 221), UINT8_C( 129),
                             UINT8_C(  82), UINT8_C(  90), UINT8_C(  53), UINT8_C( 123),
                             UINT8_C(  73), UINT8_C( 108), UINT8_C( 238), UINT8_C(  15),
                             UINT8_C(  81), UINT8_C( 229), UINT8_C( 102), UINT8_C( 118),
                             UINT8_C(  39), UINT8_C( 179), UINT8_C(  45), UINT8_C(  81)),
      simde_x_mm512_set_epu8(UINT8_C(  68), UINT8_C( 148), UINT8_C( 135), UINT8_C( 141),
                             UINT8_C( 197), UINT8_C( 178), UINT8_C( 145), UINT8_C(  56),
                             UINT8_C(   9), UINT8_C( 135), UINT8_C( 120), UINT8_C( 220),
                             UINT8_C(  36), UINT8_C( 153), UINT8_C( 152), UINT8_C(  25),
                             UINT8_C( 145), UINT8_C( 193), UINT8_C( 221), UINT8_C( 136),
                             UINT8_C(  28), UINT8_C( 212), UINT8_C( 230), UINT8_C( 170),
                             UINT8_C( 243), UINT8_C(  53), UINT8_C( 168), UINT8_C( 149),
                             UINT8_C(  68), UINT8_C(  42), UINT8_C( 138), UINT8_C( 111),
                             UINT8_C(  54), UINT8_C( 198), UINT8_C( 243), UINT8_C(  27),
                             UINT8_C(  23), UINT8_C(  41), UINT8_C( 137), UINT8_C(  44),
                             UINT8_C(   7), UINT8_C( 136), UINT8_C(  32), UINT8_C( 213),
                             UINT8_C( 114), UINT8_C( 184), UINT8_C(  73), UINT8_C( 160),
                             UINT8_C(  96), UINT8_C( 110), UINT8_C( 175), UINT8_C( 180),
                             UINT8_C( 103), UINT8_C( 156), UINT8_C( 234), UINT8_C(  18),
                             UINT8_C( 115), UINT8_C(  54), UINT8_C( 216), UINT8_C( 125),
                             UINT8_C( 110), UINT8_C(  31), UINT8_C(  51), UINT8_C( 152)),
      simde_x_mm512_set_epu8(UINT8_C( 209), UINT8_C( 255), UINT8_C( 197), UINT8_C(   0),
                             UINT8_C( 246), UINT8_C( 255), UINT8_C( 183), UINT8_C(  60),
                             UINT8_C( 101), UINT8_C( 251), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 175), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C( 138), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 157), UINT8_C(  41), UINT8_C( 248), UINT8_C( 139),
                             UINT8_C( 243), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 127), UINT8_C( 209), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 178), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 255), UINT8_C( 243),
                             UINT8_C( 149), UINT8_C( 210), UINT8_C(   0), UINT8_C(   0)) },
    { UINT64_C( 9031211210081693899),
      simde_x_mm512_set_epu8(UINT8_C( 150), UINT8_C( 238), UINT8_C(  65), UINT8_C( 152),
                             UINT8_C(  81), UINT8_C(  38), UINT8_C( 172), UINT8_C( 254),
                             UINT8_C( 242), UINT8_C(  85), UINT8_C( 176), UINT8_C(  80),
                             UINT8_C(  48), UINT8_C(  93), UINT8_C(  79), UINT8_C( 127),
                             UINT8_C(  16), UINT8_C(  41), UINT8_C(  54), UINT8_C( 140),
                             UINT8_C(  17), UINT8_C(  42), UINT8_C( 170), UINT8_C(  38),
                             UINT8_C( 138), UINT8_C( 200), UINT8_C(  60), UINT8_C(  19),
                             UINT8_C( 156), UINT8_C( 149), UINT8_C( 105), UINT8_C( 180),
                             UINT8_C(  10), UINT8_C( 247), UINT8_C( 244), UINT8_C( 200),
                             UINT8_C( 185), UINT8_C(  96), UINT8_C(   6), UINT8_C(  24),
                             UINT8_C(  69), UINT8_C(  73), UINT8_C( 229), UINT8_C(   3),
                             UINT8_C( 161), UINT8_C(  50), UINT8_C( 189), UINT8_C( 217),
                             UINT8_C(  97), UINT8_C(  62), UINT8_C( 101), UINT8_C(  56),
                             UINT8_C( 177), UINT8_C(  83), UINT8_C( 157), UINT8_C( 200),
                             UINT8_C( 239), UINT8_C( 200), UINT8_C( 248), UINT8_C( 240),
                             UINT8_C(  84), UINT8_C( 161), UINT8_C( 183), UINT8_C(  54)),
      simde_x_mm512_set_epu8(UINT8_C(  63), UINT8_C( 205), UINT8_C(  87), UINT8_C(  32),
                             UINT8_C( 191), UINT8_C(  55), UINT8_C(  14), UINT8_C(  81),
                             UINT8_C( 133), UINT8_C( 156), UINT8_C( 217), UINT8_C( 212),
                             UINT8_C(  22), UINT8_C( 112), UINT8_C(  16), UINT8_C(  15),
                             UINT8_C( 152), UINT8_C( 154), UINT8_C( 102), UINT8_C( 161),
                             UINT8_C(  69), UINT8_C(  13), UINT8_C( 235), UINT8_C(  45),
                             UINT8_C(  62), UINT8_C( 136), UINT8_C( 145), UINT8_C(  32),
                             UINT8_C( 149), UINT8_C( 226), UINT8_C(  99), UINT8_C( 192),
                             UINT8_C(   8), UINT8_C( 214), UINT8_C(  81), UINT8_C( 222),
                             UINT8_C( 210), UINT8_C(  26), UINT8_C(  31), UINT8_C( 254),
                             UINT8_C(  68), UINT8_C( 249), UINT8_C( 185), UINT8_C(  46),
                             UINT8_C( 235), UINT8_C( 183), UINT8_C(  21), UINT8_C(  83),
                             UINT8_C( 148), UINT8_C( 159), UINT8_C( 187), UINT8_C(  73),
                             UINT8_C(  57), UINT8_C( 219), UINT8_C(  21), UINT8_C(  82),
                             UINT8_C( 137), UINT8_C( 130), UINT8_C( 126), UINT8_C(  91),
                             UINT8_C( 115), UINT8_C(  31), UINT8_C( 177), UINT8_C(  28)),
      simde_x_mm512_set_epu8(UINT8_C(   0), UINT8_C( 255), UINT8_C( 152), UINT8_C( 184),
                             UINT8_C( 255), UINT8_C(  93), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C( 241), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C( 205), UINT8_C(   0), UINT8_C( 142),
                             UINT8_C(   0), UINT8_C( 195), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(  86), UINT8_C(  55), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C( 200), UINT8_C(   0), UINT8_C( 205), UINT8_C(  51),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 204), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 122), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 255), UINT8_C(  49),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 245), UINT8_C( 221), UINT8_C( 255), UINT8_C( 129),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 199), UINT8_C(   0), UINT8_C( 255), UINT8_C(  82)) },
    { UINT64_C( 6828010367541473016),
      simde_x_mm512_set_epu8(UINT8_C(  84), UINT8_C( 160), UINT8_C(  87), UINT8_C( 234),
                             UINT8_C( 251), UINT8_C( 253), UINT8_C( 129), UINT8_C(  41),
                             UINT8_C(  74), UINT8_C(  72), UINT8_C( 252), UINT8_C(  28),
                             UINT8_C( 141), UINT8_C(  93), UINT8_C( 102), UINT8_C(  44),
                             UINT8_C( 153), UINT8_C( 227), UINT8_C( 206), UINT8_C(  48),
                             UINT8_C( 160), UINT8_C( 206), UINT8_C(  46), UINT8_C( 191),
                             UINT8_C(   4), UINT8_C(  43), UINT8_C( 181), UINT8_C(  97),
                             UINT8_C( 230), UINT8_C( 153), UINT8_C(  71), UINT8_C( 149),
                             UINT8_C(  91), UINT8_C(  45), UINT8_C( 245), UINT8_C(  47),
                             UINT8_C(  29), UINT8_C(  25), UINT8_C(  26), UINT8_C( 247),
                             UINT8_C(  10), UINT8_C(  36), UINT8_C( 177), UINT8_C( 203),
                             UINT8_C(  41), UINT8_C(   1), UINT8_C( 233), UINT8_C( 193),
                             UINT8_C( 129), UINT8_C(  68), UINT8_C(  48), UINT8_C(  36),
                             UINT8_C(  89), UINT8_C( 144), UINT8_C( 225), UINT8_C( 120),
                             UINT8_C(  35), UINT8_C(  62), UINT8_C( 235), UINT8_C( 142),
                             UINT8_C( 152), UINT8_C(  57), UINT8_C(  42), UINT8_C( 145)),
      simde_x_mm512_set_epu8(UINT8_C(  96), UINT8_C( 192), UINT8_C( 233), UINT8_C( 210),
                             UINT8_C( 227), UINT8_C( 185), UINT8_C(  71), UINT8_C(  90),
                             UINT8_C(  44), UINT8_C( 164), UINT8_C( 225), UINT8_C(  26),
                             UINT8_C( 157), UINT8_C( 203), UINT8_C( 117), UINT8_C(  18),
                             UINT8_C( 193), UINT8_C(  45), UINT8_C(  12), UINT8_C(  24),
                             UINT8_C( 148), UINT8_C(  18), UINT8_C( 196), UINT8_C(  28),
                             UINT8_C(  50), UINT8_C( 245), UINT8_C( 188), UINT8_C( 225),
                             UINT8_C( 105), UINT8_C( 150), UINT8_C(  98), UINT8_C(  51),
                             UINT8_C(  58), UINT8_C( 103), UINT8_C( 111), UINT8_C( 129),
                             UINT8_C(  68), UINT8_C( 200), UINT8_C( 124), UINT8_C( 137),
                             UINT8_C(  74), UINT8_C( 194), UINT8_C( 140), UINT8_C(  37),
                             UINT8_C( 244), UINT8_C( 114), UINT8_C(   0), UINT8_C(  61),
                             UINT8_C( 103), UINT8_C( 252), UINT8_C( 151), UINT8_C( 188),
                             UINT8_C(  39), UINT8_C( 156), UINT8_C( 163), UINT8_C(  11),
                             UINT8_C( 176), UINT8_C( 237), UINT8_C( 234), UINT8_C( 217),
                             UINT8_C( 127), UINT8_C( 218), UINT8_C( 131), UINT8_C( 145)),
      simde_x_mm512_set_epu8(UINT8_C(   0), UINT8_C( 255), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 200), UINT8_C(   0),
                             UINT8_C( 118), UINT8_C( 236), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C(  62),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 218), UINT8_C(  72),
                             UINT8_C(   0), UINT8_C( 224), UINT8_C( 242), UINT8_C( 219),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 149), UINT8_C(   0), UINT8_C( 255), UINT8_C( 176),
                             UINT8_C(  97), UINT8_C( 225), UINT8_C( 150), UINT8_C( 255),
                             UINT8_C(  84), UINT8_C( 230), UINT8_C( 255), UINT8_C( 240),
                             UINT8_C( 255), UINT8_C( 115), UINT8_C( 233), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 199), UINT8_C(   0),
                             UINT8_C( 128), UINT8_C( 255), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C( 211), UINT8_C( 255), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0)) },
    { UINT64_C(18325019528117086600),
      simde_x_mm512_set_epu8(UINT8_C( 122), UINT8_C( 202), UINT8_C(  50), UINT8_C( 123),
                             UINT8_C( 225), UINT8_C( 182), UINT8_C( 192), UINT8_C(  54),
                             UINT8_C( 175), UINT8_C(  60), UINT8_C(  31), UINT8_C( 233),
                             UINT8_C( 108), UINT8_C( 137), UINT8_C( 164), UINT8_C( 176),
                             UINT8_C( 226), UINT8_C( 219), UINT8_C(  51), UINT8_C( 220),
                             UINT8_C(   8), UINT8_C(  52), UINT8_C(  97), UINT8_C( 123),
                             UINT8_C( 207), UINT8_C( 132), UINT8_C(  95), UINT8_C( 173),
                             UINT8_C(  70), UINT8_C( 206), UINT8_C( 195), UINT8_C(  25),
                             UINT8_C( 159), UINT8_C(  28), UINT8_C( 198), UINT8_C(  11),
                             UINT8_C( 242), UINT8_C( 126), UINT8_C(  81), UINT8_C(  45),
                             UINT8_C( 233), UINT8_C( 120), UINT8_C( 173), UINT8_C( 240),
                             UINT8_C(   7), UINT8_C(  51), UINT8_C( 199), UINT8_C( 206),
                             UINT8_C( 235), UINT8_C(  98), UINT8_C(  88), UINT8_C(   0),
                             UINT8_C( 190), UINT8_C(   3), UINT8_C( 124), UINT8_C( 143),
                             UINT8_C(  50), UINT8_C(  88), UINT8_C( 171), UINT8_C( 163),
                             UINT8_C( 212), UINT8_C( 243), UINT8_C( 162), UINT8_C(  17)),
      simde_x_mm512_set_epu8(UINT8_C( 243), UINT8_C( 244), UINT8_C( 132), UINT8_C(  96),
                             UINT8_C( 243), UINT8_C( 158), UINT8_C(  99), UINT8_C( 243),
                             UINT8_C(  56), UINT8_C( 211), UINT8_C( 148), UINT8_C( 237),
                             UINT8_C( 132), UINT8_C( 229), UINT8_C(  22), UINT8_C( 126),
                             UINT8_C( 150), UINT8_C( 188), UINT8_C( 196), UINT8_C(   8),
                             UINT8_C(  60), UINT8_C(  93), UINT8_C( 223), UINT8_C( 229),
                             UINT8_C( 249), UINT8_C(  27), UINT8_C( 134), UINT8_C( 218),
                             UINT8_C(  23), UINT8_C(   6), UINT8_C(  45), UINT8_C( 235),
                             UINT8_C( 233), UINT8_C( 155), UINT8_C( 116), UINT8_C( 127),
                             UINT8_C(  96), UINT8_C(  40), UINT8_C( 159), UINT8_C(  40),
                             UINT8_C(  86), UINT8_C( 212), UINT8_C(  70), UINT8_C( 185),
                             UINT8_C(  62), UINT8_C( 235), UINT8_C(  66), UINT8_C(  68),
                             UINT8_C( 169), UINT8_C( 195), UINT8_C(  48), UINT8_C( 186),
                             UINT8_C(  18), UINT8_C( 178), UINT8_C( 158), UINT8_C( 117),
                             UINT8_C(  74), UINT8_C(  32), UINT8_C(  93), UINT8_C( 125),
                             UINT8_C( 209), UINT8_C( 196), UINT8_C( 170), UINT8_C( 117)),
      simde_x_mm512_set_epu8(UINT8_C( 255), UINT8_C( 255), UINT8_C( 182), UINT8_C( 219),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 240), UINT8_C( 255), UINT8_C( 186), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(  68), UINT8_C( 145), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 229), UINT8_C(   0),
                             UINT8_C(  93), UINT8_C(   0), UINT8_C( 240), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C( 255), UINT8_C( 138),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 240), UINT8_C(  85),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 136), UINT8_C(   0),
                             UINT8_C( 208), UINT8_C(   0), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 124), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0)) },
    { UINT64_C( 7677112093376593822),
      simde_x_mm512_set_epu8(UINT8_C( 203), UINT8_C(  34), UINT8_C(   9), UINT8_C(  64),
                             UINT8_C( 195), UINT8_C( 231), UINT8_C( 141), UINT8_C( 100),
                             UINT8_C(  65), UINT8_C(   8), UINT8_C(  69), UINT8_C( 248),
                             UINT8_C( 241), UINT8_C( 205), UINT8_C(   1), UINT8_C(  90),
                             UINT8_C( 115), UINT8_C(  51), UINT8_C( 165), UINT8_C(  56),
                             UINT8_C(  64), UINT8_C( 217), UINT8_C( 137), UINT8_C( 228),
                             UINT8_C( 202), UINT8_C(  28), UINT8_C(  54), UINT8_C( 248),
                             UINT8_C( 202), UINT8_C( 128), UINT8_C( 228), UINT8_C( 185),
                             UINT8_C( 107), UINT8_C( 190), UINT8_C( 142), UINT8_C( 168),
                             UINT8_C(  34), UINT8_C( 173), UINT8_C( 235), UINT8_C( 192),
                             UINT8_C( 121), UINT8_C( 236), UINT8_C( 167), UINT8_C( 162),
                             UINT8_C( 112), UINT8_C( 229), UINT8_C(  81), UINT8_C( 202),
                             UINT8_C( 192), UINT8_C( 142), UINT8_C(  48), UINT8_C( 167),
                             UINT8_C( 195), UINT8_C(  26), UINT8_C(  43), UINT8_C(  29),
                             UINT8_C(   0), UINT8_C(  64), UINT8_C( 123), UINT8_C( 189),
                             UINT8_C(  15), UINT8_C( 120), UINT8_C(  36), UINT8_C(  40)),
      simde_x_mm512_set_epu8(UINT8_C( 113), UINT8_C(  32), UINT8_C(  71), UINT8_C(  55),
                             UINT8_C( 230), UINT8_C(  82), UINT8_C( 175), UINT8_C( 236),
                             UINT8_C( 201), UINT8_C( 112), UINT8_C(  45), UINT8_C(  37),
                             UINT8_C(  67), UINT8_C( 187), UINT8_C(  64), UINT8_C(  39),
                             UINT8_C(  72), UINT8_C(  45), UINT8_C( 120), UINT8_C( 251),
                             UINT8_C( 147), UINT8_C(  62), UINT8_C(  17), UINT8_C(  31),
                             UINT8_C( 226), UINT8_C( 198), UINT8_C(  56), UINT8_C(  21),
                             UINT8_C(  72), UINT8_C( 182), UINT8_C( 216), UINT8_C( 120),
                             UINT8_C(  95), UINT8_C( 108), UINT8_C(  32), UINT8_C(  64),
                             UINT8_C( 128), UINT8_C( 102), UINT8_C( 235), UINT8_C(  28),
                             UINT8_C( 105), UINT8_C(  52), UINT8_C(  85), UINT8_C( 152),
                             UINT8_C(  57), UINT8_C( 225), UINT8_C( 218), UINT8_C( 132),
                             UINT8_C( 149), UINT8_C( 254), UINT8_C(  55), UINT8_C(  46),
                             UINT8_C( 185), UINT8_C(  77), UINT8_C(  18), UINT8_C(  70),
                             UINT8_C(  89), UINT8_C( 125), UINT8_C( 214), UINT8_C( 131),
                             UINT8_C( 121), UINT8_C( 245), UINT8_C( 187), UINT8_C( 197)),
      simde_x_mm512_set_epu8(UINT8_C(   0), UINT8_C(  66), UINT8_C(  80), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(  65), UINT8_C(   0),
                             UINT8_C( 187), UINT8_C(   0), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C( 226), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C( 232),
                             UINT8_C( 162), UINT8_C(   0), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C( 226), UINT8_C(   0), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 103), UINT8_C( 213),
                             UINT8_C( 255), UINT8_C( 103), UINT8_C(  61), UINT8_C(  99),
                             UINT8_C(  89), UINT8_C(   0), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 136), UINT8_C( 255), UINT8_C( 223), UINT8_C(   0)) },
    { UINT64_C( 4840581956081845689),
      simde_x_mm512_set_epu8(UINT8_C( 122), UINT8_C( 148), UINT8_C( 186), UINT8_C( 193),
                             UINT8_C(  56), UINT8_C(  71), UINT8_C( 242), UINT8_C(  49),
                             UINT8_C( 183), UINT8_C(  53), UINT8_C( 227), UINT8_C(   3),
                             UINT8_C( 183), UINT8_C(  43), UINT8_C( 234), UINT8_C(  85),
                             UINT8_C( 230), UINT8_C( 247), UINT8_C(  66), UINT8_C(   1),
                             UINT8_C( 243), UINT8_C(  60), UINT8_C( 137), UINT8_C( 173),
                             UINT8_C( 134), UINT8_C( 192), UINT8_C( 173), UINT8_C( 182),
                             UINT8_C( 119), UINT8_C( 248), UINT8_C(  12), UINT8_C( 113),
                             UINT8_C( 244), UINT8_C( 172), UINT8_C(   6), UINT8_C(  69),
                             UINT8_C(   2), UINT8_C( 181), UINT8_C( 222), UINT8_C( 130),
                             UINT8_C(   3), UINT8_C( 128), UINT8_C( 247), UINT8_C(  24),
                             UINT8_C(  11), UINT8_C( 162), UINT8_C( 224), UINT8_C( 110),
                             UINT8_C(  33), UINT8_C( 232), UINT8_C( 125), UINT8_C(  35),
                             UINT8_C( 153), UINT8_C( 208), UINT8_C( 234), UINT8_C(  38),
                             UINT8_C( 175), UINT8_C(   9), UINT8_C( 245), UINT8_C( 132),
                             UINT8_C(  71), UINT8_C(  31), UINT8_C( 214), UINT8_C(  93)),
      simde_x_mm512_set_epu8(UINT8_C(  53), UINT8_C(  77), UINT8_C(  63), UINT8_C( 172),
                             UINT8_C(  27), UINT8_C(  22), UINT8_C(   3), UINT8_C( 219),
                             UINT8_C(  65), UINT8_C( 118), UINT8_C( 130), UINT8_C(  97),
                             UINT8_C( 109), UINT8_C(   7), UINT8_C( 142), UINT8_C( 181),
                             UINT8_C( 131), UINT8_C( 121), UINT8_C( 128), UINT8_C( 103),
                             UINT8_C(   0), UINT8_C( 101), UINT8_C( 215), UINT8_C(  89),
                             UINT8_C( 173), UINT8_C( 191), UINT8_C(   9), UINT8_C( 249),
                             UINT8_C( 193), UINT8_C(  13), UINT8_C( 105), UINT8_C(  92),
                             UINT8_C( 238), UINT8_C( 235), UINT8_C( 154), UINT8_C( 142),
                             UINT8_C(  74), UINT8_C( 121), UINT8_C( 211), UINT8_C(  52),
                             UINT8_C( 193), UINT8_C( 163), UINT8_C(  98), UINT8_C( 106),
                             UINT8_C( 147), UINT8_C( 209), UINT8_C(  37), UINT8_C(  70),
                             UINT8_C( 100), UINT8_C( 121), UINT8_C(  18), UINT8_C(  28),
                             UINT8_C( 139), UINT8_C( 107), UINT8_C(   3), UINT8_C( 194),
                             UINT8_C(  42), UINT8_C(  72), UINT8_C(  91), UINT8_C(  86),
                             UINT8_C( 184), UINT8_C(   9), UINT8_C( 176), UINT8_C( 118)),
      simde_x_mm512_set_epu8(UINT8_C(   0), UINT8_C( 225), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 245), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C(  50), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 194), UINT8_C( 104),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C( 182), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 160), UINT8_C( 211),
                             UINT8_C(  76), UINT8_C( 255), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C( 158), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 133), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C(   0), UINT8_C( 232),
                             UINT8_C( 217), UINT8_C(   0), UINT8_C( 255), UINT8_C( 218),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C( 211)) },
    { UINT64_C(17498311407133456191),
      simde_x_mm512_set_epu8(UINT8_C(  61), UINT8_C( 167), UINT8_C(  84), UINT8_C( 162),
                             UINT8_C(  31), UINT8_C( 140), UINT8_C( 189), UINT8_C( 154),
                             UINT8_C( 184), UINT8_C( 165), UINT8_C( 151), UINT8_C( 148),
                             UINT8_C( 212), UINT8_C(  74), UINT8_C( 228), UINT8_C( 124),
                             UINT8_C( 120), UINT8_C( 215), UINT8_C( 177), UINT8_C( 122),
                             UINT8_C(  87), UINT8_C( 137), UINT8_C(  54), UINT8_C( 254),
                             UINT8_C( 209), UINT8_C(  84), UINT8_C( 130), UINT8_C( 192),
                             UINT8_C(  14), UINT8_C(  11), UINT8_C(  37), UINT8_C( 233),
                             UINT8_C(  67), UINT8_C( 124), UINT8_C(  58), UINT8_C( 162),
                             UINT8_C(  30), UINT8_C( 223), UINT8_C(  70), UINT8_C( 232),
                             UINT8_C(  38), UINT8_C( 159), UINT8_C( 200), UINT8_C( 196),
                             UINT8_C( 197), UINT8_C(  65), UINT8_C( 182), UINT8_C(  45),
                             UINT8_C( 245), UINT8_C(  55), UINT8_C( 174), UINT8_C(  12),
                             UINT8_C( 106), UINT8_C(  22), UINT8_C( 132), UINT8_C( 252),
                             UINT8_C(   2), UINT8_C( 175), UINT8_C(  14), UINT8_C(  90),
                             UINT8_C( 156), UINT8_C( 134), UINT8_C( 221), UINT8_C(  81)),
      simde_x_mm512_set_epu8(UINT8_C(  21), UINT8_C( 108), UINT8_C( 125), UINT8_C( 196),
                             UINT8_C( 184), UINT8_C(  74), UINT8_C( 251), UINT8_C( 198),
                             UINT8_C( 215), UINT8_C(  22), UINT8_C( 141), UINT8_C( 102),
                             UINT8_C(  59), UINT8_C( 176), UINT8_C( 241), UINT8_C( 193),
                             UINT8_C(  84), UINT8_C( 185), UINT8_C(   8), UINT8_C(  12),
                             UINT8_C( 245), UINT8_C( 180), UINT8_C(  62), UINT8_C(  93),
                             UINT8_C( 181), UINT8_C( 179), UINT8_C( 172), UINT8_C( 148),
                             UINT8_C( 221), UINT8_C(  14), UINT8_C( 196), UINT8_C(  18),
                             UINT8_C(  23), UINT8_C( 196), UINT8_C( 193), UINT8_C( 142),
                             UINT8_C( 201), UINT8_C(  75), UINT8_C( 157), UINT8_C( 201),
                             UINT8_C(  58), UINT8_C(  76), UINT8_C( 154), UINT8_C( 138),
                             UINT8_C( 226), UINT8_C(  39), UINT8_C( 119), UINT8_C(  85),
                             UINT8_C( 248), UINT8_C( 184), UINT8_C( 196), UINT8_C( 162),
                             UINT8_C( 144), UINT8_C( 119), UINT8_C( 124), UINT8_C(  76),
                             UINT8_C( 214), UINT8_C( 132), UINT8_C(  54), UINT8_C(  74),
                             UINT8_C( 164), UINT8_C(  99), UINT8_C(  79), UINT8_C( 253)),
      simde_x_mm512_set_epu8(UINT8_C(  82), UINT8_C( 255), UINT8_C( 209), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C( 187), UINT8_C(   0), UINT8_C( 250),
                             UINT8_C(   0), UINT8_C( 250), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 185), UINT8_C( 134),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C( 233), UINT8_C( 251),
                             UINT8_C(  90), UINT8_C( 255), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 255), UINT8_C( 227), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C( 235), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C( 255), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C( 239), UINT8_C( 255), UINT8_C( 174),
                             UINT8_C(   0), UINT8_C( 141), UINT8_C( 255), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(  68), UINT8_C( 164),
                             UINT8_C( 255), UINT8_C( 233), UINT8_C( 255), UINT8_C( 255)) },
    { UINT64_C( 3462613925466476941),
      simde_x_mm512_set_epu8(UINT8_C(  18), UINT8_C( 182), UINT8_C(  14), UINT8_C(  36),
                             UINT8_C( 249), UINT8_C( 113), UINT8_C(  40), UINT8_C(  48),
                             UINT8_C( 149), UINT8_C( 222), UINT8_C( 181), UINT8_C(  85),
                             UINT8_C( 221), UINT8_C( 140), UINT8_C(  65), UINT8_C( 235),
                             UINT8_C(  15), UINT8_C(   3), UINT8_C(  45), UINT8_C(  21),
                             UINT8_C(  72), UINT8_C(  93), UINT8_C( 108), UINT8_C( 125),
                             UINT8_C(   1), UINT8_C(  75), UINT8_C(  21), UINT8_C( 220),
                             UINT8_C( 130), UINT8_C( 122), UINT8_C(  71), UINT8_C(  76),
                             UINT8_C(  28), UINT8_C( 200), UINT8_C(  32), UINT8_C( 101),
                             UINT8_C( 149), UINT8_C( 145), UINT8_C( 168), UINT8_C( 237),
                             UINT8_C( 179), UINT8_C(  19), UINT8_C( 235), UINT8_C( 145),
                             UINT8_C( 188), UINT8_C(  82), UINT8_C( 138), UINT8_C( 180),
                             UINT8_C(  47), UINT8_C( 127), UINT8_C(  62), UINT8_C( 240),
                             UINT8_C(  10), UINT8_C( 242), UINT8_C( 156), UINT8_C(  86),
                             UINT8_C(  29), UINT8_C( 107), UINT8_C(  56), UINT8_C(  21),
                             UINT8_C(  24), UINT8_C(  68), UINT8_C( 160), UINT8_C(  64)),
      simde_x_mm512_set_epu8(UINT8_C( 199), UINT8_C(  94), UINT8_C( 131), UINT8_C( 132),
                             UINT8_C(  36), UINT8_C(  57), UINT8_C(  68), UINT8_C( 204),
                             UINT8_C(  39), UINT8_C(  50), UINT8_C( 208), UINT8_C(  94),
                             UINT8_C(  53), UINT8_C(  11), UINT8_C(  29), UINT8_C(  65),
                             UINT8_C( 174), UINT8_C( 137), UINT8_C( 115), UINT8_C(  68),
                             UINT8_C( 187), UINT8_C( 118), UINT8_C(  17), UINT8_C(   8),
                             UINT8_C( 192), UINT8_C( 203), UINT8_C( 214), UINT8_C( 126),
                             UINT8_C(  67), UINT8_C(  50), UINT8_C( 238), UINT8_C(  76),
                             UINT8_C( 237), UINT8_C( 123), UINT8_C( 169), UINT8_C( 106),
                             UINT8_C( 182), UINT8_C(  44), UINT8_C( 117), UINT8_C( 103),
                             UINT8_C(  81), UINT8_C( 122), UINT8_C(  56), UINT8_C( 246),
                             UINT8_C(  67), UINT8_C(  79), UINT8_C(  83), UINT8_C( 218),
                             UINT8_C( 243), UINT8_C(  43), UINT8_C(  27), UINT8_C( 159),
                             UINT8_C( 102), UINT8_C( 126), UINT8_C(  38), UINT8_C( 194),
                             UINT8_C( 232), UINT8_C( 117), UINT8_C( 218), UINT8_C( 163),
                             UINT8_C( 198), UINT8_C( 132), UINT8_C( 181), UINT8_C(  10)),
      simde_x_mm512_set_epu8(UINT8_C(   0), UINT8_C(   0), UINT8_C( 145), UINT8_C( 168),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C( 151), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 189), UINT8_C(   0), UINT8_C( 160), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C( 211), UINT8_C(   0), UINT8_C( 133),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C( 172), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 255), UINT8_C( 255), UINT8_C( 201), UINT8_C( 207),
                             UINT8_C( 255), UINT8_C( 189), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C( 141), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C(   0), UINT8_C(   0), UINT8_C(   0), UINT8_C( 255),
                             UINT8_C( 255), UINT8_C(   0), UINT8_C(   0), UINT8_C(   0),
                             UINT8_C( 222), UINT8_C( 200), UINT8_C(   0), UINT8_C(  74)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_adds_epu8(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_adds_epu16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu16(UINT16_C( 57245), UINT16_C( 31803), UINT16_C(  9053), UINT16_C( 21282),
                              UINT16_C( 45515), UINT16_C( 57894), UINT16_C( 50445), UINT16_C( 50583),
                              UINT16_C( 54723), UINT16_C( 52144), UINT16_C( 13347), UINT16_C( 57624),
                              UINT16_C( 20975), UINT16_C( 65317), UINT16_C( 16493), UINT16_C( 47193),
                              UINT16_C(  8805), UINT16_C( 17012), UINT16_C( 15383), UINT16_C(  5484),
                              UINT16_C( 64480), UINT16_C( 10898), UINT16_C( 37488), UINT16_C(  8763),
                              UINT16_C( 45480), UINT16_C(  3307), UINT16_C(  9193), UINT16_C( 26229),
                              UINT16_C( 20008), UINT16_C( 59713), UINT16_C( 53564), UINT16_C( 62675)),
      simde_x_mm512_set_epu16(UINT16_C( 56619), UINT16_C( 61703), UINT16_C(  3086), UINT16_C( 41224),
                              UINT16_C(  8421), UINT16_C( 52766), UINT16_C(  7424), UINT16_C( 48822),
                              UINT16_C( 41975), UINT16_C( 22918), UINT16_C( 59915), UINT16_C( 26727),
                              UINT16_C( 53323), UINT16_C( 34707), UINT16_C( 14572), UINT16_C(   716),
                              UINT16_C( 56087), UINT16_C(  9094), UINT16_C( 20814), UINT16_C( 56962),
                              UINT16_C( 14110), UINT16_C( 52082), UINT16_C(  8468), UINT16_C( 11854),
                              UINT16_C(  1576), UINT16_C( 12690), UINT16_C( 32737), UINT16_C( 51663),
                              UINT16_C( 37115), UINT16_C( 18115), UINT16_C( 50682), UINT16_C(  1653)),
      simde_x_mm512_set_epu16(UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 12139), UINT16_C( 62506),
                              UINT16_C( 53936), UINT16_C( 65535), UINT16_C( 57869), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 31065), UINT16_C( 47909),
                              UINT16_C( 64892), UINT16_C( 26106), UINT16_C( 36197), UINT16_C( 62446),
                              UINT16_C( 65535), UINT16_C( 62980), UINT16_C( 45956), UINT16_C( 20617),
                              UINT16_C( 47056), UINT16_C( 15997), UINT16_C( 41930), UINT16_C( 65535),
                              UINT16_C( 57123), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 64328)) },
    { simde_x_mm512_set_epu16(UINT16_C(  1128), UINT16_C( 14143), UINT16_C( 19763), UINT16_C( 47819),
                              UINT16_C( 62972), UINT16_C( 13428), UINT16_C( 18537), UINT16_C( 38297),
                              UINT16_C( 21721), UINT16_C( 58861), UINT16_C( 42577), UINT16_C( 39379),
                              UINT16_C( 55643), UINT16_C( 20439), UINT16_C( 34514), UINT16_C( 49721),
                              UINT16_C( 18526), UINT16_C( 46886), UINT16_C( 20377), UINT16_C( 31658),
                              UINT16_C( 32537), UINT16_C( 50044), UINT16_C( 28503), UINT16_C(  5895),
                              UINT16_C(  1666), UINT16_C( 50137), UINT16_C( 30976), UINT16_C(  4047),
                              UINT16_C( 24589), UINT16_C( 29183), UINT16_C( 13128), UINT16_C(  3541)),
      simde_x_mm512_set_epu16(UINT16_C( 38521), UINT16_C( 64382), UINT16_C( 35873), UINT16_C( 42497),
                              UINT16_C( 46178), UINT16_C( 38904), UINT16_C( 15657), UINT16_C(  6453),
                              UINT16_C( 15377), UINT16_C( 45418), UINT16_C( 38208), UINT16_C( 44948),
                              UINT16_C( 17985), UINT16_C( 28923), UINT16_C( 11094), UINT16_C( 61254),
                              UINT16_C( 41028), UINT16_C(  4939), UINT16_C( 61438), UINT16_C( 34821),
                              UINT16_C(  9578), UINT16_C( 46329), UINT16_C( 53213), UINT16_C( 42297),
                              UINT16_C( 42846), UINT16_C( 12973), UINT16_C( 32628), UINT16_C( 10992),
                              UINT16_C( 56871), UINT16_C( 37695), UINT16_C( 16317), UINT16_C(  4356)),
      simde_x_mm512_set_epu16(UINT16_C( 39649), UINT16_C( 65535), UINT16_C( 55636), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 52332), UINT16_C( 34194), UINT16_C( 44750),
                              UINT16_C( 37098), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 49362), UINT16_C( 45608), UINT16_C( 65535),
                              UINT16_C( 59554), UINT16_C( 51825), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 42115), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 48192),
                              UINT16_C( 44512), UINT16_C( 63110), UINT16_C( 63604), UINT16_C( 15039),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 29445), UINT16_C(  7897)) },
    { simde_x_mm512_set_epu16(UINT16_C( 14747), UINT16_C( 30570), UINT16_C( 56275), UINT16_C( 58793),
                              UINT16_C( 48636), UINT16_C( 11396), UINT16_C( 12142), UINT16_C( 25883),
                              UINT16_C( 48137), UINT16_C( 31445), UINT16_C( 15524), UINT16_C( 49322),
                              UINT16_C(  6761), UINT16_C( 42681), UINT16_C(  8586), UINT16_C( 53309),
                              UINT16_C( 48898), UINT16_C( 23142), UINT16_C( 17912), UINT16_C( 20578),
                              UINT16_C( 42157), UINT16_C( 25705), UINT16_C( 39154), UINT16_C( 32521),
                              UINT16_C(  7302), UINT16_C( 65263), UINT16_C(  7623), UINT16_C( 47267),
                              UINT16_C( 41612), UINT16_C(  7723), UINT16_C( 51815), UINT16_C( 23273)),
      simde_x_mm512_set_epu16(UINT16_C( 29783), UINT16_C( 35133), UINT16_C( 18722), UINT16_C( 24788),
                              UINT16_C(  4507), UINT16_C( 64908), UINT16_C( 18120), UINT16_C( 53792),
                              UINT16_C( 40447), UINT16_C(  2013), UINT16_C( 19329), UINT16_C( 18474),
                              UINT16_C( 45067), UINT16_C( 37105), UINT16_C( 40003), UINT16_C( 11523),
                              UINT16_C( 28865), UINT16_C(  9988), UINT16_C( 16394), UINT16_C( 60605),
                              UINT16_C( 29530), UINT16_C(   612), UINT16_C( 21547), UINT16_C( 12463),
                              UINT16_C( 46586), UINT16_C(  6619), UINT16_C( 39845), UINT16_C(  4328),
                              UINT16_C( 31818), UINT16_C( 16334), UINT16_C( 57400), UINT16_C( 58095)),
      simde_x_mm512_set_epu16(UINT16_C( 44530), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 53143), UINT16_C( 65535), UINT16_C( 30262), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 33458), UINT16_C( 34853), UINT16_C( 65535),
                              UINT16_C( 51828), UINT16_C( 65535), UINT16_C( 48589), UINT16_C( 64832),
                              UINT16_C( 65535), UINT16_C( 33130), UINT16_C( 34306), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 26317), UINT16_C( 60701), UINT16_C( 44984),
                              UINT16_C( 53888), UINT16_C( 65535), UINT16_C( 47468), UINT16_C( 51595),
                              UINT16_C( 65535), UINT16_C( 24057), UINT16_C( 65535), UINT16_C( 65535)) },
    { simde_x_mm512_set_epu16(UINT16_C( 30682), UINT16_C( 30851), UINT16_C(  8660), UINT16_C( 19219),
                              UINT16_C( 10963), UINT16_C(   474), UINT16_C( 23728), UINT16_C( 46528),
                              UINT16_C( 63120), UINT16_C( 45881), UINT16_C( 63086), UINT16_C( 25197),
                              UINT16_C(  2704), UINT16_C(  4513), UINT16_C( 59294), UINT16_C( 60752),
                              UINT16_C( 50954), UINT16_C( 23266), UINT16_C( 27902), UINT16_C(  8814),
                              UINT16_C(  4863), UINT16_C(  1260), UINT16_C( 11681), UINT16_C( 46962),
                              UINT16_C( 62057), UINT16_C(  1167), UINT16_C( 44133), UINT16_C( 29723),
                              UINT16_C( 20503), UINT16_C( 47951), UINT16_C(  6645), UINT16_C( 27511)),
      simde_x_mm512_set_epu16(UINT16_C( 15286), UINT16_C(  5686), UINT16_C( 62529), UINT16_C( 52414),
                              UINT16_C( 64942), UINT16_C( 55157), UINT16_C(  1921), UINT16_C(   414),
                              UINT16_C( 57075), UINT16_C( 38949), UINT16_C( 49837), UINT16_C( 16574),
                              UINT16_C( 63519), UINT16_C( 21815), UINT16_C( 16534), UINT16_C(  5264),
                              UINT16_C( 40611), UINT16_C(  7582), UINT16_C( 27508), UINT16_C( 40060),
                              UINT16_C( 17332), UINT16_C( 48486), UINT16_C( 46457), UINT16_C(  3074),
                              UINT16_C( 49520), UINT16_C( 35694), UINT16_C( 14820), UINT16_C( 26739),
                              UINT16_C( 52931), UINT16_C( 30426), UINT16_C( 34639), UINT16_C( 60126)),
      simde_x_mm512_set_epu16(UINT16_C( 45968), UINT16_C( 36537), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 55631), UINT16_C( 25649), UINT16_C( 46942),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 41771),
                              UINT16_C( 65535), UINT16_C( 26328), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 30848), UINT16_C( 55410), UINT16_C( 48874),
                              UINT16_C( 22195), UINT16_C( 49746), UINT16_C( 58138), UINT16_C( 50036),
                              UINT16_C( 65535), UINT16_C( 36861), UINT16_C( 58953), UINT16_C( 56462),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 41284), UINT16_C( 65535)) },
    { simde_x_mm512_set_epu16(UINT16_C( 20277), UINT16_C(  2901), UINT16_C(  9119), UINT16_C( 39166),
                              UINT16_C( 42714), UINT16_C( 26596), UINT16_C( 65466), UINT16_C( 41724),
                              UINT16_C( 43509), UINT16_C( 36161), UINT16_C( 14559), UINT16_C( 39814),
                              UINT16_C( 39598), UINT16_C( 35810), UINT16_C( 32670), UINT16_C(  2206),
                              UINT16_C( 28889), UINT16_C( 41636), UINT16_C( 48990), UINT16_C( 37294),
                              UINT16_C( 39242), UINT16_C( 39778), UINT16_C(  6979), UINT16_C( 54232),
                              UINT16_C( 47738), UINT16_C( 57923), UINT16_C( 37900), UINT16_C( 45798),
                              UINT16_C( 44781), UINT16_C( 27960), UINT16_C(  1998), UINT16_C( 61149)),
      simde_x_mm512_set_epu16(UINT16_C( 20730), UINT16_C( 18080), UINT16_C( 51364), UINT16_C( 27968),
                              UINT16_C( 47873), UINT16_C( 51993), UINT16_C( 18829), UINT16_C( 15054),
                              UINT16_C( 50771), UINT16_C( 37023), UINT16_C( 39205), UINT16_C( 18252),
                              UINT16_C( 53613), UINT16_C( 36736), UINT16_C( 53958), UINT16_C( 13137),
                              UINT16_C( 43763), UINT16_C( 19789), UINT16_C(  2934), UINT16_C( 22986),
                              UINT16_C( 46964), UINT16_C( 64986), UINT16_C( 13395), UINT16_C( 61287),
                              UINT16_C( 61677), UINT16_C( 22301), UINT16_C( 41322), UINT16_C( 35063),
                              UINT16_C( 22663), UINT16_C( 28062), UINT16_C( 45351), UINT16_C( 21097)),
      simde_x_mm512_set_epu16(UINT16_C( 41007), UINT16_C( 20981), UINT16_C( 60483), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 56778),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 53764), UINT16_C( 58066),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 15343),
                              UINT16_C( 65535), UINT16_C( 61425), UINT16_C( 51924), UINT16_C( 60280),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 20374), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 56022), UINT16_C( 47349), UINT16_C( 65535)) },
    { simde_x_mm512_set_epu16(UINT16_C( 29192), UINT16_C(  8654), UINT16_C( 22850), UINT16_C( 62764),
                              UINT16_C( 24193), UINT16_C( 41054), UINT16_C( 47201), UINT16_C( 29473),
                              UINT16_C(  7514), UINT16_C( 21604), UINT16_C( 55170), UINT16_C( 17089),
                              UINT16_C( 58910), UINT16_C( 18576), UINT16_C( 50942), UINT16_C( 19375),
                              UINT16_C( 32394), UINT16_C(  6670), UINT16_C( 34291), UINT16_C( 10597),
                              UINT16_C( 63848), UINT16_C( 24622), UINT16_C( 55141), UINT16_C( 33399),
                              UINT16_C( 17676), UINT16_C( 53111), UINT16_C( 57678), UINT16_C( 53869),
                              UINT16_C(  1069), UINT16_C( 35377), UINT16_C( 20392), UINT16_C(  6033)),
      simde_x_mm512_set_epu16(UINT16_C( 30110), UINT16_C(  5765), UINT16_C( 21533), UINT16_C( 32154),
                              UINT16_C( 47788), UINT16_C(  4708), UINT16_C( 49450), UINT16_C( 15569),
                              UINT16_C( 11433), UINT16_C( 48528), UINT16_C( 18868), UINT16_C( 49914),
                              UINT16_C( 29591), UINT16_C( 10234), UINT16_C( 18676), UINT16_C( 18833),
                              UINT16_C( 53298), UINT16_C( 44056), UINT16_C(   163), UINT16_C( 14043),
                              UINT16_C( 52308), UINT16_C( 55462), UINT16_C(  1379), UINT16_C( 56593),
                              UINT16_C( 40649), UINT16_C( 10209), UINT16_C( 62181), UINT16_C(  6865),
                              UINT16_C( 33659), UINT16_C( 12643), UINT16_C( 58853), UINT16_C( 27590)),
      simde_x_mm512_set_epu16(UINT16_C( 59302), UINT16_C( 14419), UINT16_C( 44383), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 45762), UINT16_C( 65535), UINT16_C( 45042),
                              UINT16_C( 18947), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 28810), UINT16_C( 65535), UINT16_C( 38208),
                              UINT16_C( 65535), UINT16_C( 50726), UINT16_C( 34454), UINT16_C( 24640),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 56520), UINT16_C( 65535),
                              UINT16_C( 58325), UINT16_C( 63320), UINT16_C( 65535), UINT16_C( 60734),
                              UINT16_C( 34728), UINT16_C( 48020), UINT16_C( 65535), UINT16_C( 33623)) },
    { simde_x_mm512_set_epu16(UINT16_C( 54408), UINT16_C( 44486), UINT16_C( 24517), UINT16_C( 51207),
                              UINT16_C(  2791), UINT16_C( 45592), UINT16_C( 48586), UINT16_C( 36784),
                              UINT16_C( 53369), UINT16_C( 43811), UINT16_C( 65046), UINT16_C( 44235),
                              UINT16_C( 48681), UINT16_C( 21401), UINT16_C( 55417), UINT16_C(  5581),
                              UINT16_C( 56410), UINT16_C( 42092), UINT16_C( 49678), UINT16_C( 22777),
                              UINT16_C( 15839), UINT16_C( 59069), UINT16_C(  3836), UINT16_C( 55877),
                              UINT16_C( 35316), UINT16_C( 40611), UINT16_C( 30925), UINT16_C( 38925),
                              UINT16_C( 44290), UINT16_C( 48844), UINT16_C( 41388), UINT16_C( 24135)),
      simde_x_mm512_set_epu16(UINT16_C(  7663), UINT16_C(  4305), UINT16_C( 48670), UINT16_C( 39779),
                              UINT16_C(  2052), UINT16_C( 40026), UINT16_C( 47264), UINT16_C(  7297),
                              UINT16_C( 11022), UINT16_C( 64822), UINT16_C( 63292), UINT16_C( 56921),
                              UINT16_C(  8160), UINT16_C( 61766), UINT16_C( 22014), UINT16_C( 41597),
                              UINT16_C( 60919), UINT16_C( 45399), UINT16_C( 45462), UINT16_C( 35135),
                              UINT16_C( 13486), UINT16_C( 42127), UINT16_C(  6466), UINT16_C(  5931),
                              UINT16_C( 30448), UINT16_C( 27493), UINT16_C( 24921), UINT16_C( 23842),
                              UINT16_C( 21170), UINT16_C( 62954), UINT16_C(  7914), UINT16_C( 35527)),
      simde_x_mm512_set_epu16(UINT16_C( 62071), UINT16_C( 48791), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C(  4843), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 44081),
                              UINT16_C( 64391), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 56841), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 47178),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 57912),
                              UINT16_C( 29325), UINT16_C( 65535), UINT16_C( 10302), UINT16_C( 61808),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 55846), UINT16_C( 62767),
                              UINT16_C( 65460), UINT16_C( 65535), UINT16_C( 49302), UINT16_C( 59662)) },
    { simde_x_mm512_set_epu16(UINT16_C( 61865), UINT16_C( 23426), UINT16_C(  1768), UINT16_C( 58636),
                              UINT16_C( 57533), UINT16_C( 19108), UINT16_C( 59802), UINT16_C( 30792),
                              UINT16_C( 61240), UINT16_C(  6209), UINT16_C( 32476), UINT16_C( 14300),
                              UINT16_C(  7420), UINT16_C(  1985), UINT16_C( 25770), UINT16_C( 58800),
                              UINT16_C( 51482), UINT16_C( 20894), UINT16_C(   680), UINT16_C( 29487),
                              UINT16_C( 45202), UINT16_C( 20181), UINT16_C( 14712), UINT16_C( 17612),
                              UINT16_C( 16683), UINT16_C(  1453), UINT16_C( 15990), UINT16_C( 36509),
                              UINT16_C( 51737), UINT16_C( 56324), UINT16_C( 34493), UINT16_C( 17761)),
      simde_x_mm512_set_epu16(UINT16_C( 37550), UINT16_C( 51250), UINT16_C( 42874), UINT16_C(  8384),
                              UINT16_C( 13698), UINT16_C( 33689), UINT16_C( 64372), UINT16_C( 20393),
                              UINT16_C( 17692), UINT16_C( 65383), UINT16_C(   545), UINT16_C( 27220),
                              UINT16_C( 18321), UINT16_C( 26532), UINT16_C( 40835), UINT16_C( 14036),
                              UINT16_C(  3666), UINT16_C(  2040), UINT16_C( 61839), UINT16_C( 38748),
                              UINT16_C( 62699), UINT16_C( 55831), UINT16_C( 23293), UINT16_C( 13208),
                              UINT16_C( 60975), UINT16_C( 12632), UINT16_C( 13392), UINT16_C( 63703),
                              UINT16_C( 41286), UINT16_C(  6117), UINT16_C( 32645), UINT16_C( 56152)),
      simde_x_mm512_set_epu16(UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 44642), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 52797), UINT16_C( 65535), UINT16_C( 51185),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 33021), UINT16_C( 41520),
                              UINT16_C( 25741), UINT16_C( 28517), UINT16_C( 65535), UINT16_C( 65535),
                              UINT16_C( 55148), UINT16_C( 22934), UINT16_C( 62519), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 65535), UINT16_C( 38005), UINT16_C( 30820),
                              UINT16_C( 65535), UINT16_C( 14085), UINT16_C( 29382), UINT16_C( 65535),
                              UINT16_C( 65535), UINT16_C( 62441), UINT16_C( 65535), UINT16_C( 65535)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_adds_epu16(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_adds_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int16_t src[32];
    const simde__mmask32 k;
    const uint16_t a[32];
    const uint16_t b[32];
    const uint16_t r[32];
  } test_vec[] = {
    { { -INT16_C( 25480),  INT16_C( 16389),  INT16_C( 16810),  INT16_C( 20465), -INT16_C( 11763),  INT16_C(  1143),  INT16_C( 11613),  INT16_C(  5547),
        -INT16_C(   217),  INT16_C(  4795),  INT16_C(  5277), -INT16_C( 31526), -INT16_C(  1895),  INT16_C(  7976), -INT16_C(  7252),  INT16_C(  9466),
        -INT16_C(   129),  INT16_C( 10853),  INT16_C( 22080),  INT16_C( 19833), -INT16_C(  4056), -INT16_C( 31151), -INT16_C(   994),  INT16_C( 17819),
         INT16_C( 22267), -INT16_C( 26281),  INT16_C( 12651),  INT16_C(  1053),  INT16_C( 17962), -INT16_C( 10716),  INT16_C(  7721), -INT16_C( 22278) },
      UINT32_C(1590845213),
      { UINT16_C(19382), UINT16_C(57003), UINT16_C(64828), UINT16_C(23140), UINT16_C(65529), UINT16_C(62879), UINT16_C(63062), UINT16_C(49550),
        UINT16_C(43815), UINT16_C(20933), UINT16_C(59889), UINT16_C( 6695), UINT16_C( 8711), UINT16_C( 9667), UINT16_C(38273), UINT16_C(14211),
        UINT16_C(12001), UINT16_C( 7446), UINT16_C(31275), UINT16_C( 9591), UINT16_C( 5754), UINT16_C(53274), UINT16_C(43020), UINT16_C(13201),
        UINT16_C(22099), UINT16_C(17797), UINT16_C(44096), UINT16_C(18271), UINT16_C( 8910), UINT16_C(20588), UINT16_C(61368), UINT16_C(39303) },
      { UINT16_C(40222), UINT16_C(18870), UINT16_C(11544), UINT16_C(37486), UINT16_C(34883), UINT16_C(20322), UINT16_C(62256), UINT16_C(33922),
        UINT16_C( 1865), UINT16_C(35273), UINT16_C(10420), UINT16_C(33489), UINT16_C(15691), UINT16_C(  978), UINT16_C(23085), UINT16_C(19356),
        UINT16_C(21239), UINT16_C( 3988), UINT16_C(  895), UINT16_C(49825), UINT16_C(  907), UINT16_C(48145), UINT16_C(37878), UINT16_C(16448),
        UINT16_C( 2459), UINT16_C(20425), UINT16_C(39473), UINT16_C(31953), UINT16_C(42200), UINT16_C( 1407), UINT16_C( 7166), UINT16_C(62800) },
      { UINT16_C(59604), UINT16_C(16389),      UINT16_MAX, UINT16_C(60626),      UINT16_MAX, UINT16_C( 1143), UINT16_C(11613), UINT16_C( 5547),
        UINT16_C(45680), UINT16_C(56206),      UINT16_MAX, UINT16_C(40184), UINT16_C(24402), UINT16_C( 7976), UINT16_C(61358), UINT16_C( 9466),
        UINT16_C(65407), UINT16_C(11434), UINT16_C(22080), UINT16_C(19833), UINT16_C( 6661), UINT16_C(34385),      UINT16_MAX, UINT16_C(29649),
        UINT16_C(22267), UINT16_C(38222),      UINT16_MAX, UINT16_C(50224), UINT16_C(51110), UINT16_C(54820),      UINT16_MAX, UINT16_C(43258) } },
    { { -INT16_C(  7059), -INT16_C(  5115), -INT16_C( 22809),  INT16_C( 29614), -INT16_C( 16470), -INT16_C( 24529),  INT16_C( 28499), -INT16_C(  4384),
        -INT16_C( 21896), -INT16_C( 22211),  INT16_C(  3652),  INT16_C(  7206), -INT16_C( 23118), -INT16_C( 20447),  INT16_C( 29121),  INT16_C( 11942),
        -INT16_C( 21674),  INT16_C( 15643), -INT16_C( 13999), -INT16_C(  1104), -INT16_C(  8311), -INT16_C(  9060),  INT16_C( 31822), -INT16_C( 14646),
         INT16_C(  1830),  INT16_C( 27504), -INT16_C( 27115), -INT16_C( 14201), -INT16_C( 22213), -INT16_C(   904),  INT16_C(  7706),  INT16_C( 28715) },
      UINT32_C( 464406217),
      { UINT16_C(24079), UINT16_C(38934), UINT16_C(45630), UINT16_C(35956), UINT16_C(15919), UINT16_C(21843), UINT16_C(49989), UINT16_C(23488),
        UINT16_C(18521), UINT16_C(37923), UINT16_C(39921), UINT16_C( 2961), UINT16_C(48314), UINT16_C(33660), UINT16_C(10754), UINT16_C( 4510),
        UINT16_C(46472), UINT16_C(50858), UINT16_C( 7783), UINT16_C(38483), UINT16_C(42589), UINT16_C(41708), UINT16_C(44137), UINT16_C(49917),
        UINT16_C( 8436), UINT16_C(58710), UINT16_C(59324), UINT16_C(30449), UINT16_C(28067), UINT16_C(42489), UINT16_C(39063), UINT16_C( 8119) },
      { UINT16_C(24909), UINT16_C(46310), UINT16_C(14719), UINT16_C(56395), UINT16_C(14303), UINT16_C(18559), UINT16_C(31971), UINT16_C(55306),
        UINT16_C(24733), UINT16_C(22973), UINT16_C(44616), UINT16_C(60367), UINT16_C(51227), UINT16_C(45713), UINT16_C(18528), UINT16_C(44498),
        UINT16_C(47273), UINT16_C(10338), UINT16_C(44529), UINT16_C(53253), UINT16_C(34020), UINT16_C(50968), UINT16_C( 8704), UINT16_C(40351),
        UINT16_C(23938), UINT16_C(51958), UINT16_C(50443), UINT16_C(10166), UINT16_C(18318), UINT16_C(61145), UINT16_C(43919), UINT16_C(14492) },
      { UINT16_C(48988), UINT16_C(60421), UINT16_C(42727),      UINT16_MAX, UINT16_C(49066), UINT16_C(41007),      UINT16_MAX,      UINT16_MAX,
        UINT16_C(43640), UINT16_C(60896),      UINT16_MAX, UINT16_C( 7206), UINT16_C(42418), UINT16_C(45089), UINT16_C(29282), UINT16_C(11942),
        UINT16_C(43862), UINT16_C(61196), UINT16_C(52312),      UINT16_MAX, UINT16_C(57225),      UINT16_MAX, UINT16_C(31822),      UINT16_MAX,
        UINT16_C(32374),      UINT16_MAX, UINT16_C(38421), UINT16_C(40615), UINT16_C(46385), UINT16_C(64632), UINT16_C( 7706), UINT16_C(28715) } },
    { { -INT16_C(   413),  INT16_C( 21600),  INT16_C( 26027), -INT16_C( 28892),  INT16_C( 15593), -INT16_C(  5546), -INT16_C(  2466), -INT16_C(  7801),
         INT16_C( 32339),  INT16_C( 24235),  INT16_C( 24899), -INT16_C( 11899),  INT16_C( 24488),  INT16_C( 14272),  INT16_C( 23562),  INT16_C( 28271),
        -INT16_C( 12198),  INT16_C(  1474), -INT16_C(  6347),  INT16_C(  8084), -INT16_C(  5597), -INT16_C( 32247), -INT16_C( 28448),  INT16_C( 13155),
         INT16_C(  3598),  INT16_C( 21138),  INT16_C(  6000),  INT16_C(  6179), -INT16_C(  7306), -INT16_C( 32432), -INT16_C( 16577), -INT16_C( 26129) },
      UINT32_C(3315511695),
      { UINT16_C(12952), UINT16_C(48356), UINT16_C(60701), UINT16_C(64830), UINT16_C(41341), UINT16_C(35889), UINT16_C(50095), UINT16_C( 8158),
        UINT16_C(  474), UINT16_C(20792), UINT16_C(35045), UINT16_C( 9426), UINT16_C(49479), UINT16_C(55230), UINT16_C(23666), UINT16_C( 2972),
        UINT16_C(32911), UINT16_C(44231), UINT16_C( 1389), UINT16_C(60073), UINT16_C(55974), UINT16_C(21878), UINT16_C(21661), UINT16_C(30837),
        UINT16_C(44374), UINT16_C(15305), UINT16_C(39733), UINT16_C(31839), UINT16_C( 7516), UINT16_C(52819), UINT16_C(61306), UINT16_C( 2521) },
      { UINT16_C(41071), UINT16_C(56501), UINT16_C(24229), UINT16_C(19399), UINT16_C(15673), UINT16_C(54945), UINT16_C( 5778), UINT16_C(59470),
        UINT16_C( 6083), UINT16_C(63523), UINT16_C(33458), UINT16_C( 3700), UINT16_C(51360), UINT16_C( 6877), UINT16_C(46775), UINT16_C(10019),
        UINT16_C(55383), UINT16_C(64515), UINT16_C(51766), UINT16_C(28488), UINT16_C(59656), UINT16_C(39494), UINT16_C(38143), UINT16_C(49794),
        UINT16_C(42412), UINT16_C(24250), UINT16_C(11815), UINT16_C(51053), UINT16_C(19190), UINT16_C(44769), UINT16_C( 1024), UINT16_C(22485) },
      { UINT16_C(54023),      UINT16_MAX,      UINT16_MAX,      UINT16_MAX, UINT16_C(15593), UINT16_C(59990), UINT16_C(63070),      UINT16_MAX,
        UINT16_C( 6557), UINT16_C(24235), UINT16_C(24899), UINT16_C(53637),      UINT16_MAX, UINT16_C(62107), UINT16_C(23562), UINT16_C(12991),
        UINT16_C(53338),      UINT16_MAX, UINT16_C(53155),      UINT16_MAX,      UINT16_MAX, UINT16_C(33289), UINT16_C(37088),      UINT16_MAX,
             UINT16_MAX, UINT16_C(21138), UINT16_C(51548), UINT16_C( 6179), UINT16_C(58230), UINT16_C(33104), UINT16_C(62330), UINT16_C(25006) } },
    { { -INT16_C( 10020),  INT16_C(  4948), -INT16_C( 25437), -INT16_C( 21630), -INT16_C( 14203), -INT16_C( 31675), -INT16_C( 14499),  INT16_C(  2374),
         INT16_C(   108), -INT16_C( 27801), -INT16_C( 11218),  INT16_C(  9563),  INT16_C( 15390),  INT16_C(  8147), -INT16_C( 22463),  INT16_C(  7542),
        -INT16_C( 13696),  INT16_C(  9008), -INT16_C( 19610), -INT16_C(  5170),  INT16_C(  4987), -INT16_C( 10129), -INT16_C( 18982),  INT16_C( 18145),
         INT16_C( 18869), -INT16_C(  6950),  INT16_C( 13597),  INT16_C( 15369), -INT16_C(  9103), -INT16_C( 19877), -INT16_C( 11900),  INT16_C(  1232) },
      UINT32_C(  36176028),
      { UINT16_C(63155), UINT16_C(12270), UINT16_C(23818), UINT16_C(58375), UINT16_C(59667), UINT16_C(51243), UINT16_C( 1330), UINT16_C(20396),
        UINT16_C(46394), UINT16_C(43915), UINT16_C(59025), UINT16_C( 5470), UINT16_C(11960), UINT16_C(21530), UINT16_C(16942), UINT16_C(57942),
        UINT16_C(17464), UINT16_C(16913), UINT16_C( 6306), UINT16_C(46375), UINT16_C(20993), UINT16_C(13181), UINT16_C(10839), UINT16_C(37251),
        UINT16_C( 3807), UINT16_C(28988), UINT16_C(39669), UINT16_C(44422), UINT16_C(41160), UINT16_C(63233), UINT16_C(22498), UINT16_C( 7129) },
      { UINT16_C(60060), UINT16_C(15965), UINT16_C(33794), UINT16_C( 1267), UINT16_C(28886), UINT16_C(11575), UINT16_C(47770), UINT16_C(31422),
        UINT16_C(64457), UINT16_C(48875), UINT16_C(29077), UINT16_C(24171), UINT16_C(27666), UINT16_C(62549), UINT16_C(11971), UINT16_C(24335),
        UINT16_C(27928), UINT16_C( 6813), UINT16_C(37105), UINT16_C(51230), UINT16_C(22017), UINT16_C(39925), UINT16_C(46096), UINT16_C(55573),
        UINT16_C(  175), UINT16_C(17559), UINT16_C(  626), UINT16_C(33954), UINT16_C(63342), UINT16_C(12920), UINT16_C(34853), UINT16_C(15761) },
      { UINT16_C(55516), UINT16_C( 4948), UINT16_C(57612), UINT16_C(59642),      UINT16_MAX, UINT16_C(33861), UINT16_C(51037), UINT16_C(51818),
        UINT16_C(  108), UINT16_C(37735), UINT16_C(54318), UINT16_C( 9563), UINT16_C(15390), UINT16_C( 8147), UINT16_C(43073), UINT16_C( 7542),
        UINT16_C(51840), UINT16_C( 9008), UINT16_C(45926),      UINT16_MAX, UINT16_C( 4987), UINT16_C(53106), UINT16_C(46554), UINT16_C(18145),
        UINT16_C(18869), UINT16_C(46547), UINT16_C(13597), UINT16_C(15369), UINT16_C(56433), UINT16_C(45659), UINT16_C(53636), UINT16_C( 1232) } },
    { {  INT16_C( 12277), -INT16_C(  6568),  INT16_C( 30399), -INT16_C( 16210), -INT16_C( 23348), -INT16_C(  8868),  INT16_C( 29016),  INT16_C(  1974),
         INT16_C( 20082), -INT16_C(  7093), -INT16_C(  4528), -INT16_C( 16536), -INT16_C(  7963),  INT16_C(  3057), -INT16_C( 32152),  INT16_C( 23880),
        -INT16_C( 24399),  INT16_C( 28996), -INT16_C(  3561), -INT16_C(  7375), -INT16_C( 29290), -INT16_C(  4416),  INT16_C( 30719),  INT16_C( 29173),
         INT16_C( 16837),  INT16_C(  5461), -INT16_C( 17105),  INT16_C(  5332), -INT16_C( 14947),  INT16_C(  1567),  INT16_C( 26696), -INT16_C(  1693) },
      UINT32_C( 527083272),
      { UINT16_C(40090), UINT16_C(12291), UINT16_C(49961), UINT16_C(10271), UINT16_C( 5178), UINT16_C(65433), UINT16_C(61013), UINT16_C(33813),
        UINT16_C(59819), UINT16_C(18841), UINT16_C(47279), UINT16_C(63311), UINT16_C(45600), UINT16_C(10736), UINT16_C(23386), UINT16_C(62536),
        UINT16_C(19447), UINT16_C( 8228), UINT16_C(17167), UINT16_C(18761), UINT16_C(57944), UINT16_C(44361), UINT16_C(24273), UINT16_C(31794),
        UINT16_C(52039), UINT16_C(63173), UINT16_C( 5251), UINT16_C(42221), UINT16_C(57031), UINT16_C( 8653), UINT16_C( 5433), UINT16_C(12309) },
      { UINT16_C(14689), UINT16_C(28752), UINT16_C(39293), UINT16_C(54713), UINT16_C(  636), UINT16_C(19842), UINT16_C(46176), UINT16_C(43209),
        UINT16_C(36735), UINT16_C(  926), UINT16_C(36003), UINT16_C(27303), UINT16_C(29802), UINT16_C(41867), UINT16_C(41097), UINT16_C(60115),
        UINT16_C( 9178), UINT16_C(22362), UINT16_C( 5309), UINT16_C(14636), UINT16_C(44566), UINT16_C(30598), UINT16_C(20323), UINT16_C(57887),
        UINT16_C(48606), UINT16_C(33509), UINT16_C(35913), UINT16_C(46060), UINT16_C(30720), UINT16_C(35414), UINT16_C(10520), UINT16_C(62068) },
      { UINT16_C(12277), UINT16_C(58968), UINT16_C(30399), UINT16_C(64984), UINT16_C(42188), UINT16_C(56668), UINT16_C(29016), UINT16_C( 1974),
             UINT16_MAX, UINT16_C(19767),      UINT16_MAX, UINT16_C(49000), UINT16_C(57573), UINT16_C(52603), UINT16_C(33384),      UINT16_MAX,
        UINT16_C(41137), UINT16_C(30590), UINT16_C(61975), UINT16_C(33397), UINT16_C(36246),      UINT16_MAX, UINT16_C(44596), UINT16_C(29173),
             UINT16_MAX,      UINT16_MAX, UINT16_C(41164),      UINT16_MAX,      UINT16_MAX, UINT16_C( 1567), UINT16_C(26696), UINT16_C(63843) } },
    { { -INT16_C( 12467),  INT16_C(  2633),  INT16_C( 30179), -INT16_C(  1725), -INT16_C( 14044), -INT16_C( 30864), -INT16_C( 28904), -INT16_C(  2199),
         INT16_C( 20301), -INT16_C( 27015),  INT16_C( 26075), -INT16_C(  9142), -INT16_C( 24355), -INT16_C(  2458), -INT16_C(  9526),  INT16_C(  6120),
         INT16_C( 12969), -INT16_C( 29663),  INT16_C( 25767), -INT16_C( 13434), -INT16_C(  2515),  INT16_C( 17746), -INT16_C( 17274), -INT16_C( 11460),
        -INT16_C( 19189), -INT16_C(  6551), -INT16_C( 19685), -INT16_C(  1854),  INT16_C( 10324),  INT16_C(  7918), -INT16_C( 10493), -INT16_C( 21451) },
      UINT32_C(2956547593),
      { UINT16_C(49082), UINT16_C(59260), UINT16_C(52917), UINT16_C(15148), UINT16_C(27018), UINT16_C(38158), UINT16_C(30750), UINT16_C(14716),
        UINT16_C(15915), UINT16_C(32562), UINT16_C( 8295), UINT16_C(27293), UINT16_C(54007), UINT16_C(   22), UINT16_C(20264), UINT16_C(58033),
        UINT16_C(11534), UINT16_C(50377), UINT16_C(63227), UINT16_C(34559), UINT16_C( 3679), UINT16_C(32027), UINT16_C(38790), UINT16_C(45495),
        UINT16_C(59862), UINT16_C(15665), UINT16_C(52745), UINT16_C(  423), UINT16_C(48545), UINT16_C(51457), UINT16_C(45581), UINT16_C( 7084) },
      { UINT16_C(30175), UINT16_C(56287), UINT16_C(57195), UINT16_C(51809), UINT16_C(31981), UINT16_C(29512), UINT16_C(65300), UINT16_C(59940),
        UINT16_C(21992), UINT16_C(61735), UINT16_C(52772), UINT16_C(50674), UINT16_C(62603), UINT16_C(39054), UINT16_C(15014), UINT16_C(34484),
        UINT16_C(37808), UINT16_C( 7009), UINT16_C(49778), UINT16_C(24550), UINT16_C(11838), UINT16_C(21202), UINT16_C(63277), UINT16_C( 5436),
        UINT16_C(25420), UINT16_C(28678), UINT16_C(63793), UINT16_C(48437), UINT16_C(50413), UINT16_C(37717), UINT16_C( 2558), UINT16_C(44569) },
      {      UINT16_MAX, UINT16_C( 2633), UINT16_C(30179),      UINT16_MAX, UINT16_C(51492), UINT16_C(34672), UINT16_C(36632), UINT16_C(63337),
        UINT16_C(20301),      UINT16_MAX, UINT16_C(61067), UINT16_C(56394),      UINT16_MAX, UINT16_C(63078), UINT16_C(35278), UINT16_C( 6120),
        UINT16_C(49342), UINT16_C(35873), UINT16_C(25767), UINT16_C(59109), UINT16_C(15517), UINT16_C(53229), UINT16_C(48262), UINT16_C(54076),
        UINT16_C(46347), UINT16_C(58985), UINT16_C(45851), UINT16_C(63682),      UINT16_MAX,      UINT16_MAX, UINT16_C(55043), UINT16_C(51653) } },
    { {  INT16_C( 31389),  INT16_C(  4042), -INT16_C( 20420),  INT16_C( 31599),  INT16_C( 16862),  INT16_C(  3021),  INT16_C(  2616), -INT16_C( 31456),
         INT16_C(  9837), -INT16_C( 24587),  INT16_C( 11039),  INT16_C(  3164), -INT16_C( 19985), -INT16_C(  4704), -INT16_C( 17989),  INT16_C( 22684),
         INT16_C( 26164),  INT16_C( 28775), -INT16_C( 10730), -INT16_C(  2837), -INT16_C( 18152),  INT16_C( 20735),  INT16_C(  8131),  INT16_C( 12501),
        -INT16_C( 13499),  INT16_C( 26063),  INT16_C( 11254), -INT16_C(  6799),  INT16_C(  4573), -INT16_C( 26414),  INT16_C( 28363), -INT16_C(    16) },
      UINT32_C(3933165524),
      { UINT16_C(23342), UINT16_C(18142), UINT16_C(56596), UINT16_C(55190), UINT16_C(27900), UINT16_C(16903), UINT16_C(55095), UINT16_C(11687),
        UINT16_C( 6146), UINT16_C(57106), UINT16_C(58410), UINT16_C(62839), UINT16_C(26451), UINT16_C(10228), UINT16_C(25535), UINT16_C(60690),
        UINT16_C(61630), UINT16_C(53811), UINT16_C(51662), UINT16_C(51881), UINT16_C(45365), UINT16_C(27660), UINT16_C(45960), UINT16_C(35481),
        UINT16_C(43980), UINT16_C(63082), UINT16_C(57744), UINT16_C(58347), UINT16_C(57161), UINT16_C( 2058), UINT16_C( 7234), UINT16_C(  501) },
      { UINT16_C(10253), UINT16_C(56275), UINT16_C(32241), UINT16_C(10149), UINT16_C(45614), UINT16_C(46739), UINT16_C(11621), UINT16_C(12608),
        UINT16_C(43736), UINT16_C(26663), UINT16_C( 4748), UINT16_C(54603), UINT16_C(22257), UINT16_C(13533), UINT16_C(53874), UINT16_C(32565),
        UINT16_C( 2298), UINT16_C(60250), UINT16_C(  133), UINT16_C(45842), UINT16_C(42674), UINT16_C( 5993), UINT16_C(43731), UINT16_C(43849),
        UINT16_C(28756), UINT16_C(57364), UINT16_C(24451), UINT16_C(29877), UINT16_C(37557), UINT16_C(10408), UINT16_C(56676), UINT16_C(24231) },
      { UINT16_C(31389), UINT16_C( 4042),      UINT16_MAX, UINT16_C(31599),      UINT16_MAX, UINT16_C( 3021),      UINT16_MAX, UINT16_C(24295),
        UINT16_C(49882),      UINT16_MAX, UINT16_C(63158), UINT16_C( 3164), UINT16_C(48708), UINT16_C(60832),      UINT16_MAX, UINT16_C(22684),
        UINT16_C(63928),      UINT16_MAX, UINT16_C(51795),      UINT16_MAX, UINT16_C(47384), UINT16_C(33653),      UINT16_MAX, UINT16_C(12501),
        UINT16_C(52037),      UINT16_MAX, UINT16_C(11254),      UINT16_MAX, UINT16_C( 4573), UINT16_C(12466), UINT16_C(63910), UINT16_C(24732) } },
    { {  INT16_C(   742),  INT16_C( 27466),  INT16_C( 23554), -INT16_C( 19425), -INT16_C( 30718), -INT16_C( 10805),  INT16_C(  5170), -INT16_C( 30847),
        -INT16_C( 27259),  INT16_C(  2151),  INT16_C(  7668), -INT16_C( 21892),  INT16_C(  9647),  INT16_C(  5330),  INT16_C( 30978), -INT16_C(  6030),
        -INT16_C( 17285),  INT16_C( 32084),  INT16_C( 29465),  INT16_C(  6961), -INT16_C(   517),  INT16_C( 12017),  INT16_C( 29201), -INT16_C( 26955),
         INT16_C(  7175), -INT16_C(  1122),  INT16_C(  6969), -INT16_C(  5723),  INT16_C( 30528),  INT16_C( 17149),  INT16_C( 28657),  INT16_C( 27691) },
      UINT32_C(1172995884),
      { UINT16_C( 7154), UINT16_C(60768), UINT16_C(20760), UINT16_C(10779), UINT16_C(53443), UINT16_C(51904), UINT16_C(24557), UINT16_C( 9926),
        UINT16_C(27514), UINT16_C(47631), UINT16_C( 3299), UINT16_C(54524), UINT16_C(10108), UINT16_C(43072), UINT16_C(10918), UINT16_C(39149),
        UINT16_C(19782), UINT16_C(24198), UINT16_C(41375), UINT16_C(25224), UINT16_C(18802), UINT16_C(24365), UINT16_C(62376), UINT16_C( 8837),
        UINT16_C(38238), UINT16_C(16860), UINT16_C(55457), UINT16_C( 7445), UINT16_C(22016), UINT16_C(42693), UINT16_C(45696), UINT16_C(50751) },
      { UINT16_C(50432), UINT16_C(40741), UINT16_C(44390), UINT16_C(55297), UINT16_C(12022), UINT16_C(40503), UINT16_C(48417), UINT16_C(32960),
        UINT16_C(40018), UINT16_C(62401), UINT16_C(55157), UINT16_C(29969), UINT16_C(54829), UINT16_C(44315), UINT16_C(23177), UINT16_C(35188),
        UINT16_C(39199), UINT16_C(34344), UINT16_C(10566), UINT16_C(15710), UINT16_C(38488), UINT16_C(31195), UINT16_C(40019), UINT16_C(42489),
        UINT16_C(47928), UINT16_C(44440), UINT16_C(43410), UINT16_C(48930), UINT16_C(16000), UINT16_C( 2412), UINT16_C(57496), UINT16_C(47250) },
      { UINT16_C(  742), UINT16_C(27466), UINT16_C(65150),      UINT16_MAX, UINT16_C(34818),      UINT16_MAX, UINT16_C( 5170), UINT16_C(34689),
             UINT16_MAX,      UINT16_MAX, UINT16_C(58456),      UINT16_MAX, UINT16_C(64937),      UINT16_MAX, UINT16_C(34095), UINT16_C(59506),
        UINT16_C(48251), UINT16_C(58542), UINT16_C(29465), UINT16_C(40934), UINT16_C(65019), UINT16_C(55560),      UINT16_MAX, UINT16_C(51326),
             UINT16_MAX, UINT16_C(64414),      UINT16_MAX, UINT16_C(59813), UINT16_C(30528), UINT16_C(17149),      UINT16_MAX, UINT16_C(27691) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi16(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_adds_epu16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_adds_epu16 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask32 k;
    const uint16_t a[32];
    const uint16_t b[32];
    const uint16_t r[32];
  } test_vec[] = {
    { UINT32_C( 298122215),
      { UINT16_C(43877), UINT16_C(21426), UINT16_C(12985), UINT16_C(37214), UINT16_C(44901), UINT16_C(48588), UINT16_C(18525), UINT16_C(41048),
        UINT16_C(22359), UINT16_C(46043), UINT16_C(52738), UINT16_C(43622), UINT16_C(52530), UINT16_C( 6625), UINT16_C(42696), UINT16_C(11818),
        UINT16_C(56401), UINT16_C( 2689), UINT16_C(57359), UINT16_C(29852), UINT16_C(26767), UINT16_C(60465), UINT16_C(35504), UINT16_C( 2188),
        UINT16_C(26593), UINT16_C(58555), UINT16_C( 8501), UINT16_C(26510), UINT16_C(28654), UINT16_C(46976), UINT16_C(43541), UINT16_C(26597) },
      { UINT16_C(26246), UINT16_C(38257), UINT16_C( 3398), UINT16_C(54538), UINT16_C(15221), UINT16_C( 9921), UINT16_C(19909), UINT16_C(42798),
        UINT16_C(59828), UINT16_C(59787), UINT16_C( 6411), UINT16_C(63824), UINT16_C(53384), UINT16_C(40624), UINT16_C(38266), UINT16_C(  261),
        UINT16_C(30460), UINT16_C(17046), UINT16_C(41092), UINT16_C(63768), UINT16_C(55772), UINT16_C(41247), UINT16_C(19751), UINT16_C(56136),
        UINT16_C(54071), UINT16_C(17093), UINT16_C( 5612), UINT16_C(30011), UINT16_C(60646), UINT16_C(24595), UINT16_C( 6273), UINT16_C(32097) },
      {      UINT16_MAX, UINT16_C(59683), UINT16_C(16383), UINT16_C(    0), UINT16_C(    0), UINT16_C(58509), UINT16_C(38434),      UINT16_MAX,
             UINT16_MAX,      UINT16_MAX, UINT16_C(    0),      UINT16_MAX,      UINT16_MAX, UINT16_C(47249),      UINT16_MAX, UINT16_C(12079),
        UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(55255), UINT16_C(58324),
             UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { UINT32_C( 314636430),
      { UINT16_C(55448), UINT16_C(29708), UINT16_C(11185), UINT16_C(55318), UINT16_C(24185), UINT16_C(45236), UINT16_C(31026), UINT16_C( 7922),
        UINT16_C(11662), UINT16_C(29843), UINT16_C(42521), UINT16_C(39893), UINT16_C(14014), UINT16_C(19736), UINT16_C(55342), UINT16_C(51039),
        UINT16_C(27568), UINT16_C(25147), UINT16_C(20887), UINT16_C( 4154), UINT16_C(61104), UINT16_C(58048), UINT16_C(45671), UINT16_C(62976),
        UINT16_C(38111), UINT16_C(63850), UINT16_C(16186), UINT16_C(63892), UINT16_C(44150), UINT16_C(42054), UINT16_C(42373), UINT16_C(13675) },
      { UINT16_C(42769), UINT16_C(43159), UINT16_C(54008), UINT16_C(43192), UINT16_C(30912), UINT16_C(10378), UINT16_C(35626), UINT16_C( 2334),
        UINT16_C(34847), UINT16_C(22786), UINT16_C(38600), UINT16_C(15954), UINT16_C(38979), UINT16_C(51426), UINT16_C(20030), UINT16_C(20477),
        UINT16_C(38389), UINT16_C(60919), UINT16_C(44903), UINT16_C(10134), UINT16_C( 8231), UINT16_C(20815), UINT16_C(28075), UINT16_C(51802),
        UINT16_C(24054), UINT16_C(48676), UINT16_C(30451), UINT16_C(14076), UINT16_C(56847), UINT16_C(19966), UINT16_C(64556), UINT16_C( 8604) },
      { UINT16_C(    0),      UINT16_MAX, UINT16_C(65193),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(10256),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(55847), UINT16_C(52993),      UINT16_MAX,      UINT16_MAX,      UINT16_MAX,
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX,      UINT16_MAX,
        UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0) } },
    { UINT32_C(4161770385),
      { UINT16_C(42306), UINT16_C(26911), UINT16_C(28613), UINT16_C(29114), UINT16_C( 5340), UINT16_C(53819), UINT16_C(24433), UINT16_C(26000),
        UINT16_C(36054), UINT16_C(58779), UINT16_C(39531), UINT16_C(38706), UINT16_C(52886), UINT16_C(10169), UINT16_C(51297), UINT16_C(41759),
        UINT16_C(15981), UINT16_C(12812), UINT16_C(50861), UINT16_C(35491), UINT16_C(57306), UINT16_C(19548), UINT16_C(60734), UINT16_C( 5297),
        UINT16_C(19577), UINT16_C(58617), UINT16_C(11238), UINT16_C(31868), UINT16_C(13817), UINT16_C(23203), UINT16_C(49917), UINT16_C(27389) },
      { UINT16_C( 2305), UINT16_C(44700), UINT16_C(16591), UINT16_C(43576), UINT16_C(38175), UINT16_C(24054), UINT16_C(42882), UINT16_C(64370),
        UINT16_C(27635), UINT16_C(56032), UINT16_C(23703), UINT16_C(36950), UINT16_C(64145), UINT16_C(36587), UINT16_C(59580), UINT16_C(48632),
        UINT16_C(38130), UINT16_C(49516), UINT16_C(42196), UINT16_C(62315), UINT16_C(24889), UINT16_C(47953), UINT16_C(49928), UINT16_C(64695),
        UINT16_C(38702), UINT16_C(50646), UINT16_C(11507), UINT16_C(33878), UINT16_C(16678), UINT16_C(58130), UINT16_C( 2601), UINT16_C( 7072) },
      { UINT16_C(44611), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(43515), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX,
        UINT16_C(63689),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),      UINT16_MAX,
        UINT16_C(54111), UINT16_C(62328),      UINT16_MAX,      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(30495),      UINT16_MAX, UINT16_C(52518), UINT16_C(34461) } },
    { UINT32_C(1943866526),
      { UINT16_C(18609), UINT16_C(60006), UINT16_C(47018), UINT16_C(45734), UINT16_C(23930), UINT16_C(43438), UINT16_C(34036), UINT16_C(59246),
        UINT16_C(50353), UINT16_C(55147), UINT16_C(32005), UINT16_C(12218), UINT16_C(23431), UINT16_C( 9546), UINT16_C(10087), UINT16_C( 6296),
        UINT16_C(65392), UINT16_C( 6659), UINT16_C(43446), UINT16_C(12748), UINT16_C(31494), UINT16_C(64218), UINT16_C(18687), UINT16_C(45281),
        UINT16_C(19469), UINT16_C( 4744), UINT16_C(17097), UINT16_C(20545), UINT16_C(35997), UINT16_C( 1397), UINT16_C( 3763), UINT16_C( 8989) },
      { UINT16_C( 8205), UINT16_C(49981), UINT16_C( 2761), UINT16_C(53236), UINT16_C(52869), UINT16_C(33993), UINT16_C(43543), UINT16_C( 9269),
        UINT16_C(48630), UINT16_C(48950), UINT16_C(30975), UINT16_C(40207), UINT16_C(34052), UINT16_C(47010), UINT16_C(49043), UINT16_C(41179),
        UINT16_C( 6368), UINT16_C(43363), UINT16_C(22562), UINT16_C(42873), UINT16_C(16934), UINT16_C(15660), UINT16_C(25069), UINT16_C(58209),
        UINT16_C(38942), UINT16_C( 7587), UINT16_C(45584), UINT16_C( 5306), UINT16_C(23607), UINT16_C(51915), UINT16_C(42524), UINT16_C(64618) },
      { UINT16_C(    0),      UINT16_MAX, UINT16_C(49779),      UINT16_MAX,      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),      UINT16_MAX,
        UINT16_C(    0), UINT16_C(    0), UINT16_C(62980), UINT16_C(52425), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0),
             UINT16_MAX, UINT16_C(    0),      UINT16_MAX, UINT16_C(55621), UINT16_C(48428), UINT16_C(    0), UINT16_C(43756),      UINT16_MAX,
        UINT16_C(58411), UINT16_C(12331), UINT16_C(    0), UINT16_C(    0), UINT16_C(59604), UINT16_C(53312), UINT16_C(46287), UINT16_C(    0) } },
    { UINT32_C(3785739967),
      { UINT16_C( 7718), UINT16_C(19593), UINT16_C(46433), UINT16_C(20106), UINT16_C(60182), UINT16_C(13361), UINT16_C(54403), UINT16_C(37713),
        UINT16_C( 3207), UINT16_C(48807), UINT16_C(29544), UINT16_C(33929), UINT16_C(62233), UINT16_C(55424), UINT16_C( 9921), UINT16_C(59322),
        UINT16_C(17220), UINT16_C(42292), UINT16_C(48888), UINT16_C( 3827), UINT16_C( 9641), UINT16_C(11586), UINT16_C(37881), UINT16_C(32960),
        UINT16_C(26783), UINT16_C( 2111), UINT16_C(51419), UINT16_C(62604), UINT16_C( 3515), UINT16_C(32205), UINT16_C(34611), UINT16_C(30564) },
      { UINT16_C(39114), UINT16_C(49693), UINT16_C( 4182), UINT16_C(  208), UINT16_C( 4661), UINT16_C(12077), UINT16_C(60837), UINT16_C(17839),
        UINT16_C(61013), UINT16_C(12365), UINT16_C(55734), UINT16_C(29221), UINT16_C(62182), UINT16_C( 6639), UINT16_C(21369), UINT16_C(17297),
        UINT16_C(44780), UINT16_C(16901), UINT16_C(54718), UINT16_C(62530), UINT16_C(28647), UINT16_C(35875), UINT16_C(53853), UINT16_C(45777),
        UINT16_C( 7873), UINT16_C(30691), UINT16_C( 2296), UINT16_C(57065), UINT16_C(55546), UINT16_C(29688), UINT16_C(35116), UINT16_C( 6326) },
      { UINT16_C(46832),      UINT16_MAX, UINT16_C(50615), UINT16_C(20314), UINT16_C(64843), UINT16_C(25438), UINT16_C(    0), UINT16_C(55552),
        UINT16_C(    0), UINT16_C(61172),      UINT16_MAX, UINT16_C(63150), UINT16_C(    0), UINT16_C(    0), UINT16_C(31290),      UINT16_MAX,
        UINT16_C(62000), UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(47461), UINT16_C(    0),      UINT16_MAX,
        UINT16_C(34656), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(61893),      UINT16_MAX, UINT16_C(36890) } },
    { UINT32_C(4116364087),
      { UINT16_C(40336), UINT16_C(30697), UINT16_C( 3084), UINT16_C(26883), UINT16_C(54751), UINT16_C(40988), UINT16_C(65523), UINT16_C(60183),
        UINT16_C(  263), UINT16_C(  458), UINT16_C(49881), UINT16_C( 1396), UINT16_C(10827), UINT16_C(33309), UINT16_C(30949), UINT16_C(30071),
        UINT16_C(24853), UINT16_C( 8684), UINT16_C(61293), UINT16_C(19595), UINT16_C(42948), UINT16_C(47340), UINT16_C( 1190), UINT16_C(44451),
        UINT16_C(27909), UINT16_C(57006), UINT16_C( 8751), UINT16_C(31460), UINT16_C(  332), UINT16_C(12796), UINT16_C(29817), UINT16_C(36518) },
      { UINT16_C(37589), UINT16_C(17072), UINT16_C(15233), UINT16_C(18063), UINT16_C(31714), UINT16_C(35070), UINT16_C(41343), UINT16_C(33845),
        UINT16_C(58127), UINT16_C(15971), UINT16_C(18181), UINT16_C(20921), UINT16_C(46408), UINT16_C(49794), UINT16_C(10281), UINT16_C(65104),
        UINT16_C(  186), UINT16_C(15169), UINT16_C(53307), UINT16_C( 7553), UINT16_C(32587), UINT16_C(52133), UINT16_C(55841), UINT16_C(12367),
        UINT16_C(45757), UINT16_C(49774), UINT16_C(10233), UINT16_C(16915), UINT16_C(38365), UINT16_C( 1540), UINT16_C(21693), UINT16_C(30469) },
      {      UINT16_MAX, UINT16_C(47769), UINT16_C(18317), UINT16_C(    0),      UINT16_MAX,      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),
        UINT16_C(58390), UINT16_C(16429), UINT16_C(    0), UINT16_C(22317), UINT16_C(57235),      UINT16_MAX, UINT16_C(    0),      UINT16_MAX,
        UINT16_C(    0), UINT16_C(23853), UINT16_C(    0), UINT16_C(27148),      UINT16_MAX, UINT16_C(    0), UINT16_C(57031), UINT16_C(    0),
             UINT16_MAX, UINT16_C(    0), UINT16_C(18984), UINT16_C(    0), UINT16_C(38697), UINT16_C(14336), UINT16_C(51510),      UINT16_MAX } },
    { UINT32_C(2427668053),
      { UINT16_C(13334), UINT16_C(25006), UINT16_C(21428), UINT16_C(54572), UINT16_C(31790), UINT16_C(60165), UINT16_C(29486), UINT16_C(10414),
        UINT16_C(49563), UINT16_C(30826), UINT16_C(28247), UINT16_C( 5246), UINT16_C(33730), UINT16_C( 6028), UINT16_C(16329), UINT16_C(57256),
        UINT16_C(22131), UINT16_C(10049), UINT16_C(28073), UINT16_C(55292), UINT16_C(  489), UINT16_C( 6339), UINT16_C(29045), UINT16_C( 4160),
        UINT16_C(43570), UINT16_C(35208), UINT16_C( 1560), UINT16_C(55966), UINT16_C(10890), UINT16_C(21490), UINT16_C(39529), UINT16_C(56371) },
      { UINT16_C(29936), UINT16_C(39172), UINT16_C(  225), UINT16_C(52081), UINT16_C(13314), UINT16_C(30691), UINT16_C( 9125), UINT16_C(55175),
        UINT16_C( 4045), UINT16_C(58721), UINT16_C(65301), UINT16_C(40895), UINT16_C(45353), UINT16_C(37619), UINT16_C( 9803), UINT16_C(15214),
        UINT16_C(29338), UINT16_C(31701), UINT16_C(18035), UINT16_C(30022), UINT16_C(10618), UINT16_C( 8172), UINT16_C(29516), UINT16_C( 6646),
        UINT16_C(22402), UINT16_C(38910), UINT16_C(48726), UINT16_C(32567), UINT16_C(10863), UINT16_C(47889), UINT16_C(32848), UINT16_C(60150) },
      { UINT16_C(43270), UINT16_C(    0), UINT16_C(21653), UINT16_C(    0), UINT16_C(45104), UINT16_C(    0), UINT16_C(38611), UINT16_C(    0),
        UINT16_C(    0),      UINT16_MAX,      UINT16_MAX, UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(26132), UINT16_C(    0),
        UINT16_C(51469), UINT16_C(41750), UINT16_C(    0), UINT16_C(    0), UINT16_C(11107), UINT16_C(14511), UINT16_C(    0), UINT16_C(10806),
        UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(    0), UINT16_C(21753), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX } },
    { UINT32_C(1701170162),
      { UINT16_C(44049), UINT16_C(35802), UINT16_C(50901), UINT16_C( 8874), UINT16_C(41273), UINT16_C(47931), UINT16_C(15096), UINT16_C(20307),
        UINT16_C(35576), UINT16_C(26574), UINT16_C(57524), UINT16_C( 1058), UINT16_C( 6496), UINT16_C(21230), UINT16_C(21476), UINT16_C(63160),
        UINT16_C(37631), UINT16_C(54657), UINT16_C(11353), UINT16_C(37623), UINT16_C(13005), UINT16_C(50510), UINT16_C(41324), UINT16_C(25620),
        UINT16_C(58155), UINT16_C(57292), UINT16_C(61123), UINT16_C( 9187), UINT16_C(53511), UINT16_C(60533), UINT16_C(11556), UINT16_C( 9442) },
      { UINT16_C(25536), UINT16_C( 6649), UINT16_C(61583), UINT16_C(23723), UINT16_C(63778), UINT16_C(36642), UINT16_C(13978), UINT16_C(50675),
        UINT16_C(48921), UINT16_C(56484), UINT16_C(34734), UINT16_C(46591), UINT16_C(30040), UINT16_C(32161), UINT16_C(33698), UINT16_C(25249),
        UINT16_C(39655), UINT16_C(30331), UINT16_C(10122), UINT16_C(44243), UINT16_C(62752), UINT16_C(47931), UINT16_C(12075), UINT16_C(17792),
        UINT16_C( 9710), UINT16_C(39969), UINT16_C( 8620), UINT16_C( 1362), UINT16_C(62358), UINT16_C(14466), UINT16_C( 9079), UINT16_C(24219) },
      { UINT16_C(    0), UINT16_C(42451), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX,      UINT16_MAX, UINT16_C(29074),      UINT16_MAX,
             UINT16_MAX,      UINT16_MAX, UINT16_C(    0), UINT16_C(47649), UINT16_C(    0), UINT16_C(    0), UINT16_C(55174),      UINT16_MAX,
             UINT16_MAX, UINT16_C(    0), UINT16_C(21475), UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(53399), UINT16_C(    0),
             UINT16_MAX, UINT16_C(    0),      UINT16_MAX, UINT16_C(    0), UINT16_C(    0),      UINT16_MAX, UINT16_C(20635), UINT16_C(    0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi16(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi16(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_adds_epu16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_u16x32(r, simde_mm512_loadu_epi16(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_x_mm_adds_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[4];
    const int32_t b[4];
    const int32_t r[4];
  } test_vec[] = {
    { {  INT32_C(   795043823),  INT32_C(  1677977077), -INT32_C(  1932655024), -INT32_C(   289899915) },
      { -INT32_C(  1044210742),  INT32_C(   668338078),  INT32_C(  1187560581),  INT32_C(    40306706) },
      { -INT32_C(   249166919),              INT32_MAX, -INT32_C(   745094443), -INT32_C(   249593209) } },
    { {  INT32_C(  1697761904), -INT32_C(     3525457), -INT32_C(  1333028806), -INT32_C(   593607662) },
      {  INT32_C(  2107466207), -INT32_C(   291146904),  INT32_C(  1110732080), -INT32_C(   431711370) },
      {              INT32_MAX, -INT32_C(   294672361), -INT32_C(   222296726), -INT32_C(  1025319032) } },
    { {  INT32_C(   357266789), -INT32_C(   468445782), -INT32_C(  1080778579), -INT32_C(  1013238812) },
      { -INT32_C(    62900076), -INT32_C(   588585556), -INT32_C(   920773037),  INT32_C(   531588025) },
      {  INT32_C(   294366713), -INT32_C(  1057031338), -INT32_C(  2001551616), -INT32_C(   481650787) } },
    { { -INT32_C(  2110457128), -INT32_C(  1117304816), -INT32_C(   864158744), -INT32_C(  1030809554) },
      { -INT32_C(    37760943),  INT32_C(   148482485), -INT32_C(  2116945976),  INT32_C(   866156634) },
      {              INT32_MIN, -INT32_C(   968822331),              INT32_MIN, -INT32_C(   164652920) } },
    { { -INT32_C(  1951017861),  INT32_C(    88611869),  INT32_C(  1188218136),  INT32_C(   789144030) },
      { -INT32_C(   416495567),  INT32_C(   971965809),  INT32_C(  1471922684), -INT32_C(  1131783359) },
      {              INT32_MIN,  INT32_C(  1060577678),              INT32_MAX, -INT32_C(   342639329) } },
    { {  INT32_C(  1296514864),  INT32_C(  1951567964),  INT32_C(   867837013), -INT32_C(  1218264186) },
      { -INT32_C(    39940469), -INT32_C(  1875472749), -INT32_C(  1863847601),  INT32_C(  2102227277) },
      {  INT32_C(  1256574395),  INT32_C(    76095215), -INT32_C(   996010588),  INT32_C(   883963091) } },
    { {  INT32_C(   214602928),  INT32_C(  2055216164), -INT32_C(   944948415), -INT32_C(  1971449602) },
      {  INT32_C(   847715742), -INT32_C(    87900757), -INT32_C(    58021457), -INT32_C(   897984742) },
      {  INT32_C(  1062318670),  INT32_C(  1967315407), -INT32_C(  1002969872),              INT32_MIN } },
    { { -INT32_C(  1864940692), -INT32_C(  1609935009), -INT32_C(  1855473518),  INT32_C(  1713104584) },
      { -INT32_C(  1365728765),  INT32_C(   245914207),  INT32_C(   487207427),  INT32_C(  1994949386) },
      {              INT32_MIN, -INT32_C(  1364020802), -INT32_C(  1368266091),              INT32_MAX } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi32(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi32(test_vec[i].b);
    simde__m128i r = simde_x_mm_adds_epi32(a, b);
    simde_test_x86_assert_equal_i32x4(r, simde_x_mm_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i a = simde_test_x86_random_i32x4();
    simde__m128i b = simde_test_x86_random_i32x4();
    simde__m128i r = simde_x_mm_adds_epi32(a, b);

    simde_test_x86_write_i32x4(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x4(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x4(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_x_mm256_adds_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[8];
    const int32_t b[8];
    const int32_t r[8];
  } test_vec[] = {
    { {  INT32_C(   637976518), -INT32_C(  1446637290), -INT32_C(  1858457911),  INT32_C(   402085140),  INT32_C(  1455788279), -INT32_C(   295342614), -INT32_C(  1458868321), -INT32_C(  1189088269) },
      { -INT32_C(   908122446),  INT32_C(     7513655), -INT32_C(   393106220), -INT32_C(   117471231),  INT32_C(    55493656), -INT32_C(   789466319),  INT32_C(   377158691), -INT32_C(  1563452944) },
      { -INT32_C(   270145928), -INT32_C(  1439123635),              INT32_MIN,  INT32_C(   284613909),  INT32_C(  1511281935), -INT32_C(  1084808933), -INT32_C(  1081709630),              INT32_MIN } },
    { { -INT32_C(   160714817),  INT32_C(   704044373), -INT32_C(  1978562679),  INT32_C(   679612432),  INT32_C(    86757844), -INT32_C(  1479205756),  INT32_C(   163401497), -INT32_C(  1465152023) },
      { -INT32_C(  1851844804),  INT32_C(  2126157300),  INT32_C(   755551005), -INT32_C(  1353348133), -INT32_C(   525041316), -INT32_C(  1232565859), -INT32_C(  1027652135),  INT32_C(   241855442) },
      { -INT32_C(  2012559621),              INT32_MAX, -INT32_C(  1223011674), -INT32_C(   673735701), -INT32_C(   438283472),              INT32_MIN, -INT32_C(   864250638), -INT32_C(  1223296581) } },
    { {  INT32_C(  1990134146), -INT32_C(  1124836961), -INT32_C(     1442524), -INT32_C(   458342776),  INT32_C(  1573151423), -INT32_C(   988590868),  INT32_C(  1686623122), -INT32_C(  1066143426) },
      { -INT32_C(  1707732229), -INT32_C(  1873400980), -INT32_C(  1332723928),  INT32_C(  1033125501), -INT32_C(  1919264607),  INT32_C(   928165285), -INT32_C(  1097082496), -INT32_C(   981594422) },
      {  INT32_C(   282401917),              INT32_MIN, -INT32_C(  1334166452),  INT32_C(   574782725), -INT32_C(   346113184), -INT32_C(    60425583),  INT32_C(   589540626), -INT32_C(  2047737848) } },
    { { -INT32_C(  1923107551),  INT32_C(   136164832),  INT32_C(  1924705780), -INT32_C(  1917891348),  INT32_C(  1226459556),  INT32_C(  2004905206),  INT32_C(   255138885),  INT32_C(  1272296490) },
      {  INT32_C(  1238905961), -INT32_C(   565053718), -INT32_C(  1890580061), -INT32_C(   115540139),  INT32_C(  1061369416), -INT32_C(   407452766),  INT32_C(   184019935),  INT32_C(   139840671) },
      { -INT32_C(   684201590), -INT32_C(   428888886),  INT32_C(    34125719), -INT32_C(  2033431487),              INT32_MAX,  INT32_C(  1597452440),  INT32_C(   439158820),  INT32_C(  1412137161) } },
    { { -INT32_C(   363778560), -INT32_C(   943087068),  INT32_C(     5708203),  INT32_C(  1643803417),  INT32_C(  1285569962), -INT32_C(   533441024), -INT32_C(   504747198), -INT32_C(   135643145) },
      { -INT32_C(  1847444627), -INT32_C(  1990612002), -INT32_C(   578113340), -INT32_C(   851475421), -INT32_C(  1055203391),  INT32_C(  2023837238),  INT32_C(  1884916601),  INT32_C(   946291659) },
      {              INT32_MIN,              INT32_MIN, -INT32_C(   572405137),  INT32_C(   792327996),  INT32_C(   230366571),  INT32_C(  1490396214),  INT32_C(  1380169403),  INT32_C(   810648514) } },
    { {  INT32_C(  1573538175), -INT32_C(  1176099852), -INT32_C(   157912877), -INT32_C(  1245391372), -INT32_C(   344465739), -INT32_C(  1520232404),  INT32_C(  1863695524),  INT32_C(  2141682688) },
      { -INT32_C(  1159958074),  INT32_C(  1735639700),  INT32_C(   660474419), -INT32_C(  1797446945),  INT32_C(   746542080),  INT32_C(   282190444), -INT32_C(  1619007841),  INT32_C(   673064802) },
      {  INT32_C(   413580101),  INT32_C(   559539848),  INT32_C(   502561542),              INT32_MIN,  INT32_C(   402076341), -INT32_C(  1238041960),  INT32_C(   244687683),              INT32_MAX } },
    { {  INT32_C(   769915544), -INT32_C(   275491140),  INT32_C(  1075311200),  INT32_C(   349500436), -INT32_C(  1270852536), -INT32_C(   708505290),  INT32_C(  1517569271),  INT32_C(    75666027) },
      {  INT32_C(  1227974028),  INT32_C(   473482684), -INT32_C(   866365512), -INT32_C(  1948176061), -INT32_C(  1153425019),  INT32_C(   714147123), -INT32_C(  1249639095),  INT32_C(   616105880) },
      {  INT32_C(  1997889572),  INT32_C(   197991544),  INT32_C(   208945688), -INT32_C(  1598675625),              INT32_MIN,  INT32_C(     5641833),  INT32_C(   267930176),  INT32_C(   691771907) } },
    { {  INT32_C(   678292076),  INT32_C(  1732617903),  INT32_C(   959750645),  INT32_C(  1472468434),  INT32_C(  1762853942),  INT32_C(  1402250249),  INT32_C(  1091049641), -INT32_C(  1939422945) },
      {  INT32_C(  1521800107),  INT32_C(  1875048825),  INT32_C(  1839789723),  INT32_C(  1103457291),  INT32_C(  2058082417),  INT32_C(   634208124),  INT32_C(  2003293527),  INT32_C(  1090768278) },
      {              INT32_MAX,              INT32_MAX,              INT32_MAX,              INT32_MAX,              INT32_MAX,  INT32_C(  2036458373),              INT32_MAX, -INT32_C(   848654667) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m256i a = simde_x_mm256_loadu_epi32(test_vec[i].a);
    simde__m256i b = simde_x_mm256_loadu_epi32(test_vec[i].b);
    simde__m256i r = simde_x_mm256_adds_epi32(a, b);
    simde_test_x86_assert_equal_i32x8(r, simde_x_mm256_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m256i a = simde_test_x86_random_i32x8();
    simde__m256i b = simde_test_x86_random_i32x8();
    simde__m256i r = simde_x_mm256_adds_epi32(a, b);

    simde_test_x86_write_i32x8(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_x_mm512_adds_epi32 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(   446478240),  INT32_C(  1284071089),  INT32_C(  1605972308),  INT32_C(   245399197), -INT32_C(   762754218), -INT32_C(   487041398), -INT32_C(  1034330324), -INT32_C(   872129492),
        -INT32_C(   991518701),  INT32_C(  1376808958),  INT32_C(  1051839136), -INT32_C(  1622388408),  INT32_C(   661771677),  INT32_C(  1477011756), -INT32_C(   199597368), -INT32_C(   759095618) },
      { -INT32_C(  1130977346), -INT32_C(  1223776489), -INT32_C(  1175077007), -INT32_C(  1369947631),  INT32_C(  1138084375), -INT32_C(    56893645),  INT32_C(    15775297), -INT32_C(  1814908460),
         INT32_C(  1884252249), -INT32_C(  2144903665),  INT32_C(   775560477),  INT32_C(  1994166879), -INT32_C(  1866812836), -INT32_C(   762554991), -INT32_C(   523076597), -INT32_C(  2022464466) },
      { -INT32_C(   684499106),  INT32_C(    60294600),  INT32_C(   430895301), -INT32_C(  1124548434),  INT32_C(   375330157), -INT32_C(   543935043), -INT32_C(  1018555027),              INT32_MIN,
         INT32_C(   892733548), -INT32_C(   768094707),  INT32_C(  1827399613),  INT32_C(   371778471), -INT32_C(  1205041159),  INT32_C(   714456765), -INT32_C(   722673965),              INT32_MIN } },
    { {  INT32_C(   485999117),  INT32_C(  1050484256), -INT32_C(  1704143045), -INT32_C(   971945623), -INT32_C(  1940468741),  INT32_C(   744415776), -INT32_C(  1945357986), -INT32_C(   502038571),
         INT32_C(  1660881473),  INT32_C(  1688247337), -INT32_C(   587264909),  INT32_C(  1352798293), -INT32_C(    69404453),  INT32_C(   958872538),  INT32_C(  1103442796), -INT32_C(   198911566) },
      {  INT32_C(   206971875),  INT32_C(   846329535),  INT32_C(  1477341186),  INT32_C(  1537782144), -INT32_C(  2074704471),  INT32_C(   750616256),  INT32_C(  1684898481),  INT32_C(  1062768987),
         INT32_C(  1934339764), -INT32_C(  1499087708), -INT32_C(  1392593876),  INT32_C(   235382629), -INT32_C(   325951956), -INT32_C(  1927786532),  INT32_C(   770803154), -INT32_C(   882095849) },
      {  INT32_C(   692970992),  INT32_C(  1896813791), -INT32_C(   226801859),  INT32_C(   565836521),              INT32_MIN,  INT32_C(  1495032032), -INT32_C(   260459505),  INT32_C(   560730416),
                     INT32_MAX,  INT32_C(   189159629), -INT32_C(  1979858785),  INT32_C(  1588180922), -INT32_C(   395356409), -INT32_C(   968913994),  INT32_C(  1874245950), -INT32_C(  1081007415) } },
    { { -INT32_C(  1690322697), -INT32_C(  1589451660), -INT32_C(    45268840),  INT32_C(   319575527), -INT32_C(  1879073101), -INT32_C(  1071900690), -INT32_C(  1259532643),  INT32_C(  1333746263),
        -INT32_C(  2031436270),  INT32_C(   992423075),  INT32_C(  1413051757),  INT32_C(  2103985610), -INT32_C(   787716125),  INT32_C(   479275135), -INT32_C(  1898938570), -INT32_C(   337817639) },
      { -INT32_C(  1317943538),  INT32_C(  1642895860), -INT32_C(   659217138),  INT32_C(  1314200938),  INT32_C(    69165445), -INT32_C(  1071599223),  INT32_C(   156168240),  INT32_C(  1307847487),
        -INT32_C(   419535374),  INT32_C(   206040062),  INT32_C(  2078604560), -INT32_C(  1614202598),  INT32_C(   614721690), -INT32_C(   907754599), -INT32_C(   204328269),  INT32_C(  1329645149) },
      {              INT32_MIN,  INT32_C(    53444200), -INT32_C(   704485978),  INT32_C(  1633776465), -INT32_C(  1809907656), -INT32_C(  2143499913), -INT32_C(  1103364403),              INT32_MAX,
                     INT32_MIN,  INT32_C(  1198463137),              INT32_MAX,  INT32_C(   489783012), -INT32_C(   172994435), -INT32_C(   428479464), -INT32_C(  2103266839),  INT32_C(   991827510) } },
    { {  INT32_C(   708198188),  INT32_C(   976715050), -INT32_C(  1783293062), -INT32_C(   281772459),  INT32_C(     1300583),  INT32_C(  1338701723), -INT32_C(  2042454999), -INT32_C(  1881767325),
        -INT32_C(   340194111),  INT32_C(    69660809),  INT32_C(  1637473036), -INT32_C(  1051669158),  INT32_C(  1103193253), -INT32_C(  2054124709), -INT32_C(  1962159576),  INT32_C(   370860372) },
      {  INT32_C(  1996608493), -INT32_C(   797235260),  INT32_C(  1563497475), -INT32_C(  2028043807),  INT32_C(  1103683557), -INT32_C(  1815717781),  INT32_C(  2115948842), -INT32_C(  1600898893),
        -INT32_C(   820537845), -INT32_C(  1063284035), -INT32_C(  2011311962),  INT32_C(   923745106), -INT32_C(  2038900965),  INT32_C(  1494826543), -INT32_C(  1009305840),  INT32_C(  2053336175) },
      {              INT32_MAX,  INT32_C(   179479790), -INT32_C(   219795587),              INT32_MIN,  INT32_C(  1104984140), -INT32_C(   477016058),  INT32_C(    73493843),              INT32_MIN,
        -INT32_C(  1160731956), -INT32_C(   993623226), -INT32_C(   373838926), -INT32_C(   127924052), -INT32_C(   935707712), -INT32_C(   559298166),              INT32_MIN,              INT32_MAX } },
    { { -INT32_C(  1085637886), -INT32_C(  1266620147),  INT32_C(   205299130), -INT32_C(   196916263),  INT32_C(  1366998050),  INT32_C(   178951162),  INT32_C(   969769418), -INT32_C(   273468947),
        -INT32_C(  1179648596), -INT32_C(  1586679833), -INT32_C(  1515345460),  INT32_C(   379187444), -INT32_C(  1486416980),  INT32_C(  1890652582), -INT32_C(  2136375405),  INT32_C(  1550867888) },
      {  INT32_C(  1091903322),  INT32_C(   451052366),  INT32_C(   566267692),  INT32_C(   741824896),  INT32_C(   332636013),  INT32_C(  1132758448), -INT32_C(  1262211836), -INT32_C(   451922806),
        -INT32_C(  1591335597), -INT32_C(   709162584),  INT32_C(   418806680),  INT32_C(  1111829973),  INT32_C(  2102728908), -INT32_C(  1581196899), -INT32_C(  1856666617),  INT32_C(   192308664) },
      {  INT32_C(     6265436), -INT32_C(   815567781),  INT32_C(   771566822),  INT32_C(   544908633),  INT32_C(  1699634063),  INT32_C(  1311709610), -INT32_C(   292442418), -INT32_C(   725391753),
                     INT32_MIN,              INT32_MIN, -INT32_C(  1096538780),  INT32_C(  1491017417),  INT32_C(   616311928),  INT32_C(   309455683),              INT32_MIN,  INT32_C(  1743176552) } },
    { {  INT32_C(   866950539),  INT32_C(  1040738470), -INT32_C(  1202192669), -INT32_C(   117793748),  INT32_C(  1383420084),  INT32_C(   821245481),  INT32_C(  1942112698),  INT32_C(   964573358),
         INT32_C(  2070752213),  INT32_C(  1991931283), -INT32_C(  1607528076),  INT32_C(  1637362093), -INT32_C(  1548546439), -INT32_C(    19683516), -INT32_C(  1636723216), -INT32_C(  1546063666) },
      { -INT32_C(  1373682405),  INT32_C(   774166970), -INT32_C(  1748085782), -INT32_C(   151492739), -INT32_C(  1181111179),  INT32_C(  1136094547), -INT32_C(   790550270),  INT32_C(   880064793),
        -INT32_C(  1176333314),  INT32_C(  1457981036), -INT32_C(   672287142), -INT32_C(  1832000227), -INT32_C(   464820335), -INT32_C(   702086444),  INT32_C(  1151797547), -INT32_C(  1049093182) },
      { -INT32_C(   506731866),  INT32_C(  1814905440),              INT32_MIN, -INT32_C(   269286487),  INT32_C(   202308905),  INT32_C(  1957340028),  INT32_C(  1151562428),  INT32_C(  1844638151),
         INT32_C(   894418899),              INT32_MAX,              INT32_MIN, -INT32_C(   194638134), -INT32_C(  2013366774), -INT32_C(   721769960), -INT32_C(   484925669),              INT32_MIN } },
    { {  INT32_C(   444226222), -INT32_C(  1150262943),  INT32_C(   882007319), -INT32_C(   725196990), -INT32_C(  1699212858),  INT32_C(  1064427540), -INT32_C(  1417406231), -INT32_C(   512951245),
        -INT32_C(  1191451049),  INT32_C(  1601399624),  INT32_C(   177472968),  INT32_C(   735992420), -INT32_C(  2134534292),  INT32_C(  1623144055), -INT32_C(  2129968306), -INT32_C(  1755154368) },
      { -INT32_C(  1504748194), -INT32_C(  1878605112),  INT32_C(   731617991),  INT32_C(  1616280052), -INT32_C(  2015355888), -INT32_C(  1578655662),  INT32_C(   589493219), -INT32_C(   910523029),
        -INT32_C(  1418720798), -INT32_C(  1841596725),  INT32_C(    79615504),  INT32_C(  1617237072), -INT32_C(  2081929936), -INT32_C(   920334363),  INT32_C(   770459330), -INT32_C(  1359501365) },
      { -INT32_C(  1060521972),              INT32_MIN,  INT32_C(  1613625310),  INT32_C(   891083062),              INT32_MIN, -INT32_C(   514228122), -INT32_C(   827913012), -INT32_C(  1423474274),
                     INT32_MIN, -INT32_C(   240197101),  INT32_C(   257088472),              INT32_MAX,              INT32_MIN,  INT32_C(   702809692), -INT32_C(  1359508976),              INT32_MIN } },
    { {  INT32_C(  2086234032), -INT32_C(   317811491), -INT32_C(  1141781397),  INT32_C(   287004385), -INT32_C(  2121006180), -INT32_C(  1790265133), -INT32_C(   893175809), -INT32_C(  1904690466),
        -INT32_C(    32845535), -INT32_C(   773121690), -INT32_C(   963846939), -INT32_C(   807885005),  INT32_C(  2119199915),  INT32_C(   605264677), -INT32_C(  1326524718), -INT32_C(  1321310320) },
      { -INT32_C(  1632679624),  INT32_C(  1198496354), -INT32_C(  1441858697),  INT32_C(  1316611747),  INT32_C(  2009909586),  INT32_C(   916184932),  INT32_C(  1189513910),  INT32_C(   704128497),
        -INT32_C(   792156306), -INT32_C(  1189660862), -INT32_C(   698145485),  INT32_C(  1579473931),  INT32_C(   181792933), -INT32_C(  2059374129), -INT32_C(   322164741), -INT32_C(  1172978612) },
      {  INT32_C(   453554408),  INT32_C(   880684863),              INT32_MIN,  INT32_C(  1603616132), -INT32_C(   111096594), -INT32_C(   874080201),  INT32_C(   296338101), -INT32_C(  1200561969),
        -INT32_C(   825001841), -INT32_C(  1962782552), -INT32_C(  1661992424),  INT32_C(   771588926),              INT32_MAX, -INT32_C(  1454109452), -INT32_C(  1648689459),              INT32_MIN } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_x_mm512_adds_epi32(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_x_mm512_adds_epi32(a, b);

    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_adds_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_adds_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_adds_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_adds_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_adds_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_adds_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_adds_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_adds_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_adds_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_adds_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_adds_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_adds_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_adds_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_adds_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_mask_adds_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm256_maskz_adds_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_adds_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_adds_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_adds_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_adds_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_adds_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_adds_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_adds_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_adds_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_adds_epu8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_adds_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_adds_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_adds_epu16)
  SIMDE_TEST_FUNC_LIST_ENTRY(x_mm_adds_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(x_mm256_adds_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(x_mm512_adds_epi32)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
