package TreePath;
$TreePath::VERSION = '0.16';
use utf8;
use Moose;
with 'MooseX::Object::Pluggable';

use Moose::Util::TypeConstraints;
use Config::JFDI;
use Carp qw/croak/;
use Data::Dumper;

subtype MyConf => as 'HashRef';
coerce 'MyConf'
  => from 'Str' => via {
    my $conf = shift;
    my ($jfdi_h, $jfdi) = Config::JFDI->open($conf)
      or croak "Error (conf: $conf) : $!\n";
    return $jfdi->get;
  };

has conf => ( is => 'rw',
              isa => 'MyConf',
              coerce => 1,
              trigger  => sub {
                my $self = shift;
                my $args = shift;

                # if conf exist
                if ( defined $args->{$self->configword} ) {
                  croak "Error: Can not find " . $self->configword . " in your conf !"
                    if ( ! $args->{$self->configword});

                  $self->config($args->{$self->configword});

                  $self->debug($self->config->{'debug'})
                    if ( ! defined $self->debug && defined $self->config->{'debug'} );

                  $self->_load_backend if ! $self->can('backend');
                }
                # it's a hash
                else {
                  $self->tree($args);
                  $self->_build_tree;
                }
              }
            );



has config => (
               isa      => "HashRef",
               is       => "rw",
);

has 'configword' => (
                is       => 'rw',
                default => sub { __PACKAGE__ },
               );

has 'debug' => (
                is       => 'rw',
               );

has '_backend' => (
                is       => 'rw',
                isa      => 'Str',
               );

has _plugin_ns => (
                is       => 'rw',
                required => 1,
                isa      => 'Str',
                default  => sub{ 'Backend' },
                  );

has _source_field => (
                is       => 'rw',
                isa      => 'Str',
                default  => sub{ 'source' },
                  );

has _default_source => (
                is       => 'rw',
                isa      => 'Str',
                default  => sub{ 'Page' },
                  );

has _search_field => (
                is       => 'rw',
                isa      => 'Str',
                default  => sub{ 'name' },
                  );

has tree => (
                isa      => "HashRef",
                is       => "rw",
);


has root => (
                isa      => "HashRef",
                is       => "rw",
);



sub _load_backend {
  my $self = shift;
  my $backend = $self->config->{'backend'}->{name};
  $self->_backend($backend);

  $self->_log("Loading $backend backend ...");
  $self->load_plugin( $backend );
  $self->_load_tree;
  $self->_build_tree;
}

sub reload {
    my $self = shift;

    $self->_populate_backend(0)
        if $self->can('_populate_backend');

    $self->_load_tree;
}

sub _log{
  my ($self, $msg ) = @_;

  return if ! $self->debug;

  say STDERR "[debug] $msg";
}

# Load tree from backend
sub _load_tree {
  my $self = shift;

  $self->tree($self->_load);
}

# Build Tree
# build root and relationships (just belongs_to)
sub _build_tree {
  my $self = shift;

  my $tree = $self->tree;

  foreach my $id ( sort { $a cmp $b } keys %$tree ) {


      my $node = $self->tree->{$id};

      # transform obj_id to object reference
      foreach my $col ( keys %$node ) {

          # hash = belongs_to
          if ( ref($node->{$col}) eq 'HASH') {

              my @values = keys %{$node->{$col}};
              my $value = $values[0];

              if ( $value =~ m/^\w*_\d*$/ ) {
                  $node->{$col} = $self->tree->{$value};
              }
              else {
                  die "Error: Relationship belongs_to is not well formatted : $value, use Obj_ID as form"
              }
          }
          # array = has_may
          elsif ( ref($node->{$col}) eq 'ARRAY') {
              my $newarray = [];
              foreach my $child (@{$node->{$col}}) {
                  if ( $child =~ m/^\w*_\d*$/ ) {
                      push(@$newarray, $self->tree->{$child});
                  }
                  elsif ( ref($child) eq 'HASH') {
                      push(@$newarray, $child);
                  }
                  else {
                      die "Error : has_many relationship $col (=$child) is not formatted, ex: 'Page_3'\n";
                  }
              }
              $node->{$col} = $newarray;
          }
      }

      my $parent_field = $self->_parent_field($node);
      $self->add($node, $node->{$parent_field});
  }
}

sub _parent_field{
    my $self = shift;
    my $node = shift;

    my $source;
    if ( ref($node)) {
        $source = $node->{$self->_source_field};
    }
    else {
        $source = $node;
    }
    my $parent_field;
    if ( defined $self->config && defined $self->config->{backend}->{args}->{sources_name}->{$source}->{parent_field}) {
        $parent_field = $self->config->{backend}->{args}->{sources_name}->{$source}->{parent_field};
    }
    else {
        $parent_field = 'parent';
    }
    return $parent_field;
}
# sub  _bckd_create {
#     my $self = shift;
#     my $node = shift;
#     my $msg  = shift;

#     return if ( ! $self->_backend || ! $self->_sync );
#     $self->_log("[" . $self->_backend . "] CREATE " . $node->{name} . " | $msg");
#     $self->_create($node);
# }

# sub  _bckd_delete {
#     my $self  = shift;
#     my $nodes = shift;
#     my $msg   = shift;

#     return if ( ! $self->_backend || ! $self->_sync );
#     my @nodes_name = map { $_->{$self->_search_field} } @$nodes;
#     $self->_log("[" . $self->_backend . "] DELETE @nodes_name | $msg");

#     $self->_delete($nodes);
# }

# sub  _bckd_update {
#     my $self = shift;
#     my $node = shift;
#     my $msg  = shift;

#     return if ( ! $self->_backend || ! $self->_sync );
#     $self->_log("[" . $self->_backend . "] UPDATE " . $node->{name} . " | $msg");
#     $self->_update($node);
# }




# removes the child's father
sub _remove_child_father{
    my $self = shift;
    my $node = shift;

    my $father = $node->{parent};
    my $key_children = $self->_key_children($node, $father);
    my $id = 0;
    foreach my $child ( @{$father->{$key_children}}) {
        if ( $child->{$self->_search_field} eq $node->{$self->_search_field} &&
                 $child->{parent} eq $node->{parent} ){
            return splice ( @{$father->{$key_children}},$id,1);
        }
        $id++;
    }
}


# sub _clone_node {
#     my $self = shift;
#     my $node = shift;

#     my $clone = {};
#     foreach my $k (keys %$node) {
#         if ( $k eq 'parent'){
#             if ( $node->{'parent'}) {
#                 $clone->{$self->_parent_field} = $node->{$k}->{id};
#             }
#             # is root
#             else {
#                 $clone->{$self->_parent_field} = 0;
#             }
#         }
#         else { $clone->{$k} = $node->{$k} }
#     }
#     return $clone;
# }


sub _key_obj {
  my $self = shift;
  my $node = shift;

  return $node->{source} . '_' . $node->{id};
}


# return key name of children parent
sub _key_children {
    my ($self, $node, $parent) = @_;

    my $key_children;
    #if parent have a source
    if ( defined $parent->{$self->_source_field}) {

        # if node have a source
        if ( defined $node->{$self->_source_field}) {

            # if node and parent have the same source
            if ( $node->{$self->_source_field} eq $parent->{$self->_source_field} ) {
                $key_children = 'children';
            }
            else {
                $key_children = 'children_' . $node->{$self->_source_field};
            }
        }
        # else node have the same source as parent
        else {
            $node->{$self->_source_field} = $parent->{$self->_source_field};
            $key_children = 'children';
        }
    }
    # parent haven't source
    else {
        if ( defined $node->{$self->_source_field}) {
            die "node " . $node->{$self->_search_field} . " [id:". $node->{id} . "] have a source but not the parent [id:" . $parent->{id} . "]!";
        }
        else {
            $key_children = 'children';
        }
    }
    return $key_children;
}

sub search {
  my ( $self, $args ) = @_;

  my $results = [];
  my $tree = $self->tree;
  foreach my $id  ( sort {$a cmp $b} keys %$tree ) {

      my $found = 1;

      # next search source field and it's not same source'
      next if ( $args->{$self->_source_field}
                    && $tree->{$id}->{$self->_source_field} ne $args->{$self->_source_field}
                );

    foreach my $key ( keys %$args ) {
      my $current;
      if ( $key =~ m/(.*)\.(.*)/) {
        # ex: parent.name
        if ( defined $tree->{$id}->{$1} && ref($tree->{$id}->{$1})) {
          $current = $tree->{$id}->{$1}->{$2};
        }
        else { next }
      }
      else {
          if ( defined $tree->{$id}->{$key} ){
              $current = $tree->{$id}->{$key};
          }
          else {
              die "'$key' is not a key hash [node obj_key:$id]";
          }
      }
      my $value = $args->{$key};
      if ( $current ne $value ) {
        $found = 0;
        last;
      }
    }

    if ( $found ){
      if ( wantarray) {
        push(@$results, $tree->{$id});
      }
      # if found and scalar context
      else {
        return $tree->{$id};
      }
    }
  }

  return 0 if (  ! wantarray && ! $$results[0] );

  # wantarray
  return @$results;
}


# ex : search_path(/A/B/C')
#      or search_path(/A/B/C, { by => 'name', source => 'Page'} )
sub search_path {
  my ( $self, $path, $opts ) = @_;


  # search by 'name' if not defined
  $opts->{by} = $self->_search_field if ! defined $opts->{by};

  # source => 'Page' if not defined
  $opts->{$self->_source_field} = $self->_default_source if ! defined $opts->{$self->_source_field};


  croak "path must be start by '/' !: $!\n" if ( $path !~ m|^/| );

  my $nodes = [ split m%/%, $path ];
  $$nodes[0] = '/';

  my (@found, @not_found);
  my $parent = '/';
  foreach my $node ( @$nodes ) {

    my $args = { $opts->{by} => $node, "parent\.$opts->{by}" => $parent, source => $opts->{source}};
    my $result = $self->search( $args );

    $parent = $result->{$opts->{by}} if $result;

    if ( $result ) {
      push(@found, $result);
    }
    else {

      push(@not_found, $node);
    }
  }

  if ( wantarray ) {
    return ( \@found, \@not_found );
  }
  else {
    if ( $not_found[-1] ) {
      return '';
    }
    else {
      return $found[-1];
    }
  }
}


sub count {
  my $self = shift;

  return scalar keys %{$self->tree};
}

sub dump {
  my $self = shift;
  my $var  = shift;

  $var = $self->tree if ! defined $var;
  $Data::Dumper::Maxdepth = 3;
  $Data::Dumper::Sortkeys = 1;
  $Data::Dumper::Terse = 1;
  return Dumper($var);
}

sub traverse {
  my ($self, $node, $funcref, $args) = @_;

  return 0 if ( ! $node );
  $args ||= {};
  $args->{_count} = 1 if ! defined ($args->{_count});

  my $nofunc = 0;
  if ( ! $funcref ) {
    $nofunc = 1;
    $funcref = sub {    my ($node, $args) = @_;
                        $args->{_each_nodes} = []
                          if ( ! defined $args->{_each_nodes});
                        if(defined($node)) {
                          push(@{$args->{_each_nodes}}, $node);
                          return 1;
                        }
                      }
  }
  # if first node
  if ( $args->{_count} == 1 ) {
    return 0 if ( ! &$funcref( $node, $args ) )
  }

  foreach my $field ( keys %$node) {
      if ( ref($node->{$field}) eq 'ARRAY' ) {
        foreach my $child ( @{$node->{$field}} ) {
          return 0 if ( ! &$funcref( $child, $args ) );
          $args->{_count}++;
          $self->traverse( $child, $funcref, $args );
        }
      }
  }

  return $args->{_each_nodes} if $nofunc;
  return 1;
}


sub del {
  my ($self, @nodes) = @_;

  my @deleted;
  foreach my $node ( @nodes ) {

      my $father = $node->{parent};

      $self->_remove_child_father($node);

      my $key_children = $self->_key_children($node, $node->{parent});

      # traverse child branches and delete it
      my $nodes = $self->traverse($node);

      push(@deleted,map { delete $self->tree->{$self->_key_obj($_)} } @$nodes);
      #$self->_bckd_delete($nodes, "delete " . @$nodes . " node(s)");
  }
  return @deleted;
}

# Inserts a node beneath the parent
sub add {
  my ($self, $node, $parent) = @_;


  if ( ! $parent ){
      if ( $self->root) {
          die "root already exist [ node: " .$node->{$self->_source_field}. '_' . $node->{id} . " ]!";
      }
      $self->root($node);
  }

  $node->{parent} = $parent;

  # save ref node in tree
  $self->tree->{$self->_key_obj($node)} = $node;

  my $key_children;
  my $update_children = 0;

  # node != root
  if ( defined $parent && $parent ) {
      $key_children = $self->_key_children($node, $parent);

      if ( $parent ) {
          push(@{$parent->{$key_children}}, $node);
      }
  }
  # add node in backend
  #$self->_bckd_create($node, 'add node ' . $node->{$self->_search_field})
  #    if ! $is_exist_in_backend;

  return $self->root if ! $parent;

  return $node;
}

sub update {
  my ($self, $node, $datas) = @_;

  foreach my $k ( sort keys %$datas ) {

      if ( ! defined $node->{$k} ){
              $self->_log("update: can not update, node->{$k} is not defined but _force_create_key is not equal 1");
              next;
      }
      elsif ( $node->{$k} eq $datas->{$k} ){next}

      my $previous = $node->{$k};
      my $parent   = $node->{parent};
      my $key_children = $self->_key_children($node, $parent);
      my $children = $parent->{$key_children};


      if ( $k eq 'parent') {

          my $old = $self->_remove_child_father($node);
          my $new_parent = $datas->{$k};
          push(@{$new_parent->{$key_children}}, $old);

          $node->{parent} = $new_parent;

          my $msg = 'updating parent of '. $node->{$self->_search_field} .
              " ( " . $old->{name} . " -> ". $node->{parent}->{name} .")";

          #$self->_bckd_update($node, $msg);
      }
      else {
          $node->{$k} = $datas->{$k};
      }
  }
  return $node;
}


sub move {
    my ($self, $node, $parent) = @_;

    return $self->update($node, { parent => $parent });
}

=head1 NAME

TreePath - Simple Tree Path!

=head1 VERSION


=head1 SYNOPSIS

 use TreePath;

 my $tp = TreePath->new(  conf  => $conf  );
 my $tree = $tp->tree;

 # All nodes are hash
 # The first is called 'root'
 my $root = $tp->root;

 # a node can have children
 my $children = $root->{children};

 # or children from another source
 my $children = $root->{children_Comment};

=head1 SUBROUTINES/METHODS

=head2 new($method => $value)

 # for now there are two backends : DBIX and File
 $tp = TreePath->new( conf => 't/conf/treefromdbix.yml')

 # see t/conf/treepath.yml for hash structure
 $tp = TreePath->new( conf => $datas);

 also see t/01-tpath.t

=cut

=head2 tree

 $tree = $tp->tree;

=cut

=head2 reload

 # reload tree from backend
 $tree = $tp->reload;

=cut

=head2 nodes

 $root = $tp->root;
 # $root and $tree->{1} are the same node

 This is the root node ( a simple hashref )
 it has no parent.
     {
       'Page_1' => {
                'id' => '1',
                'name' => '/',
                'parent' => '0'
                'source' => 'Page'
              }
     }

  $A = $tp->search( { name => 'A', source => 'Page' } );
  See the dump :

    {
      'children' => [
                      {
                        'children' => 'ARRAY(0x293ce00)',
                        'id' => '3',
                        'name' => 'B',
                        'parent' => $VAR1
                        'source' => 'Page'
                     },
                      {
                        'children' => 'ARRAY(0x2fd69b0)',
                        'id' => '7',
                        'name' => 'F',
                        'parent' => $VAR1
                        'source' => 'Page'
                      }
                    ],
      'id' => '2',
      'name' => 'A',
      'parent' => {
                    'children' => [
                                    $VAR1
                                  ],
                    'id' => '1',
                    'name' => '/',
                    'parent' => '0',
                    'source' => 'Page'
                  }
     'source' => 'Page'
    }

    => 'parent' is a reference on root node and 'children' is an array containing 2 nodes

    => if nodes have 'source', it's possible to 'link' some 'source' of node on some others sources

       For example, A:(source1), B:(source1, parent:A), C:(source2, parent:A) becomes :
       A:(source1, children:[B], children_source2[C])

=cut

=head2 search (hashref)

 # in scalar context return the first result
 my $E = $tp->search( { name => 'E', source => 'Page' } );

 # return all result in array context
 my @allE = $tp->search( { name => 'E', source => 'Page' } );

 # It is also possible to specify a particular field of a hash
 my $B = $tp->search( { name => 'B', source => 'Page', 'parent.name' => 'A'} );

=cut

=head2 search_path (PATH)

 # Search a path in a tree
 # in scalar context return last node
 my $c = $tp->search_path('/A/B/C');

 # in array context return found and not_found nodes
 my ($found, $not_found) = $tp->search_path('/A/B/X/D/E');

=cut

=head2 dump

 # dump whole tree
 print $tp->dump;

 # dump a node
 print $tp->dump($c);;

=cut

=head2 count

 # return the number of nodes
 print $tp->count;

=cut

=head2 traverse ($node, [\&function], [$args])

 # return an arrayref of nodes
 my $nodes = $tp->traverse($node);

 # or use a function on each nodes
 $tp->traverse($node, \&function, $args);

=cut

=head2 del ($node)

 # delete recursively all children and node
 $deleted = $tp->del($node);

 # delete several nodes at once
 @del = $tp->del($n1, $n2, ...);

=cut

=head2 add ($node, $parent)

 # add the root
 $root = $tp->add({ name => '/', source => 'Node' }, 0);

 # add a node beneath the parent.
 $Z = $tp->add({ name => 'Z', source => 'Node' }, $parent);

=cut

=head2 update ($node, $datas)

 # update node with somes datas
 $Z = $tp->update($node, { name => 'new_name' });


=cut


=head2 move ($node, $parent)

 # move a node as child of given parent
 $Z = $tp->move($Z, $X);

=cut

=head2 insert_before ($sibling, $node)

 # Inserts a node beneath the parent before the given sibling.
 $Y = $tp->insert_before($Z, { name => 'Y' });

=cut



=head1 AUTHOR

Daniel Brosseau, C<< <dab at catapulse.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-tpath at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=TreePath>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc TreePath


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=TreePath>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/TreePath>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/TreePath>

=item * Search CPAN

L<http://search.cpan.org/dist/TreePath/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2014 Daniel Brosseau.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of TreePath
