﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/m2/MainframeModernization_EXPORTS.h>
#include <aws/m2/model/JobIdentifier.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace MainframeModernization {
namespace Model {

/**
 * <p>A batch job identifier in which the batch jobs to run are identified by an
 * Amazon S3 location.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/m2-2021-04-28/S3BatchJobIdentifier">AWS
 * API Reference</a></p>
 */
class S3BatchJobIdentifier {
 public:
  AWS_MAINFRAMEMODERNIZATION_API S3BatchJobIdentifier() = default;
  AWS_MAINFRAMEMODERNIZATION_API S3BatchJobIdentifier(Aws::Utils::Json::JsonView jsonValue);
  AWS_MAINFRAMEMODERNIZATION_API S3BatchJobIdentifier& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_MAINFRAMEMODERNIZATION_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon S3 bucket that contains the batch job definitions.</p>
   */
  inline const Aws::String& GetBucket() const { return m_bucket; }
  inline bool BucketHasBeenSet() const { return m_bucketHasBeenSet; }
  template <typename BucketT = Aws::String>
  void SetBucket(BucketT&& value) {
    m_bucketHasBeenSet = true;
    m_bucket = std::forward<BucketT>(value);
  }
  template <typename BucketT = Aws::String>
  S3BatchJobIdentifier& WithBucket(BucketT&& value) {
    SetBucket(std::forward<BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Identifies the batch job definition. This identifier can also point to any
   * batch job definition that already exists in the application or to one of the
   * batch job definitions within the directory that is specified in
   * <code>keyPrefix</code>.</p>
   */
  inline const JobIdentifier& GetIdentifier() const { return m_identifier; }
  inline bool IdentifierHasBeenSet() const { return m_identifierHasBeenSet; }
  template <typename IdentifierT = JobIdentifier>
  void SetIdentifier(IdentifierT&& value) {
    m_identifierHasBeenSet = true;
    m_identifier = std::forward<IdentifierT>(value);
  }
  template <typename IdentifierT = JobIdentifier>
  S3BatchJobIdentifier& WithIdentifier(IdentifierT&& value) {
    SetIdentifier(std::forward<IdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key prefix that specifies the path to the folder in the S3 bucket that
   * has the batch job definitions.</p>
   */
  inline const Aws::String& GetKeyPrefix() const { return m_keyPrefix; }
  inline bool KeyPrefixHasBeenSet() const { return m_keyPrefixHasBeenSet; }
  template <typename KeyPrefixT = Aws::String>
  void SetKeyPrefix(KeyPrefixT&& value) {
    m_keyPrefixHasBeenSet = true;
    m_keyPrefix = std::forward<KeyPrefixT>(value);
  }
  template <typename KeyPrefixT = Aws::String>
  S3BatchJobIdentifier& WithKeyPrefix(KeyPrefixT&& value) {
    SetKeyPrefix(std::forward<KeyPrefixT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_bucket;

  JobIdentifier m_identifier;

  Aws::String m_keyPrefix;
  bool m_bucketHasBeenSet = false;
  bool m_identifierHasBeenSet = false;
  bool m_keyPrefixHasBeenSet = false;
};

}  // namespace Model
}  // namespace MainframeModernization
}  // namespace Aws
