# frozen_string_literal: true

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class ResponsibilityClassifier
        RESPONSIBLE = 'responsible'
        DEPENDENT = 'dependent'

        # @param test_to_sources [Hash<String, Array<String>>] Test files mapped to source files they cover
        # @param responsible_patterns [Array<Regexp>] Patterns for unit tests
        # @param dependent_patterns [Array<Regexp>] Patterns for integration/E2E tests
        def initialize(test_to_sources, responsible_patterns:, dependent_patterns:)
          @test_to_sources = test_to_sources
          @responsible_patterns = responsible_patterns
          @dependent_patterns = dependent_patterns
        end

        # Classifies each test file as responsible or dependent
        # @return [Hash<String, String>] Test file path => classification
        def classify_tests
          @test_to_sources.keys.each_with_object({}) do |test_file, result|
            result[test_file] = classify_test(test_file)
          end
        end

        private

        # Classifies a test file as responsible (unit) or dependent (integration/E2E).
        #
        # Dependent patterns are checked first because it's the safer default:
        # - is_responsible: true claims "this file has unit test coverage"
        # - is_dependent: true claims "this file has integration test coverage"
        #
        # If uncertain (overlapping patterns or no match), we default to dependent
        # to avoid incorrectly inflating unit test coverage metrics.
        def classify_test(test_file)
          return DEPENDENT if @dependent_patterns.any? { |p| test_file.match?(p) }
          return RESPONSIBLE if @responsible_patterns.any? { |p| test_file.match?(p) }

          # Default to dependent for unknown test types
          DEPENDENT
        end
      end
    end
  end
end
