# frozen_string_literal: true

require 'net/http'
require 'yaml'

require_relative 'utils'

module GitlabQuality
  module TestTooling
    module CodeCoverage
      class CategoryOwners
        include Utils

        SOURCE_URL  = URI('https://gitlab.com/gitlab-com/www-gitlab-com/raw/master/data/stages.yml')
        BASE_DELAY  = 1 # seconds
        MAX_RETRIES = 3

        # @return [Hash] Feature category ownership hierarchy, section -> stage -> group -> [feature_categories]
        # @example Return value
        #   {
        #     "team_planning" => {             # section
        #       "project_management" => {      # stage
        #         "plan" => [                  # group
        #           "dev",                     # feature_category
        #           "service_desk"             # feature_category
        #         ],
        #         "product_planning" => [      # group
        #           "portfolio_management",    # feature_category
        #           ...
        #         ]
        #       }
        #     },
        #     ...
        #   }
        attr_reader :hierarchy

        def initialize
          @feature_categories_map = {}
          @hierarchy = {}

          yaml_file = fetch_yaml_file
          yaml_content = YAML.load(yaml_file)
          populate_ownership_hierarchy(yaml_content)
        end

        # @return [Array<Hash>] Flattened feature category ownership
        # @example Return value
        #   [
        #     { feature_category: "team_planning", group: "project_management", stage: "plan", section: "dev" },
        #     { feature_category: "service_desk", group: "project_management", stage: "plan", section: "dev" },
        #     { feature_category: "portfolio_management", group: "product_planning", stage: "plan", section: "dev" }
        #     ...
        #   ]
        def as_db_table
          hierarchy.each_with_object([]) do |(section, stages), flattened_hierarchy|
            next unless stages

            stages.each do |stage, groups|
              next unless groups

              groups.each do |group, feature_categories|
                Array(feature_categories).each do |feature_category|
                  flattened_hierarchy << {
                    feature_category: feature_category,
                    group: group,
                    stage: stage,
                    section: section
                  }
                end
              end
            end
          end
        end

        # @return [Hash] Mapping of feature categories to teams (i.e., groups, stages, sections)
        # @example Return value
        #   {
        #     "team_planning" => { group: "project_management", stage: "plan", section: "dev" },
        #     "service_desk" => { group: "project_management", stage: "plan", section: "dev" },
        #     "portfolio_management" => { group: "product_planning", stage: "plan", section: "dev" },
        #     ...
        #   }
        def feature_categories_to_teams
          populate_feature_categories_map(hierarchy)
          @feature_categories_map
        end

        private

        def fetch_yaml_file
          retries = 0

          begin
            response = Net::HTTP.get_response(SOURCE_URL)
            raise "Failed to fetch file: HTTP #{response.code}" unless response.is_a?(Net::HTTPSuccess)

            response.body
          rescue Net::OpenTimeout, Net::ReadTimeout, Errno::ECONNREFUSED, Errno::ETIMEDOUT, IOError, RuntimeError => e
            retries += 1
            raise "Failed to fetch YAML after #{MAX_RETRIES} retries: #{e.message}" if retries >= MAX_RETRIES

            delay = exponential_delay_with_jitter(retries)
            warn "Fetch attempt #{retries} failed: #{e.message}. Retrying in #{delay.round(2)}s..."
            sleep(delay)
            retry
          end
        end

        def populate_ownership_hierarchy(data)
          stages = data['stages'] || {}

          stages.each do |stage, stage_data|
            next unless stage_data.is_a?(Hash)

            section = stage_data['section']
            groups = stage_data['groups'] || {}
            next unless section

            groups.each { |group, group_data| add_hierarchy_entry(section, stage, group, group_data['categories']) }
          end
        end

        def add_hierarchy_entry(section, stage, group, categories)
          @hierarchy[section] ||= {}
          @hierarchy[section][stage] ||= {}
          @hierarchy[section][stage][group] = categories || []
        end

        def populate_feature_categories_map(data, current_section = nil, current_stage = nil, current_group = nil)
          case data
          when Hash # Sections / Stages / Groups
            data.each do |key, value|
              if current_section.nil? # Sections
                populate_feature_categories_map(value, key, nil, nil)
              elsif current_stage.nil? # Stages
                populate_feature_categories_map(value, current_section, key, nil)
              elsif current_group.nil? # Groups
                populate_feature_categories_map(value, current_section, current_stage, key)
              else                     # Feature categories
                populate_feature_categories_map(value, current_section, current_stage, current_group)
              end
            end
          when Array # Feature categories array
            data.each do |feature_category|
              next unless feature_category.is_a?(String)

              @feature_categories_map[feature_category] = {
                section: current_section,
                stage: current_stage,
                group: current_group
              }
            end
          end
        end
      end
    end
  end
end
